/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 *
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */


package org.netbeans.modules.j2ee.verification;

import java.lang.reflect.Modifier;
import org.netbeans.editor.TokenID;
import org.netbeans.editor.TokenItem;
import org.netbeans.editor.ext.java.JavaSyntaxSupport;
import org.netbeans.editor.ext.java.JavaTokenContext;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.editor.BaseDocument;
import org.netbeans.editor.Utilities;
import org.netbeans.modules.j2ee.verification.persistence.PersistenceAPIAnnotations;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.openide.ErrorManager;
import org.openide.text.Line;

import javax.swing.text.BadLocationException;
import java.util.*;
import org.netbeans.modules.db.api.sql.SQLKeywords;
import org.netbeans.modules.j2ee.persistence.dd.JavaPersistenceQLKeywords;
import org.openide.text.PositionBounds;

/**
 * Generic utility methods useful for finding problems within the persistence API
 */
public class ProblemFindingUtils{
    
    static{
        ProblemFindersRegistry.intit();
    }
    
    public static Annotation findAnnotation(AnnotableElement element, String annotationName){
        assert element != null && annotationName != null;
        return findFirstAnnotationFromGivenSet(element, Collections.singleton(annotationName));
    }
    
    public static Annotation findFirstAnnotationFromGivenSet(AnnotableElement element, Set annotationNames){
        assert annotationNames != null;
        Annotation annots[] = (Annotation[]) element.getAnnotations().toArray(new Annotation[]{});
        
        for (Annotation ann : annots) {
            AnnotationType atype = ann.getType();
            // atype might be null in case of unresolved annotation class
            if (atype != null && annotationNames.contains(atype.getName())){
                return ann;
            } else if (atype == null){
                JEEVerificationAnnotationProvider.tmpDbg("findFirstAnnotationFromGivenSet(): unresolved annotation type: " //NOI18N
                        + ann.getTypeName().getName());
            }
        }
        
        return null;
    }
    
    public static Line.Part getLinePartForElement(Line.Set lineSet, BaseDocument baseDoc, Element elem){
        JavaMetamodel jmm = JavaMetamodel.getManager();
        PositionBounds pos = jmm.getElementPosition(elem);
        
        if (elem instanceof JavaClass || elem instanceof Field || elem instanceof Method){
            pos = jmm.getElementPartPosition(elem, ElementPartKindEnum.NAME, 0);
        }
        
        Line.Part ret = null;
        
        try {
            Line startLine = lineSet.getCurrent(pos.getBegin().getLine());
            int startColumn = pos.getBegin().getColumn();
            int endColumn = pos.getEnd().getColumn();
            
            // handle a line break, currently editor insfrastructure doesn't support
            // error annotations spanned across multiple lines
            if (lineSet.getCurrent(pos.getEnd().getLine()).getLineNumber() > startLine.getLineNumber()){
                endColumn = startLine.getText().length();
            }
            
            ret = startLine.createPart(startColumn, endColumn - startColumn);
            
        } catch (Exception ex) {
            ErrorManager.getDefault().notify(ErrorManager.EXCEPTION, ex);
        }
        
        return ret;
    }
    
    public static Line.Part getLinePartForModifier(Line.Set lineSet, BaseDocument baseDoc, Element elem, int modifier){
        TokenID searchedTokenId;
        
        switch (modifier){
            case Modifier.PUBLIC:
                searchedTokenId = JavaTokenContext.PUBLIC;
                break;
            case Modifier.PROTECTED:
                searchedTokenId = JavaTokenContext.PROTECTED;
                break;
            case Modifier.PRIVATE:
                searchedTokenId = JavaTokenContext.PRIVATE;
                break;
            case Modifier.STATIC:
                searchedTokenId = JavaTokenContext.STATIC;
                break;
            case Modifier.FINAL:
                searchedTokenId = JavaTokenContext.FINAL;
                break;
                // ...
                
            default:
                searchedTokenId = null;
                break;
        }
        
        if (searchedTokenId == null){
            throw new IllegalArgumentException("Unsupported modifier constant"); ///NOI18N
        }
        
        int headerStart = elem.getPartStartOffset(ElementPartKindEnum.HEADER);
        int headerEnd = elem.getPartEndOffset(ElementPartKindEnum.HEADER);
        try {
            TokenItem tokenItm = ((JavaSyntaxSupport)baseDoc.getSyntaxSupport().get(JavaSyntaxSupport.class)).getTokenChain(headerStart, headerEnd);
            
            for (; tokenItm != null && tokenItm.getOffset() <= headerEnd; tokenItm = tokenItm.getNext()){
                if (searchedTokenId == tokenItm.getTokenID()){
                    int startOffset = tokenItm.getOffset();
                    int endOffset = startOffset + tokenItm.getImage().length();
                    
                    int startLineNumber = Utilities.getLineOffset(baseDoc, startOffset);
                    int startLineOffset = Utilities.getRowStartFromLineOffset(baseDoc, startLineNumber);
                    Line startLine = lineSet.getCurrent(startLineNumber);
                    Line.Part linePart = startLine.createPart(startOffset - startLineOffset, endOffset - startOffset);
                    
                    return linePart;
                }
            }
        } catch (BadLocationException ex) {
            ErrorManager.getDefault().notify(ErrorManager.ERROR, ex);
        }
        
        return null;
    }
    
    /**
     * Utility method which behaves same way as
     * {@link java.lang.Class#isAssignableFrom}.
     *
     * @param thisClass name of the JavaClass which corresponds to implicit this
     * object in Class.isAssignableFrom method call.
     * @param from JavaClass to be checked
     * @return the boolean value indicating whether objects of the type from
     * can be assigned to objects of type thisClass
     */
    public static boolean isAssignableFrom(String thisClass, JavaClass from) {
        Set<String> visited = new HashSet<String>();
        return isAssignableFrom0(thisClass, from, visited);
    }
    
    // intrenal method used by isAssignableFrom to avoid infinite recursion
    private static boolean isAssignableFrom0(String thisClass,
            JavaClass from,
            Set<String> visited) {
        if(from instanceof ParameterizedType) {
            // this turns java.util.Collection<java.lang.Object> to java.lang.Collection
            from = ParameterizedType.class.cast(from).getDefinition();
        }
        if(!visited.add(from.getName())) {
            return false; // already checked this class, so avoid infinite recursion
        }
        
        if(thisClass.equals(from.getName())) return true;
        
        // search interface list
        for (JavaClass i : (List<JavaClass>)from.getInterfaces()) {
            // recursively search the interface chain
            if(isAssignableFrom0(thisClass, i, visited)) {
                return true;
            }
        }
        
        // now recursively search the hierarchy
        final JavaClass superClass = from.getSuperClass();
        if(superClass != null) { // java.lang.Object has null superclass
            return isAssignableFrom0(thisClass, from.getSuperClass(), visited);
        }
        return false;
    }
    
    public static String getPrimaryTableName(JavaClass entityClass) {
        String tableName = entityClass.getSimpleName();
        
        // TODO: Use Persistence model (Entity.getName())
        Annotation annTable = ProblemFindingUtils.findAnnotation(entityClass,
                PersistenceAPIAnnotations.TABLE);
        
        String nameDefinedByTableAnn = getNameAttribute(annTable);
        
        if (nameDefinedByTableAnn != null){
            tableName = nameDefinedByTableAnn;
        }else{
            Annotation annEntity = ProblemFindingUtils.findAnnotation(entityClass,
                    PersistenceAPIAnnotations.ENTITY);
            
            String nameDefinedByEntityAnn = getNameAttribute(annEntity);
            tableName = nameDefinedByEntityAnn == null ? tableName : nameDefinedByEntityAnn;
        }
        tableName = nameDefinedByTableAnn == null ? tableName : nameDefinedByTableAnn;
        return tableName;
    }
    
    public static String getColumnName(Field field) {
        String columnName = field.getName();
        
        //TODO: Use Persistence Model instead
        Annotation annColumn = ProblemFindingUtils.findAnnotation(field,
                PersistenceAPIAnnotations.COLUMN);
        
        if (annColumn == null){
            annColumn = ProblemFindingUtils.findAnnotation(field,
                    PersistenceAPIAnnotations.JOIN_COLUMN);
        }
        
        if (annColumn != null){
            String nameDefinedByColumnAttr = getNameAttribute(annColumn);
            
            if (nameDefinedByColumnAttr != null){
                columnName = nameDefinedByColumnAttr;
            }
        }
        
        return columnName;
    }
    
    public static String getColumnName(Method persistentFieldAccessor) {
        String columnName = persistentFieldAccessor.getName().substring("get".length()); //NOI18N
        
        //TODO: Use Persistence Model instead
        Annotation annColumn = ProblemFindingUtils.findAnnotation(persistentFieldAccessor,
                PersistenceAPIAnnotations.COLUMN);
        
        if (annColumn == null){
            annColumn = ProblemFindingUtils.findAnnotation(persistentFieldAccessor,
                    PersistenceAPIAnnotations.JOIN_COLUMN);
        }
        
        if (annColumn != null){
            String nameDefinedByColumnAttr = getNameAttribute(annColumn);
            
            if (nameDefinedByColumnAttr != null){
                columnName = nameDefinedByColumnAttr;
            }
        }
        return columnName;
    }
    
    private static String getNameAttribute(Annotation ann){
        if (ann != null){
            for (AttributeValue av : (List<AttributeValue>)ann.getAttributeValues()){
                if ("name".equals(av.getName())){ //NOI18N
                    
                    if (av.getValue() instanceof StringLiteral){
                        return ((StringLiteral)av.getValue()).getValue();
                    }
                }
            }
        }
        
        return null;
    }
    
    public static boolean isSQLAndNotPersistenceQLKeyword(String text){
        return SQLKeywords.isSQL99Keyword(text) && !JavaPersistenceQLKeywords.isKeyword(text);
    }
}
