/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.j2ee.verification;

import javax.swing.text.Document;
import org.netbeans.modules.j2ee.persistence.api.PersistenceScope;
import org.netbeans.modules.j2ee.persistence.api.PersistenceScopes;
import org.netbeans.modules.j2ee.persistence.dd.PersistenceMetadata;
import org.netbeans.modules.j2ee.persistence.dd.PersistenceUtils;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.*;
import java.util.concurrent.atomic.AtomicBoolean;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.api.mdr.events.AttributeEvent;
import org.netbeans.api.mdr.events.MDRChangeEvent;
import org.netbeans.api.mdr.events.MDRChangeListener;
import org.netbeans.api.mdr.events.MDRChangeSource;
import org.netbeans.api.project.FileOwnerQuery;
import org.netbeans.api.project.Project;
import org.netbeans.editor.BaseDocument;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.modules.editor.NbEditorUtilities;
import org.netbeans.modules.editor.java.JMIUtils;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.spi.java.classpath.ClassPathProvider;
import org.openide.ErrorManager;
import org.openide.cookies.EditorCookie;
import org.openide.cookies.EditorCookie.Observable;
import org.openide.cookies.LineCookie;
import org.openide.filesystems.FileAttributeEvent;
import org.openide.filesystems.FileChangeListener;
import org.openide.filesystems.FileEvent;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileRenameEvent;
import org.openide.loaders.DataObject;
import org.openide.text.AnnotationProvider;
import org.openide.text.CloneableEditorSupport;
import org.openide.text.Line;
import org.openide.util.Lookup;
import org.openide.util.RequestProcessor;
import org.netbeans.modules.j2ee.persistence.dd.orm.model_1_0.EntityMappings;
import org.netbeans.modules.j2ee.verification.ejb.EJBApiHelper;
import static org.netbeans.modules.j2ee.verification.persistence.PersistenceAPIHelper.*;
import static org.netbeans.modules.j2ee.verification.ejb.EJBApiHelper.*;

/**
 * Class' responsibilieties:
 * - listening for opening file for the first time (using the AnnotationProvider mechanism)
 * - listening for closing and subsequent opening the file
 * - listening for changes in the Java Model (MDR)
 *
 * - registering problem finders
 * - upon changes: processing the source file using registered problem finders
 *
 * @author Tomasz Slota
 */
public class JEEVerificationAnnotationProvider implements AnnotationProvider {
    private static Collection<ProblemFinder> problemFinders = new ArrayList();
    
    // Document property keys used for caching problem data in the document (communication with the HintsProvider)
    static final String JEE_PROBLEM_SCANNING_RESULT = "jee.problem_scaning.result"; //NOI18N
    static final String JEE_PROBLEM_SCANNING_LINESET = "jee.problem_scaning.lineset"; //NOI18N
    
    public static void registerProblemFinderClass(String className){
        
        ProblemFinder problemFinder = null;
        try{
            Class classDef = Class.forName(className);
            problemFinder = (ProblemFinder) classDef.newInstance();
            
        } catch(Exception e){
            ErrorManager.getDefault().notify(ErrorManager.EXCEPTION, e);
            throw new RuntimeException(e);
        }
        
        if (problemFinder != null){
            problemFinders.add(problemFinder);
        }
    }
    
    // called when the document is opened for the first time
    public void annotate(final Line.Set set, Lookup context) {
        
        final DataObject dataObj = (DataObject)context.lookup(DataObject.class);
        if (dataObj == null) {
            return;
        }
        
        final FileObject fileObj = dataObj.getPrimaryFile();
        
        final BaseDocument baseDoc = (BaseDocument)(((EditorCookie)dataObj.getCookie(EditorCookie.class)).getDocument());
        
        // is it a java source file?
        if ("text/x-java".equals(fileObj.getMIMEType())) //NOI18N
        {
            Runnable firstTimeScanning = new Runnable(){
                public void run() {
                    // set a listener for changes
                    ChangeListener changeListener = createMDRListenerOnDocument(baseDoc);
                    EntityMappings mappings = PersistenceUtils.getEntityMappings(NbEditorUtilities.getFileObject(baseDoc));
                    
                    if (mappings != null){
                        mappings.addPropertyChangeListener(changeListener);
                    }
                    
                    // set a listener for closing / subsequent opening the file
                    EditorCookie.Observable editor = (Observable) dataObj.getCookie(EditorCookie.Observable.class);
                    EditorCookieListener editorListener = new EditorCookieListener(changeListener);
                    editor.addPropertyChangeListener(editorListener);
                    setPersistenceUnitPresenceListener(fileObj, changeListener);
                }

                private void setPersistenceUnitPresenceListener(FileObject fileObj, ChangeListener changeListener) {
                    Project project = FileOwnerQuery.getOwner(fileObj);
                    
                    if (project != null){
                        PersistenceScopes psScopes = PersistenceScopes.getPersistenceScopes(project);
                        
                        if (psScopes != null){
                            psScopes.addPropertyChangeListener(changeListener);
			    PersistenceScope scopes[] = psScopes.getPersistenceScopes();
			    
			    for (PersistenceScope scope : scopes){
				try {
				    PersistenceMetadata.getDefault().getRoot(scope.getPersistenceXml()).addPropertyChangeListener(changeListener);
				    
				} catch (Exception ex) {
				    ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, ex);
				}
			    }
                        }
                    }
                }
            };
            
            RequestProcessor.getDefault().post(firstTimeScanning);
        }
    }
    
    /**
     * Find persistence-related problems in the java code using registered problem finders
     */
    private static Collection processDocument(BaseDocument baseDoc, Line.Set lineSet){
        tmpDbg("looking for problems"); //NOI18N
        
        // manage data cached in the document
        Collection<JEEVerificationProblemMark> scanningResult = (Collection) baseDoc.getProperty(JEE_PROBLEM_SCANNING_RESULT);
        
        if (scanningResult == null){
            // document is being processed for the first time
            tmpDbg("document is being processed for the first time"); //NOI18N
            scanningResult = new ArrayList<JEEVerificationProblemMark>();
            baseDoc.putProperty(JEE_PROBLEM_SCANNING_RESULT, scanningResult);
        } else{
            // document is being processed subsequentially
            tmpDbg("document is being processed subsequentially"); //NOI18N
            
            synchronized (scanningResult){
                // remove existing problem marks from the document
                tmpDbg("removing " + scanningResult.size() + " existing problem marks"); //NOI18N
                
                for (JEEVerificationProblemMark problem : scanningResult){
                    problem.detach();
                }
                
                scanningResult.clear();
            }
        }
        baseDoc.putProperty(JEE_PROBLEM_SCANNING_LINESET, lineSet);
        
        // prepare common infrastructure for the problem finders
        JavaMetamodel javaMdl = JavaMetamodel.getManager();
        javaMdl.waitScanFinished();
        Resource resource = null;
        JavaModel.getJavaRepository().beginTrans(false);
        
        try {
            JMIUtils utils = JMIUtils.get(baseDoc);
            resource = utils.getResource();
            
            if (resource != null) {
                JavaModel.setClassPath(resource);
                
                for (JavaClass javaClass : getAllClassesDefinedInResource(resource)){
                    tmpDbg("Processing class: " + javaClass.getName()); //NOI18N
                    
                    ProblemFinderContext ctx = new ProblemFinderContext();
                    ctx.setLineSet(lineSet);
                    ctx.setDocument(baseDoc);
                    ctx.setResource(resource);
                    ctx.setMainJavaClass(javaClass);
                    EntityMappings mappings = PersistenceUtils.getEntityMappings(NbEditorUtilities.getFileObject(baseDoc));
                    ctx.setEntityMapping(mappings);

                    // because of a bug in ORM model population code,
                    // we are not calling isEntity first.
                    if(isEmbeddable(javaClass)) {
                        ctx.setEmbeddable(true);
                        // TODO: set embedding class access type
                    } else if(isMappedSuperclass(javaClass)) {
                        ctx.setMappedSuperclass(true);
                        ctx.setBeanAccessType(findAccessTypeOfHierarchy(javaClass));
                    } else if(isEntity(javaClass)){
                        ctx.setIsEntityClass(true);
                        ctx.setBeanAccessType(findAccessTypeOfHierarchy(javaClass));
                    } else if(isIdClass(javaClass)) {
                        ctx.setIdClass(true);
                        ctx.setBeanAccessType(findIdClassAccessType(javaClass));
                    } else if(isStateless(javaClass)) {
                        ctx.setSLSB(true);
                        Set<String> lbis = new HashSet<String>();
                        Set<String> rbis = new HashSet<String>();
                        EJBApiHelper.getBusinessInterfaces(javaClass, lbis, rbis);
                        ctx.setLBIs(lbis);
                        ctx.setRBIs(rbis);
                    } else if(isStateful(javaClass)) {
                        ctx.setSFSB(true);
                        Set<String> lbis = new HashSet<String>();
                        Set<String> rbis = new HashSet<String>();
                        EJBApiHelper.getBusinessInterfaces(javaClass, lbis, rbis);
                        ctx.setLBIs(lbis);
                        ctx.setRBIs(rbis);
                    } else if(isRBI(javaClass)) {
                        ctx.setRBI(true);
                    } else if(isLBI(javaClass)) {
                        ctx.setLBI(true);
                    }
                    
                    // iterate through the registered problem finders
                    for (ProblemFinder problemFinder : problemFinders){
                        problemFinder.reset();
                        problemFinder.setContext(ctx);
                        problemFinder.parseDocument();
                        
                        synchronized(scanningResult){
                            scanningResult.addAll(problemFinder.getProblemMarks());
                        }
                    }
                }
            } else{
                ErrorManager.getDefault().log("resource was null"); //NOI18N
            }
        } catch (Exception e) {
            // report only exceptions from code, do not consider problems
            // when user edited the code during hints computation
            if (resource == null || !javaMdl.isModified(javaMdl.getFileObject(resource))) {
                
                throw new RuntimeException(e);
            }
        } finally {
            JavaModel.getJavaRepository().endTrans();
        }
        
        updateHintsProvider(baseDoc);
        return scanningResult;
    }
    
    private static ChangeListener createMDRListenerOnDocument(BaseDocument baseDoc){
        ChangeListener changeListener = new ChangeListener(baseDoc);
        JavaMetamodel javaMdl = JavaMetamodel.getManager();
        FileObject fileObj = NbEditorUtilities.getFileObject(baseDoc);
        Project project = FileOwnerQuery.getOwner(fileObj);
        
        if (project != null) {
            javaMdl.waitScanFinished();
            ClassPathProvider classPathProvider = (ClassPathProvider) project.getLookup().lookup(ClassPathProvider.class);
            ClassPath classPath = classPathProvider.findClassPath(fileObj, ClassPath.SOURCE);   
            
            if (classPath != null){
                JavaModel.getJavaRepository().beginTrans(false);
                
                try {
                    List<MDRChangeSource> changeSources = new ArrayList<MDRChangeSource>(1); // normally there should be 1 source root
                    
                    for (FileObject root : classPath.getRoots()){
                        MDRChangeSource mdrChangeSource = (MDRChangeSource)JavaMetamodel.getManager().resolveJavaExtent(root);
                        mdrChangeSource.addListener(changeListener);
                        changeSources.add(mdrChangeSource);
                    }
                    
                    changeListener.setMDRChangeSources(changeSources.toArray(new MDRChangeSource[0]));
                    
                } catch (Exception e){
                    ErrorManager.getDefault().notify(ErrorManager.EXCEPTION, e);
                } finally{
                    JavaModel.getJavaRepository().endTrans(false);
                }
            }
        }
        
        changeListener.scheduleScanningTask();
        return changeListener;
    }
    
    private static List<JavaClass> getAllClassesDefinedInResource(Resource resource){
        ArrayList<JavaClass> classes = new ArrayList<JavaClass>(resource.getClassifiers());
        
        for (JavaClass javaClass : (List<JavaClass>)resource.getClassifiers()){
            classes.addAll(getAllSubClasses(javaClass));
        }
        
        return classes;
    }
    
    private static List<JavaClass> getAllSubClasses(JavaClass parentClass){
        ArrayList<JavaClass> kids = new ArrayList<JavaClass>();
        
        for (Object o : parentClass.getFeatures()){
            if (o instanceof JavaClass){
                JavaClass javaClass = (JavaClass)o;
                kids.add(javaClass);
                kids.addAll(getAllSubClasses(javaClass));
            }
        }
        
        return kids;
    }
    
    /*
     * Listens for changes in the Java model
     */
    private static class ChangeListener implements MDRChangeListener, PropertyChangeListener, FileChangeListener {
        private MDRChangeSource mdrChangeSources[];
        private final Object mdrChangeSourcesLock = new Object();
        private Collection<JEEVerificationProblemMark> generatedProblemMarks;
        private final RequestProcessor.Task task;
        private final AtomicBoolean active = new AtomicBoolean(true);
        private static RequestProcessor requestProcessor = new RequestProcessor("Java EE Verification Request Processor", 1); //NOI18N
        
        public ChangeListener(final BaseDocument baseDoc){
            assert baseDoc != null;
            
            Runnable action = new Runnable(){
                public void run() {
                    LineCookie lcookie = (LineCookie) NbEditorUtilities.getDataObject(baseDoc).getCookie(LineCookie.class);
                    final Line.Set lineSet = lcookie.getLineSet();
                    setGeneratedProblemMarks(processDocument(baseDoc, lineSet));
                }
            };
            
            task = requestProcessor.create(action, true);
        }
        
        public void change(MDRChangeEvent mdrEvent) {
            if (active.get() && isMeaningfulMDREvent(mdrEvent)){
                tmpDbg("a change in the java model detected"); // NOI18N
                
                // a series of events should be aggregated
                scheduleScanningTask();
            }
        }
        
        public void detach(){
            active.set(false);
            task.cancel();
            task.waitFinished();
            
            synchronized(mdrChangeSourcesLock){
                if (mdrChangeSources != null){
                    for (MDRChangeSource mcs : mdrChangeSources){
                        mcs.removeListener(this);
                    }
                    
                    mdrChangeSources = null;
                }
            }
        }
        
        public void setMDRChangeSources(MDRChangeSource mdrChangeSources[]){
            synchronized(mdrChangeSourcesLock){
                this.mdrChangeSources = mdrChangeSources;
            }
        }
        
        public synchronized Collection<JEEVerificationProblemMark> getGeneratedProblemMarks() {
            return generatedProblemMarks;
        }
        
        public synchronized void setGeneratedProblemMarks(Collection generatedProblemMarks) {
            this.generatedProblemMarks = generatedProblemMarks;
        }
        
        private static boolean isMeaningfulMDREvent(MDRChangeEvent mdrEvent){
            if (mdrEvent instanceof AttributeEvent){
                AttributeEvent attrEvent = (AttributeEvent)mdrEvent;
                
                String attrName = attrEvent.getAttributeName();
                tmpDbg("mdr event: " + attrName); //NOI18N
                
                if (!"timestamp".equals(attrName)){ //NOI18N
                    return true;
                }
            }
            
            return false;
        }
        
        public void propertyChange(PropertyChangeEvent evt) {
            tmpDbg("model update"); //NOI18N
            scheduleScanningTask();
        }
        
        public void scheduleScanningTask(){
            task.schedule(500);
        }
        
        public void fileFolderCreated(FileEvent fileEvent) {}
        
        public void fileDataCreated(FileEvent fileEvent) {
            if (isPersistentXMLFile(fileEvent.getFile())){
                scheduleScanningTask();
            }
        }
        
        public void fileChanged(FileEvent fileEvent) {}
        
        public void fileDeleted(FileEvent fileEvent) {
            if (isPersistentXMLFile(fileEvent.getFile())){
                scheduleScanningTask();
            }
        }
        
        public void fileRenamed(FileRenameEvent fileEvent) {}
        
        public void fileAttributeChanged(FileAttributeEvent fileEvent) {}
        
        private boolean isPersistentXMLFile(FileObject fileObj){
            return fileObj.getNameExt().equals("persistence.xml");
        }
    }
    
    /**
     * Listens for opening (subsequent) and closing file in the editor
     */
    private class EditorCookieListener implements PropertyChangeListener{
        private ChangeListener mdrListener;
        
        public EditorCookieListener(ChangeListener initialMDRListener){
            this.mdrListener = initialMDRListener;
        }
        
        public void propertyChange(PropertyChangeEvent evt) {
            if (EditorCookie.Observable.PROP_DOCUMENT.equals(evt.getPropertyName())){
                CloneableEditorSupport editor = (CloneableEditorSupport) evt.getSource();
                final BaseDocument baseDoc = (BaseDocument) editor.getDocument();
                
                // the document is being closed
                if (baseDoc == null){
                    tmpDbg("Document being closed"); //NOI18N
                    
                    final ChangeListener listenerCopy = mdrListener;
                    
                    RequestProcessor.getDefault().post(new Runnable(){
                        public void run() {
                            // remove the mdr listener from the java class
                            listenerCopy.detach();
                            
                            // remove all the problem marks (annotations)
                            if (listenerCopy.getGeneratedProblemMarks() != null){
                                for (JEEVerificationProblemMark problem : listenerCopy.getGeneratedProblemMarks()){
                                    problem.detach();
                                }
                            }
                        }
                    });
                    
                    mdrListener = null;
                    
                } // the document is being subsequentially opened
                else{
                    tmpDbg("Document being subsequentially opened"); //NOI18N
                    RequestProcessor.getDefault().post(new Runnable(){
                        public void run() {
                            mdrListener = createMDRListenerOnDocument(baseDoc);
                        }
                    });
                }
            }
        }
    }
    
    @Deprecated public static void tmpDbg(String debugMsg){
        // System.err.println(debugMsg);
    }
    
    private static void updateHintsProvider(Document doc){
        if (JEEVerificationHintsProvider.getInstance() != null){
            JEEVerificationHintsProvider.getInstance().update(doc);
        }
    }
}
