/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.j2ee.common;

import java.io.IOException;
import java.lang.reflect.Modifier;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import org.netbeans.jmi.javamodel.Annotation;
import org.netbeans.jmi.javamodel.AttributeValue;
import org.netbeans.jmi.javamodel.Constructor;
import org.netbeans.jmi.javamodel.Element;
import org.netbeans.jmi.javamodel.Field;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.JavaModelPackage;
import org.netbeans.jmi.javamodel.Method;
import org.netbeans.jmi.javamodel.MultipartId;
import org.netbeans.jmi.javamodel.Parameter;
import org.netbeans.jmi.javamodel.PrimitiveType;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.jmi.javamodel.StringLiteral;
import org.netbeans.jmi.javamodel.Type;
import org.netbeans.jmi.javamodel.TypeReference;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.javacore.internalapi.JavaModelUtil;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileSystem;
import org.openide.filesystems.Repository;
import org.openide.loaders.DataFolder;
import org.openide.loaders.DataObject;
import org.openide.util.NbBundle;

/**
 * This class need to be moved to some reasonable place, this is just temporary.
 * It is used from persistence(ide cluster) and from ejbcore(enterprise cluster)
 *
 * @author Martin Adamek, Andrei Badea
 */
public class JMIGenerationUtil {
    
    /**
     * The template used for new Entities, same template as for normal Java class.
     */
    private static final String CLASS_TEMPLATE = "Templates/Classes/Class.java"; // NOI18N
    private static final String INTERFACE_TEMPLATE = "Templates/Classes/Interface.java"; // NOI18N
    
    /**
     * Creates a new ordinary Java class, but enforces default
     * public constructor which is required by persistence specification. Uses 
     * the same file template as a regular Java class (see {@link #CLASS_TEMPLATE}).
     * Note that the created JavaClass doesn't contain any annotations. The class
     * is not created under an MDR transaction, a transaction is only entered to
     * retrieve a {@link JavaClass} instance for the new class.
     *
     * <p>This method does not need to be called in a MDR transaction, but if
     * it is, that transaction needs to be enclosed in an 
     * {@link FileSystem#runAtomicAction atomic action}, 
     * otherwise a deadlock can occur, like in issue 77500.</p>
     * 
     * @param targetFolder the folder / package for entity. Must not be null.
     * @param targetName the name of entity. Must not be null or empty.
     * @return JavaClass with default public constructor or null if JavaClass couldn't
     *         be created.
     */
    public static JavaClass createEntityClass(FileObject targetFolder, String targetName) throws IOException{
        DataObject jcDob = createClassFromTemplate(CLASS_TEMPLATE, targetFolder, targetName);
        JavaClass javaClass = null;
        
        JavaModel.getJavaRepository().beginTrans(true);
        boolean rollback = true;
        try {
            javaClass = getJavaClassFromDataObject(jcDob);
            String templateJavaDoc = javaClass.getJavadocText();
            javaClass.setJavadocText(NbBundle.getMessage(JMIGenerationUtil.class, "MSG_Javadoc_Class", 
                targetName) + templateJavaDoc);

            // public default constructor required
            ensurePublicConstructor(javaClass);
            
            rollback = false;
        } finally {
            JavaModel.getJavaRepository().endTrans(rollback);
        }
        
        return javaClass;
    }
    
    /**
     * Creates a new ordinary Java class. Uses the
     * same file template as a regular Java class (see {@link #CLASS_TEMPLATE}). 
     * The class is not created under an MDR transaction, a transaction is only 
     * entered to retrieve a {@link JavaClass} instance for the new class.
     *
     * <p>This method does not need to be called in a MDR transaction, but if
     * it is, that transaction needs to be enclosed in an 
     * {@link org.openide.filesystems.FileSystem#runAtomicAction atomic action}, 
     * otherwise a deadlock can occur, like in issue 77500.</p>
     */
    public static JavaClass createClass(FileObject targetFolder, String targetName) throws IOException{
        DataObject dobj = createClassFromTemplate(CLASS_TEMPLATE, targetFolder, targetName);
        return getJavaClassFromDataObject(dobj);
    }
    
    /**
     * Creates a new ordinary Java interface. Uses the
     * same file template as a regular Java interface (see {@link #INTERFACE_TEMPLATE}). 
     * The class is not created under an MDR transaction, a transaction is only 
     * entered to retrieve a {@link JavaClass} instance for the new interface.
     *
     * <p>This method does not need to be called in a MDR transaction, but if
     * it is, that transaction needs to be enclosed in an 
     * {@link org.openide.filesystems.FileSystem#runAtomicAction atomic action}, 
     * otherwise a deadlock can occur, like in issue 77500.</p>
     */
    public static JavaClass createInterface(FileObject targetFolder, String targetName) throws IOException{
        DataObject dobj = createClassFromTemplate(INTERFACE_TEMPLATE, targetFolder, targetName);
        return getJavaClassFromDataObject(dobj);
    }
    
    private static DataObject createClassFromTemplate(String template, FileObject targetFolder, String targetName) throws IOException {
        if (null == targetFolder || null == targetName || "".equals(targetName.trim())) { // NOI18N
            throw new IllegalArgumentException("Target folder and target name must be given."); // NOI18N
        }
        
        FileSystem dfs = Repository.getDefault().getDefaultFileSystem();
        FileObject fo = dfs.findResource(template);
        DataObject dob = DataObject.find(fo);
        DataFolder dataFolder = DataFolder.findFolder(targetFolder);
        return dob.createFromTemplate(dataFolder, targetName);
    }

    public static Annotation createAnnotation(Element element, String annotationClassName, List attributeValues) {
        JavaClass annotationClass = (JavaClass) JavaModel.getDefaultExtent().getType().resolve(annotationClassName);
        return getJavaModelPackage(element).getAnnotation().createAnnotation(
                JavaModelUtil.resolveImportsForClass(element, annotationClass),
                attributeValues
                );
    }
    
    public static AttributeValue createAttributeValue(Element element, String name, String value) {
        return getJavaModelPackage(element).getAttributeValue().createAttributeValue(
                name, getJavaModelPackage(element).getStringLiteral().createStringLiteral(value)
                );
    }
    
    public static AttributeValue createAttributeValue(Element element, String name, boolean value) {
        return getJavaModelPackage(element).getAttributeValue().createAttributeValue(
                name, getJavaModelPackage(element).getBooleanLiteral().createBooleanLiteral(value)
                );
    }
    
    public static AttributeValue createAttributeValue(Element element, String name, List arrayItems) {
        return getJavaModelPackage(element).getAttributeValue().createAttributeValue(
                name, getJavaModelPackage(element).getArrayInitialization().createArrayInitialization(arrayItems)
                );
    }
    
    public static AttributeValue createAttributeValue(Element element, String name, String type, String variable) {
        JavaClass typeClass = (JavaClass) JavaModel.getDefaultExtent().getType().resolve(type);
        String typeSimpleName = JavaModelUtil.resolveImportsForType(element, typeClass).getName();
        return getJavaModelPackage(element).getAttributeValue().createAttributeValue(
                name,
                getJavaModelPackage(element).getVariableAccess().createVariableAccess(typeSimpleName + (variable == null ? null : "." + variable), null, false)
                );
    }
    
    public static Field createField(Element element, String name, int modifiers, String type) {
        return getJavaModelPackage(element).getField().createField(
                name, null, modifiers, null, null, false, getTypeReference(element, type), 0, null, null
                );
    }
    
    /** Create field with type set to a one dimensional array of type given in <i>type</i> parameter.
     * @param type is the type of an item in array
     */
    public static Field createFieldArray(Element element, String name, int modifiers, String type) {
        return getJavaModelPackage(element).getField().createField(
                name, null, modifiers, null, null, false, getArrayTypeReference(element, type), 0, null, null
                );
    }
    
    public static Method createMethod(Element element, String name, int modifiers, String type) {
        return getJavaModelPackage(element).getMethod().createMethod(
                name, null, modifiers, null, null, null, null, null, null, null, getTypeReference(element, type), 0
                );
    }
    
    /** Create method that returns a one dimensional array of type given in <i>type</i> parameter.
     * @param type is the type of an item in array
     */
    public static Method createMethodArray(Element element, String name, int modifiers, String type) {
        return getJavaModelPackage(element).getMethod().createMethod(
                name, null, modifiers, null, null, null, null, null, null, null, getArrayTypeReference(element, type), 0
                );
    }

    public static Method createGetterMethod(String fieldName, String capitalizedFieldName, String fieldType, JavaClass javaClass) {
        Method getter = createMethod(javaClass, 
                "get" + capitalizedFieldName, Modifier.PUBLIC, fieldType); //NOI18N
        getter.setBodyText("return this." + fieldName + ";"); //NOI18N
        getter.setJavadocText(NbBundle.getMessage(
                JMIGenerationUtil.class, "MSG_Javadoc_Getter", fieldName, javaClass.getSimpleName()));
        return getter;
    }

    public static Method createSetterMethod(String fieldName, String capitalizedFieldName, String fieldType, JavaClass javaClass) {
        Method setter = createMethod(javaClass, 
                "set" + capitalizedFieldName, Modifier.PUBLIC, "void"); //NOI18N
        Parameter idParameter = createParameter(javaClass, fieldName, fieldType);
        setter.getParameters().add(idParameter);
        setter.setBodyText(getAssignmentLineForField(fieldName));
        setter.setJavadocText(NbBundle.getMessage(
                JMIGenerationUtil.class, "MSG_Javadoc_Setter", fieldName, javaClass.getSimpleName()));
        return setter;
    }

    private static String getAssignmentLineForField(String fieldName){
        return "this." + fieldName + " = " + fieldName + ";"; //NOI18N
    }

    /**
     * Create an equals method that compares two objects based on the fields list in <i>idFields</i> parameter.
     * This method is used to create equals methods for Entity classes and related PK classes.
     */
    public static Method createEntityEqualsMethod (JavaClass javaClass, List idFields) {
        Method equals = createMethod(javaClass, "equals", Modifier.PUBLIC, "boolean"); // NOI18N
        Parameter objectParameter = createParameter(javaClass, "object", "java.lang.Object"); // NOI18N
        equals.getParameters().add(objectParameter);
        String javaClassName = javaClass.getSimpleName();
        StringBuilder equalsMethodBody = 
            new StringBuilder("// TODO: Warning - this method won't work in the case the id fields are not set\n");  // NOI18N

        equalsMethodBody.append("if (!(object instanceof "); // NOI18N
        equalsMethodBody.append(javaClassName + ")) {\nreturn false;\n}\n"); // NOI18N
        equalsMethodBody.append(javaClassName + " other = (" + javaClassName + ")object;\n"); // NOI18N

        for(Iterator it = idFields.iterator(); it.hasNext();) {
            Field nextField = (Field)it.next();
            equalsMethodBody.append(getEqualsLineForField(nextField.getName(), nextField.getType()));
        }

        equalsMethodBody.append("return true;\n"); // NOI18N    
        equals.setBodyText(equalsMethodBody.toString());
        
        equals.setJavadocText(NbBundle.getMessage(
                JMIGenerationUtil.class, "MSG_Javadoc_Equals", javaClassName));

        Annotation overrideAnnotation = createAnnotation(javaClass, "java.lang.Override", Collections.EMPTY_LIST); //NOI18N
        equals.getAnnotations().add(overrideAnnotation);

        return equals;
    }

    private static String getEqualsLineForField(String fieldName, Type resolvedType){
        if (resolvedType instanceof PrimitiveType) {
            return "if (this." + fieldName + " != other." // NOI18N
                     + fieldName + ") return false;\n"; // NOI18N
        }

        return "if (this." + fieldName + " != other." + fieldName + // NOI18N
                 " && (this." + fieldName + " == null || !this." + // NOI18N
                 fieldName + ".equals(other." + fieldName // NOI18N
                 + "))) return false;\n"; // NOI18N
    }

    /** Create a hashCode method that creates a hash code based on the fields list in <i>fields</i> parameter.
     */
    public static Method createHashCodeMethod (JavaClass javaClass, List fields) {
        Method hashCode = createMethod(javaClass, "hashCode", Modifier.PUBLIC, "int"); // NOI18N
        StringBuilder hashCodeMethodBody = new StringBuilder("int hash = 0;\n"); // NOI18N

        for(Iterator it = fields.iterator(); it.hasNext();) {
            Field nextField = (Field)it.next();
            hashCodeMethodBody.append(getHashCodeLineForField(nextField.getName(), nextField.getType()));
        }

        hashCodeMethodBody.append("return hash;"); // NOI18N
        hashCode.setBodyText(hashCodeMethodBody.toString());

        hashCode.setJavadocText(NbBundle.getMessage(JMIGenerationUtil.class, "MSG_Javadoc_HashCode"));

        Annotation overrideAnnotation = createAnnotation(javaClass, "java.lang.Override", Collections.EMPTY_LIST); //NOI18N
        hashCode.getAnnotations().add(overrideAnnotation);

        return hashCode;
    }

    private static String getHashCodeLineForField(String fieldName, Type resolvedType){
        if (resolvedType instanceof PrimitiveType) {
            if (resolvedType.getName().equals("boolean")) { // NOI18N
                return "hash += (" + fieldName + " ? 1 : 0);\n"; // NOI18N
            }
            return "hash += (int)" + fieldName + ";\n"; // NOI18N
        }

        return "hash += (this." + fieldName + " != null ? this." + fieldName // NOI18N
            + ".hashCode() : 0);\n"; // NOI18N
    }

   /** Create a toString method that returns a string based on the fields list in <i>fields</i> parameter.
     */
    public static Method createToStringMethod (JavaClass javaClass, List fields) {
        Method toString = createMethod(javaClass, "toString", Modifier.PUBLIC, "java.lang.String"); // NOI18N
        StringBuilder toStringMethodBody = new StringBuilder("return \"" + javaClass.getName() + "["); // NOI18N
        int fieldsCount = fields.size();

        for(int i = 0; i < fieldsCount; i++) {
            String nextFieldName = ((Field)fields.get(i)).getName();
            toStringMethodBody.append(nextFieldName + "=\" + " + nextFieldName + " + \""); //NOI18N
            toStringMethodBody.append((i < (fieldsCount - 1)) ? ", " : "]\";"); //NOI18N
        }

        toString.setBodyText(toStringMethodBody.toString());
        toString.setJavadocText(NbBundle.getMessage(JMIGenerationUtil.class, "MSG_Javadoc_ToString"));

        Annotation overrideAnnotation = createAnnotation(javaClass, "java.lang.Override", Collections.EMPTY_LIST); //NOI18N
        toString.getAnnotations().add(overrideAnnotation);

        return toString;
    }

    public static Constructor createConstructor(JavaClass javaClass, int modifiers) {
        return getJavaModelPackage(javaClass).getConstructor().createConstructor(
                javaClass.getSimpleName(), null, modifiers, null, null, null, null, null, null, null
                );
    }

    public static Constructor createConstructor (JavaClass javaClass, List fields) {
        Constructor constructor = createConstructor(javaClass, Modifier.PUBLIC);
        StringBuilder constructorBody = new StringBuilder();
        String javaClassName = javaClass.getSimpleName();
        StringBuilder constructorJavaDoc = new StringBuilder(
            NbBundle.getMessage(JMIGenerationUtil.class, "MSG_Javadoc_Constructor", 
                javaClassName));
        List params = constructor.getParameters();

        for(Iterator it = fields.iterator(); it.hasNext();) {
            Field nextField = (Field)it.next();
            String fieldName = nextField.getName();
            Parameter fieldParameter = createParameter(
                    javaClass, fieldName, nextField.getType().getName());

            params.add(fieldParameter);
            constructorBody.append(getAssignmentLineForField(fieldName) + "\n"); //NOI18N
            constructorJavaDoc.append(NbBundle.getMessage(JMIGenerationUtil.class, "MSG_Javadoc_ConstructorParam", 
                fieldName, javaClassName));
        }

        constructor.setBodyText(constructorBody.toString());
        constructor.setJavadocText(constructorJavaDoc.toString());
        return constructor;
    }

    /** Checks the class for a public no-arg constructor and adds one if necessary.
     *  If a no-arg constructor is found with a different permission, it is changed to public.
     */
    public static void ensurePublicConstructor(JavaClass javaClass){
        Constructor existing = javaClass.getConstructor(Collections.EMPTY_LIST, false);
        if (null == existing){
            Constructor constructor = createConstructor(javaClass, Modifier.PUBLIC);
            javaClass.getFeatures().add(constructor);
        } else if ( existing.getModifiers() != Modifier.PUBLIC){
            existing.setModifiers(Modifier.PUBLIC);
        }
    }

    public static Parameter createParameter(Element element, String name, String type) {
        return getJavaModelPackage(element).getParameter().createParameter(
                name, null, false, getTypeReference(element, type), 0, false
                );
    }
    
    public static Parameter createParameterArray(Element element, String name, String type) {
        return getJavaModelPackage(element).getParameter().createParameter(
                name, null, false, getArrayTypeReference(element, type), 0, false
                );
    }
    
    public static StringLiteral createStringLiteral(Element element, String value) {
        return getJavaModelPackage(element).getStringLiteral().createStringLiteral(value);
    }
    
    public static void setAttribute(Annotation annotation, String name, String value) {
        for (Iterator it = annotation.getAttributeValues().iterator(); it.hasNext();) {
            AttributeValue attributeValue = (AttributeValue) it.next();
            // TODO: what should be used? getDefinition().getName() or getName()?
            if (attributeValue.getDefinition().getName().equals(name)) {
                StringLiteral stringLiteral = createStringLiteral(annotation, value);
                attributeValue.setValue(stringLiteral);
                return;
            }
        }
    }
    
    /** Create import for element that will be used in method body so that the
     * generated code can use simple names (not FQN).
     */
    public static TypeReference createImport(Element element, String name) {
        return getTypeReference(element, name);
    }
    
    public static void addInterface(JavaClass clazz, String interfaceName) {
        int lastDot = interfaceName.lastIndexOf('.');
        String simpleInterfaceName = lastDot > 0 ? interfaceName.substring(lastDot + 1) : interfaceName;
        List names = clazz.getInterfaceNames();
        for(Iterator it = names.iterator(); it.hasNext();) {
            MultipartId name = (MultipartId) it.next();
            if (interfaceName.equals(name.getName()) ||
                    simpleInterfaceName.equals(name.getName())) {
                return;
            }
        }
        createImport(clazz, interfaceName);
        names.add(getJavaModelPackage(clazz).getMultipartId().createMultipartId(simpleInterfaceName, null, null));
    }
    
    private static TypeReference getTypeReference(Element element, String name) {
        Type resolvedType = JavaModel.getDefaultExtent().getType().resolve(name);
        if (resolvedType instanceof PrimitiveType) {
            return getJavaModelPackage(element).getMultipartId().createMultipartId(name, null, null);
        } else {
            return JavaModelUtil.resolveImportsForType(element, resolvedType);
        }
    }
    
    /** Generate reference to "name[]" */
    private static TypeReference getArrayTypeReference(Element element, String name) {
        Type resolvedType = JavaModel.getDefaultExtent().getType().resolve(name);
        MultipartId itemType = getJavaModelPackage(element).getMultipartId().createMultipartId(name, null, null);
        return getJavaModelPackage(element).getArrayReference().createArrayReference(null, itemType, 1);
    }
    
    private static JavaModelPackage getJavaModelPackage(Element element) {
        return element != null ? (JavaModelPackage) element.refImmediatePackage() : JavaModel.getDefaultExtent();
    }
    
    /**
     * Gets JavaClass from given DataObject.
     * @param dobj must not be null.
     * @return JavaClass or null if couldn't get JavaClass
     *  from given dobj.
     */
    public static JavaClass getJavaClassFromDataObject(DataObject dobj) {
        if (null == dobj){
            throw new NullPointerException("DataObject must be given.");
        }
        JavaModel.getJavaRepository().beginTrans(false);
        try {
            Resource res = JavaModel.getResource(dobj.getPrimaryFile());
            if (res != null) {
                JavaModel.setClassPath(res);
                List/*<JavaClass>*/ classes = res.getClassifiers();
                if (classes.size() == 1) {
                    return (JavaClass)classes.get(0);
                }
            }
        } finally {
            JavaModel.getJavaRepository().endTrans(false);
        }
        return null;
    }
    
}
