/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.j2ee.refactoring.test;

import java.io.File;
import java.io.IOException;
import junit.textui.TestRunner;
import org.netbeans.junit.NbTestSuite;
import org.netbeans.modules.web.taglib.TLDDataObject;
import org.netbeans.modules.j2ee.refactoring.test.util.Helper;
import org.openide.cookies.EditorCookie;
import org.openide.cookies.OpenCookie;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileStateInvalidException;
import org.openide.filesystems.FileUtil;
import org.openide.loaders.DataObject;

/**
 *
 * @author Martin Grebac
 */
public class J2EERenameRefactoringTest extends J2EERefactoringTest {
    private static final String PROJECT_EJBMOD1 = "ejbmod1";
    private static final String PROJECT_WEBAPP1 = "webapp1";

    public J2EERenameRefactoringTest(String testName) {
        super(testName);
    }

    public static NbTestSuite suite() {
        NbTestSuite suite = new NbTestSuite("J2EERenameRefactoringTest");
        suite.addTest(new J2EERenameRefactoringTest("testPrepare"));
        suite.addTest(new J2EERenameRefactoringTest("testTldRenameClass"));
        suite.addTest(new J2EERenameRefactoringTest("testWebRenameClass"));
        suite.addTest(new J2EERenameRefactoringTest("testEjbRenameClass"));
        suite.addTest(new J2EERenameRefactoringTest("testWSRenameClass"));
        suite.addTest(new J2EERenameRefactoringTest("testCleanup"));
        return suite;
    }

    /**
     * Opens projects for tests
     */
    public void testPrepare() {
        openProject(PROJECT_WEBAPP1);
        openProject(PROJECT_EJBMOD1);
    }

    /** Renames
     *      tag handler
     *  and checks renamed files and mytld.tld for differences
     */
    public void testTldRenameClass() throws FileStateInvalidException, IOException {
        File f = getProjectFile(PROJECT_WEBAPP1, "/web/WEB-INF/tlds/mytld.tld");
        FileObject fo = FileUtil.toFileObject(f);

        // this part is here to test if the documents opened in editor are changed, too, which is a bug 55405
        TLDDataObject tdo = ((TLDDataObject)DataObject.find(fo));
        OpenCookie oc = (OpenCookie)tdo.getCookie(OpenCookie.class);
        oc.open();
        // end of 55405 part

        doRenameRefactoring(PROJECT_WEBAPP1, "abc.NewTagHandler", "RenamedTagHandler", "rename tag handler");

        Helper.waitForAWTDispatchThread();

        // another 55405 specific part
        EditorCookie ec = (EditorCookie)tdo.getCookie(EditorCookie.class);
        try {
            ec.getDocument().insertString(0, " ", null);
            ec.saveDocument();
            ec.getDocument().remove(0, 1);
            ec.saveDocument();
        } catch (Exception e) {
            fail(e.toString());
        }
        // end of 55405 specific part

        Helper.waitForAWTDispatchThread();
        // check modified files
        try {
            checkGoldenFile(PROJECT_WEBAPP1, "src/java/abc/RenamedTagHandler.java", "RenamedTagHandler.pass");

            // check TLD
            checkFile(PROJECT_WEBAPP1, "web/WEB-INF/tlds/mytld.tld", new String[]{">abc.RenamedTagHandler<"},
                    new String[]{">abc.NewTagHandler<"});

        } catch (FileStateInvalidException e) {
            fail(e.getMessage());
        } catch (IOException e) {
            fail(e.getMessage());
        }
    }

    /** Renames
     *      servlet
     *      filter
     *      listener
     *  and checks renamed files and web.xml for differences
     */
    public void testWebRenameClass() throws FileStateInvalidException, IOException {

        doRenameRefactoring(PROJECT_WEBAPP1, "abc.NewServlet", "RenamedServlet", "rename servlet");
        doRenameRefactoring(PROJECT_WEBAPP1, "abc.NewSimpleFilter", "RenamedFilter", "rename filter");
        doRenameRefactoring(PROJECT_WEBAPP1, "abc.NewServletListener", "RenamedListener", "rename listener");

        Helper.waitForAWTDispatchThread();

        // check modified files
        try {
            checkGoldenFile(PROJECT_WEBAPP1, "src/java/abc/RenamedServlet.java", "RenamedServlet.pass");
            checkGoldenFile(PROJECT_WEBAPP1, "src/java/abc/RenamedFilter.java", "RenamedFilter.pass");
            checkGoldenFile(PROJECT_WEBAPP1, "src/java/abc/RenamedListener.java", "RenamedListener.pass");

            // check web.xml
            checkFile(PROJECT_WEBAPP1, "web/WEB-INF/web.xml",
                    new String[]{">abc.RenamedServlet<", ">abc.RenamedFilter<", ">abc.RenamedListener<"},
                    new String[]{">abc.NewServlet<", ">abc.NewSimpleFilter<", ">abc.NewServletListener<"});

        } catch (FileStateInvalidException e) {
            fail(e.getMessage());
        } catch (IOException e) {
            fail(e.getMessage());
        }
    }

    /** Renames
     *      session bean implementation class
     *      session bean local interface
     *      session bean local home interface
     *      session bean remote interface
     *      session bean remote home interface
     *      queue message bean impl class
     *      topic message bean impl class
     *      CMP entity bean impl class
     *      CMP entity bean local interface
     *      CMP entity bean local home interface
     *      BMP entity bean impl class
     *      BMP entity bean local interface
     *      BMP entity bean local home interface
     * and checks renamed files and ejb-jar.xml for differences
     */
    public void testEjbRenameClass() throws FileStateInvalidException, IOException {
        Helper.waitForAWTDispatchThread();
        doRenameRefactoring(PROJECT_EJBMOD1, "my.enterprise.beans.NewSessionBean", "RenamedSessionBean", "rename session bean");
        doRenameRefactoring(PROJECT_EJBMOD1, "my.enterprise.beans.NewSessionLocal", "RenamedSessionLocal",
                "rename session bean's local interface");
        doRenameRefactoring(PROJECT_EJBMOD1, "my.enterprise.beans.NewSessionLocalHome", "RenamedSessionLocalHome",
                "rename session bean's local home interface");
        doRenameRefactoring(PROJECT_EJBMOD1, "my.enterprise.beans.NewSessionRemote", "RenamedSessionRemote",
                "rename session bean's remote interface");
        doRenameRefactoring(PROJECT_EJBMOD1, "my.enterprise.beans.NewSessionRemoteHome", "RenamedSessionRemoteHome",
                "rename session bean's remote home interface");
        doRenameRefactoring(PROJECT_EJBMOD1, "my.enterprise.beans.NewMessageBean", "RenamedMessageBean", "rename queue message bean");
        doRenameRefactoring(PROJECT_EJBMOD1, "my.enterprise.beans.NewMessageTopicBean", "RenamedMessageTopicBean",
                "rename topic message bean");
        doRenameRefactoring(PROJECT_EJBMOD1, "my.enterprise.beans.NewEntityContainerBean", "RenamedEntityContainerBean",
                "rename entity container bean");
        doRenameRefactoring(PROJECT_EJBMOD1, "my.enterprise.beans.NewEntityContainerLocal", "RenamedEntityContainerLocal",
                "rename entity container bean's local interface");
        doRenameRefactoring(PROJECT_EJBMOD1, "my.enterprise.beans.NewEntityContainerLocalHome", "RenamedEntityContainerLocalHome",
                "rename entity container bean's local home interface");
        doRenameRefactoring(PROJECT_EJBMOD1, "my.enterprise.beans.NewEntityBeanMPBean", "RenamedEntityBeanMPBean",
                "rename entity bmp bean");
        doRenameRefactoring(PROJECT_EJBMOD1, "my.enterprise.beans.NewEntityBeanMPLocal", "RenamedEntityBeanMPLocal",
                "rename entity bmp bean's local interface");
        doRenameRefactoring(PROJECT_EJBMOD1, "my.enterprise.beans.NewEntityBeanMPLocalHome", "RenamedEntityBeanMPLocalHome",
                "rename entity bmp bean's local home interface");
        
        Helper.waitForAWTDispatchThread();

        // check modified files
        try {
            checkGoldenFile(PROJECT_EJBMOD1, "src/java/my/enterprise/beans/RenamedSessionBean.java",
                    "RenamedSessionBean.pass");
            checkGoldenFile(PROJECT_EJBMOD1, "src/java/my/enterprise/beans/RenamedSessionLocal.java",
                    "RenamedSessionLocal.pass");
            checkGoldenFile(PROJECT_EJBMOD1, "src/java/my/enterprise/beans/RenamedSessionLocalHome.java",
                    "RenamedSessionLocalHome.pass");
            checkGoldenFile(PROJECT_EJBMOD1, "src/java/my/enterprise/beans/RenamedSessionRemote.java",
                    "RenamedSessionRemote.pass");
            checkGoldenFile(PROJECT_EJBMOD1, "src/java/my/enterprise/beans/RenamedSessionRemoteHome.java",
                    "RenamedSessionRemoteHome.pass");
            checkGoldenFile(PROJECT_EJBMOD1, "src/java/my/enterprise/beans/RenamedMessageBean.java",
                    "RenamedMessageBean.pass");
            checkGoldenFile(PROJECT_EJBMOD1, "src/java/my/enterprise/beans/RenamedMessageTopicBean.java",
                    "RenamedMessageTopicBean.pass");
            checkGoldenFile(PROJECT_EJBMOD1, "src/java/my/enterprise/beans/RenamedEntityContainerBean.java",
                    "RenamedEntityContainerBean.pass");
            checkGoldenFile(PROJECT_EJBMOD1, "src/java/my/enterprise/beans/RenamedEntityContainerLocal.java",
                    "RenamedEntityContainerLocal.pass");
            checkGoldenFile(PROJECT_EJBMOD1, "src/java/my/enterprise/beans/RenamedEntityContainerLocalHome.java",
                    "RenamedEntityContainerLocalHome.pass");
            checkGoldenFile(PROJECT_EJBMOD1, "src/java/my/enterprise/beans/RenamedEntityBeanMPBean.java",
                    "RenamedEntityBeanMPBean.pass");
            checkGoldenFile(PROJECT_EJBMOD1, "src/java/my/enterprise/beans/RenamedEntityBeanMPLocal.java",
                    "RenamedEntityBeanMPLocal.pass");
            checkGoldenFile(PROJECT_EJBMOD1, "src/java/my/enterprise/beans/RenamedEntityBeanMPLocalHome.java",
                    "RenamedEntityBeanMPLocalHome.pass");

            // check ejb-jar
            String[] mustContain = new String[]{
                    ">my.enterprise.beans.RenamedSessionBean<",
                    ">my.enterprise.beans.RenamedSessionLocal<",
                    ">my.enterprise.beans.RenamedSessionLocalHome<",
                    ">my.enterprise.beans.RenamedSessionRemote<",
                    ">my.enterprise.beans.RenamedSessionRemoteHome<",
                    ">my.enterprise.beans.RenamedMessageBean<",
                    ">my.enterprise.beans.RenamedMessageTopicBean<",
                    ">my.enterprise.beans.RenamedEntityContainerBean<",
                    ">my.enterprise.beans.RenamedEntityContainerLocal<",
                    ">my.enterprise.beans.RenamedEntityContainerLocalHome<",
                    ">my.enterprise.beans.RenamedEntityBeanMPBean<",
                    ">my.enterprise.beans.RenamedEntityBeanMPLocal<",
                    ">my.enterprise.beans.RenamedEntityBeanMPLocalHome<"
            };
            String[] mustNotContain = new String[]{
                    ">my.enterprise.beans.NewSessionBean<",
                    ">my.enterprise.beans.NewSessionLocal<",
                    ">my.enterprise.beans.NewSessionLocalHome<",
                    ">my.enterprise.beans.NewSessionRemote<",
                    ">my.enterprise.beans.NewSessionRemoteHome<",
                    ">my.enterprise.beans.NewMessageBean<",
                    ">my.enterprise.beans.NewMessageTopicBean<",
                    ">my.enterprise.beans.NewEntityContainerBean<",
                    ">my.enterprise.beans.NewEntityContainerLocal<",
                    ">my.enterprise.beans.NewEntityContainerLocalHome<",
                    ">my.enterprise.beans.NewEntityBeanMPBean<",
                    ">my.enterprise.beans.NewEntityBeanMPLocal<",
                    ">my.enterprise.beans.NewEntityBeanMPLocalHome<"
            };

            // wait for saving data to ejb-jar.xml  
            try {
                Thread.sleep(2000);
            } catch (InterruptedException e) {
                //
            }

            checkFile(PROJECT_EJBMOD1, "src/conf/ejb-jar.xml", mustContain, mustNotContain);

        } catch (FileStateInvalidException e) {
            fail(e.getMessage());
        } catch (IOException e) {
            fail(e.getMessage());
        }
    }

    /** Renames
     *      WS server impl bean
     *      WS server SEI interface
     * and checks renamed files and wscompile config and ejb-jar.xml and webservices.xml for differences
     */
    public void testWSRenameClass() throws FileStateInvalidException, IOException {

        doRenameRefactoring(PROJECT_EJBMOD1, "my.ws.server.NewWebServiceBean", "RenamedWebServiceBean", "rename ws server impl bean");
        doRenameRefactoring(PROJECT_EJBMOD1, "my.ws.server.NewWebServiceSEI", "RenamedWebServiceSEI",
                "rename ws server servlet endpoint interface");

        Helper.waitForAWTDispatchThread();

        // check modified files
        try {
            checkGoldenFile(PROJECT_EJBMOD1, "src/java/my/ws/server/RenamedWebServiceBean.java", "RenamedWebServiceBean.pass");
            checkGoldenFile(PROJECT_EJBMOD1, "src/java/my/ws/server/RenamedWebServiceSEI.java", "RenamedWebServiceSEI.pass");

            // check wscompile config file
            checkFile(PROJECT_EJBMOD1, "src/java/my/ws/server/NewWebService-config.xml",
                    new String[]{"my.ws.server.RenamedWebServiceSEI", "my.ws.server.RenamedWebServiceBean"},
                    new String[]{"my.ws.server.NewWebServiceSEI", "my.ws.server.NewWebServiceBean"});

            // check ejb-jar
            checkFile(PROJECT_EJBMOD1, "src/conf/ejb-jar.xml", new String[]{">my.ws.server.RenamedWebServiceBean<"},
                    new String[]{">my.ws.server.NewWebServiceBean<"});

            // check webservices.xml
            checkFile(PROJECT_EJBMOD1, "src/conf/webservices.xml", new String[]{">my.ws.server.RenamedWebServiceSEI<"},
                    new String[]{">my.ws.server.NewWebServiceSEI<"});

            // check webservices.xml
            checkFile(PROJECT_EJBMOD1, "src/conf/webservices.xml", new String[]{">wsdl-port_ns:RenamedWebServiceSEIPort<"},
                    new String[]{">wsdl-port_ns:NewWebServiceSEIPort<"});
                    
        } catch (FileStateInvalidException e) {
            fail(e.getMessage());
        } catch (IOException e) {
            fail(e.getMessage());
        }
    }

    /**
     * Opens projects for tests
     */
    public void testCleanup() {
        closeProject(PROJECT_WEBAPP1);
        closeProject(PROJECT_EJBMOD1);
    }

    /**
     * Used for running test from inside the IDE by internal execution.
     *
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        TestRunner.run(suite());
    }

}
