/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 *
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.j2ee.refactoring.safedelete;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import org.netbeans.modules.j2ee.dd.api.web.Filter;
import org.netbeans.modules.j2ee.dd.api.web.FilterMapping;
import org.netbeans.modules.j2ee.dd.api.web.Listener;
import org.netbeans.modules.j2ee.dd.api.web.Servlet;
import org.netbeans.modules.j2ee.dd.api.web.ServletMapping;
import org.netbeans.modules.j2ee.dd.api.web.WebApp;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.modules.j2ee.refactoring.DefaultPositionBoundsResolver;
import org.netbeans.modules.j2ee.refactoring.J2EERefactoring;
import org.netbeans.modules.j2ee.refactoring.Utility;
import org.netbeans.modules.javacore.internalapi.ExternalChange;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.refactoring.api.Problem;
import org.netbeans.modules.refactoring.api.SafeDeleteRefactoring;
import org.netbeans.modules.refactoring.spi.RefactoringElementsBag;
import org.netbeans.modules.refactoring.spi.RefactoringElementImplementation;
import org.netbeans.modules.web.api.webmodule.WebModule;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;
import org.openide.loaders.DataObject;
import org.openide.loaders.DataObjectNotFoundException;
import org.openide.text.PositionBounds;
import org.openide.util.NbBundle;

/**
 * Safe delete refactoring for web.xml. Handles removing of servlets, listeners,
 * filters and their associated mappings.
 *
 * @author Erno Mononen
 */
public final class WebXmlSafeDeleteRefactoring implements J2EERefactoring {
    
    private JavaClass javaClass;
    private SafeDeleteRefactoring safeDeleteRefactoring;
    
    public WebXmlSafeDeleteRefactoring(SafeDeleteRefactoring safeDeleteRefactoring, JavaClass javaClass) {
        this.javaClass = javaClass;
        this.safeDeleteRefactoring = safeDeleteRefactoring;
    }
    
    /**
     * @return list containing DDs from web modules that are relevant
     * to our <code>javaClass</class>.
     */
    private List<FileObject> getDeploymentDescriptors(){
        Collection modules = Utility.getRelevantWebModules(javaClass);
        
        List<FileObject> result = new ArrayList<FileObject>();
        for (Iterator it = modules.iterator(); it.hasNext();) {
            WebModule each = (WebModule) it.next();
            FileObject webXmlFo = each.getDeploymentDescriptor();
            if (webXmlFo != null){
                result.add(webXmlFo);
            }
        }
        return result;
    }
    
    /**
     * Handles removing of servlets and their associated servlet mappings.
     */
    private void processServlets(WebApp webXmlDD, FileObject webXmlFO, RefactoringElementsBag refactoringElements){
        Servlet[] servlets = webXmlDD.getServlet();
        
        for (int s=0; s < servlets.length; s++) {
            final Servlet servlet = servlets[s];
            final String servletClass = servlet.getServletClass();
            
            if (servletClass == null || !servletClass.equals(javaClass.getName())) {
                continue;
            }
            
            final List<ServletMapping> mappings = getServletMappings(webXmlDD, servlet);
            
            RefactoringElementImplementation elem = new WebXmlSafeDeleteRefactoringElement(webXmlFO, webXmlDD,servletClass) {
                protected void doChange(){
                    webApp.removeServlet(servlet);
                    for (ServletMapping elem : mappings) {
                        webApp.removeServletMapping(elem);
                    }
                }
                protected void undoChange(){
                    webApp.addServlet(servlet);
                    for (ServletMapping elem : mappings) {
                        webApp.addServletMapping(elem);
                    }
                }
                public String getDisplayText(){
                    return NbBundle.getMessage(WebXmlSafeDeleteRefactoring.class, "TXT_WebXmlServletSafeDelete", servletClass);
                }
            };
            
            refactoringElements.add(safeDeleteRefactoring, elem);
        }
        
    }
    
    /**
     * @return List of servlet mappings that are associated
     * with the given <code>servlet</code>.
     */
    private List<ServletMapping> getServletMappings(WebApp webApp, Servlet servlet){
        ServletMapping[] mappings = webApp.getServletMapping();
        List<ServletMapping> result = new ArrayList<ServletMapping>();
        
        for (int i = 0; i < mappings.length; i++) {
            if (servlet.getServletName().equals(mappings[i].getServletName())){
                result.add(mappings[i]);
            }
        }
        return result;
    }
    
    /**
     * Handles removing of filters and their associated filter mappings.
     */
    private void processFilters(WebApp webXmlDD, FileObject webXmlFO, RefactoringElementsBag refactoringElements){
        Filter[] filters = webXmlDD.getFilter();
        
        for (int i = 0; i < filters.length; i++) {
            final Filter filter = filters[i];
            final String filterClass = filter.getFilterClass();
            
            if (filterClass == null || !filterClass.equals(javaClass.getName())) {
                continue;
            }
            
            final List<FilterMapping> mappings = getFilterMappings(webXmlDD, filter);
            
            RefactoringElementImplementation elem = new WebXmlSafeDeleteRefactoringElement(webXmlFO, webXmlDD, filterClass) {
                protected void doChange() {
                    webApp.removeFilter(filter);
                    for (FilterMapping elem : mappings) {
                        webApp.removeFilterMapping(elem);
                    }
                }
                protected void undoChange() {
                    webApp.addFilter(filter);
                    for (FilterMapping elem : mappings) {
                        webApp.addFilterMapping(elem);
                    }
                }
                public String getDisplayText(){
                    return NbBundle.getMessage(WebXmlSafeDeleteRefactoring.class, "TXT_WebXmlFilterSafeDelete", filterClass);
                }
            };
            
            refactoringElements.add(safeDeleteRefactoring, elem);
        }
        
    }
    
    /**
     * @return list of filter mappings that are associated with
     * the given <code>filter</code>.
     */
    private List<FilterMapping> getFilterMappings(WebApp webApp, Filter filter){
        FilterMapping[] mappings = webApp.getFilterMapping();
        List<FilterMapping> result = new ArrayList<FilterMapping>();
        
        for (int i = 0; i < mappings.length; i++) {
            if (filter.getFilterName().equals(mappings[i].getFilterName())){
                result.add(mappings[i]);
            }
        }
        return result;
    }
    
    /**
     * Handles removing of listeners.
     */
    private void processListeners(WebApp webXmlDD, FileObject webXmlFO, RefactoringElementsBag refactoringElements){
        Listener[] listeners = webXmlDD.getListener();
        for (int i = 0; i < listeners.length; i++) {
            final Listener listener = listeners[i];
            final String listenerClass = listener.getListenerClass();
            if (listenerClass == null || !listenerClass.equals(javaClass.getName())) {
                continue;
            }
            
            RefactoringElementImplementation elem = new WebXmlSafeDeleteRefactoringElement(webXmlFO, webXmlDD, listenerClass) {
                protected void doChange() {
                    webApp.removeListener(listener);
                }
                protected void undoChange() {
                    webApp.addListener(listener);
                }
                public String getDisplayText(){
                    return NbBundle.getMessage(WebXmlSafeDeleteRefactoring.class, "TXT_WebXmlListenerSafeDelete", listenerClass);
                }
            };
            
            refactoringElements.add(safeDeleteRefactoring, elem);
        }
        
    }
    
    public Problem prepare(RefactoringElementsBag refactoringElements) {
        
        Problem problem = null;
        
        List<FileObject> webXmlDDs = getDeploymentDescriptors();
        
        for (FileObject webXmlFO : webXmlDDs) {
            WebApp webXmlDD = null;
            
            try {
                webXmlDD = org.netbeans.modules.j2ee.dd.api.web.DDProvider.getDefault().getDDRoot(webXmlFO);
            } catch (IOException ioe) {
                ErrorManager.getDefault().notify(ioe);
            }
            
            if (webXmlDD != null && webXmlDD.getStatus() != WebApp.STATE_INVALID_UNPARSABLE){
                processServlets(webXmlDD, webXmlFO, refactoringElements);
                processFilters(webXmlDD, webXmlFO, refactoringElements);
                processListeners(webXmlDD, webXmlFO, refactoringElements);
                
            } else {
                Problem newProblem = new Problem(false, NbBundle.getMessage(WebXmlSafeDeleteRefactoring.class, "TXT_WebXmlInvalidProblem", webXmlFO.getPath()));
                problem = Utility.addProblemsToEnd(problem, newProblem);
            }
            
        }
        
        return problem;
    }
    
    public Problem preCheck() {
        return null;
    }
    
    public Problem checkParameters() {
        return null;
    }
    
    public Problem fastCheckParameters() {
        return null;
    }
    
    
    private abstract static class WebXmlSafeDeleteRefactoringElement extends AbstractSafeDeleteRefactoringElement implements ExternalChange{
        
        protected WebApp webApp;
        private String className;
        
        public WebXmlSafeDeleteRefactoringElement(FileObject parentFile, WebApp webApp, String className){
            super(parentFile);
            this.webApp = webApp;
            this.className = className;
        }
        
        public final void performChange() {
            JavaMetamodel.getManager().registerExtChange(this);
        }
        
        /**
         * Do the actual change.
         */
        protected abstract void doChange();
        
        /**
         * Undo the change.
         */
        protected abstract void undoChange();
        
        public final void performExternalChange() {
            doChange();
            try{
                webApp.write(parentFile);
            } catch (IOException ioe){
                ErrorManager.getDefault().notify(ioe);
            }
        }
        
        public final void undoExternalChange() {
            undoChange();
            try{
                webApp.write(parentFile);
            } catch (IOException ioe){
                ErrorManager.getDefault().notify(ioe);
            }
        }
        
        public PositionBounds getPosition() {
            try {
                return new DefaultPositionBoundsResolver(DataObject.find(getParentFile()), className).getPositionBounds();
            } catch (DataObjectNotFoundException ex) {
                ErrorManager.getDefault().notify(ex);
            }
            return null;
        }
        
    }
}
