/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.j2ee.refactoring;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import org.netbeans.api.project.FileOwnerQuery;
import org.netbeans.api.project.Project;
import org.netbeans.jmi.javamodel.AnnotableElement;
import org.netbeans.jmi.javamodel.Annotation;
import org.netbeans.jmi.javamodel.AnnotationType;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.modules.j2ee.common.JMIUtils;
import org.netbeans.modules.j2ee.persistence.api.PersistenceScope;
import org.netbeans.modules.j2ee.persistence.dd.PersistenceUtils;
import org.netbeans.modules.j2ee.persistence.dd.orm.model_1_0.Entity;
import org.netbeans.modules.javacore.api.JavaModel;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;

/**
 * Utility methods for persistence related refactorings.
 * @author Erno Mononen
 */
public class PersistenceRefactoringUtil {
    
    private PersistenceRefactoringUtil() {
    }
    
    /**
     * @return list of FileObjects representing persistence dd files
     * in the project of the given <code>javaClass</code>.
     */
    public static List<FileObject> getPersistence(JavaClass javaClass){
        
        List<FileObject> result = new ArrayList<FileObject>();
        
        Project project = getProject(javaClass);
        if (project != null){
            PersistenceScope[] persistenceScopes = PersistenceUtils.getPersistenceScopes(project);
            for (int i = 0; i < persistenceScopes.length; i++) {
                result.add(persistenceScopes[i].getPersistenceXml());
            }
        }
        
        return result;
    }
    
    /**
     * @return the project of the given <code>javaClass</code>.
     */
    public static Project getProject(JavaClass javaClass){
        Resource res = javaClass.getResource();
        FileObject fo = JavaModel.getFileObject(res);
        return FileOwnerQuery.getOwner(fo);
    }
    
    /**
     * Gets all entities from the project to which given <code>javaClass</code>
     * belongs to.
     */
    public static List<JavaClass> getEntitiesInProject(JavaClass javaClass){
        List<JavaClass> result = new ArrayList<JavaClass>();
        Project project = getProject(javaClass);
        
        Set<Entity> entities = null;
        try {
            entities = PersistenceUtils.getEntityClasses(project);
        } catch (IOException e) {
            ErrorManager.getDefault().notify(e);
            return Collections.emptyList();
        }
        
        JMIUtils.beginJmiTransaction();
        boolean rollback = true;
        try{
            for (Entity entity : entities) {
                JavaClass jc = (JavaClass) JavaModel.getDefaultExtent().getType().resolve(entity.getClass2());
                result.add(jc);
            }
            rollback = false;
        } finally {
            JMIUtils.endJmiTransaction(rollback);
        }
        return result;
    }
    
    /**
     * @return true if given element represents an annotated entity.
     */
    public static boolean isEntity(AnnotableElement element){
        
        if (element == null || element.getAnnotations() == null){
            return false;
        }
        
        String entityAnnotation = "javax.persistence.Entity";
        Annotation annots[] = (Annotation[]) element.getAnnotations().toArray(new Annotation[]{});
        
        for (Annotation ann : annots) {
            AnnotationType atype = ann.getType();
            if (atype != null && entityAnnotation.equals(atype.getName())){
                return true;
            }
        }
        
        return false;
    }
    
}
