/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.j2ee.persistence.dd;

import java.io.IOException;
import java.lang.ref.Reference;
import java.lang.ref.WeakReference;
import java.util.Collections;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.modules.j2ee.metadata.MetadataUnit;
import org.netbeans.modules.j2ee.persistence.api.PersistenceScope;
import org.netbeans.modules.j2ee.persistence.dd.persistence.model_1_0.PersistenceUnit;
import org.netbeans.modules.j2ee.persistence.spi.PersistenceScopeFactory;
import org.netbeans.modules.j2ee.persistence.spi.PersistenceScopeImplementation;
import org.netbeans.modules.j2ee.persistence.unit.PUDataObjectTestBase;
import org.netbeans.spi.java.classpath.support.ClassPathSupport;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;

/**
 *
 * @author Andrei Badea
 */
public class PersistenceUtilsTest extends PUDataObjectTestBase {

    private FileObject dataDir;
    private FileObject workDir;

    public PersistenceUtilsTest(String testName) {
        super(testName);
    }

    public void setUp() throws Exception {
        clearWorkDir();
        dataDir = FileUtil.toFileObject(getDataDir());
        workDir = FileUtil.toFileObject(getWorkDir());
    }

    private static FileObject copyFile(FileObject source, FileObject destFolder) throws IOException {
        return FileUtil.copyFile(source, destFolder, source.getName());
    }

    public void testNoLeaks() throws Exception {
        FileObject persistenceXml = copyFile(dataDir.getFileObject("persistence.xml"), workDir);
        assertNotNull(persistenceXml);

        ClassPath classPath = ClassPathSupport.createClassPath(Collections.emptyList());
        PersistenceScopeImpl persistenceScopeImpl = new PersistenceScopeImpl(persistenceXml, classPath);
        PersistenceScope persistenceScope = PersistenceScopeFactory.createPersistenceScope(persistenceScopeImpl);

        PersistenceUtils.EntityMappingsCache emCache = PersistenceUtils.getEntityMappingsCache(persistenceScope);
        emCache.getMetadataUnit("em");

        Reference<ClassPath> classPathRef = new WeakReference<ClassPath>(classPath);
        classPath = null;
        persistenceScopeImpl.setClassPath(null);
        assertGC("Should be possible to GC classPath", classPathRef);

        Reference<FileObject> persistenceXmlRef = new WeakReference<FileObject>(persistenceXml);
        persistenceXml = null;
        persistenceScopeImpl.setPersistenceXml(null);
        assertGC("Should be possible to GC persistenceXml", persistenceXmlRef);

        Reference<PersistenceScope> persistenceScopeRef = new WeakReference<PersistenceScope>(persistenceScope);
        persistenceScope = null;
        assertGC("Should be possible to GC persistenceScope", persistenceScopeRef);
    }

    public void testCaching() throws Exception {
        FileObject persistenceXml = copyFile(dataDir.getFileObject("persistence.xml"), workDir);
        assertNotNull(persistenceXml);

        ClassPath classPath = ClassPathSupport.createClassPath(Collections.emptyList());
        PersistenceScopeImpl persistenceScopeImpl = new PersistenceScopeImpl(persistenceXml, classPath);
        PersistenceScope persistenceScope = PersistenceScopeFactory.createPersistenceScope(persistenceScopeImpl);

        // EntityMappingsCache instances are cached

        PersistenceUtils.EntityMappingsCache emCache1 = PersistenceUtils.getEntityMappingsCache(persistenceScope);
        PersistenceUtils.EntityMappingsCache emCache2 = PersistenceUtils.getEntityMappingsCache(persistenceScope);

        // MetadataUnit's are cached

        assertNotNull(emCache1);
        assertSame(emCache1, emCache2);

        MetadataUnit emmu1 = emCache1.getMetadataUnit("em");
        MetadataUnit emmu2 = emCache1.getMetadataUnit("em");
        assertNotNull(emmu1);
        assertSame(emmu1, emmu2);

        MetadataUnit em2mu1 = emCache2.getMetadataUnit("em2");
        MetadataUnit em2mu2 = emCache2.getMetadataUnit("em2");

        assertNotNull(em2mu1);
        assertSame(emmu1, emmu2);

        assertNotSame(emmu1, em2mu1); // that would be too much caching ;-)

        // modifying persistence.xml should cause EntityMappingCache to be invalidated

        PersistenceUnit persistenceUnit = new PersistenceUnit();
        persistenceUnit.setName("em3");

        emCache1.getPUDataObject().addPersistenceUnit(persistenceUnit);
        emCache1.getPUDataObject().save();

        // the EntityMappingsCache should have been invalidated by now
        // it still should be the same instance...

        PersistenceUtils.EntityMappingsCache emCache3 = PersistenceUtils.getEntityMappingsCache(persistenceScope);
        assertSame(emCache1, emCache3);

        // ... and the same MetadataUnit's will be retrieved from it...

        MetadataUnit emmu3 = emCache1.getMetadataUnit("em");
        MetadataUnit emmu4 = emCache1.getMetadataUnit("em");
        assertSame(emmu3, emmu4);
        assertSame(emmu1, emmu3);

        // ... and it will contain a new MetadataUnit for em3

        MetadataUnit em3mu1 = emCache1.getMetadataUnit("em3");
        MetadataUnit em3mu2 = emCache1.getMetadataUnit("em3");
        assertNotNull(em3mu1);
        assertSame(em3mu1, em3mu2);

        // removing persistence.xml should cause EntityMappingCache to be removed from the cache

        emCache1.getPUDataObject().delete();

        PersistenceUtils.EntityMappingsCache emCache4 = PersistenceUtils.getEntityMappingsCache(persistenceScope);
        assertNull(emCache4);
    }

    public void testMetadataUnitContents() throws Exception {
        FileObject persistenceXml = copyFile(dataDir.getFileObject("persistence.xml"), workDir);
        assertNotNull(persistenceXml);
        FileObject ormXml = copyFile(dataDir.getFileObject("orm.xml"), workDir);
        assertNotNull(ormXml);

        ClassPath classPath = ClassPathSupport.createClassPath(Collections.emptyList());
        PersistenceScopeImpl persistenceScopeImpl = new PersistenceScopeImpl(persistenceXml, classPath);
        PersistenceScope persistenceScope = PersistenceScopeFactory.createPersistenceScope(persistenceScopeImpl);

        PersistenceUtils.EntityMappingsCache emCache1 = PersistenceUtils.getEntityMappingsCache(persistenceScope);
        MetadataUnit mu = emCache1.getMetadataUnit("em");

        assertSame(classPath, mu.getClassPath());
        assertSame(ormXml, mu.getDeploymentDescriptor());

        // getDeploymentDescriptor() should be null when there is no orm.xml

        ormXml.delete();
        MetadataUnit mu2 = emCache1.getMetadataUnit("em2");

        assertSame(classPath, mu2.getClassPath());
        assertNull(mu2.getDeploymentDescriptor());
    }

    private static final class PersistenceScopeImpl implements PersistenceScopeImplementation {

        private FileObject persistenceXml;
        private ClassPath classPath;

        public PersistenceScopeImpl(FileObject persistenceXml, ClassPath classPath) {
            this.persistenceXml = persistenceXml;
            this.classPath = classPath;
        }

        public FileObject getPersistenceXml() {
            return persistenceXml;
        }

        public ClassPath getClassPath() {
            return classPath;
        }

        public void setPersistenceXml(FileObject persistenceXml) {
            this.persistenceXml = persistenceXml;
        }

        public void setClassPath(ClassPath classPath) {
            this.classPath = classPath;
        }
    }
}
