/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 *
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.j2ee.metadata;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import org.netbeans.jmi.javamodel.Annotation;
import org.netbeans.jmi.javamodel.AnnotationType;
import org.netbeans.jmi.javamodel.Element;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.NamedElement;

import org.netbeans.jmi.javamodel.TypedElement;
import org.netbeans.modules.schema2beans.BaseBean;
import org.openide.ErrorManager;

/**
 * Annotation listener implementation for ORM Metadata model building.
 *
 * @author Marek Fukala
 */
public abstract class NNModelBuilder implements NNListener {
    private static final ErrorManager LOGGER = ErrorManager.getDefault().getInstance(NNModelBuilder.class.getName());
    private static final boolean LOG = LOGGER.isLoggable(ErrorManager.INFORMATIONAL);
    protected BaseBean ddRoot;
    private Collection supportedNNs;
    public NNModelBuilder(BaseBean ddRoot) {
        this.ddRoot = ddRoot;
    }
    protected static final String JAVA_CLASS_REF = "$JAVACLASS"; //NOI18N
    protected static final String JAVA_CLASS_SIMPLE_REF = "$JAVACLASS_SIMPLE"; //NOI18N
    protected static final String MEMBER_NAME_REF = "$MEMBER"; //NOI18N
    protected static final String MEMBER_CLASS_REF = "$MEMBERCLASS"; //NOI18N
    protected static final String MEMBER_CLASS_SIMPLE_REF = "$MEMBERCLASS_SIMPLE"; //NOI18N
    
    protected static final String JAVA_CLASS_MEMBERNAME_REF = "$JAVA_CLASS-MEMBER"; //NOI18N
    
    protected static final String CUSTOM_ATTRIBUTE_MARKER = "<CUSTOM>";
    //mapping parser symbols
    protected static final String ELEMENTS_DELIMITER = "/"; //NOI18N
    protected static final String ATTRS_BLOCK_DELIMITER = "#"; //NOI18N
    protected static final String ATTRS_DELIMITER = ";"; //NOI18N
    protected static final String ATTRS_VALUE_DELIMITER = "="; //NOI18N
    //marker for annotation's attribute which name doesn't correspond to the s2b bean property name
    protected static final String ATTR_MAPPING_MARKER = "!"; //NOI18N
    //for construction methods which doesn't correspond to the mapping item name
    // like. Embeddable.newAttributes() -> Embeddable.newEmbeddableAttributes()
    protected static final String NEW_ATTR_MAPPING_MARKER = "*"; //NOI18N
    protected static final String ATTRS_KEY_MARK = "$"; //NOI18N
    protected static final String MAPPING_KEY_CONDITION_DELIMITER = "?"; //NOI18N
    /** Returns a Map of annotations class names to schema2beans paths. */
    protected abstract Map<String,String> getAnnotation2ModelMapping();
    public Collection<String> getSupportedAnnotations() {
        synchronized (this) {
            if(supportedNNs == null) {
                supportedNNs = getSupportedAnnotationsFromMapping();
            }
            return supportedNNs;
        }
    }
    public void addClassAnnotation(JavaClass javaClass, Annotation annotation, AnnotationType t) {
        String annotationClass = t.getName();
        String javaClassName = javaClass.getName();
        Map attributes = Utils.getNNAttributes(annotation);
        addNNToModel(annotationClass, javaClassName, null, null, attributes, false);
    }
    public void removeClassAnnotation(JavaClass javaClass, Annotation annotation, AnnotationType t) {
        String annotationClass = t.getName();
        String javaClassName = javaClass.getName();
        removeNNFromModel(annotationClass, javaClassName, null, null, null, false);
    }
    public void addMemberAnnotation(boolean field, JavaClass javaClass, Element member, Annotation annotation, AnnotationType t) {
        String annotationClass = t.getName();
        String javaClassName = javaClass.getName();
        String memberName = ((NamedElement)member).getName();
        String memberType = ((TypedElement)member).getType().getName();
        Map attributes = Utils.getNNAttributes(annotation);
        addNNToModel(annotationClass, javaClassName, memberName, memberType, attributes, field);
    }
    public void removeMemberAnnotation(boolean field, JavaClass javaClass, String memberName, String memberType, Annotation annotation, AnnotationType t) {
        String annotationClass = t.getName();
        String javaClassName = javaClass.getName();
        removeNNFromModel(annotationClass, javaClassName, memberName, memberType, null, field);
    }
    public void addClassAttribute(JavaClass javaClass, Annotation annotation, AnnotationType t, String attributeName, String attributeValue) {
        String annotationClass = t.getName();
        String javaClassName = javaClass.getName();
        updateNNModelAttribute(annotationClass, javaClassName, null, null, attributeName, attributeValue);
    }
    public void removeClassAttribute(JavaClass javaClass, Annotation annotation, AnnotationType t, String attributeName) {
        String annotationClass = t.getName();
        String javaClassName = javaClass.getName();
        updateNNModelAttribute(annotationClass, javaClassName, null, null, attributeName, null);
    }
    public void addMemberAttribute(JavaClass javaClass, Element member, Annotation annotation, AnnotationType t, String attributeName, String attributeValue) {
        String annotationClass = t.getName();
        String javaClassName = javaClass.getName();
        String memberType = ((TypedElement)member).getType().getName();
        updateNNModelAttribute(annotationClass, javaClassName, ((NamedElement)member).getName(), memberType, attributeName, attributeValue);
    }
    public void removeMemberAttribute(JavaClass javaClass, Element member, Annotation annotation, AnnotationType t, String attributeName) {
        String annotationClass = t.getName();
        String javaClassName = javaClass.getName();
        String memberType = ((TypedElement)member).getType().getName();
        updateNNModelAttribute(annotationClass, javaClassName, ((NamedElement)member).getName(), memberType, attributeName, null);
    }
    /**
     * the mapping syntax:
     * Entity#class2=$JAVACLASS;name=value;name2=value2/OneToOne#name=$MEMBER;!name=ejbName
     */
    private void addNNToModel(String annotationName, String javaClass, String memberName, String memberClass, Map attributes, boolean field) {
        NNMappingTokenizer mt = null;
        try {
            mt = findMapping(annotationName, javaClass, memberName, memberClass);
            if(mt == null) {
                LOGGER.log(ErrorManager.INFORMATIONAL, "No suitable mapping found for " + annotationName + " in java class " + javaClass + " for member " + memberName + " of type " + memberClass);
                return ;
            }
            BaseBean parentBean = ddRoot;
            //the first bean which doesn't exist and is created in the mapping sequence is stored here.
            //we need to first create all descendants and then add the first new bean to its parent
            BaseBean firstNewBean = null;
            BaseBean firstNewBeanParent = null;
            String firstNewBeanName = null;
            Class firstNewBeanClass = null;
            while(mt.hasMoreElements()) {
                NNMappingItem nnmi = mt.nextElement();
                
                //re-sets the refAttrName to the annotation attribute value if there is any
                if(nnmi.getRefAttrName() != null) { //can it be null???
                    if(nnmi.getAttrNamesMapping().containsValue(nnmi.getRefAttrName())) {
                        for(Object key : nnmi.getAttrNamesMapping().keySet()) {
                            String _key = (String)key;
                            String val = (String)nnmi.getAttrNamesMapping().get(_key);
                            if(nnmi.getRefAttrName().equals(val)) {
                                String attrv = (String)attributes.get(_key);
                                if(attrv != null) {
                                    nnmi.setRefAttrValue(attrv);
                                }
                            }
                        }
                    }
                }
                
                String beanName = nnmi.getName();
                //get all beans of the type
                Class beanClass = identifyBeanType(parentBean, beanName);
                if(beanClass != null) {
                    if(String.class.isAssignableFrom(beanClass)) {
                        //the bean property is just a string
                        //how to handle???
                    } else {
                        BaseBean newBean = createBean(parentBean, beanName, nnmi.getNewAttrNamesMapping());
                        
                        //set attributes from the mapping
                        setBeanProperties(newBean, nnmi.getAttrs(), Collections.EMPTY_MAP);
                        
                        //set reference attributes to the bean
                        if(nnmi.getRefAttrName() != null) {
                            setBeanProperty(newBean, nnmi.getRefAttrName(), String.class, nnmi.getRefAttrValue(), Collections.EMPTY_MAP);
                        }
                        
                        //if the mapping element is last in the path set all attributes from annotation to the bean
                        if(!mt.hasMoreElements()) {
                            setBeanProperties(newBean, attributes, nnmi.getAttrNamesMapping());
                        }
                        Object beanObj = findBeans(parentBean, beanName);
                        if(beanObj != null && !(beanObj instanceof BaseBean || beanObj instanceof BaseBean[])) {
                            throw new IllegalStateException("Found unsupported bean (not instanceof BaseBean or BaseBean[]; type=" + beanObj.getClass().getName());
                        }
                        if(!beanClass.isArray()) {
                            //not an array - it should be an instance of BaseBean - do the nasty cast
                            if(beanObj == null) {
                                if(mt.hasMoreElements() && nnmi.getRefAttrName() != null) { //little of woodo magic - it seems that if the item doesn't have any attributes we can create it - it doesn't change the semantic info.
                                    //path element is missing => ignore this mapping
                                    LOGGER.log(ErrorManager.INFORMATIONAL, "path element for " + nnmi.getName() + " is missing => skipping.");
                                    return ;
                                }
                                if(firstNewBean == null) {
                                    //first new bean in the mapping sequence - just store it and add it to its parent at the end
                                    firstNewBean = newBean;
                                    firstNewBeanParent = parentBean;
                                    firstNewBeanClass = beanClass;
                                    firstNewBeanName = beanName;
                                } else {
                                    //not first new bean => physically add the bean to the tree
                                    setBeanProperty(parentBean, beanName, beanClass, newBean, Collections.EMPTY_MAP);
                                }
                                parentBean = newBean;
                            } else {
                                parentBean = (BaseBean)beanObj;
                                
                                //if the bean is the last one in the path, set annotation attributes to it
                                if(!mt.hasMoreElements()) {
                                    setBeanProperties((BaseBean)beanObj, attributes, nnmi.getAttrNamesMapping());
                                }
                            }
                        } else {
                            //an array
                            BaseBean[] beans = (BaseBean[])beanObj;
                            if(beans == null || beans.length == 0) {
                                if(mt.hasMoreElements() && nnmi.getRefAttrName() != null) { //little of woodo magic - it seems that if the item doesn't have any attributes we can create it - it doesn't change the semantic info.
                                    //path element is missing => ignore this mapping
                                    LOGGER.log(ErrorManager.INFORMATIONAL, "path element for " + nnmi.getName() + " is missing => skipping.");
                                    return ;
                                }
                                if(firstNewBean == null) {
                                    //first new bean in the mapping sequence - just store it and add it to its parent at the end
                                    firstNewBean = newBean;
                                    firstNewBeanParent = parentBean;
                                    firstNewBeanClass = beanClass;
                                    firstNewBeanName = beanName;
                                } else {
                                    //not first new bean
                                    addBeanArrayProperty(parentBean, beanName, beanClass.getComponentType(), newBean);
                                }
                                parentBean = newBean;
                            } else {
                                //there are some beans - try to find the right one
                                BaseBean matchingBean = findBeanInArray(beans, nnmi.getRefAttrName(), nnmi.getRefAttrValue());
                                if(matchingBean != null) {
                                    parentBean = matchingBean;
                                    //if the bean is the last one in the path, set annotation attributes to it
                                    if(!mt.hasMoreElements()) {
                                        setBeanProperties((BaseBean)matchingBean, attributes, nnmi.getAttrNamesMapping());
                                    }
                                } else {
                                    if(mt.hasMoreElements() && nnmi.getRefAttrName() != null) { //little of woodo magic - it seems that if the item doesn't have any attributes we can create it - it doesn't change the semantic info.
                                        //path element is missing => ignore this mapping
                                        LOGGER.log(ErrorManager.INFORMATIONAL, "path element for " + nnmi.getName() + " is missing => skipping.");
                                        return ;
                                    }
                                    //we need to create a new bean
                                    if(firstNewBean == null) {
                                        //first new bean in the mapping sequence - just store it and add it to its parent at the end
                                        firstNewBean = newBean;
                                        firstNewBeanParent = parentBean;
                                        firstNewBeanClass = beanClass;
                                        firstNewBeanName = beanName;
                                    } else {
                                        //not first new bean
                                        addBeanArrayProperty(parentBean, beanName, beanClass.getComponentType(), newBean);
                                    }
                                    parentBean = newBean;
                                }
                            }
                        }
                    }
                }
            }
            if(firstNewBean != null) {
                //set the created s2b tree piece to the existing tree
                if(firstNewBeanClass.isArray()) {
                    addBeanArrayProperty(firstNewBeanParent, firstNewBeanName, firstNewBeanClass.getComponentType(), firstNewBean);
                } else {
                    setBeanProperty(firstNewBeanParent, firstNewBeanName, firstNewBeanClass, firstNewBean, Collections.EMPTY_MAP);
                }
            }
            //>>>debug
            if(memberName == null) {
                LOGGER.log(ErrorManager.INFORMATIONAL, "+++ added class annotation " + annotationName + " into class " + javaClass);
            } else {
                LOGGER.log(ErrorManager.INFORMATIONAL, "+ added member annotation " + annotationName + " to member " + memberName +  " in class " + javaClass);
            }
            if(LOG) {
                ddRoot.dumpXml();
            }
        }catch(Exception iae) {
            //a problem occured during the mapping processing => print some context information
            String eMsg = "NNModelBuilder.addNNToModel(): An error occured during processing annotation event ";
            eMsg += "[annotation='" + annotationName + "', javaClass='" + javaClass + "', memberName='" + memberName + "', memberClass='" + memberClass + "', field?=" + field + "] ";
            eMsg += mt == null ? "(mapping not found)" : mt.toString();
            
            ErrorManager.getDefault().log(ErrorManager.INFORMATIONAL, eMsg);
            ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, iae);
        }
    }
    
    /** Adds/removes annotations attributes.*/
    private void updateNNModelAttribute(String annotationName, String javaClass, String memberName, String memberClass, String attributeName, String attributeValue) {
        NNMappingTokenizer mt = null;
        try {
            mt = findMapping(annotationName, javaClass, memberName, memberClass);
            if(mt == null) {
                LOGGER.log(ErrorManager.INFORMATIONAL, "No suitable mapping found for " + annotationName + " in java class " + javaClass + " for member " + memberName + " of type " + memberClass);
                return ;
            }
            BaseBean parentBean = ddRoot;
            while(mt.hasMoreElements()) {
                NNMappingItem nnmi = mt.nextElement();
                
                //re-sets the refAttrName to the annotation attribute value if there is any
                if(nnmi.getRefAttrName() != null) { //can it be null???
                    for(Object key : nnmi.getAttrNamesMapping().keySet()) {
                        String _key = (String)key;
                        String val = (String)nnmi.getAttrNamesMapping().get(_key);
                        if(nnmi.getRefAttrName().equals(val)) {
                            if(attributeName.equals(_key)) {
                                if(attributeValue != null) {
                                    nnmi.setRefAttrValue(attributeValue);
                                }
                            }
                        }
                    }
                }
                
                String beanName = nnmi.getName();
                Object beanObj = findBeans(parentBean, beanName);
                if(beanObj != null) {
                    if(beanObj instanceof BaseBean[]) {
                        //array of beans
                        BaseBean[] beans = (BaseBean[])beanObj;
                        BaseBean matchingBean = null;
                        if(beans != null && beans.length > 0) {
                            //there are some beans - try to find the right one
                            matchingBean = findBeanInArray(beans, nnmi.getRefAttrName(), nnmi.getRefAttrValue());
                            if(matchingBean != null) {
                                if(mt.hasMoreElements()) {
                                    //not last element - go on with traversing
                                    parentBean = matchingBean;
                                } else {
                                    //last element - remove the attribute from it
                                    //XXX this is probably wrong - the attribute can be of arbitrary type, not just string!
                                    setBeanProperty(matchingBean, attributeName, String.class, attributeValue, nnmi.getAttrNamesMapping());
                                }
                            } else {
                                //nothing to delete
                                break;
                            }
                        }
                    } else if(beanObj instanceof BaseBean) {
                        if(mt.hasMoreElements()) {
                            //not last element - go on with traversing
                            parentBean = (BaseBean)beanObj;
                        } else {
                            //last element - delete
                            //XXX this is probably wrong - the attribute can be of arbitrary type, not just string!
                            setBeanProperty((BaseBean)beanObj, attributeName, String.class, attributeValue, nnmi.getAttrNamesMapping());
                        }
                    }
                }
            }
            //>>>debug
            if(memberName == null) {
                LOGGER.log(ErrorManager.INFORMATIONAL, "* set attribute " + attributeName + " to value '" + attributeValue + "' of class annotation " + annotationName + " from class " + javaClass);
            } else {
                LOGGER.log(ErrorManager.INFORMATIONAL, "* set attribute " + attributeName + " to value '" + attributeValue + "' of member annotation " + annotationName + " for member " + memberName + " from class " + javaClass);
            }
            if(LOG) {
                ddRoot.dumpXml();
            }
        }catch(Exception iae) {
            //a problem occured during the mapping processing => print some context information
            String eMsg = "NNModelBuilder.updateNNModelAttribute(): An error occured during processing annotation event ";
            eMsg += "[annotation='" + annotationName + "', javaClass='" + javaClass + "', memberName='" + memberName + "', memberClass='" + memberClass + "', attributeName='" + attributeName+ "', attributeValue='" + attributeValue + "'] ";
            eMsg += mt == null ? "(mapping not found)" : mt.toString();
            
            ErrorManager.getDefault().log(ErrorManager.INFORMATIONAL, eMsg);
            ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, iae);
        }
    }
    
    
    private void removeNNFromModel(String annotationName, String javaClass, String memberName, String memberClass, Map attributes, boolean field) {
        NNMappingTokenizer mt = null;
        try {
            mt = findMapping(annotationName, javaClass, memberName, memberClass);
            if(mt == null) {
                LOGGER.log(ErrorManager.INFORMATIONAL, "No suitable mapping found for " + annotationName + " in java class " + javaClass + " for member " + memberName + " of type " + memberClass);
                return ;
            }
            BaseBean parentBean = ddRoot;
            while(mt.hasMoreElements()) {
                NNMappingItem nnmi = mt.nextElement();
                String beanName = nnmi.getName();
                Object beanObj = findBeans(parentBean, beanName);
                if(beanObj != null) {
                    if(beanObj instanceof BaseBean[]) {
                        //array of beans
                        BaseBean[] beans = (BaseBean[])beanObj;
                        BaseBean matchingBean = null;
                        if(beans != null && beans.length > 0) {
                            //there are some beans - try to find the right one
                            matchingBean = findBeanInArray(beans, nnmi.getRefAttrName(), nnmi.getRefAttrValue());
                            if(matchingBean != null) {
                                if(mt.hasMoreElements()) {
                                    //not last element - go on with traversing
                                    parentBean = matchingBean;
                                } else {
                                    //last element - remove the element from the beans array
                                    Class beanClass = identifyBeanType(parentBean, beanName);
                                    if(beanClass.isArray()) beanClass = beanClass.getComponentType();
                                    removeBeanArrayProperty(parentBean, beanName, matchingBean, beanClass);
                                }
                            } else {
                                //nothing to delete
                                LOGGER.log(ErrorManager.INFORMATIONAL, "path element " + beanName + " not found -> ignoring.");
                                return ;
                            }
                        } else {
                            //either the element to remove or one of it's ancestors are missing -> ignore
                            LOGGER.log(ErrorManager.INFORMATIONAL, "path element " + beanName + " not found -> ignoring.");
                            return ;
                        }
                    } else if(beanObj instanceof BaseBean) {
                        if(mt.hasMoreElements()) {
                            //not last element - go on with traversing
                            parentBean = (BaseBean)beanObj;
                        } else {
                            //last element - delete
                            Class beanClass = identifyBeanType(parentBean, beanName);
                            if(beanClass.isArray()) beanClass = beanClass.getComponentType();
                            nullizeBeanProperty(parentBean, beanName, beanClass);
                        }
                    } else if(beanObj instanceof String) {
                        nullizeBeanProperty(parentBean, beanName, String.class);
                    }
                } else {
                    //either the element to remove or one of it's ancestors are missing -> ignore
                    LOGGER.log(ErrorManager.INFORMATIONAL, "path element " + beanName + " not found -> ignoring.");
                    return ;
                }
            }
            //>>>debug
            if(memberName == null) {
                LOGGER.log(ErrorManager.INFORMATIONAL, "--- removed class annotation " + annotationName + " from class " + javaClass);
            } else {
                LOGGER.log(ErrorManager.INFORMATIONAL, "- removed member annotation " + annotationName + " from member " + memberName +  " in class " + javaClass);
            }
            if(LOG) {
                ddRoot.dumpXml();
            }
        }catch(Exception iae) {
            //a problem occured during the mapping processing => print some context information
            String eMsg = "NNModelBuilder.removeNNFromModel(): An error occured during processing annotation event ";
            eMsg += "[annotation='" + annotationName + "', javaClass='" + javaClass + "', memberName='" + memberName + "', memberClass='" + memberClass + "', field?=" + field + "] ";
            eMsg += mt == null ? "(mapping not found)" : mt.toString();
            
            ErrorManager.getDefault().log(ErrorManager.INFORMATIONAL, eMsg);
            ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, iae);
        }
    }
    private boolean verifyModelMapping(NNMappingTokenizer mt) {
        BaseBean parentBean = ddRoot;
        while(mt.hasMoreElements()) {
            NNMappingItem nnmi = mt.nextElement();
            String beanName = nnmi.getName();
            Object beanObj = findBeans(parentBean, beanName);
            if(beanObj != null) {
                if(beanObj instanceof BaseBean[]) {
                    //array of beans
                    BaseBean[] beans = (BaseBean[])beanObj;
                    BaseBean matchingBean = null;
                    if(beans != null && beans.length > 0) {
                        //there are some beans - try to find the right one
                        matchingBean = findBeanInArray(beans, nnmi.getRefAttrName(), nnmi.getRefAttrValue());
                        if(matchingBean != null) {
                            if(mt.hasMoreElements()) {
                                //not last element - go on with traversing
                                parentBean = matchingBean;
                            } else {
                                //last element found
                                return true;
                            }
                        } else {
                            //nothing found
                            return false;
                        }
                    }
                } else if(beanObj instanceof BaseBean) {
                    if(mt.hasMoreElements()) {
                        //not last element - go on with traversing
                        parentBean = (BaseBean)beanObj;
                    } else {
                        //last element
                        return true;
                    }
                }
                //                } else if(beanObj instanceof String) {
                //                    nullizeBeanProperty(parentBean, beanName, String.class);
                //                }
            } else {
                //if any of the beans in the path doesn't exist return false (condition test failed)
                break;
            }
        }
        return false;
    }
    private BaseBean findBeanInArray(BaseBean[] beans, String refAttrName, String refAttrValue) {
        if(refAttrName == null || refAttrValue == null) {
            String msg = "Looking for bean in array of BaseBeans[";
            for(BaseBean bb : beans) {
                msg += bb.toString();
                msg += ", ";
            }
            msg = msg.substring(0, msg.length() - 2);
            msg += "]: arguments (refAttrName = " + refAttrName + ", refAttrValue = " + refAttrValue + " cannot be null.";
            
            throw new IllegalArgumentException(msg);
        }
        for(int i = 0; i < beans.length; i++) {
            BaseBean b = beans[i];
            if(hasProperty(b, refAttrName)) {
                String value = (String)findBeans(b, refAttrName);
                if(refAttrValue.equals(value)) {
                    return b;
                }
            }
        }
        return null;
    }
    
    /** This method sets properties specified in attributes map into the bean.
     * <b>Only string properties of the bean can be set! If the bean set-property
     * has different argument type than j.l.String it won't be set!</b>
     */
    private static void setBeanProperties(BaseBean bean, Map attributes, Map attrNamesMap) {
        Class c = bean.getClass();
        Method setter = null;
        try {
            for(Iterator i = attributes.keySet().iterator(); i.hasNext();) {
                String attrName = (String)i.next();
                String attrValue = (String)attributes.get(attrName);
                //test if the attribute name should be translated according to attrNamesMap
                if(attrNamesMap.containsKey(attrName)) {
                    attrName = (String)attrNamesMap.get(attrName);
                    //test if the attribute value doesn't state
                    //that the attribute is handled in custom code ( <CUSTOM> )
                    if(CUSTOM_ATTRIBUTE_MARKER.equals(attrName)) {
                        //ignore this attribute
                        continue;
                    }
                }
                setter = c.getMethod("set" + propertize(attrName), new Class[]{String.class}); //NOI18N
                setter.invoke(bean, new Object[]{attrValue});
            }
        } catch (NoSuchMethodException nsme) {
            //bean setter method not found - probably a wrong parameter type - ignore if not logging enabled
            if(LOG) {
                throw new IllegalStateException("The '" + bean + "' property doesn't exist", nsme);
            }
        } catch (Exception e) {
            throw new RuntimeException(
                    "Cannot set attributes of bean " + bean, e);
        }
    }
    private static void addBeanArrayProperty(BaseBean bean, String propertyName, Class valueType, Object value) {
        Class c = bean.getClass();
        Method setter;
        Object result;
        try {
            setter = c.getMethod("add" + propertize(propertyName), new Class[]{valueType}); //NOI18N
            setter.invoke(bean, new Object[]{value});
        } catch (NoSuchMethodException nsme) {
            throw new IllegalStateException("The bean property doesn't exist!", nsme);
        } catch (Exception e) {
            throw new RuntimeException(
                    "Cannot call add method for " + propertyName + " of bean " + bean + " to value " + value, e);
        }
    }
    private static void removeBeanArrayProperty(BaseBean bean, String propertyName, BaseBean itemToRemove, Class itemToRemoveType) {
        Class c = bean.getClass();
        Method setter;
        Object result;
        try {
            setter = c.getMethod("remove" + propertize(propertyName), new Class[]{itemToRemoveType}); //NOI18N
            setter.invoke(bean, new Object[]{itemToRemove});
        } catch (Exception e) {
            throw new RuntimeException(
                    "Cannot call remove method for " + propertyName + " of bean " + bean + " to value " + itemToRemove, e);
        }
    }
    private static void nullizeBeanProperty(BaseBean bean, String propertyName, Class propertyType) {
        Class c = bean.getClass();
        Method setter;
        Object result;
        try {
            setter = c.getMethod("set" + propertize(propertyName), new Class[]{propertyType}); //NOI18N
            setter.invoke(bean, new Object[]{null});
        } catch (Exception e) {
            throw new RuntimeException(
                    "Cannot nullize property " + propertyName + " of bean " + bean, e);
        }
    }
    private static void setBeanProperty(BaseBean bean, String propertyName, Class valueType, Object value, Map attrNamesMap) {
        Class c = bean.getClass();
        Method setter = null;
        Object result;
        try {
            //Class valueType = value != null ? value.getClass() : String.class; //hack!!!
            //test if the attribute name should be translated according to attrNamesMap
            if(attrNamesMap.containsKey(propertyName)) {
                propertyName = (String)attrNamesMap.get(propertyName);
            }
            propertyName = propertize(propertyName);
            setter = c.getMethod("set" + propertize(propertyName), new Class[]{valueType}); //NOI18N
            setter.invoke(bean, new Object[]{value});
        } catch (NoSuchMethodException nsme) {
            throw new IllegalStateException("The bean property doesn't exist (bean="+bean+")", nsme);
        } catch (Exception e) {
            throw new RuntimeException(
                    "Cannot set property " + propertyName + " of bean " + bean + " to value " + value, e);
        }
    }
    private static BaseBean createBean(BaseBean parent, String beanName, Map newAttrsNamesMap) {
        Class c = parent.getClass();
        Method getter;
        Object result;
        try {
            if(newAttrsNamesMap.containsKey(beanName)) {
                beanName = (String)newAttrsNamesMap.get(beanName);
            }
            getter = c.getMethod("new" + propertize(beanName), new Class[]{}); //NOI18N
            result = getter.invoke(parent, new Object[]{});
            if (result != null && result instanceof BaseBean) {
                return (BaseBean)result;
            } else {
                return null;
            }
        } catch (Exception e) {
            throw new RuntimeException(
                    "parent = " + parent + ", beanProperty = " + beanName, e);
        }
    }
    private static boolean hasProperty(BaseBean bean, String propertyName) {
        try {
            Class c = bean.getClass();
            Method getter = c.getMethod("get" + propertize(propertyName), new Class[]{}); //NOI18N
            return true;
        } catch (NoSuchMethodException nsme) {
            return false;
        } catch (Exception e) {
            throw new RuntimeException(
                    "cannot test presence of property " + propertyName + " of bean " + bean, e);
        }
    }
    private static Object findBeans(BaseBean parent, String beanProperty) {
        Class c = parent.getClass();
        Method getter;
        Object result;
        try {
            getter = c.getMethod("get" + propertize(beanProperty), new Class[]{}); //NOI18N
            return getter.invoke(parent, new Object[]{});
        } catch (Exception e) {
            throw new RuntimeException(
                    "parent = " + parent + ", beanProperty = " + beanProperty, e);
        }
    }
    private static Class identifyBeanType(BaseBean bean, String beanProperty) {
        Class c = bean.getClass();
        Method getter;
        Object result;
        try {
            getter = c.getMethod("get" + propertize(beanProperty), new Class[]{}); //NOI18N
            return getter.getReturnType();
        } catch (Exception e) {
            throw new RuntimeException(
                    "Cannot identify type of " + beanProperty + " of bean " + bean, e);
        }
    }
    private static BaseBean findBeanByName(BaseBean parent, String beanProperty, String nameProperty, String value) {
        BaseBean[] beans = (BaseBean[])findBeans(parent, beanProperty);
        try {
            for (int i=0;i<beans.length;i++) {
                Class c1 = beans[i].getClass();
                Method getter1;
                Object result1;
                getter1 = c1.getMethod("get" + propertize(nameProperty), new Class[]{}); //NOI18N
                result1 = getter1.invoke(beans[i], new Object[]{});
                if (result1 instanceof String) {
                    if (value.equals((String)result1)) {
                        return beans[i];
                    }
                }
            }
            return null;
        } catch (Exception e) {
            throw new RuntimeException(
                    "parent = " + parent + ", beanProperty = " + beanProperty +
                    ", nameProperty = " + nameProperty + ", value = " + value,
                    e);
        }
    }
    private static String propertize(String propertyName) {
        return Character.toUpperCase(propertyName.charAt(0)) + propertyName.substring(1);
    }
    
    //NOTE: there should not be multiple mapping keys for one annotation where one key
    //specifies CLASSNAME and/or MEMBERNAME and another one not!!! ***
    //TODO: We need some effecient caching of the results returned from this method based on arguments since it is called quite often
    private NNMappingTokenizer findMapping(String annotationName, String javaClassName, String memberName, String memberClass) {
        Iterator mappingKeys = getAnnotation2ModelMapping().keySet().iterator();
        //iterate all mapping keys and try to find a proper one
        while(mappingKeys.hasNext()) {
            String mkey = (String)mappingKeys.next();
            NNMappingKeyParser nnmkp = new NNMappingKeyParser(mkey);
            if(nnmkp.getAnnotationName().equals(annotationName)) {
                //test if there is/are a condition/s defined in the key
                List<String> conditionsList = nnmkp.getMappingConditions();
                if(!conditionsList.isEmpty()) {
                    Iterator<String> condsItr = conditionsList.iterator();
                    boolean allConditionsPasses = true;
                    //there are some conds so evaluate them
                    while(condsItr.hasNext()) {
                        //if(DEBUG) System.out.println("testing condtions for mapping key " + mkey);
                        String condition = condsItr.next();
                        //test if the condition is true
                        boolean pass = verifyModelMapping(new NNMappingTokenizer(condition, javaClassName, memberName, memberClass));
                        if(pass) {
                            //if(DEBUG) System.out.println("condition " + condition + " passes.");
                        } else {
                            //if(DEBUG) System.out.println("condition " + condition + " failed.");
                            allConditionsPasses = false;
                            break;
                        }
                    }
                    if(!allConditionsPasses) continue; //take next key
                }
                
                boolean passes = true;
                
                //test whether the key mapping contains a javaclass or member class conditions
                String javaCR = nnmkp.getPropertyValue(JAVA_CLASS_REF);
                String memberCR = nnmkp.getPropertyValue(MEMBER_CLASS_REF);
                String mappingValue = getAnnotation2ModelMapping().get(mkey);
                
                //now we need to test whether the specified javac class and member class are correspond from the class/member which the annotation comes from
                if(javaCR != null && !javaCR.equals(javaClassName)) {
                    passes = false;
                }
                
                if(memberCR != null && !memberCR.equals(memberClass)) {
                    passes = false;
                }
                
                String javaCRsimple = nnmkp.getPropertyValue(JAVA_CLASS_SIMPLE_REF);
                String memberCRsimple = nnmkp.getPropertyValue(MEMBER_CLASS_SIMPLE_REF);
                
                //now we need to test whether the specified javac class and member class are correspond from the class/member which the annotation comes from
                if(javaCRsimple != null && !javaCRsimple.equals(getSimpleClassName(javaClassName))) {
                    passes = false;
                }
                if(memberCRsimple != null && !memberCR.equals(getSimpleClassName(memberClass))) {
                    passes = false;
                }
                
                //                String javaCRmember = nnmkp.getPropertyValue(JAVA_CLASS_MEMBERNAME_REF);
                //                if(javaCRmember != null) {
                //                    int delimIndex = javaCRmember.indexOf('/');
                //                    String javaClass = javaCRmember.substring(0, delimIndex);
                //                    String member = javaCRmember.substring(delimIndex + 1);
                //                    if(!(javaClassName.equals(javaClass) && memberName.equals(member))) {
                //                        passes = false;
                //                    }
                //                }
                
                if(passes) {
                    return new NNMappingTokenizer(mappingValue, javaClassName, memberName, memberClass);
                }
            }
        }
        //no suitable mapping found
        return null;
    }
    private String getSimpleClassName(String fullyQualifiedClassName) {
        int lastDotIndex = fullyQualifiedClassName.lastIndexOf('.');
        if(lastDotIndex != -1) {
            return fullyQualifiedClassName.substring(lastDotIndex + 1);
        } else {
            return fullyQualifiedClassName;
        }
    }
    //note: there should not be multiple mapping keys for one annotation where one key
    //specifies CLASSNAME and/or MEMBERNAME and another one not!!! ***
    private Collection getSupportedAnnotationsFromMapping() {
        Iterator mappingKeys = getAnnotation2ModelMapping().keySet().iterator();
        HashSet supportedNNs = new HashSet();
        //iterate all mapping keys and try to find a proper one
        while(mappingKeys.hasNext()) {
            String mkey = (String)mappingKeys.next();
            NNMappingKeyParser nnmkp = new NNMappingKeyParser(mkey);
            supportedNNs.add(nnmkp.getAnnotationName());
        }
        return Collections.unmodifiableCollection(supportedNNs);
    }
    //nn2ModelMapping.put("javax.common.Resource#$JAVACLASS=...;$MEMBERCLASS=...","...");
    private class NNMappingKeyParser {
        private String nnmkey, nnname = null;
        private HashMap<String,String> properties = new HashMap();
        private ArrayList<String> mappingConditions = null;
        public NNMappingKeyParser(String nnmkey) {
            parse(nnmkey);
        }
        public String getAnnotationName() {
            return nnname;
        }
        /** returns a string like condition which has to be true if the mapping should be used */
        public List<String> getMappingConditions() {
            return this.mappingConditions;
        }
        public String getPropertyValue(String propertyName) {
            return properties.get(propertyName);
        }
        private void parse(String nnmk) {
            //first find out whether the mapping key contains a condition/s
            StringTokenizer condt = new StringTokenizer(nnmk, MAPPING_KEY_CONDITION_DELIMITER);
            mappingConditions = new ArrayList();
            String firstPart = null;
            while(condt.hasMoreTokens()) {
                if(firstPart != null) {
                    mappingConditions.add(condt.nextToken());
                } else {
                    firstPart = condt.nextToken();
                }
            }
            StringTokenizer nnparam = new StringTokenizer(firstPart, ATTRS_BLOCK_DELIMITER);
            while(nnparam.hasMoreTokens()) {
                //the first token must be the nn name
                if(nnname == null) {
                    nnname = nnparam.nextToken();
                } else {
                    //properties
                    StringTokenizer params = new StringTokenizer(nnparam.nextToken(), ATTRS_DELIMITER);
                    while(params.hasMoreTokens()) {
                        String parampair = params.nextToken();
                        int idx = parampair.indexOf(ATTRS_VALUE_DELIMITER);
                        if(idx != -1) {
                            String key = parampair.substring(0, idx);
                            String value = parampair.substring(idx + 1);
                            if(!key.startsWith(ATTRS_KEY_MARK)) {
                                //add the $
                                key = "$" + key;
                            }
                            properties.put(key, value);
                        } else {
                            throw new IllegalStateException("Bad mapping key: " + nnmkey);
                        }
                    }
                }
            }
            if(nnname == null) {
                throw new IllegalStateException("Bad mapping key: " + nnmkey);
            }
        }
    }
    private class NNMappingTokenizer {
        private String mapping;
        private StringTokenizer baseTokenizer;
        private String javaClassName, memberName, memberClass;
        public NNMappingTokenizer(String mapping, String javaClassName, String memberName, String memberClass) {
            this.mapping = mapping;
            this.javaClassName = javaClassName;
            this.memberName = memberName;
            this.baseTokenizer = new StringTokenizer(mapping, ELEMENTS_DELIMITER);
            this.memberClass = memberClass;
        }
        public boolean hasMoreElements() {
            return baseTokenizer.hasMoreTokens();
        }
        //    Entity/Table#name=hahaha
        //doesn't check syntax errors in the mapping -> in this case it will throw NoSuchElement or sg. like that
        public NNMappingItem nextElement() {
            if(hasMoreElements()) {
                //parse the element
                String element = baseTokenizer.nextToken(); //Table#name=value
                StringTokenizer attrsBlockTokenizer = new StringTokenizer(element, ATTRS_BLOCK_DELIMITER);
                HashMap attrsMap = new HashMap();
                HashMap attrNamesMapping = new HashMap();
                HashMap newAttrNamesMapping = new HashMap();
                String elementName = attrsBlockTokenizer.nextToken(); //Entity
                String refName = null;
                String refValue = null;
                if(attrsBlockTokenizer.hasMoreTokens()) {
                    String attrpair = attrsBlockTokenizer.nextToken(); //name=value;type=value2
                    StringTokenizer attrsTokenizer = new StringTokenizer(attrpair, ATTRS_DELIMITER);
                    while(attrsTokenizer.hasMoreTokens()) {
                        //name=value
                        String attrNameValue = attrsTokenizer.nextToken();
                        StringTokenizer nameValueTokenizer = new StringTokenizer(attrNameValue, ATTRS_VALUE_DELIMITER);
                        String name = nameValueTokenizer.nextToken();
                        String value = nameValueTokenizer.nextToken();
                        
                        if(MEMBER_CLASS_SIMPLE_REF.equals(value)) {
                            refName = name;
                            value = getSimpleClassName(memberClass);
                            refValue = value;
                        } else if(MEMBER_CLASS_REF.equals(value)) {
                            refName = name;
                            value = memberClass;
                            refValue = value;
                        } else if(JAVA_CLASS_REF.equals(value)) {
                            refName = name;
                            refValue = javaClassName;
                        } else if(JAVA_CLASS_SIMPLE_REF.equals(value)) {
                            refName = name;
                            refValue = getSimpleClassName(javaClassName);
                        } else if(MEMBER_NAME_REF.equals(value)) {
                            refName = name;
                            refValue = memberName;
                        } else if(JAVA_CLASS_MEMBERNAME_REF.equals(value)) {
                            refName = name;
                            value = javaClassName + "/" + memberName;
                            refValue = value;
                        } else {
                            if(name.startsWith(ATTR_MAPPING_MARKER)) {
                                //it is a special attribute name mapping
                                attrNamesMapping.put(name.substring(1), value);
                            } else if(name.startsWith(NEW_ATTR_MAPPING_MARKER)) {
                                newAttrNamesMapping.put(name.substring(1), value);
                            } else {
                                attrsMap.put(name, value);
                            }
                        }
                    }
                    
                }
                //if(DEBUG) System.out.println("returning NNMappingItem(" + elementName + ", refName=" + refName + ", refValue=" + refValue + ", " + attrsMap + ", " + attrNamesMapping);
                return new NNMappingItem(elementName, refName, refValue, attrsMap, attrNamesMapping, newAttrNamesMapping);
            } else {
                //no more items
                return null;
            }
        }
        
        public String toString() {
            return "NNMappingTokenizer[mapping=" + mapping + "]";
        }
        
    }
    
    private class NNMappingItem {
        private String name, refAttrName, refAttrValue;
        private Map attrs;
        private Map attrNamesMapping;
        private Map newAttrNamesMapping;
        public NNMappingItem(String name, String refAttrName, String refAttrValue, Map attrs, Map attrNamesMapping, Map newAttrNamesMapping) {
            this.name = name;
            this.attrs = attrs;
            this.refAttrName = refAttrName;
            this.refAttrValue = refAttrValue;
            this.attrNamesMapping = attrNamesMapping;
            this.newAttrNamesMapping = newAttrNamesMapping;
        }
        public String getName() {
            return name;
        }
        public Map getAttrs() {
            return attrs;
        }
        public Map getAttrNamesMapping() {
            return attrNamesMapping;
        }
        //for construction methods which doesn't correspond to the mapping item name
        // like. Embeddable.newAttributes() -> Embeddable.newEmbeddableAttributes()
        public Map getNewAttrNamesMapping() {
            return newAttrNamesMapping;
        }
        public String getRefAttrName() {
            return refAttrName;
        }
        public String getRefAttrValue() {
            return refAttrValue;
        }
        public void setRefAttrValue(String val) {
            this.refAttrValue = val;
        }
        
        public String toString() {
            return "NNMappingItem(name=" + getName() +
                    "; refAttrName=" + getRefAttrName() +
                    "; refAttrValue=" + getRefAttrValue() +
                    ")";
        }
    }
}

