/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.j2ee.metadata;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.jmi.reflect.RefFeatured;
import javax.jmi.reflect.RefObject;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.api.mdr.events.AssociationEvent;
import org.netbeans.api.mdr.events.AttributeEvent;
import org.netbeans.api.mdr.events.MDRChangeEvent;
import org.netbeans.api.mdr.events.MDRChangeSource;
import org.netbeans.api.mdr.events.MDRPreChangeListener;
import org.netbeans.api.mdr.events.VetoChangeException;
import org.netbeans.jmi.javamodel.AnnotableElement;
import org.netbeans.jmi.javamodel.Annotation;
import org.netbeans.jmi.javamodel.AnnotationType;
import org.netbeans.jmi.javamodel.AttributeValue;
import org.netbeans.jmi.javamodel.ClassDefinition;
import org.netbeans.jmi.javamodel.ClassMember;
import org.netbeans.jmi.javamodel.Element;
import org.netbeans.jmi.javamodel.ErrorInfo;
import org.netbeans.jmi.javamodel.Feature;
import org.netbeans.jmi.javamodel.Field;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.JavaModelPackage;
import org.netbeans.jmi.javamodel.Method;
import org.netbeans.jmi.javamodel.NamedElement;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.jmi.javamodel.Type;
import org.netbeans.jmi.javamodel.TypedElement;
import org.netbeans.jmi.javamodel.UnresolvedClass;
import org.netbeans.modules.javacore.ClassIndex;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.javacore.jmiimpl.javamodel.ResourceImpl;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import org.openide.util.RequestProcessor;
import org.openide.util.Utilities;
/**
 *
 * @author Martin Adamek
 * @author Marek Fukala
 */
public final class NNMDRListener implements MDRPreChangeListener {

    //special request processor for adding annotation listeners. Throughput is set to just one since
    //the task accesses MDR which uses ExclusiveMutex anyway.
    private static final RequestProcessor nnMDRListenerRP = new RequestProcessor(NNMDRListener.class.getName(), 1);
    
    // for logging of classpah-related actions (registration, unregistration, changes)
    private static final ErrorManager CP_LOGGER = ErrorManager.getDefault().getInstance("org.netbeans.modules.j2ee.metadata.NNMDRListener.cp"); // NOI18N
    private static final boolean CP_LOG = CP_LOGGER.isLoggable(ErrorManager.INFORMATIONAL);
    
    // for logging of events handling
    private static final ErrorManager LOGGER = ErrorManager.getDefault().getInstance("org.netbeans.modules.j2ee.metadata.NNMDRListener.nnListener"); // NOI18N
    private static final boolean LOG = LOGGER.isLoggable(ErrorManager.INFORMATIONAL);
    
    private static final boolean isNoScan = "true".equals(System.getProperty("netbeans.javacore.noscan")); // NOI18N
    
    /**
     * list of classpath roots where this class is registered as listener
     */
    private Set<FileObject> classpathRoots = new HashSet<FileObject>();
    private static NNMDRListener instance = new NNMDRListener();
    private volatile Map<NNListener, ClassPathSourceCache> listeners = new HashMap<NNListener, ClassPathSourceCache>(3);
    private CountLatch latch = new CountLatch();
    
    private NNMDRListener() {
    }
    
    public static NNMDRListener getDefault() {
        return instance;
    }
    
    public void addAnnotationListener(final NNListener annotationListener) {
        if (CP_LOG) CP_LOGGER.log("Adding NNListener: " + annotationListener);
        final ClassPath annotationListenerClassPath = annotationListener.getClassPath();
        final ClassPathSourceCache classPathSourceCache = ClassPathSourceCache.newInstance(annotationListenerClassPath, !annotationListener.getPrimaryAnnotations().isEmpty());
        if (annotationListenerClassPath == null) {
            return;
        }
        synchronized (this) {
            Map<NNListener, ClassPathSourceCache> listenersCopy = new HashMap<NNListener, ClassPathSourceCache>(listeners);
            listenersCopy.put(annotationListener, classPathSourceCache);
            listeners = listenersCopy;
        }
        classPathSourceCache.addPropertyChangeListener(new PropertyChangeListener() {
            public void propertyChange(PropertyChangeEvent evt) {
                if (ClassPathSourceCache.PROP_ROOTS.equals(evt.getPropertyName())) {
                    handleClasspathListenerChange(classPathSourceCache);
                }
            }
        });
        // initial registration on classpath roots
        handleClasspathListenerChange(classPathSourceCache);
        latch.countUp();
        
        nnMDRListenerRP.post( new Runnable() {
            public void run() {
                boolean waited = JavaMetamodel.getManager().waitScanFinished();
                if (CP_LOG && waited) CP_LOGGER.log("Was waiting for scan finished (while setting NNListener)");
                JavaModel.getJavaRepository().beginTrans(false);
                try {
                    // set classpath for THIS TRANSACTION
                    JavaModel.setClassPath(annotationListenerClassPath);
                    // first primary annotations should be resolved, so they are reflected in model before any field/method annotations
                    Collection<String> primaryAnnotations = annotationListener.getPrimaryAnnotations();
                    for (String annotationName : primaryAnnotations) {
                        for (Annotation annotation : getAnnotationUsages(annotationName, annotationListenerClassPath)) {
                            AnnotationType type = annotation.getType();
                            if (type != null) {
                                primaryAnnotationAddedForListener(annotationListener, annotation, type);
                            }
                        }
                    }
                    Collection<String> supportedAnnotations  = annotationListener.getSupportedAnnotations();
                    for (String annotationName : supportedAnnotations) {
                        if (!primaryAnnotations.contains(annotationName)) {
                            for (Annotation annotation : getAnnotationUsages(annotationName, annotationListenerClassPath)) {
                                AnnotationType type = annotation.getType();
                                if (type != null) {
                                    annotationAddedForListener(annotationListener, annotation, type);
                                }
                            }
                        }
                    }
                } finally {
                    JavaModel.getJavaRepository().endTrans();
                    latch.countDown();
                }
            }
        });
    }
    
    public void waitScanFinished() {
        try {
            latch.await();
        } catch (InterruptedException ex) {
            ErrorManager.getDefault().notify(ex);
        }
    }
    
    public boolean isScanInProgress() {
        return latch.getCount() > 0;
    }
    
    public void change(MDRChangeEvent event) {
        JavaModel.getJavaRepository().beginTrans(false);
        try {
            switch (event.getType()) {
                case AttributeEvent.EVENT_ATTRIBUTE_ADD:
                    if(LOG) LOGGER.log("AttributeEvent.EVENT_ATTRIBUTE_ADD: " + event);
                    eventAttributeAdd((AttributeEvent) event);
                    break;
                case AttributeEvent.EVENT_ATTRIBUTE_SET:
                    if(LOG) LOGGER.log("AttributeEvent.EVENT_ATTRIBUTE_SET: " + event);
                    eventAttributeSet((AttributeEvent)event);
                    break;
                case AssociationEvent.EVENT_ASSOCIATION_SET:
                    if (LOG) LOGGER.log("AssociationEvent.EVENT_ASSOCIATION_SET: " + event);
                    postEventAssociationSet((AssociationEvent) event);
                    break;
                default:
            }
        } finally {
            JavaModel.getJavaRepository().endTrans();
        }
    }
    
    public void plannedChange(MDRChangeEvent event) throws VetoChangeException {
        
//        if (event instanceof AttributeEvent) {
//            AttributeEvent ae = (AttributeEvent) event;
//            System.out.println("### AE: " + Integer.toHexString(event.getType()) + " " + ae.getAttributeName() + ", " + ae.getOldElement() + " -> " + ae.getNewElement());
//        } else if (event instanceof InstanceEvent) {
//            InstanceEvent ie = (InstanceEvent) event;
//            Object o = ie.getInstance();
//            if (o instanceof MultipartId) {
//                System.out.println("### IE: " + Integer.toHexString(event.getType()) + " MI " + ((MultipartId) o).getName());
//            } else if (o instanceof StringLiteral) {
//                System.out.println("### IE: " + Integer.toHexString(event.getType()) + " SL " + ((StringLiteral) o).getValue());
//            } else {
//                System.out.println("### IE: " + Integer.toHexString(event.getType()) + " " + o);
//            }
//        } else if (event instanceof AssociationEvent) {
//            AssociationEvent ae = (AssociationEvent) event;
//            System.out.println("### AS: " + Integer.toHexString(event.getType()) + ", " + ae.getEndName() + ", " + ae.getFixedElement() + ", " + ae.getOldElement() + " -> " + ae.getNewElement());
//        } else {
//            System.out.println("### " + Integer.toHexString(event.getType()));
//        }
        
        JavaModel.getJavaRepository().beginTrans(false);
        try {
            switch (event.getType()) {
                case AttributeEvent.EVENT_ATTRIBUTE_REMOVE:
                    if(LOG) LOGGER.log("PLANNED: AttributeEvent.EVENT_ATTRIBUTE_REMOVE - planned: " + event);
                    eventAttributeRemove((AttributeEvent) event);
                    break;
                case AttributeEvent.EVENT_ATTRIBUTE_SET:
                    if(LOG) LOGGER.log("PLANNED: AttributeEvent.EVENT_ATTRIBUTE_SET - planned: " + event);
                    eventAttributeRemoveSet((AttributeEvent)event);
                    break;
                case AssociationEvent.EVENT_ASSOCIATION_SET:
                    if (LOG) LOGGER.log("AssociationEvent.EVENT_ASSOCIATION_SET - planned: " + event);
                    preEventAssociationSet((AssociationEvent) event);
                    break;
                default:
            }
        } finally {
            JavaModel.getJavaRepository().endTrans();
        }
    }
    
    public void changeCancelled(MDRChangeEvent event) {
        if(LOG) LOGGER.log("### ATTENTION !!! changeCancelled !!!");
    }
    
    // private stuff ===========================================================
    
    // classpath handling ------------------------------------------------------
    
    private void handleClasspathListenerChange(final ClassPathSourceCache classPathSourceCache) {
        nnMDRListenerRP.post( new Runnable() {
            public void run() {
                boolean waited = JavaMetamodel.getManager().waitScanFinished();
                if (CP_LOG && waited) CP_LOGGER.log("Was waiting for scan finished (while setting)");
                FileObject[] roots = classPathSourceCache.getRoots();
                JavaModel.getJavaRepository().beginTrans(false);
                try {
                    for (int i = 0; i < roots.length; i++) {
                        // check if we are already registered on this root
                        if (!classpathRoots.contains(roots[i])) {
                            if (CP_LOG) CP_LOGGER.log("New root added to classpath: " + FileUtil.getFileDisplayName(roots[i]));
                            JavaModelPackage jmp = JavaMetamodel.getManager().getJavaExtent(roots[i]);
                            if (jmp == null) {
                                jmp = JavaMetamodel.getManager().resolveJavaExtent(roots[i]);
                            }
                            if (CP_LOG && (jmp == null)) CP_LOGGER.log("Didn't resolve Java Extent for classpath root " + FileUtil.getFileDisplayName(roots[i]));
                            if (!isNoScan) {
                                assert jmp != null;
                            }
                            if (jmp != null) {
                                ((MDRChangeSource) jmp).addListener(NNMDRListener.this);
                                classpathRoots.add(roots[i]);
                                if (CP_LOG) CP_LOGGER.log("Added listener to " + FileUtil.getFileDisplayName(roots[i]));
                            }
                        }
                    }
                    // if root disappeared from classpath, we cannot resolve Java extent for that and call removeListener
                    // according to javacore (dprusa) it should be ok
                } finally {
                    JavaModel.getJavaRepository().endTrans();
                }
            }
        });
    }
    
    // event handling ----------------------------------------------------------
    
    /** Handle class rename */
    private void eventAttributeRemoveSet(AttributeEvent event) {
        Object oldValue = event.getOldElement();
        Object newValue = event.getNewElement();
        if (Utilities.compareObjects(oldValue, newValue)) {
            return;
        }
        String eventName = event.getAttributeName();
        if("name".equals(eventName)) {
            Object source = event.getSource();
            if(source instanceof JavaClass) {
                //fire remove event
                classRemoved((JavaClass)source);
            }
        }
    }
    
    private void eventAttributeSet(AttributeEvent event) {
        Object oldValue = event.getOldElement();
        Object newValue = event.getNewElement();
        if (Utilities.compareObjects(oldValue, newValue)) {
            return;
        }
//        handle class members renaming
        if("name".equals(event.getAttributeName())) {
            Object source = event.getSource();
            if(source instanceof Field || source instanceof Method) {
                //update class member annotation
                ClassDefinition declClass = ((ClassMember)source).getDeclaringClass();
                if (declClass instanceof JavaClass) {
                    JavaClass dclass = (JavaClass)declClass;
                    AnnotableElement anne = (AnnotableElement)source;
                    Iterator<Annotation> nns = anne.getAnnotations().iterator();
                    while(nns.hasNext()) {
                        Annotation nn = nns.next();
                        //removed annotation from the old field
                        AnnotationType type = nn.getType();
                        if (type != null) {
                            memberAnnotationRemoved(nn, type, dclass, (String)event.getOldElement(), ((TypedElement)source).getType().getName(), source instanceof Field);
                        }
                        //and add it to the new one
                        memberAnnotationAdded(nn, (Element)source);
                    } // while
                } // if
            } else if(source instanceof JavaClass) {
                //handle class rename
                classAdded((JavaClass)source);
            }
        } else if ("typeName".equals(event.getAttributeName())) {
            Object source = event.getSource();
            if(source != null) {
                if(source instanceof Annotation) {
                    Annotation a = (Annotation)source;
                        //called when an annotation class type has been resolved (for example when the import has been added)
                        annotationAdded(a); //resolved nn class
                }
            }
        } else if ("value".equals(event.getAttributeName())) {
            Object source = event.getSource();
            if (source instanceof AttributeValue) {
                attributeAdded((AttributeValue) source);
            }
        }
    }
    
    private void eventAttributeAdd(AttributeEvent event) {
        Object newElement = event.getNewElement();
        if (newElement instanceof Annotation) {
            Annotation annotation = (Annotation) newElement;
                annotationAdded(annotation);
        } else if (newElement instanceof AttributeValue) {
            attributeAdded((AttributeValue) event.getNewElement());
        } else if ("classifiers".equals(event.getAttributeName())) {
            // new class added, no annotations event are fired, let's find them
            // this is the case when class from template was created and annotations were part of template
            // for JMI it is just moving of the class from one extent to another
            classAdded((JavaClass) event.getNewElement());
        }
    }
    
    // e.g. @Stateless changed to @Stateful, part 1 of 2
    // removing of old annotation
    // this can be transformed to single event once we have support for it in model builder
    // see postEventAssociationSet
    private void preEventAssociationSet(AssociationEvent event) {
        if ("type".equals(event.getEndName())) {
            RefObject fixedElement = event.getFixedElement();
            if (fixedElement instanceof Annotation) {
                annotationRemoved((Annotation) fixedElement, (Type)event.getOldElement());
            }
        }
    }
    
    // e.g. @Stateless changed to @Stateful, part 2 of 2
    // adding new annotation
    // this can be transformed to single event once we have support for it in model builder
    // see preEventAssociationSet
    private void postEventAssociationSet(AssociationEvent event) {
        if ("type".equals(event.getEndName())) {
            RefObject fixedElement = event.getFixedElement();
            if (fixedElement instanceof Annotation) {
                annotationAdded((Annotation) fixedElement);
            }
        }
    }
    
    private void classAdded(JavaClass javaClass) {
        if (!javaClass.isValid())
            return;
        List annotations = javaClass.getAnnotations();
        //fire add events for class annotations
        for (Iterator it = annotations.iterator(); it.hasNext();) {
            annotationAdded((Annotation) it.next());
        }
        //fire add events for member annotations
        Iterator features = javaClass.getFeatures().iterator();
        while(features.hasNext()) {
            Object f = features.next();
            if(f instanceof Method || f instanceof Field) {
                for (Iterator it = ((AnnotableElement)f).getAnnotations().iterator(); it.hasNext();) {
                    memberAnnotationAdded((Annotation)it.next(), (Element)f);
                }
            }
        }
        
    }
    
    private void eventAttributeRemove(AttributeEvent event) {
        Object oldElement = event.getOldElement();
        if (oldElement instanceof Annotation) {
            Annotation annotation = (Annotation) oldElement;
            Resource resource = annotation.getResource();
            if (resource == null || resource.getStatus() == ResourceImpl.HAS_SYNTAX_ERROR) {
                return;
            }
            List errors = resource.getErrors();
            if (errors.size() > 0) {
                for (Object elem : resource.getErrors()) {
                    if (elem instanceof ErrorInfo) {
                        ErrorInfo ei = (ErrorInfo) elem;
                        // if field with annotation was removed and this field is referenced somewhere in class,
                        // we want to fire annotation removal event
                        // TODO: discuss this with javacore
                        if (!"cant.resolve.location".equals(ei.getErrorId())) {
                            return;
                        }
                    } else {
                        return;
                    }
                }
            }
            AnnotationType type = annotation.getType();
            if (type != null) { // issue #64877
                annotationRemoved(annotation, type);
            }
        } else if (oldElement instanceof AttributeValue) {
            attributeRemoved((AttributeValue) oldElement);
        } else if (oldElement instanceof JavaClass) {
            classRemoved((JavaClass) oldElement);
        }
    }
    
    private void classRemoved(JavaClass javaClass) {
        for (NNListener annotationListener : listeners.keySet()) {
            annotationListener.classRemoved(javaClass.getName());
        }
    }
    
    /**
     * Searches for all usages of annotation of given type
     *
     * @param annotationClassName fully-qualified name of type of the annotation
     * @return list of annotation instances or empty list if no elements found.
     */
    private List<Annotation> getAnnotationUsages(String annotationClassName, ClassPath cp) {
        List<Annotation> result = new ArrayList<Annotation>();
        AnnotationType annotationClass = (AnnotationType) resolveType(annotationClassName, cp);
        if (annotationClass == null) {
            return result;
        }
        if (CP_LOG && (annotationClass == null)) CP_LOGGER.log("Didn't resolve " + annotationClassName + " annotation");
        Resource[] resources = ClassIndex.findResourcesForIdentifier(annotationClass.getSimpleName(), true);
        for (Resource resource : resources) {
            List<JavaClass> classes = resource.getClassifiers();
            for (JavaClass jc : classes) {
                findAnnotationsInClass(jc, annotationClass, result);
            }
        }
        return result;
    }
    
    private void findAnnotationsInClass(JavaClass jc, AnnotationType annotationType, List<Annotation> result) {
        findAnnotationsInList(jc.getAnnotations(), annotationType, result);
        List<Feature> features = jc.getFeatures();
        for (Feature feature : features) {
            findAnnotationsInList(feature.getAnnotations(), annotationType, result);
        }
    }
    
    private void findAnnotationsInList(List<Annotation> list, AnnotationType annotationType, List<Annotation> result) {
        for (Annotation an : list) {
            if (annotationType.equals(an.getType())) {
                result.add(an);
            }
        }
    }
    
    // we don't have access to JMIUtils
    private Type resolveType(String typeName, ClassPath cp) {
        JavaModelPackage jmp = JavaModel.getDefaultExtent();
        if (!isNoScan) {
            assert jmp != null : "Default Java Extent wasn't resolved. Probably called BEFORE the roots were added to GlobalPathRegistry?";
        }
        if (jmp == null) {
            return null;
        }
        Type result = jmp.getType().resolve(typeName);
        if (!(result instanceof UnresolvedClass)) {
            return result;
        }
        return null;
    }
    
    private void classAnnotationAddedForListener(NNListener nnListener, Annotation annotation, AnnotationType annotationType, JavaClass clazz, boolean findAllInClass) {
        if (annotationType != null) {
            nnListener.addClassAnnotation(clazz, annotation, annotationType);
        }
        if (findAllInClass) {
            String annotationName = annotationType.getName();
            if(nnListener.getPrimaryAnnotations().contains(annotationName)) {
                //primary annotation has just been found
                //we need to scan the javaclass for all other annotations
                //and fire nn events for them
                List annotations = clazz.getAnnotations();
                //fire add events for class annotations
                for (Annotation nn : (List<Annotation>)annotations) {
                    AnnotationType type = nn.getType();
                    if (type != null) {
                        nnListener.addClassAnnotation(clazz, nn, type);
                    }
                }
                //fire add events for member annotations
                for(Object f : clazz.getFeatures()) {
                    if(f instanceof Method || f instanceof Field) {
                        for (Annotation memberNN : (List<Annotation>)((AnnotableElement)f).getAnnotations()) {
                            AnnotationType type = memberNN.getType();
                            if (type != null) {
                                nnListener.addMemberAnnotation(f instanceof Field, clazz, (Element)f, memberNN, type);
                            }
                        }
                    }
                }
            }


        }
    }
    
    private void classAnnotationRemoved(Annotation annotation, Type annoType, JavaClass javaClass) {
        for (Map.Entry<NNListener, ClassPathSourceCache> entry : listeners.entrySet()) {
            // if class is from listener's classpath
            if (entry.getValue().contains(JavaModel.getFileObject(javaClass.getResource()))) {
                NNListener nnListener = entry.getKey();
                if (annoType != null && nnListener.getSupportedAnnotations().contains(annoType.getName())) {
                    nnListener.removeClassAnnotation(javaClass, annotation, (AnnotationType) annoType);
                }
            }
        }
    }
    
    private void memberAnnotationAdded(Annotation annotation, Element te) {
        AnnotationType type = annotation.getType();
        if (type != null) {
        for (Map.Entry<NNListener, ClassPathSourceCache> entry : listeners.entrySet()) {
            // if element is from listener's classpath
            if (entry.getValue().contains(JavaModel.getFileObject(te.getResource()))) {
                NNListener nnListener = entry.getKey();
                    if (nnListener.getSupportedAnnotations().contains(type.getName())) {
                    //field or method
                    ClassDefinition dclass = ((ClassMember)te).getDeclaringClass();
                    if (dclass instanceof JavaClass) {
                            nnListener.addMemberAnnotation(te instanceof Field, (JavaClass)dclass, te, annotation, type);
                    }
                }
            }
        }
    }
    }
    
    private void memberAnnotationAddedForListener(NNListener nnListener, Annotation annotation, AnnotationType type, Element te) {
        if (type != null) {
            ClassPathSourceCache cpSourceCache = listeners.get(nnListener);
            if (cpSourceCache.contains(JavaModel.getFileObject(te.getResource()))) {
                //field or method
                ClassDefinition dclass = ((ClassMember)te).getDeclaringClass();
                if (dclass instanceof JavaClass) {
                    nnListener.addMemberAnnotation(te instanceof Field, (JavaClass)dclass, te, annotation, type);
                }
            }
        }
    }
    
    private void memberAnnotationRemoved(Annotation annotation, Type annoType, JavaClass javaClass, String memberName, String memberType, boolean field) {
        for (Map.Entry<NNListener, ClassPathSourceCache> entry : listeners.entrySet()) {
            // if class is from listener's classpath
            if (entry.getValue().contains(JavaModel.getFileObject(javaClass.getResource()))) {
                NNListener nnListener = entry.getKey();
                if (annoType != null && nnListener.getSupportedAnnotations().contains(annoType.getName())) {
                    //field or method
                    nnListener.removeMemberAnnotation(field, javaClass, memberName, memberType, annotation, (AnnotationType) annoType);
                }
            }
        }
    }
    
    private void annotationAdded(Annotation annotation) {
        AnnotationType type = annotation.getType();
        if (type != null) {
        for (Map.Entry<NNListener, ClassPathSourceCache> entry : listeners.entrySet()) {
            // if class is from listener's classpath
            if (entry.getValue().contains(JavaModel.getFileObject(annotation.getResource()))) {
                NNListener nnListener = entry.getKey();
                    if (nnListener.getSupportedAnnotations().contains(type.getName())) {
                        annotationAddedForListener(nnListener, annotation, type);
                }
            }
        }
    }
    }
    
    private void annotationAddedForListener(NNListener nnListener, Annotation annotation, AnnotationType type) {
        RefFeatured refFeatured = annotation.refImmediateComposite();
        if(refFeatured instanceof JavaClass) {
            classAnnotationAddedForListener(nnListener, annotation, type, (JavaClass)refFeatured, true);
        } else if(refFeatured instanceof Field ||
                refFeatured instanceof Method) {
            memberAnnotationAddedForListener(nnListener, annotation, type, (Element)refFeatured);
        }
    }
    
    private void primaryAnnotationAddedForListener(NNListener nnListener, Annotation annotation, AnnotationType type) {
        RefFeatured refFeatured = annotation.refImmediateComposite();
        if(refFeatured instanceof JavaClass) {
            classAnnotationAddedForListener(nnListener, annotation, type, (JavaClass)refFeatured, false);
        }
    }
    
    private void annotationRemoved(Annotation annotation, Type annoType) {
        RefFeatured refFeatured = annotation.refImmediateComposite();
        if (annoType != null) {
            if(refFeatured instanceof JavaClass) {
                classAnnotationRemoved(annotation, annoType, (JavaClass)refFeatured);
            } else if(refFeatured instanceof Field ||
                    refFeatured instanceof Method) {
                ClassDefinition dclass = ((ClassMember)refFeatured).getDeclaringClass();
                if (dclass instanceof JavaClass) {
                    memberAnnotationRemoved(annotation, annoType, (JavaClass)dclass, ((NamedElement)refFeatured).getName(), ((TypedElement)refFeatured).getType().getName(), refFeatured instanceof Field);
                }
            }
        }
    }
    
    private void attributeAdded(AttributeValue attributeValue) {
        RefFeatured refFeatured = attributeValue.refImmediateComposite();
        if (refFeatured instanceof Annotation) {
            Annotation annotation = (Annotation) refFeatured;
            AnnotationType type = annotation.getType();
            if (type != null) {
            for (Map.Entry<NNListener, ClassPathSourceCache> entry : listeners.entrySet()) {
                // if class is from listener's classpath
                if (entry.getValue().contains(JavaModel.getFileObject(annotation.getResource()))) {
                    NNListener nnListener = entry.getKey();
                        if (nnListener.getSupportedAnnotations().contains(type.getName())) {
                            _attributeAdded(nnListener, annotation, type, attributeValue);
                    }
                }
            }
        }
    }
    }
    
    private void _attributeAdded(NNListener nnListener, Annotation annotation, AnnotationType type, AttributeValue attributeValue) {
        RefFeatured refFeatured = annotation.refImmediateComposite();
        if (refFeatured instanceof JavaClass) {
            JavaClass jc = (JavaClass) refFeatured;
            nnListener.addClassAttribute(jc, annotation, type, attributeValue.getDefinition().getName(), Utils.stringValue(attributeValue));
        } else if(refFeatured instanceof Field || refFeatured instanceof Method) {
            ClassDefinition dclass = ((ClassMember)refFeatured).getDeclaringClass();
            if (dclass instanceof JavaClass) {
                nnListener.addMemberAttribute((JavaClass)dclass, (Element)refFeatured, annotation, type, attributeValue.getDefinition().getName(), Utils.stringValue(attributeValue));
            }
        }
    }
    
    private void attributeRemoved(AttributeValue attributeValue)  {
        RefFeatured refFeatured = attributeValue.refImmediateComposite();
        if (refFeatured instanceof Annotation) {
            Annotation annotation = (Annotation) refFeatured;
            AnnotationType type = annotation.getType();
            if (type != null) {
            for (Map.Entry<NNListener, ClassPathSourceCache> entry : listeners.entrySet()) {
                // if class is from listener's classpath
                if (entry.getValue().contains(JavaModel.getFileObject(annotation.getResource()))) {
                    NNListener nnListener = entry.getKey();
                        if (nnListener.getSupportedAnnotations().contains(type.getName())) {
                            _attributeRemoved(nnListener, annotation, type, attributeValue);
                    }
                }
            }
        }
    }
    }
    
    private void _attributeRemoved(NNListener nnListener, Annotation annotation, AnnotationType type, AttributeValue attributeValue) {
        RefFeatured refFeatured = annotation.refImmediateComposite();
        if (refFeatured instanceof JavaClass) {
            JavaClass jc = (JavaClass) refFeatured;
            nnListener.removeClassAttribute(jc, annotation, type, attributeValue.getDefinition().getName());
        } else if (refFeatured instanceof Field || refFeatured instanceof Method) {
            ClassDefinition dclass = ((ClassMember)refFeatured).getDeclaringClass();
            if (dclass instanceof JavaClass) {
                nnListener.removeMemberAttribute((JavaClass)dclass, (Element)refFeatured, annotation, type, attributeValue.getDefinition().getName());
            }
        }
    }

}
