/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 *
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.j2ee.metadata;

import java.util.Collection;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.jmi.javamodel.Annotation;
import org.netbeans.jmi.javamodel.AnnotationType;
import org.netbeans.jmi.javamodel.Element;
import org.netbeans.jmi.javamodel.JavaClass;

/**
 * Annotation listener that receives events related to annotations.
 * This listener affects the amount and type of annotations that it is receiving
 * events for by specifiing type names of annotations and the classpath that 
 * restricts amount of sources of events.
 * 
 * @author Marek Fukala
 * @author Martin Adamek
 */
public interface NNListener {
    
    /**
     * Returns classpath that should be covered by this listener.
     * In other words, this listener will receive events about annotation changes
     * that happen in defined classpath.
     *
     * @return classpath that this listener covers
     */
    ClassPath getClassPath();
    
    /**
     * Returns all supported annotations.
     * These annotation are then taken into account by {@link NNMDRListener} and hence
     * by classes extending NNListener
     *
     * @return Fully qualified names of annotations to listen on
     */
    Collection<String> getSupportedAnnotations();
    
    /**
     * Returns a list of class annotations which presence and
     * subsequent model update is very important to be properly
     * able to update the model when other annotation are found.
     * For example: @Entity, @Stateless, @Statefull, @MessageDriven
     *
     * When such annotation is found then the source javaclass is searched
     * for annotations and NNListener evens are fired for all of them.
     *
     * The reason why we need this is that it may happen that a non-primary
     * event (like @OneToMany on a class member) is fired but the model cannot
     * be properly updated since the primary annotation builds a part of the
     * model which needs to exist to be able to reflect the @OneToMany annotation.
     *
     */
    Collection<String> getPrimaryAnnotations();
    
    /**
     * Class was annotated by annotation.
     *
     * @param javaClass class which was annotated
     * @param annotation annotation that was added to class
     * @param type non-null Java type of annotation. It can be obtained also by
     * {@link Annotation#getType()}, but it is expensive operation. Please use this
     * instance instead.
     */
    void addClassAnnotation(JavaClass javaClass, Annotation annotation, AnnotationType type);
    
    /**
     * Annotation annotating class was removed.
     * 
     * @param className class which was annotated
     * @param annotation annotation that was removed
     * @param type non-null Java type of annotation. It can be obtained also by
     * {@link Annotation#getType()}, but it is expensive operation. Please use this
     * instance instead.
     */
    void removeClassAnnotation(JavaClass className, Annotation annotation, AnnotationType type);
    
    /**
     * Field or method was annotated.
     * {@link org.netbeans.jmi.javamodel.Field} or {@link org.netbeans.jmi.javamodel.Method} 
     * can be passed as member argument (there is no suitable common interface for both)
     * 
     * @param field true if annotated member is fiels, false if it was method
     * @param javaClass parent class of annotated element
     * @param member element that was annotated
     * @param annotation annotaion that was added
     * @param type non-null Java type of annotation. It can be obtained also by
     * {@link Annotation#getType()}, but it is expensive operation. Please use this
     * instance instead.
     */
    void addMemberAnnotation(boolean field, JavaClass javaClass, Element member, Annotation annotation, AnnotationType type);
    
    /**
     * Annotation annotatin field or methos was removed.
     * 
     * @param field true if annotated member is fiels, false if it was method
     * @param javaClass parent class of affected element
     * @param memberName name of the affected member
     * @param memberType type of the affected member
     * @param annotation annotaion that was removed
     * @param type non-null Java type of annotation. It can be obtained also by
     * {@link Annotation#getType()}, but it is expensive operation. 
     * Please use this instance instead.
     */
    void removeMemberAnnotation(boolean field, JavaClass javaClass, String memberName, String memberType, Annotation annotation, AnnotationType type);
    
    /**
     * Attribute was added to annotation annotating the class.
     * 
     * @param javaClass class that is annotated
     * @param annotation annotation where attribute was added
     * @param type non-null Java type of annotation. It can be obtained also by
     * {@link Annotation#getType()}, but it is expensive operation. Please use this
     * instance instead.
     * @param attributeName name of the attribute
     * @param attributeValue value of the attribute
     */
    //XXX attributes support has to be rewritten
    void addClassAttribute(JavaClass javaClass, Annotation annotation, AnnotationType type, String attributeName, String attributeValue);
    
    /**
     * Attribute was removed from annotation annotating the class.
     * 
     * @param javaClass class that is annotated
     * @param annotation annotation where attribute was removed
     * @param type non-null Java type of annotation. It can be obtained also by
     * {@link Annotation#getType()}, but it is expensive operation. Please use this
     * instance instead.
     * @param attributeName name of the attribute
     */
    void removeClassAttribute(JavaClass javaClass, Annotation annotation, AnnotationType type, String attributeName);
    
    /**
     * Attribute was added to annotation annotating field or method.
     * 
     * @param javaClass parent class of affected element
     * @param member field or method that is annotated
     * @param annotation annotation where attribute was added
     * @param type non-null Java type of annotation. It can be obtained also by
     * {@link Annotation#getType()}, but it is expensive operation. Please use this
     * instance instead.
     * @param attributeName name of the attribute
     * @param attributeValue value of the attribute
     */
    void addMemberAttribute(JavaClass javaClass, Element member, Annotation annotation, AnnotationType type, String attributeName, String attributeValue);
    
    /**
     * Attribute was removed from annotation annotating field or method
     * 
     * @param javaClass parent class of affected element
     * @param member field or method that is annotated
     * @param annotation annotation where attribute was removed
     * @param type non-null Java type of annotation. It can be obtained also by
     * {@link Annotation#getType()}, but it is expensive operation. Please use this
     * instance instead.
     * @param attributeName name of the attribute
     */
    void removeMemberAttribute(JavaClass javaClass, Element member, Annotation annotation, AnnotationType type, String attributeName);
    
    /**
     * Class was removed.
     * 
     * @param fqn fully-qualified name of the class that was removed
     */
    void classRemoved(String fqn);
    
}

