/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.j2ee.metadata;

/**
 * A synchronization aid that allows one or more threads to wait until
 * a set of operations being performed in other threads completes.
 * 
 * <p>A <tt>CountLatch</tt> is initialized with <em>count</em> 0 (zero).
 * The {@link #await await} methods block until the current
 * {@link #getCount count} reaches zero due to invocations of the
 * {@link #countDown} method, after which all waiting threads are
 * released and any subsequent invocations of {@link #await await} return
 * immediately. Count can be increased by invoking {@link #countUp} method.
 *
 * @author Martin Adamek
 */
public final class CountLatch {
    
    private int count = 0;

    public CountLatch() {
    }
    
    /**
     * Causes the current thread to wait until the latch has counted down to 
     * zero, unless the thread is {@link Thread#interrupt interrupted}.
     * 
     * <p>If the current {@link #getCount count} is zero then this method
     * returns immediately.
     * 
     * @throws java.lang.InterruptedException 
     */
    public void await() throws InterruptedException {
        synchronized (this) {
            while (count > 0) {
                wait();
            }
        }
    }
    
    /**
     * Increments the count of the latch.
     */
    public void countUp() {
        synchronized (this) {
            count++;
        }
    }
    
    /**
     * Decrements the count of the latch, releasing all waiting threads if
     * the count reaches zero.
     * <p>If the current {@link #getCount count} is greater than zero then
     * it is decremented. If the new count is zero then all waiting threads
     * are re-enabled for thread scheduling purposes.
     * <p>If the current {@link #getCount count} equals zero then nothing
     * happens.
     */
    public void countDown() {
        synchronized (this) {
            if (count != 0) {
                count--;
                if (count == 0) {
                    notifyAll();
                }
            }
        }
    }
    
    /**
     * Returns the current count.
     * 
     * @return the current count.
     */
    public int getCount() {
        synchronized (this) {
            return count;
        }
    }
    
}
