/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 *
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.j2ee.ejbcore.ui.logicalview.entres;

import java.io.IOException;
import java.lang.reflect.Modifier;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import org.netbeans.api.project.Project;
import org.netbeans.api.project.ProjectInformation;
import org.netbeans.api.project.ProjectUtils;
import org.netbeans.jmi.javamodel.Annotation;
import org.netbeans.jmi.javamodel.AttributeValue;
import org.netbeans.jmi.javamodel.Feature;
import org.netbeans.jmi.javamodel.Field;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.JavaModelPackage;
import org.netbeans.jmi.javamodel.Method;
import org.netbeans.jmi.javamodel.Parameter;
import org.netbeans.modules.j2ee.common.queries.api.InjectionTargetQuery;
import org.netbeans.modules.j2ee.dd.api.common.MessageDestinationRef;
import org.netbeans.modules.j2ee.dd.api.common.ResourceRef;
import org.netbeans.modules.j2ee.api.ejbjar.EnterpriseReferenceContainer;
import org.netbeans.modules.j2ee.common.JMIGenerationUtil;
import org.netbeans.modules.j2ee.common.JMIUtils;
import org.netbeans.modules.j2ee.ejbcore.Utils;
import org.netbeans.modules.javacore.api.JavaModel;


/**
 * Provide an encapsulation of a JMS Destination.
 * @author Chris Webster
 * @author Martin Adamek
 */
public class JMSDestination {
    private static final String PRODUCES = org.netbeans.modules.j2ee.dd.api.common.MessageDestinationRef.MESSAGE_DESTINATION_USAGE_PRODUCES;
    
    private String destination;
    private Project project;
    private String type;
    /**
     * Indicates whether the target class supports injection.
     */
    private boolean supportsInjection;
    
    public JMSDestination(String destination, Project p, String type) {
        this.destination = destination;
        project = p;
        this.type = type;
    }
    
    public String toString() {
        return destination;
    }
    
    public String getDestination() {
        return destination;
    }
    
    public String getType() {
        return type;
    }
    
    public Project getProject() {
        return project;
    }
    
    private String generateConnectionFactoryReference(
            EnterpriseReferenceContainer erc,
            String className) throws IOException {
        ResourceRef ref = erc.createResourceRef(className);
        ref.setResRefName(getConnectionFactoryName()); // NOI18N
        ref.setResAuth(org.netbeans.modules.j2ee.dd.api.common.ResourceRef.RES_AUTH_CONTAINER);
        ref.setResSharingScope(org.netbeans.modules.j2ee.dd.api.common.ResourceRef.RES_SHARING_SCOPE_SHAREABLE);
        ref.setResType("javax.jms.ConnectionFactory"); //NOI18N
        return erc.addResourceRef(ref, className);
    }
    
    
    private String generateDestinationReference(
            EnterpriseReferenceContainer erc,
            String className,
            Project thisProject) throws IOException {
        
        MessageDestinationRef ref = erc.createDestinationRef(className);
        ref.setMessageDestinationUsage(PRODUCES);
        ref.setMessageDestinationType(getType());
        ref.setMessageDestinationRefName(getDestinationName());
        // this may need to generalized later if jms producers are expected
        // in web modules
        ProjectInformation pi = ProjectUtils.getInformation(getProject());
        ref.setMessageDestinationLink(pi.getName()+".jar#"+getDestination());
        if (getProject().equals(thisProject)) {
            String linkWithoutModule = ref.getMessageDestinationLink();
            linkWithoutModule = linkWithoutModule.substring(linkWithoutModule.indexOf('#')+1);
            ref.setMessageDestinationLink(linkWithoutModule);
        }
        return erc.addDestinationRef(ref, className);
    }
    
    private String getConnectionFactoryName(){
        return "jms/" + getDestination() + "Factory"; //NO18N
    }
    
    private String getDestinationName(){
        return "jms/" + getDestination();//NO18N
    }
    
    public void genMethods(EnterpriseReferenceContainer erc,
            String className,
            Project currentProject,
            JavaClass target,
            ServiceLocatorStrategy sl)
            throws IOException {
        
        
        this.supportsInjection = InjectionTargetQuery.isInjectionTarget(target);
        
        String destinationFieldName = null;
        String connectionFactoryFieldName = null;
        String factoryName = null;
        String destinationName = null;
        
        List<Feature> addedFeatures = new ArrayList<Feature>();
        if (supportsInjection){
            factoryName = getConnectionFactoryName();
            destinationName = getDestinationName();
            Field connectionFactoryField =
                    createInjectedField(target, factoryName, "javax.jms.ConnectionFactory"); //NO18N
            Field destinationField =
                    createInjectedField(target, destinationName, type);
            destinationFieldName = destinationField.getName();
            connectionFactoryFieldName = connectionFactoryField.getName();
            if (!Utils.containsFeature(target, connectionFactoryField)){
                target.getFeatures().add(0, connectionFactoryField);
                addedFeatures.add(connectionFactoryField);
            }
            if (!Utils.containsFeature(target, destinationField)){
                target.getFeatures().add(1, destinationField);
                addedFeatures.add(destinationField);
            }
        } else {
            factoryName = generateConnectionFactoryReference(erc, className);
            destinationName = generateDestinationReference(erc, className, currentProject);
        }
        
        // TODO add code to make method names unique
        Method sendMethod = createSendMethod(target, getDestination());
        Method producer = createJMSProducer(target, factoryName, connectionFactoryFieldName, destinationName,
                destinationFieldName,sendMethod.getName(), target, sl);
        
        if (!Utils.containsFeature(target, sendMethod)){
            target.getContents().add(sendMethod);
            addedFeatures.add(sendMethod);
        }
        if (!Utils.containsFeature(target, producer)){
            target.getContents().add(producer);
            addedFeatures.add(producer);
        }
        
        boolean failed = true;
        JavaModel.getJavaRepository().beginTrans(true);
        try {
            for(Feature feature : addedFeatures){
                JMIUtils.fixImports(target, feature);
            }
            failed = false;
        } finally {
            JavaModel.getJavaRepository().endTrans(failed);
        }
    }
    
    
    /**
     * Creates an injected resource field for the given <code>target</code>. The name 
     * of the field will be derivated from the given <code>mappedName</code>.
     * @param target the target class
     * @param mappedName the value for resource's mappedName attribute
     * @param clazz the class of the field.
     * @return the created Field.
     */
    private Field createInjectedField(JavaClass target, String mappedName, String clazz) {
        String fieldName = Utils.jndiNameToCamelCase(mappedName, true, "jms"); // NO18N
        int modifier = InjectionTargetQuery.isStaticReferenceRequired(target) ? (Modifier.STATIC | Modifier.PRIVATE) : Modifier.PRIVATE;
        Field field = JMIGenerationUtil.createField(target, fieldName, modifier, clazz);
        AttributeValue av = JMIGenerationUtil.createAttributeValue(target, "mappedName", mappedName); //NO18N
        Annotation a = JMIGenerationUtil.createAnnotation(target, "javax.annotation.Resource", Collections.singletonList(av)); //NO18N
        field.getAnnotations().add(a);
        return field;
    }
    
    
    private Method createSendMethod(JavaClass jc, String destination) {
        Method me = JMIUtils.createMethod(jc);
        me.setModifiers(Modifier.PRIVATE);
        // TODO make method unique
        me.setName("createJMSMessageFor"+destination); // NOI18N
        me.setType(JMIUtils.resolveType("javax.jms.Message"));
        JMIUtils.addException(me, "javax.jms.JMSException");
        JavaModelPackage jmp = (JavaModelPackage) me.refImmediatePackage();
        Parameter param = jmp.getParameter().createParameter(
                "session",
                Collections.EMPTY_LIST,
                false,
                jmp.getMultipartId().createMultipartId("javax.jms.Session", null, null), // type name
                0,
                false);
        Parameter context = jmp.getParameter().createParameter(
                "messageData",
                Collections.EMPTY_LIST,
                false,
                jmp.getMultipartId().createMultipartId(Object.class.getName(), null, null), // type name
                0,
                false);
        Parameter[] params = new Parameter[] { param, context };
        me.getParameters().addAll(Arrays.asList(params));
        String methodBody =
                "// TODO create and populate message to send\n" +
                "// javax.jms.TextMessage tm = session.createTextMessage();\n" +
                "// tm.setText(messageData.toString());\n"+
                "// return tm;\n";
        me.setBodyText(methodBody);
        return me;
    }
    
    private Method createJMSProducer(JavaClass jc,
            String connectionFactoryName,
            String connectionFactoryFieldName,
            String destinationName,
            String destinationFieldName,
            String sendMethodName,
            JavaClass target,
            ServiceLocatorStrategy sl) {
        Method me = JMIUtils.createMethod(jc);
        me.setModifiers(Modifier.PRIVATE);
        String destName = destinationName.substring(destinationName.lastIndexOf('/')+1);
        StringBuffer destBuff = new StringBuffer(destName);
        destBuff.setCharAt(0, Character.toUpperCase(destBuff.charAt(0)));
        me.setName("sendJMSMessageTo"+destBuff); //NOI18N
        JavaModelPackage jmp = (JavaModelPackage) me.refImmediatePackage();
        Parameter context = jmp.getParameter().createParameter(
                "messageData",
                Collections.EMPTY_LIST,
                false,
                jmp.getMultipartId().createMultipartId(Object.class.getName(), null, null), // type name
                0,
                false);
        Parameter[] params = new Parameter[] { context };
        me.getParameters().addAll(Arrays.asList(params));
        me.setType(JMIUtils.resolveType("void")); //NOI18N
        JMIUtils.addException(me, javax.naming.NamingException.class.getName());
        JMIUtils.addException(me, "javax.jms.JMSException"); //NOI18N
        if (supportsInjection){
            me.setBodyText(getSendJMSCodeWithInjectedFields(connectionFactoryFieldName,
                    destinationFieldName,
                    sendMethodName));
        } else if (sl == null) {
            me.setBodyText(getSendJMSCode(connectionFactoryName,
                    destinationName,
                    sendMethodName));
        } else {
            me.setBodyText(getSendJMSCode(connectionFactoryName, destinationName,
                    sendMethodName, sl, target));
        }
        return me;
    }
    
    /**
     * @return String representing the code for send jms method using injected
     * fields.
     */
    private String getSendJMSCodeWithInjectedFields(String connectionFactoryFieldName,
            String destinationFieldName,
            String messageMethodName){
        
        return MessageFormat.format(
                "javax.jms.Connection connection = null;\n" +
                "javax.jms.Session session = null;\n" +
                "try '{' \n" +
                "connection = {0}.createConnection();\n" +
                "session = connection.createSession(false,javax.jms.Session.AUTO_ACKNOWLEDGE);\n" +
                "javax.jms.MessageProducer messageProducer = session.createProducer({1});\n" +
                "messageProducer.send({2}(session, messageData));\n" +
                " '}' finally '{'\n" +
                "if (session != null) '{'\n"+
                " session.close();\n" +
                "'}'\n" +
                "if (connection != null) '{'\n" +
                "connection.close();\n" +
                "'}'\n" +
                "'}'\n",
                connectionFactoryFieldName, destinationFieldName, messageMethodName);
    }
    
    private String getSendJMSCode(String connectionName, String destinationName,
            String messageMethodName, ServiceLocatorStrategy sls,
            JavaClass targetClass) {
        String connectionFactory = sls.genJMSFactory(connectionName, targetClass);
        String destination = sls.genDestinationLookup(destinationName, targetClass);
        return MessageFormat.format(
                "javax.jms.ConnectionFactory cf = (javax.jms.ConnectionFactory) " + connectionFactory + ";\n" +
                "javax.jms.Connection conn = null;\n" +
                "javax.jms.Session s = null;\n" +
                "try '{' \n" +
                "conn = cf.createConnection();\n" +
                "s = conn.createSession(false,s.AUTO_ACKNOWLEDGE);\n" +
                "javax.jms.Destination destination = (javax.jms.Destination) " + destination + ";\n" +
                "javax.jms.MessageProducer mp = s.createProducer(destination);\n" +
                "mp.send({2}(s,messageData));\n" +
                " '}' finally '{'\n" +
                "if (s != null) '{'\n"+
                " s.close();\n" +
                "'}'\n" +
                "if (conn != null) '{'\n" +
                "conn.close();\n" +
                "'}'\n" +
                "'}'\n",
                new Object[] {connectionName, destinationName, messageMethodName});
    }
    
    private String getSendJMSCode(String connectionName, String destinationName,
            String messageMethodName) {
        return MessageFormat.format(
                "javax.naming.Context c = new javax.naming.InitialContext();\n" +
                "javax.jms.ConnectionFactory cf = (javax.jms.ConnectionFactory) c.lookup(\"java:comp/env/{0}\");\n" +
                "javax.jms.Connection conn = null;\n" +
                "javax.jms.Session s = null;\n" +
                "try '{' \n" +
                "conn = cf.createConnection();\n" +
                "s = conn.createSession(false,s.AUTO_ACKNOWLEDGE);\n" +
                "javax.jms.Destination destination = (javax.jms.Destination) c.lookup(\"java:comp/env/{1}\");\n" +
                "javax.jms.MessageProducer mp = s.createProducer(destination);\n" +
                "mp.send({2}(s,messageData));\n" +
                " '}' finally '{'\n" +
                "if (s != null) '{'\n"+
                " s.close();\n" +
                "'}'\n" +
                "if (conn != null) '{'\n" +
                "conn.close();\n" +
                "'}'\n" +
                "'}'\n",
                new Object[] {connectionName, destinationName, messageMethodName});
    }
}

