/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.j2ee.ejbcore.ui.logicalview.entres;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import javax.swing.AbstractListModel;
import javax.swing.ComboBoxModel;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JComboBox;
import javax.swing.JList;
import javax.swing.JSeparator;
import javax.swing.SwingUtilities;
import org.netbeans.modules.j2ee.deployment.common.api.Datasource;
import org.openide.util.NbBundle;

/**
 *
 * DatasourceUIHelper populates and manages the content of the combobox for a data sources management.
 *
 * @author Libor Kotouc
 *
 * @since 1.6
 */
final class DatasourceComboBoxHelper {
    
    private static class DatasourceImpl implements Datasource {
        
        private String jndiName;
        private String url;
        private String username;
        private String password;
        private String driverClassName;
        private String displayName;
        
        public DatasourceImpl(String jndiName, String url, String username, String password, String driverClassName) {
            this.jndiName = jndiName;
            this.url = url;
            this.username = username;
            this.password = password;
            this.driverClassName = driverClassName;
        }
        
        public String getJndiName() {
            return jndiName;
        }
        
        public String getUrl() {
            return url;
        }
        
        public String getUsername() {
            return username;
        }
        
        public String getPassword() {
            return password;
        }
        
        public String getDriverClassName() {
            return driverClassName;
        }
        
        public String getDisplayName() {
            if (displayName == null) {
                displayName = getJndiName() + " [" + getUrl() + "]"; // NOI18N
            }
            return displayName;
        }
        
    }
    
    private static final class Separator extends JSeparator {
        Separator() {
            setPreferredSize(new Dimension(getWidth(), 1));
            setForeground(Color.BLACK);
        }
    }
    
    static final Separator SEPARATOR_ITEM = new Separator();
    static final Object NEW_ITEM = new Object() {
        public String toString() {
            return NbBundle.getMessage(DatasourceComboBoxHelper.class, "LBL_NEW_DATASOURCE"); // NOI18N
        }
    };
    
    private static class DatasourceComboBoxModel extends AbstractListModel implements ComboBoxModel {
        
        private Object[] items;
        private Object selectedItem;
        private Object previousItem;
        
        DatasourceComboBoxModel(Object[] items) {
            this.items = items;
            if (items.length > 0 && items[0] instanceof Datasource) {
                selectedItem = items[0];
            }
        }
        
        public void setSelectedItem(Object anItem) {
            
            if (selectedItem == null || !selectedItem.equals(anItem)) {
                
                previousItem = selectedItem;
                selectedItem = anItem;
                
                fireContentsChanged(this, 0, -1);
            }
        }
        
        public Object getSelectedItem() {
            return selectedItem;
        }
        
        public Object getElementAt(int index) {
            return items[index];
        }
        
        public int getSize() {
            return items.length;
        }
        
        Object getPreviousItem() {
            return previousItem;
        }
    }
    
    private static class DatasourceListCellRenderer extends DefaultListCellRenderer {
        
        public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
            
            if (isSelected) {
                setBackground(list.getSelectionBackground());
                setForeground(list.getSelectionForeground());
            } else {
                setBackground(list.getBackground());
                setForeground(list.getForeground());
            }
            
            if (value instanceof Datasource) {
                Datasource ds = (Datasource) value;
                setText(ds != null ? ds.getDisplayName() : ""); // NOI18N
                setToolTipText(ds.toString());
            } else
                if (value == SEPARATOR_ITEM) {
                return SEPARATOR_ITEM;
                } else {
                setText(value != null ? value.toString() : ""); // NOI18N
                setToolTipText(""); // NOI18N
                }
            
            return this;
        }
        
    }
    
    private static class DatasourceComparator implements Comparator<Datasource> {
        
        public int compare(Datasource ds1, Datasource ds2) {
            
            if (ds1 == null) {
                return ds2 == null ? 0 : -1;
            } else {
                if (ds2 == null) {
                    return 1;
                } else {
                    String dispName1 = ds1.getDisplayName();
                    String dispName2 = ds2.getDisplayName();
                    if (dispName1 == null) {
                        return dispName2 == null ? 0 : -1;
                    } else {
                        return dispName2 == null ? 1 : dispName1.compareToIgnoreCase(dispName2);
                    }
                }
            }
        }
    }
    
    private Set<Datasource> datasources = new TreeSet(new DatasourceComparator());
    
    /**
     * Entry point for the combobox initialization. It connects combobox with its content and
     * add items for the combobox content management.
     *
     * @param combo combobox to manage
     */
    DatasourceComboBoxHelper(final JComboBox combo) {
        
        combo.setRenderer(new DatasourceListCellRenderer());
        
        populate(combo, null, false);
        
        combo.addKeyListener(new KeyAdapter() {
            public void keyPressed(KeyEvent e) {
                int keyCode = e.getKeyCode();
                if (KeyEvent.VK_ENTER == keyCode) {
                    Object selectedItem = combo.getSelectedItem();
                    if (selectedItem == NEW_ITEM) {
                        Datasource ds = performCreateDsAction(combo);
                        combo.setPopupVisible(false);
                        e.consume();
                        if (ds != null)
                            populate(combo, ds, false);
                    }
                }
            }
        });
        
        combo.addActionListener(new ActionListener() {
            
            Object previousItem;
            int previousIndex = combo.getSelectedIndex();
            
            public void actionPerformed(ActionEvent e) {
                
                Object selectedItem = combo.getSelectedItem();
                // skipping of separator
                if (selectedItem == SEPARATOR_ITEM) {
                    int selectedIndex = combo.getSelectedIndex();
                    if (selectedIndex > previousIndex) {
                        previousIndex = selectedIndex + 1;
                        previousItem = combo.getItemAt(previousIndex);
                    } else {
                        previousIndex = selectedIndex - 1;
                        previousItem = combo.getItemAt(previousIndex);
                    }
                    combo.setSelectedItem(previousItem);
                    // handling mouse click, see KeyEvent.getKeyModifiersText(e.getModifiers())
                } else if ((e.getModifiers() & InputEvent.BUTTON1_MASK) != 0) {
                    if (selectedItem == NEW_ITEM) {
                        Datasource ds = performCreateDsAction(combo);
                        combo.setPopupVisible(false);
                        if (ds != null)
                            populate(combo, ds, true);
                    }
                }
            }
        });
        
    }
    
    private Datasource performCreateDsAction(final JComboBox combo) {
        
        final DatasourceComboBoxModel model = (DatasourceComboBoxModel)combo.getModel();
        
        DatasourceComboBoxCustomizer dsc = new DatasourceComboBoxCustomizer(datasources);
        boolean accept = dsc.showDialog();
        Datasource ds = null;
        if (accept) {
            if (dsc.getPassword() != null) {
                ds = new DatasourceImpl(
                        dsc.getJndiName(),
                        dsc.getUrl(),
                        dsc.getUsername(),
                        dsc.getPassword(),
                        dsc.getDriverClassName());
                datasources.add(ds);
            }
        } 
        
        if (ds == null) {
            SwingUtilities.invokeLater(new Runnable() { // postpone item selection to enable event firing from JCombobox.setSelectedItem()
                public void run() {
                    combo.setSelectedItem(model.getPreviousItem());
                }
            });
        }
        
        return ds;
    }
    
    private void populate(final JComboBox combo, final Datasource selectedDatasource, boolean selectItemLater) {
        
        List<Object> items = new LinkedList<Object>(datasources);
        
        if (items.size() > 0) {
            items.add(SEPARATOR_ITEM);
        }
        
        items.add(NEW_ITEM);
        
        Object[] itemsArray = items.toArray();
        DatasourceComboBoxModel model = new DatasourceComboBoxModel(itemsArray);
        
        combo.setModel(model);
        
        if (selectedDatasource != null) {
            
            // Ensure that the correct item is selected before listeners like FocusListener are called.
            // ActionListener.actionPerformed() is not called if this method is already called from
            // actionPerformed(), in that case selectItemLater should be set to true and setSelectedItem()
            // below is called asynchronously so that the actionPerformed() is called
            combo.setSelectedItem(selectedDatasource);
            
            if (selectItemLater) {
                SwingUtilities.invokeLater(new Runnable() { // postpone item selection to enable event firing from JCombobox.setSelectedItem()
                    public void run() {
                        combo.setSelectedItem(selectedDatasource);
                    }
                });
            }
            
        }
    }
    
}
