/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 *
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.j2ee.ejbcore.ui.logicalview.ejb.dnd;

import java.lang.reflect.Modifier;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import javax.naming.NamingException;
import org.netbeans.api.project.FileOwnerQuery;
import org.netbeans.api.project.Project;
import org.netbeans.api.project.ant.AntArtifact;
import org.netbeans.jmi.javamodel.Annotation;
import org.netbeans.jmi.javamodel.Feature;
import org.netbeans.jmi.javamodel.Field;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.Method;
import org.netbeans.modules.j2ee.api.ejbjar.EjbReference;
import org.netbeans.modules.j2ee.common.JMIUtils;
import org.netbeans.modules.j2ee.common.queries.api.InjectionTargetQuery;
import org.netbeans.modules.j2ee.dd.api.common.EjbLocalRef;
import org.netbeans.modules.j2ee.dd.api.common.EjbRef;
import org.netbeans.modules.j2ee.dd.api.ejb.EntityAndSession;
import org.netbeans.modules.j2ee.dd.api.ejb.Session;
import org.netbeans.modules.j2ee.ejbcore.Utils;
import org.netbeans.modules.j2ee.ejbcore.ui.logicalview.entres.ServiceLocatorStrategy;
import org.netbeans.modules.j2ee.common.JMIGenerationUtil;
import org.netbeans.modules.javacore.api.JavaModel;
import org.openide.filesystems.FileObject;
import org.openide.nodes.Node;

/**
 *
 * @author Martin Adamek
 */
public class EjbReferenceImpl implements Node.Cookie, EjbReference {
    
    private AntArtifact precursor;
    private EntityAndSession dd;
    private boolean isSimplified;
    
    public EjbReferenceImpl(AntArtifact precursor, EntityAndSession dd) {
        this.precursor = precursor;
        this.dd = dd;
        this.isSimplified = dd.getRoot().getVersion().doubleValue() > 2.1;
    }
    
    public AntArtifact getClientJarTarget() {
        return precursor;
    }
    
    public EjbRef createRef() {
        EjbRef ref  = dd.newEjbRef();
        populateReference(ref);
        return ref;
    }
    
    public EjbLocalRef createLocalRef() {
        EjbLocalRef ref = dd.newEjbLocalRef();
        populateReference(ref);
        return ref;
    }
    
    public boolean supportsLocalInvocation() {
        if (isSimplified) {
            return dd.getLocal() != null;
        } else {
            return dd.getLocal() != null && dd.getLocalHome() != null;
        }
    }
    
    public boolean supportsRemoteInvocation() {
        if (isSimplified) {
            return dd.getRemote() != null;
        } else {
            return dd.getRemote() != null && dd.getHome() != null;
        }
    }
    
    public void populateReference(EjbRef ref) {
        ref.setEjbRefName("ejb/"+dd.getEjbName());
        if (dd instanceof Session) {
            ref.setEjbRefType("Session");
        } else {
            ref.setEjbRefType("Entity");
        }
        ref.setRemote(dd.getRemote());
        if (dd.getHome() != null) {
            ref.setHome(dd.getHome());
        }
        // only first URI is taken, if more of them are defined, just first one is taken
        String jarName = "";
        if (getClientJarTarget() != null) {
            String[] names = getClientJarTarget().getArtifactLocations()[0].getPath().split("/");  //NOI18N
            jarName = names[names.length - 1] + "#";
        }
        ref.setEjbLink(jarName + dd.getEjbName());
    }
    
    public void populateReference(EjbLocalRef ref) {
        ref.setEjbRefName("ejb/"+dd.getEjbName());
        if (dd instanceof Session) {
            ref.setEjbRefType("Session");
        } else {
            ref.setEjbRefType("Entity");
        }
        ref.setLocal(dd.getLocal());
        if (dd.getLocalHome() != null) {
            ref.setLocalHome(dd.getLocalHome());
        }
        // only first URI is taken, if more of them are defined, just first one is taken
        String jarName = "";
        if (getClientJarTarget() != null) {
            String[] names = getClientJarTarget().getArtifactLocations()[0].getPath().split("/");  //NOI18N
            jarName = names[names.length - 1] + "#";
        }
        ref.setEjbLink(jarName + dd.getEjbName());
    }
    
    public Feature generateServiceLocatorLookup(JavaClass target, EjbRef ref, String serviceLocatorName, boolean throwExceptions) {
        return generateServiceLocatorJNDI(target, ref.getHome(), ref.getEjbRefName(), true, ref.getRemote(), throwExceptions, serviceLocatorName);
    }
    
    public Feature generateServiceLocatorLookup(JavaClass target, EjbLocalRef ref, String serviceLocatorName, boolean throwExceptions) {
        return generateServiceLocatorJNDI(target, ref.getLocalHome(), ref.getEjbRefName(), false, ref.getLocal(), throwExceptions, serviceLocatorName);
    }
    
    public Feature generateReferenceCode(JavaClass target, EjbRef ref, boolean throwExceptions) {
        boolean isInjectionTarget = InjectionTargetQuery.isInjectionTarget(target);
        if (isInjectionTarget) {
            return generateInjection(target, ref.getEjbRefName(), ref.getRemote());
        } else {
            return generateJNDI(target, ref.getHome(), ref.getEjbRefName(), true, ref.getRemote(), throwExceptions);
        }
    }
    
    public Feature generateReferenceCode(JavaClass target, EjbLocalRef ref, boolean throwExceptions) {
        boolean isInjectionTarget = InjectionTargetQuery.isInjectionTarget(target);
        if (isInjectionTarget) {
            return generateInjection(target, ref.getEjbRefName(), ref.getLocal());
        } else {
            return generateJNDI(target, ref.getLocalHome(), ref.getEjbRefName(), false, ref.getLocal(), throwExceptions);
        }
    }
    
    // private stuff ===========================================================
    
    private static final String LOG_STATEMENT =
            "java.util.logging.Logger.getLogger(getClass().getName()).log(java.util.logging.Level.SEVERE,\"exception caught\" ,{0});\n";
    
    private static final String JNDI_LOOKUP_LOCAL =
            "javax.naming.Context c = new javax.naming.InitialContext();\n" +
            "{1} rv = ({1}) c.lookup(\"java:comp/env/{0}\");\n" +
            "return rv{2};\n";
    
    private static final String JNDI_LOOKUP_REMOTE =
            "javax.naming.Context c = new javax.naming.InitialContext();\n" +
            "Object remote = c.lookup(\"java:comp/env/{0}\");\n" +
            "{1} rv = ({1}) javax.rmi.PortableRemoteObject.narrow(remote, {1}.class);\n" +
            "return rv{2};\n";

    /**
     * Lookup code for EJB 2.x beans in Java SE environments.
     */
    private static final String JNDI_LOOKUP_REMOTE_JAVASE =
            "javax.naming.Context c = new javax.naming.InitialContext();\n" +
            "Object remote = c.lookup(\"{0}\");\n" +
            "{1} rv = ({1}) javax.rmi.PortableRemoteObject.narrow(remote, {1}.class);\n" +
            "return rv{2};\n";
    
    private static final String JNDI_LOOKUP_EJB3 =
            "javax.naming.Context c = new javax.naming.InitialContext();\n" +
            "return ({1}) c.lookup(\"java:comp/env/{0}\");\n";
    
    /**
     * Lookup code for EJB3 beans in Java SE environments.
     */
    private static final String JNDI_LOOKUP_EJB3_JAVASE =
            "javax.naming.Context c = new javax.naming.InitialContext();\n" +
            "return ({1}) c.lookup(\"{0}\");\n";
    
    private Feature generateInjection(JavaClass target, String refName, String componentName) {
        String name = refName.substring(refName.lastIndexOf('/') + 1);
        name = Character.toLowerCase(name.charAt(0)) + name.substring(1);
        int modifier = InjectionTargetQuery.isStaticReferenceRequired(target) ? (Modifier.STATIC | Modifier.PRIVATE) : Modifier.PRIVATE;
        Field f = JMIGenerationUtil.createField(target, JMIUtils.uniqueMemberName(target, name, "ejb"), modifier, componentName); //NOI18N
        Annotation a = JMIGenerationUtil.createAnnotation(target, "javax.ejb.EJB", Collections.EMPTY_LIST);
        f.getAnnotations().add(a);
        target.getContents().add(0, f);
        return f;
    }
    
    private Method generateJNDI(JavaClass target, String homeName, String refName, boolean narrow, String componentName, boolean throwCheckedExceptions) {
        boolean rollback = true; // rollback the transaction by default
        JMIUtils.beginJmiTransaction(true);
        try {
            JavaClass ejbClass = JMIUtils.findClass(dd.getEjbClass());
            if (ejbClass==null) {
                // likely the lookup method for session bean from another project is generated
                // set classpath for java model from another project
                JavaModel.setClassPath(precursor.getProject().getProjectDirectory());
                ejbClass = JMIUtils.findClass(dd.getEjbClass());
            }
            String name = "lookup"+refName.substring(refName.lastIndexOf('/')+1);
            Method me = JMIGenerationUtil.createMethod(target, JMIUtils.uniqueMemberName(ejbClass, name, "ejb"), Modifier.PRIVATE, isSimplified ? componentName : homeName); //NOI18N
            String body = null;
            List<String> exceptions = new ArrayList<String>(3);
            exceptions.add(NamingException.class.getName());
            String sessionCreate = "";
            if (isSession()) {
                me.setType(JMIUtils.findClass(componentName));
                sessionCreate = ".create()";
                if (!isSimplified) {
                    exceptions.add("javax.ejb.CreateException");
                }
                if (narrow && !isSimplified) {
                    exceptions.add("java.rmi.RemoteException");
                }
            }
            boolean isTargetJavaSE = Utils.isTargetJavaSE(target);
            if (isSimplified && isTargetJavaSE){
                body = MessageFormat.format(JNDI_LOOKUP_EJB3_JAVASE, new Object[] {ejbClass.getName(), componentName});
            } else if (isSimplified) {
                body = MessageFormat.format(JNDI_LOOKUP_EJB3, new Object[] {refName, componentName});
            } else if (isTargetJavaSE){
                body = MessageFormat.format(JNDI_LOOKUP_REMOTE_JAVASE, new Object[] {homeName, homeName, sessionCreate});
            } else if (narrow) {
                body = MessageFormat.format(JNDI_LOOKUP_REMOTE, new Object[] {refName, homeName, sessionCreate});
            } else {
                body = MessageFormat.format(JNDI_LOOKUP_LOCAL, new Object[] {refName, homeName, sessionCreate});
            }
            if (throwCheckedExceptions) {
                me.setBodyText(body);
                JMIUtils.addExceptions(me, exceptions);
            } else {
                Iterator exIt = exceptions.iterator();
                StringBuffer catchBody = new StringBuffer("try {\n" + body + "}\n"); // NOI18N
                while (exIt.hasNext()) {
                    String exceptionName = (String) exIt.next();
                    catchBody.append("catch("); // NOI18N
                    catchBody.append(exceptionName);
                    catchBody.append(' ');  //NOI18N
                    String id = extractAllCapitalLetters(exceptionName);
                    catchBody.append(id);
                    catchBody.append(") {\n"); //NOI18N
                    catchBody.append(MessageFormat.format(LOG_STATEMENT,
                            new Object[] {id}));
                    catchBody.append("throw new RuntimeException("+id+");\n");
                    catchBody.append("}\n"); //NOI18N
                }
                me.setBodyText(catchBody.toString());
            }
            target.getFeatures().add(me);
            rollback = false;
            return me;
        } finally {
            JMIUtils.endJmiTransaction(rollback);
        }
    }
    
    
    private Feature generateServiceLocatorJNDI(JavaClass target, String homeName, String refName,
            boolean narrow, String componentName,
            boolean throwCheckedExceptions,
            String serviceLocatorName) {
        boolean rollback = true; // rollback the transaction by default
        JMIUtils.beginJmiTransaction(true);
        try {
            String name = "lookup"+refName.substring(refName.lastIndexOf('/')+1);
            Method me = JMIGenerationUtil.createMethod(target, JMIUtils.uniqueMemberName(target, name, "ejb"), Modifier.PRIVATE, isSimplified ? componentName : homeName); //NOI18N
            String body = null;
            List<String> exceptions = new ArrayList<String>(3);
            exceptions.add(NamingException.class.getName());
            boolean genCreate = isSession();
            if (genCreate) {
                me.setType(JMIUtils.findClass(componentName));
                exceptions.add("javax.ejb.CreateException"); //NOI18N
                if (narrow) {
                    exceptions.add("java.rmi.RemoteException"); //NOI18N
                }
            }
            FileObject srcFile = JavaModel.getFileObject(target.getResource());
            Project enterpriseProject = FileOwnerQuery.getOwner(srcFile);
            ServiceLocatorStrategy sls = ServiceLocatorStrategy.create(enterpriseProject, srcFile, serviceLocatorName);
            if (narrow) {
                body = sls.genRemoteEjbStringLookup(refName, homeName, target, genCreate);
            } else {
                body = sls.genLocalEjbStringLookup(refName, homeName, target, genCreate);
            }
            if (throwCheckedExceptions) {
                me.setBodyText(body);
                JMIUtils.addExceptions(me, exceptions);
            } else {
                Iterator exIt = exceptions.iterator();
                StringBuffer catchBody = new StringBuffer("try {\n" + body + "\n}"); // NOI18N
                while (exIt.hasNext()) {
                    String exceptionName = (String) exIt.next();
                    catchBody.append(" catch("); // NOI18N
                    catchBody.append(exceptionName);
                    catchBody.append(' ');  //NOI18N
                    String id = extractAllCapitalLetters(exceptionName);
                    catchBody.append(id);
                    catchBody.append(") {\n"); //NOI18N
                    catchBody.append(MessageFormat.format(LOG_STATEMENT, new Object[] {id}));
                    catchBody.append("throw new RuntimeException("+id+");\n");
                    catchBody.append("}"); //NOI18N
                }
                me.setBodyText(catchBody.toString());
            }
            target.getFeatures().add(me);
            rollback = false;
            return me;
        } finally {
            JMIUtils.endJmiTransaction(rollback);
        }
    }
    
    private String extractAllCapitalLetters(String word) {
        StringBuffer caps = new StringBuffer(4);
        for (int i =0; i < word.length(); i++) {
            char c = word.charAt(i);
            if (Character.isUpperCase(c)) {
                caps.append(Character.toLowerCase(c));
            }
        }
        return caps.toString();
    }
    
    private boolean isSession() {
        return dd instanceof Session;
    }
}
