/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 *
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.j2ee.dd.impl.ejb;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.netbeans.jmi.javamodel.AnnotableElement;
import org.netbeans.jmi.javamodel.Annotation;
import org.netbeans.jmi.javamodel.ArrayInitialization;
import org.netbeans.jmi.javamodel.AttributeValue;
import org.netbeans.jmi.javamodel.ClassDefinition;
import org.netbeans.jmi.javamodel.ClassExpression;
import org.netbeans.jmi.javamodel.Element;
import org.netbeans.jmi.javamodel.Field;
import org.netbeans.jmi.javamodel.InitialValue;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.JavaModelPackage;
import org.netbeans.jmi.javamodel.NamedElement;
import org.netbeans.jmi.javamodel.StringLiteral;
import org.netbeans.jmi.javamodel.Type;
import org.netbeans.jmi.javamodel.TypeReference;
import org.netbeans.jmi.javamodel.TypedElement;
import org.netbeans.modules.j2ee.dd.api.common.MessageDestination;
import org.netbeans.modules.j2ee.dd.api.common.PortComponentRef;
import org.netbeans.modules.j2ee.dd.api.common.ServiceRef;
import org.netbeans.modules.j2ee.dd.api.common.VersionNotSupportedException;
import org.netbeans.modules.j2ee.dd.api.ejb.ActivationConfig;
import org.netbeans.modules.j2ee.dd.api.ejb.ActivationConfigProperty;
import org.netbeans.modules.j2ee.dd.api.ejb.AssemblyDescriptor;
import org.netbeans.modules.j2ee.dd.api.ejb.ContainerTransaction;
import org.netbeans.modules.j2ee.dd.api.ejb.DDProvider;
import org.netbeans.modules.j2ee.dd.api.ejb.Ejb;
import org.netbeans.modules.j2ee.dd.api.ejb.EjbJar;
import org.netbeans.modules.j2ee.dd.api.ejb.EnterpriseBeans;
import org.netbeans.modules.j2ee.dd.api.ejb.MessageDriven;
import org.netbeans.modules.j2ee.dd.api.ejb.Session;
import org.netbeans.modules.j2ee.metadata.Utils;
import org.openide.ErrorManager;

/**
 * Utils used for building EJB model from annotations.
 * Mostly useful in cases where common NNModelBuilder cannot help and custom handling is needed.
 *
 * @author Martin Adamek
 * @author Marek Fukala
 */
public final class EjbBuilderUtil {
    
    private static final ErrorManager LOGGER = ErrorManager.getDefault().getInstance(EjbBuilderUtil.class.getName());
    private static final boolean LOG = LOGGER.isLoggable(ErrorManager.INFORMATIONAL);
    
    public static final String JAVAX_EJB_REMOTE = "javax.ejb.Remote";
    public static final String JAVAX_EJB_LOCAL = "javax.ejb.Local";
    public static final String JAVAX_EJB_STATELESS = "javax.ejb.Stateless";
    public static final String JAVAX_EJB_STATEFUL = "javax.ejb.Stateful";
    public static final String JAVAX_EJB_MESSAGEDRIVEN = "javax.ejb.MessageDriven";
    public static final String JAVAX_EJB_EJB = "javax.ejb.EJB";
    
    public static final String WS_PKG = "javax.xml.ws.";
    public static final String WS_SERVICE_CLASS_NAME = "Service";
    public static final String WS_ENDPOINT_CLASS_NAME = "Endpoint";
    
    public static final String JAVAX_ANNOTATION_RESOURCE = "javax.annotation.Resource";
    public static final String JAVAX_ANNOTATION_WEBSERVICEREF = "javax.xml.ws.WebServiceRef";
    
    private EjbJar ejbJar;
    
    public EjbBuilderUtil(EjbJar ejbJar) {
        this.ejbJar = ejbJar;
    }
    
    public void webServiceRefNNAdded(JavaClass javaClass, Element member, Annotation annotation) {
        //The default framework creates the ServiceRef element
        //and sets the ServiceRefName and WsdlFile properties
        //
        //Now the special cases described in JSR-109 specifications needs to be handled:
        
        //At first, find the <service-ref> bean
        ServiceRef sref = null;
        try {
            String javaClassName = javaClass.getName();
            Session sb = getSession(javaClassName);
            MessageDriven md = getMessageDriven(javaClassName);
            String serviceRefName = javaClassName + "/" + ((NamedElement)member).getName();
            ServiceRef[] srefs = (ServiceRef[])(sb == null ? ( md == null ? null : md.getServiceRef() ) : sb.getServiceRef());
            if(srefs != null) {
                for(ServiceRef sr : srefs) {
                    if(serviceRefName.equals(sr.getServiceRefName())) {
                        sref = sr;
                        break;
                    }
                }
            }
        } catch (VersionNotSupportedException e) {
            ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, e);
        }
        if(sref == null) {
            //<service-ref> element not defined for the class member
            //once the primary annotations are fired, this event will be refired again
            return ;
        }
        
        
        JavaModelPackage jmp = (JavaModelPackage)javaClass.refImmediatePackage();
        Type _ServiceType = jmp.getJavaClass().resolve(WS_PKG + WS_SERVICE_CLASS_NAME);
        Type _EndpointType = jmp.getJavaClass().resolve(WS_PKG + WS_ENDPOINT_CLASS_NAME);
        
        JavaClass typeJC = null;
        JavaClass valueJC = null;
        
        //get annotation member type
        if(member instanceof Field) {
            typeJC = (JavaClass)((Field)member).getTypeName().getElement();
        }
        
        String wsdlFileAttr = null;
        //get the java classes from attribute values
        for(AttributeValue av : (List<AttributeValue>)annotation.getAttributeValues()) {
            String aName = av.getDefinition().getName();
            InitialValue val = av.getValue();
            if(val instanceof ClassExpression) {
                TypeReference tr = ((ClassExpression)val).getClassName();
                JavaClass jc = (JavaClass)tr.getElement();
                if("type".equals(aName)) {
                    typeJC = jc;
                }
                if("value".equals(aName)) {
                    valueJC = jc;
                }
            } else if (val instanceof StringLiteral) {
                if("wsdlLocation".equals(aName)) {
                    StringLiteral sl = (StringLiteral)val;
                    wsdlFileAttr = sl.getValue();
                }
            }
        }
        
        //<service-ref>/<service-interface>
        if(valueJC == null) {
            //= @WebServiceRef.type (annotation declares a Service)
            sref.setServiceInterface(typeJC.getName());
        }
        
        if(Utils.isOfType(typeJC, _EndpointType)) {
            //= @WebServiceRef.value
            if(valueJC != null) {
                sref.setServiceInterface(valueJC.getName());
            }
        }
        
        //<service-ref>/<port-component-ref>/<service-endpoint-interface>
        if(valueJC != null && Utils.isOfType(valueJC, _ServiceType)) {
            //= @WebServiceRef.type
            PortComponentRef pcr = null;
            try {
                pcr = sref.newPortComponentRef();
            } catch (VersionNotSupportedException ex) {
                ErrorManager.getDefault().notify(ex);
            }
            if (pcr != null) {
                pcr.setServiceEndpointInterface(typeJC.getName());
                sref.addPortComponentRef(pcr);
            }
        }
        
        //<service-ref>/<wsdl-file>
        if(wsdlFileAttr != null) {
            try {
                URI wsdlFileURI = new URI(wsdlFileAttr);
                sref.setWsdlFile(wsdlFileURI);
            }catch(URISyntaxException e) {
                //just ignore
            }
        }
        //<service-ref>/<service-ref-type> = @WebServiceRef.type
        //there doesn't seem to be <service-ref-type> property in the current schema!?!?
    }
    
    public String handleMessageDriven(JavaClass javaClass, Annotation annotation) {
        if (LOG) LOGGER.log("handleMessageDriven: " + javaClass.getName() + ", " + annotation.getType().getName());
        Map<String, String> activationConfigProperties = getActivationConfigProperties(annotation);
        MessageDriven mdb = getMessageDriven(javaClass.getName());
        try {
            mdb.setMessageDestinationType(activationConfigProperties.get("destinationType"));
        } catch (VersionNotSupportedException ex) {
            ErrorManager.getDefault().notify(ex);
        }
        addOrUpdateActivationConfigProperty(mdb, "destinationType", activationConfigProperties.get("destinationType"));
        addOrUpdateActivationConfigProperty(mdb, "acknowledgeMode", activationConfigProperties.get("acknowledgeMode"));
        addOrUpdateActivationConfigProperty(mdb, "subscriptionDurability", activationConfigProperties.get("subscriptionDurability"));
        addOrUpdateActivationConfigProperty(mdb, "messageSelector", activationConfigProperties.get("messageSelector"));
        try {
            addOrUpdateAssemblyDescriptorForMDB(mdb);
        } catch (VersionNotSupportedException ex) {
            ErrorManager.getDefault().notify(ex);
        }
        return mdb.getEjbName();
    }
    
    public void updateLocal(JavaClass javaClass) {
        if (javaClass.isInterface()) {
            Collection<ClassDefinition> implementors = javaClass.getImplementors();
            for (ClassDefinition cd : implementors) {
                if (cd instanceof JavaClass && !((JavaClass) cd).isInterface()) {
                    try {
                        handleInterfaces((JavaClass) cd);
                    } catch (VersionNotSupportedException ex) {
                        ErrorManager.getDefault().notify(ex);
                    }
                }
            }
        } else {
            try {
                handleInterfaces(javaClass);
            } catch (VersionNotSupportedException ex) {
                ErrorManager.getDefault().notify(ex);
            }
        }
    }
    
    public void updateRemote(JavaClass javaClass) {
        // it is ok to delegate to updateLocal, because finally impl class handles all interfaces
        updateLocal(javaClass);
    }
    
    public void handleInterfaces(JavaClass jc) throws VersionNotSupportedException {
        Session session = getSession(jc.getName());
        if (session == null) {
            ErrorManager.getDefault().log("Session bean for " + jc.getName() + " class not found.");
            return;
        }
        // find all business interface candidates
        List<JavaClass> businessInterfaces = new ArrayList<JavaClass>();
        List<JavaClass> interfaces = jc.getInterfaces();
        for (JavaClass interfaceClass : interfaces) {
            String fqn = interfaceClass.getName();
            if (!"java.io.Serializable".equals(fqn) && !"java.io.Externalizable".equals(fqn) && !fqn.startsWith("javax.ejb")) {
                businessInterfaces.add(interfaceClass);
            }
        }
        // session bean must have at least one business interface
        int biCount = businessInterfaces.size();
        if (biCount < 1) {
            return;
        }
        // find remote interfaces
        List<String> remoteInterfaces = new ArrayList<String>();
        Annotation aRemote = findAnnotation(jc, JAVAX_EJB_REMOTE);
        if (aRemote != null) {
            remoteInterfaces = getInterfaceNames(aRemote);
        }
        remoteInterfaces.addAll(getAnnotatedInterfaces(businessInterfaces, JAVAX_EJB_REMOTE));
        //  find local interfaces
        List<String> localInterfaces = new ArrayList<String>();
        Annotation aLocal = findAnnotation(jc, JAVAX_EJB_LOCAL);
        if (aLocal != null) {
            localInterfaces = getInterfaceNames(aLocal);
        }
        localInterfaces.addAll(getAnnotatedInterfaces(businessInterfaces, JAVAX_EJB_LOCAL));
        // if there is no explicitly annotated local/remote interface and exactly one interface exists
        if (localInterfaces.isEmpty() && remoteInterfaces.isEmpty() && biCount == 1) {
            String name = businessInterfaces.get(0).getName();
            if (!containsName(session.getBusinessLocal(), name) && !name.endsWith("Remote")) { //NO18N
                session.addBusinessLocal(name);
                // TODO: this is hack, we don't support multiple business interfaces in UI
                // this way what was working for EJB 2.1 works also for EJB 3.0
                // see #83196
                session.setLocal(name);
            }
            return;
        } 
        if (localInterfaces.isEmpty()) {
            session.setBusinessLocal(new String[0]);
        } else {
            for (String elem : localInterfaces) {
                if (!containsName(session.getBusinessLocal(), elem)) {
                    session.addBusinessLocal(elem);
                }
            }
            // TODO: this is hack, we don't support multiple business interfaces in UI
            // this way what was working for EJB 2.1 works also for EJB 3.0
            // see #83196
            session.setLocal(localInterfaces.get(0));
        }
        if (remoteInterfaces.isEmpty()) {
            session.setBusinessRemote(new String[0]);
        } else {
            for (String elem : remoteInterfaces) {
                if (!containsName(session.getBusinessRemote(), elem)) {
                    session.addBusinessRemote(elem);
                }
            }
            // TODO: this is hack, we don't support multiple business interfaces in UI
            // this way what was working for EJB 2.1 works also for EJB 3.0
            // see #83196
            session.setRemote(remoteInterfaces.get(0));
        }
    }
    
    public static Session findReferencedSessionBean(TypedElement element, Annotation annotation) {
        List<Session> localInterfaceExposers = new ArrayList<Session>();
        List<Session> remoteInterfaceExposers = new ArrayList<Session>();
        String typeName = element.getType().getName();
        List<Session> sbs = getSessionBeansFromAllProjects();
        for (Session session : sbs) {
            if (exposesLocalInterface(session, typeName)) {
                localInterfaceExposers.add(session);
            }
            if (exposesRemoteInterface(session, typeName)) {
                remoteInterfaceExposers.add(session);
            }
        }
        AttributeValue av = findAttribute(annotation, "beanName");
        if (av == null) {
            int localExposers = localInterfaceExposers.size();
            int remoteExposers = remoteInterfaceExposers.size();
            if (localExposers != 0) {
                return localInterfaceExposers.get(0);
            } else if (remoteExposers != 0) {
                return remoteInterfaceExposers.get(0);
            }
        } else {
            Object value = av.getValue();
            if (value instanceof StringLiteral) {
                String beanName = ((StringLiteral) value).getValue();
                if (beanName != null) {
                    return getSessionByEjbNameFromAllProjects(sbs, beanName);
                }
            }
        }
        return null;
    }

    public static boolean exposesLocalInterface(Session session, String typeName) {
        try {
            for (String businessLocal : session.getBusinessLocal()) {
                if (typeName.equals(businessLocal)) {
                    return true;
                }
            }
        } catch (VersionNotSupportedException ex) {
            ErrorManager.getDefault().notify(ex);
        }
        return false;
    }
    
    public static boolean exposesRemoteInterface(Session session, String typeName) {
        try {
            for (String businessRemote : session.getBusinessRemote()) {
                if (typeName.equals(businessRemote)) {
                    return true;
                }
            }
        } catch (VersionNotSupportedException ex) {
            ErrorManager.getDefault().notify(ex);
        }
        return false;
    }
    
    public Session getSession(String ejbClass) {
        return (Session) getEnterpriseBeans().findBeanByName(EnterpriseBeans.SESSION, Ejb.EJB_CLASS, ejbClass);
    }
    
    public MessageDriven getMessageDriven(String ejbClass) {
        return (MessageDriven) getEnterpriseBeans().findBeanByName(EnterpriseBeans.MESSAGE_DRIVEN, Ejb.EJB_CLASS, ejbClass);
    }
    
    public EnterpriseBeans getEnterpriseBeans() {
        EnterpriseBeans enterpriseBeans = ejbJar.getEnterpriseBeans();
        if (enterpriseBeans == null) {
            enterpriseBeans = ejbJar.newEnterpriseBeans();
            ejbJar.setEnterpriseBeans(enterpriseBeans);
        }
        return enterpriseBeans;
    }
    
    private static Session getSessionByEjbNameFromAllProjects(List<Session> sbs, String ejbName) {
        if (ejbName != null) {
            for (Session session : sbs) {
                if (ejbName.equals(session.getEjbName())) {
                    return session;
                }
            }
        }
        return null;
    }
    
    public static List<Session> getSessionBeansFromAllProjects() {
        List<Session> result = new ArrayList<Session>();
        for (EjbJar ejbJar : DDProvider.getDefault().getRoots()) {
            EnterpriseBeans ebs = ejbJar.getEnterpriseBeans();
            if (ebs != null) {
                for (Session session : ebs.getSession()) {
                    result.add(session);
                }
            }
        }
        return result;
    }
    
    // private =================================================================
    
    private static boolean containsName(String[] array, String name) {
        for (String elem : array) {
            if (elem.equals(name)) {
                return true;
            }
        }
        return false;
    }
    
    private void addOrUpdateActivationConfigProperty(MessageDriven mdb, String propertyName, String propertyValue) {
        if (LOG) LOGGER.log("addOrUpdateActivationConfigProperty : " + mdb.getEjbClass() + ", " + propertyName + ", " + propertyValue);
        try {
            ActivationConfig activationConfig = mdb.getActivationConfig();
            if (activationConfig == null) {
                if (LOG) LOGGER.log("addOrUpdateActivationConfigProperty 2 - creating new ActivationConfig");
                activationConfig = mdb.newActivationConfig();
                mdb.setActivationConfig(activationConfig);
            }
            ActivationConfigProperty[] acps = activationConfig.getActivationConfigProperty();
            if (LOG) LOGGER.log("addOrUpdateActivationConfigProperty 3: " + acps.length);
            for (int i = 0; i < acps.length; i++) {
                if (LOG) LOGGER.log("addOrUpdateActivationConfigProperty 4: " + acps[i].getActivationConfigPropertyName() + ", " + acps[i].getActivationConfigPropertyValue());
                if (acps[i].getActivationConfigPropertyName().equals(propertyName)) {
                    if (propertyValue == null) {
                        activationConfig.removeActivationConfigProperty(acps[i]);
                    } else {
                        acps[i].setActivationConfigPropertyValue(propertyValue);
                    }
                    return;
                }
            }
            if (propertyValue != null) {
                if (LOG) LOGGER.log("addOrUpdateActivationConfigProperty 5");
                ActivationConfigProperty newAcp = activationConfig.newActivationConfigProperty();
                newAcp.setActivationConfigPropertyName(propertyName);
                newAcp.setActivationConfigPropertyValue(propertyValue);
                activationConfig.addActivationConfigProperty(newAcp);
            }
        } catch (VersionNotSupportedException ex) {
            ErrorManager.getDefault().notify(ex);
        }
    }
    
    private void addOrUpdateAssemblyDescriptorForMDB(MessageDriven mb) throws VersionNotSupportedException {
        // add transaction requirements
        AssemblyDescriptor ad = ejbJar.getSingleAssemblyDescriptor();
        if (ad == null) {
            ad = ejbJar.newAssemblyDescriptor();
            ejbJar.setAssemblyDescriptor(ad);
        }
        MessageDestination[] messageDestinations = ad.getMessageDestination();
        String ejbName = mb.getEjbName();
        for (int i = 0; i < messageDestinations.length; i++) {
            if (messageDestinations[i].getMessageDestinationName().equals(mb.getMessageDestinationLink())) {
                return;
            }
        }
        MessageDestination md = ad.newMessageDestination();
        md.setDisplayName("Destination for " + ejbName);
        md.setMessageDestinationName(ejbName);
        ad.addMessageDestination(md);
        
        mb.setMessageDestinationLink(ejbName);
        ContainerTransaction ct = ad.newContainerTransaction();
        ct.setTransAttribute("Required"); //NOI18N
        org.netbeans.modules.j2ee.dd.api.ejb.Method m = ct.newMethod();
        m.setEjbName(ejbName);
        m.setMethodName("*"); //NOI18N
        ct.addMethod(m);
        ad.addContainerTransaction(ct);
    }
    
    /**
     * Tries to find all activation config properties from <code>javax.ejb.MessageDriven</code> annotation
     *
     * @param annotation expected is instance of <code>javax.ejb.MessageDriven</code> annotation
     * @return map where key is <code>propertyName</code> and value is <code>propertyValue</code> of
     * <code>javax.ejb.ActivationConfigProperty</code> annotation
     */
    private Map<String, String> getActivationConfigProperties(Annotation annotation) {
        if (annotation == null) {
            throw new IllegalArgumentException("null passed to EjbBuilderUtil.getActivationConfigProperties(Annotation)");
        }
        if (!JAVAX_EJB_MESSAGEDRIVEN.equals(annotation.getType().getName())) {
            throw new IllegalArgumentException(JAVAX_EJB_MESSAGEDRIVEN + " annotation was expected, but " + annotation.getType().getName() + " was passed");
        }
        Map<String, String> resultMap = new HashMap<String, String>();
        // go thru all attributes of annotation
        for (Iterator it = annotation.getAttributeValues().iterator(); it.hasNext();) {
            AttributeValue av = (AttributeValue) it.next();
            // if activationConfig attribute is found ...
            if ("activationConfig".equals(av.getName())) {
                ArrayInitialization ai = (ArrayInitialization) av.getValue();
                // ... go thru all ActivationConfigProperty annotations
                for (Iterator it2 = ai.getElementValues().iterator(); it2.hasNext();) {
                    InitialValue iv = (InitialValue) it2.next();
                    if (iv instanceof Annotation) {
                        Annotation acp = (Annotation) iv;
                        List<AttributeValue> nameAndValue = acp.getAttributeValues();
                        // every ActivationConfigProperty annotation has 2 attributes (propertyName and propertyValue)
                        assert nameAndValue.size() == 2;
                        AttributeValue av1 = nameAndValue.get(0);
                        AttributeValue av2 = nameAndValue.get(1);
                        String attribute1 = ((StringLiteral) av1.getValue()).getValue();
                        String attribute2 = ((StringLiteral) av2.getValue()).getValue();
                        // order of attributes is unknown, so let's find out
                        if ("propertyName".equals(av1.getDefinition().getName())) {
                            resultMap.put(attribute1, attribute2);
                        } else {
                            resultMap.put(attribute2, attribute1);
                        }
                    }
                }
            }
        }
        return resultMap;
    }
    
    private static Annotation findAnnotation(AnnotableElement element, String fqnAnnotation) {
        List<Annotation> annotations = element.getAnnotations();
        for (Annotation a : annotations) {
            Type t = a.getType();
            if (t != null && t.getName().equals(fqnAnnotation)) {
                return a;
            }
        }
        return null;
    }
    
    /**
     * Searches for elements annotated with annotation of given fully-qualified name
     * within list of annotable elements
     *
     * @param list list of annotable elements
     * @param fqnAnnotation fully-qualified name of annotation
     * @return set of found fully-qualified class names or empty set if nothing was found
     */
    private static Set<String> getAnnotatedInterfaces(List<? extends AnnotableElement> list, String fqnAnnotation) {
        Set<String> result = new HashSet<String>();
        for (AnnotableElement element : list) {
            if (findAnnotation(element, fqnAnnotation) != null) {
                result.add(element.getName());
            }
        }
        return result;
    }
    
    /**
     * Searches for attribute of given name within annotation
     *
     * @param annotation annotation where attribute should be found
     * @param attributeName name of attribute to search
     * @return found attribute or null, if not found
     */
    private static AttributeValue findAttribute(Annotation annotation, String attributeName) {
        List<AttributeValue> attributeValues = annotation.getAttributeValues();
        for (AttributeValue av : attributeValues) {
            String avName = av.getName();
            if (avName != null && avName.equals(attributeName)) {
                return av;
            }
        }
        return null;
    }
    
    /**
     * Searches for interfaces names within @Local/@Remote annotations that are
     * defined in 'value' attribute. Note, that this is applicaple only to
     * annotation defined on Session bean class
     *
     * @param annotation @Local or @Remote annotation on Session bean class
     * @return set of fully-qualified names of interfaces, empty set if none was found
     */
    private static List<String> getInterfaceNames(Annotation annotation) {
        List<String> result = new ArrayList<String>();
        AttributeValue valueAV = findAttribute(annotation, "value");
        if (valueAV != null) {
            for (Object elem : ((ArrayInitialization) valueAV.getValue()).getElementValues()) {
                result.add(((ClassExpression) elem).getClassName().getName());
            }
        }
        return result;
    }
    
}
