/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 *
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.j2ee.dd.impl.ejb;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.jmi.javamodel.Annotation;
import org.netbeans.jmi.javamodel.AnnotationType;
import org.netbeans.jmi.javamodel.Element;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.TypedElement;
import org.netbeans.modules.j2ee.dd.api.common.VersionNotSupportedException;
import org.netbeans.modules.j2ee.dd.api.ejb.*;
import org.netbeans.modules.j2ee.metadata.NNModelBuilder;
import org.netbeans.modules.schema2beans.BaseBean;
import org.openide.ErrorManager;

/**
 *
 * @author Martin Adamek
 * @author Marek Fukala
 */
public class EjbAnnotationListener extends NNModelBuilder {
    
    private static final ErrorManager LOGGER = ErrorManager.getDefault().getInstance(EjbAnnotationListener.class.getName());
    private static final boolean LOG = LOGGER.isLoggable(ErrorManager.INFORMATIONAL);
    
    //mapping parts
    private static final String PATH_TO_SESSION = "EnterpriseBeans/Session#EjbClass=$JAVACLASS";
    private static final String PATH_TO_MESSAGE_DRIVEN = "EnterpriseBeans/MessageDriven#EjbClass=$JAVACLASS";
    private static final String ENV_ENTRY =  "/EnvEntry#EnvEntryName=$MEMBER;!name=EnvEntryName;!type=EnvEntryType";
    private static final String RES_REF_ENTRY = "/ResourceRef#ResRefName=$MEMBER;ResType=$MEMBERCLASS;!name=ResRefName;!type=ResType";
    private static final String SERVICE_REF_ENTRY = "/ServiceRef#ServiceRefName=$JAVA_CLASS-MEMBER;!name=ServiceRefName";
    private static final String RES_ENV_REF_ENTRY = "/ResourceEnvRef#ResourceEnvRefName=$MEMBER!;name=ResourceEnvRefName;!type=ResourceEnvRefType";
    private static final String MSG_DEST_ENTRY = "/MessageDestinationRef#MessageDestinationRefName=$MEMBER;!name=MessageDestinationRefName";
    private static final String RES_KEY_PART = EjbBuilderUtil.JAVAX_ANNOTATION_RESOURCE + "#" +MEMBER_CLASS_REF+"=";
    
    private static final Collection<String> primaryAnnotations = Arrays.asList(new String[]{
        EjbBuilderUtil.JAVAX_EJB_STATEFUL,
        EjbBuilderUtil.JAVAX_EJB_STATELESS,
        EjbBuilderUtil.JAVAX_EJB_MESSAGEDRIVEN
    });
    
    /** A map storing annotations to model structure mapping. */
    private Map<String, String> nn2ModelMapping;
    private EjbBuilderUtil util;
    private ClassPath cp;
    
    public EjbAnnotationListener(EjbJar ejbJar, ClassPath cp) {
        super((BaseBean) ejbJar);
        this.util = new EjbBuilderUtil(ejbJar);
        this.cp = cp;
        initNN2ModelMapping();
        if (LOG) dumpMappings();
    }
    
    public ClassPath getClassPath() {
        return cp;
    }
    
    public Collection<String> getPrimaryAnnotations() {
        return primaryAnnotations;
    }
    
    public Map<String,String> getAnnotation2ModelMapping() {
        return nn2ModelMapping;
    }
    
    public void classRemoved(String fqn) {
        Ejb ejb = util.getSession(fqn);
        if (ejb == null) {
            ejb = util.getMessageDriven(fqn);
        }
        if (ejb != null) {
            // problem? all listeners will receive this event, there is no sorting on annotation class
            util.getEnterpriseBeans().removeEjb(ejb);
        }
        handleEJBInterfaceRemove(fqn);
        
    }
    
    /**
     * Checks whether the given <code>fqn</code> is an interface of a session 
     * bean and removes respective entries from the model if it was. See IZ 84602.
     * @param fqn the class that was removed.
     */
    private void handleEJBInterfaceRemove(String fqn){
    
        for (Session sessionBean : util.getEnterpriseBeans().getSession()){
            
            //local
            if (fqn.equals(sessionBean.getLocal())){
                sessionBean.setLocal(null);
            }
            //remote
            if (fqn.equals(sessionBean.getRemote())){
                sessionBean.setRemote(null);
            }
            
            // local and remote business interfaces
            try {
                for (String businessLocal : sessionBean.getBusinessLocal()){
                    if (fqn.equals(businessLocal)){
                        sessionBean.removeBusinessLocal(businessLocal);
                    }
                }
                for (String remote : sessionBean.getBusinessRemote()){
                    if (fqn.equals(remote)){
                        sessionBean.removeBusinessRemote(remote);
                    }
                }
            } catch (VersionNotSupportedException ex) {
                ErrorManager.getDefault().notify(ex);
            }
        }
        
    }
    
    @Override
    public void addClassAnnotation(JavaClass javaClass, Annotation annotation, AnnotationType t) {
        super.addClassAnnotation(javaClass, annotation, t);
        
        //more complicated cases are handled here
        String annotationClass = t.getName();
        if (EjbBuilderUtil.JAVAX_EJB_STATELESS.equals(annotationClass)) {
            try {
                util.handleInterfaces(javaClass);
            } catch (VersionNotSupportedException ex) {
                ErrorManager.getDefault().notify(ex);
            }
        } else if (EjbBuilderUtil.JAVAX_EJB_STATEFUL.equals(annotationClass)) {
            try {
                util.handleInterfaces(javaClass);
            } catch (VersionNotSupportedException ex) {
                ErrorManager.getDefault().notify(ex);
            }
        } else if (EjbBuilderUtil.JAVAX_EJB_MESSAGEDRIVEN.equals(annotationClass)) {
            util.handleMessageDriven(javaClass, annotation);
        } else if (EjbBuilderUtil.JAVAX_EJB_LOCAL.equals(annotationClass)) {
            util.updateLocal(javaClass);
        } else if (EjbBuilderUtil.JAVAX_EJB_REMOTE.equals(annotationClass)) {
            util.updateRemote(javaClass);
        }
    }

    @Override 
    public void addMemberAnnotation(boolean field, JavaClass javaClass, Element member, Annotation annotation, AnnotationType t) {


        String annotationClass = t.getName();
        if (EjbBuilderUtil.JAVAX_EJB_EJB.equals(annotationClass)) {
            Session session = EjbBuilderUtil.findReferencedSessionBean((TypedElement) member, annotation);
            if (session != null) {
                Map imaps = new HashMap();
                if (util.exposesLocalInterface(session, ((TypedElement) member).getType().getName())) {
                    imaps.put(EjbBuilderUtil.JAVAX_EJB_EJB, "/EjbLocalRef#EjbRefName=$JAVA_CLASS-MEMBER;!name=EjbRefName");
                } else {
                    imaps.put(EjbBuilderUtil.JAVAX_EJB_EJB, "/EjbRef#EjbRefName=$JAVA_CLASS-MEMBER;!name=EjbRefName");
                }
                generateItemsMappings(PATH_TO_SESSION, PATH_TO_SESSION, nn2ModelMapping, imaps);
                generateItemsMappings(PATH_TO_MESSAGE_DRIVEN, PATH_TO_MESSAGE_DRIVEN, nn2ModelMapping, imaps);
            }
        }
        super.addMemberAnnotation(field, javaClass, member, annotation, t);
        if(EjbBuilderUtil.JAVAX_ANNOTATION_WEBSERVICEREF.equals(annotationClass)) {
            util.webServiceRefNNAdded(javaClass, member, annotation);
        }
    }
    
    // private -----------------------------------------------------------------
    
    private void initNN2ModelMapping() {
        nn2ModelMapping = new HashMap();
        Map imaps = new HashMap();
        //<env-entry> element mappings
        imaps.put(RES_KEY_PART + "java.lang.String", ENV_ENTRY);
        imaps.put(RES_KEY_PART + "java.lang.Character", ENV_ENTRY);
        imaps.put(RES_KEY_PART + "java.lang.Integer", ENV_ENTRY);
        imaps.put(RES_KEY_PART + "java.lang.Boolean", ENV_ENTRY);
        imaps.put(RES_KEY_PART + "java.lang.Double", ENV_ENTRY);
        imaps.put(RES_KEY_PART + "java.lang.Byte", ENV_ENTRY);
        imaps.put(RES_KEY_PART + "java.lang.Short", ENV_ENTRY);
        imaps.put(RES_KEY_PART + "java.lang.Long", ENV_ENTRY);
        imaps.put(RES_KEY_PART + "java.lang.Float", ENV_ENTRY);
        //<service-ref>
        imaps.put(RES_KEY_PART + "javax.xml.rpc.Service", SERVICE_REF_ENTRY);
        //<resource-ref>
        imaps.put(RES_KEY_PART + "javax.sql.DataSource", RES_REF_ENTRY);
        imaps.put(RES_KEY_PART + "javax.jms.ConnectionFactory", RES_REF_ENTRY);
        imaps.put(RES_KEY_PART + "javax.jms.QueueConnectionFactory", RES_REF_ENTRY);
        imaps.put(RES_KEY_PART + "javax.jms.TopicConnectionFactory", RES_REF_ENTRY);
        imaps.put(RES_KEY_PART + "javax.mail.Session", RES_REF_ENTRY);
        imaps.put(RES_KEY_PART + "javax.net.URL", RES_REF_ENTRY);
        imaps.put(RES_KEY_PART + "javax.resource.cci.ConnectionFactory", RES_REF_ENTRY);
        imaps.put(RES_KEY_PART + "org.omg.CORBA_2_3.ORB", RES_REF_ENTRY);
        //XXX any other connection factory defined by a resource adapter?!?!?
        //<message-destination-ref>
        imaps.put(RES_KEY_PART + "javax.jms.Queue", MSG_DEST_ENTRY);
        imaps.put(RES_KEY_PART + "javax.jms.Topic", MSG_DEST_ENTRY);
        //<resource-env-ref>
        imaps.put(RES_KEY_PART + "javax.resource.cci.InteractionSpec", RES_ENV_REF_ENTRY);
        imaps.put(RES_KEY_PART + "javax.transaction.UserTransaction", RES_ENV_REF_ENTRY);
        //everything else is also <resource-env-ref>
        //imaps.put(RES_KEY_PART + "java.lang.Object", RES_ENV_REF_ENTRY);
        //<ejb-ref> handled directly on event
        imaps.put("javax.ejb.EJB", "");
        //@WebServiceRef -> <service-ref> mapping
        imaps.put(EjbBuilderUtil.JAVAX_ANNOTATION_WEBSERVICEREF, SERVICE_REF_ENTRY + ";!wsdlLocation=<CUSTOM>;!value=<CUSTOM>;!type=<CUSTOM>");
        //generate mapping items for Session beans
        generateItemsMappings(PATH_TO_SESSION, PATH_TO_SESSION, nn2ModelMapping, imaps);
        //generate mapping items for MessageDriven beans
        generateItemsMappings(PATH_TO_MESSAGE_DRIVEN, PATH_TO_MESSAGE_DRIVEN, nn2ModelMapping, imaps);
        // Annotations to Specify Bean Type
        nn2ModelMapping.put("javax.ejb.Stateless", PATH_TO_SESSION + ";!name=ejbName;SessionType=Stateless");
        nn2ModelMapping.put("javax.ejb.Stateful", PATH_TO_SESSION + ";!name=ejbName;SessionType=Stateful");
        nn2ModelMapping.put("javax.ejb.MessageDriven", PATH_TO_MESSAGE_DRIVEN + ";!name=ejbName");
        // Annotations to Specify Local or Remote Interfaces
        nn2ModelMapping.put("javax.ejb.Remote", ""); // handled in custom code
        nn2ModelMapping.put("javax.ejb.Local", ""); // handled in custom code
    }
    
    //generates real nnmodel builder mappings
    private void generateItemsMappings(String condition, String mappingPrefix, Map<String,String> mappings, Map<String,String> items) {
        Iterator<String> i = items.keySet().iterator();
        while(i.hasNext()) {
            String key = i.next();
            String value = items.get(key);
            //generate and store the real mapping
            mappings.put(key + "?" + condition, mappingPrefix + value);
        }
    }
    
    private void dumpMappings() {
        LOGGER.log(ErrorManager.INFORMATIONAL, "EJB NN model builder mappings:\n==============================\n");
        Iterator<String> i = getAnnotation2ModelMapping().keySet().iterator();
        while(i.hasNext()) {
            String key = i.next();
            String val = getAnnotation2ModelMapping().get(key);
            LOGGER.log(ErrorManager.INFORMATIONAL, key + " -> " + val);
        }
    }
    
}


