/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.welcome.content;

import java.awt.GridBagConstraints;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import javax.xml.parsers.ParserConfigurationException;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.Repository;
import org.openide.util.Lookup;
import org.openide.xml.XMLUtil;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.DefaultHandler;

/**
 * XML parser for Start Page content layout files.
 *
 * @author S. Aubrecht
 */
class ContentParser extends DefaultHandler {
    
    private static final String ELEM_ROOT = "welcomepage"; // NOI18N
    private static final String ELEM_BACKGROUND = "background"; // NOI18N
    private static final String ELEM_PANEL = "panel"; // NOI18N
    private static final String ELEM_CONSTRAINTS = "constraints"; // NOI18N

    private static final String CURRENT_VERSION = "1.0"; // NOI18N

    private Map anchors = new HashMap( 10 );
    private Map fills = new HashMap( 4 );

    private Map/*<String,ComponentDescriptor>*/ descriptors = new HashMap( 5 );
    private ArrayList/*<ComponentDescriptor>*/ contentIds = new ArrayList( 5 );
    private String backgroundPainterClassName;
    
    private ComponentDescriptor currentDescriptor;
    
    private static final String CONTENT_ROOT_FOLDER = "WelcomePage"; // NOI18N
    
    public ContentParser() {
        anchors.put( "CENTER",    new Integer( GridBagConstraints.CENTER ) ); // NOI18N
        anchors.put( "WEST",      new Integer( GridBagConstraints.WEST ) ); // NOI18N
        anchors.put( "NORTH",     new Integer( GridBagConstraints.NORTH ) ); // NOI18N
        anchors.put( "NORTHEAST", new Integer( GridBagConstraints.NORTHEAST ) ); // NOI18N
        anchors.put( "NORTHEAST", new Integer( GridBagConstraints.NORTHEAST ) ); // NOI18N
        anchors.put( "EAST",      new Integer( GridBagConstraints.EAST ) ); // NOI18N
        anchors.put( "SOUTH",     new Integer( GridBagConstraints.SOUTH ) ); // NOI18N
        anchors.put( "SOUTHEAST", new Integer( GridBagConstraints.SOUTHEAST ) ); // NOI18N
        anchors.put( "SOUTHWEST", new Integer( GridBagConstraints.SOUTHWEST ) ); // NOI18N

        fills.put( "NONE",       new Integer( GridBagConstraints.NONE ) ); // NOI18N
        fills.put( "HORIZONTAL", new Integer( GridBagConstraints.HORIZONTAL ) ); // NOI18N
        fills.put( "VERTICAL",   new Integer( GridBagConstraints.VERTICAL ) ); // NOI18N
        fills.put( "BOTH",       new Integer( GridBagConstraints.BOTH ) ); // NOI18N
    }
    
    public void parse() throws ParserConfigurationException, SAXException, IOException {
        FileObject root = Repository.getDefault().getDefaultFileSystem().findResource( CONTENT_ROOT_FOLDER );
        if( null != root ) {
            FileObject[] children = root.getChildren();
            for( int i=0; i<children.length; i++ ) {
                FileObject contentFile = children[i];
                if( contentFile.canRead() && contentFile.isData() ) {
                    parse( contentFile.getInputStream() );
                }
            }
        }
    }

    public void parseLocal() throws ParserConfigurationException, SAXException, IOException {
        String layoutFilePath = BundleSupport.getURL( "Layout" ); // NOI18N
        ClassLoader loader = (ClassLoader)Lookup.getDefault().lookup( ClassLoader.class );
        if( null == loader )
            loader = ClassLoader.getSystemClassLoader();
        parse( loader.getResourceAsStream( layoutFilePath ) );
    }
    
    private void parse( InputStream input ) throws ParserConfigurationException, SAXException, IOException {
        XMLReader reader = XMLUtil.createXMLReader(false, false);
        reader.setContentHandler(this);
        reader.parse(new InputSource(input));
    }
    
    public ComponentDescriptor[] getContents() {
        ComponentDescriptor[] res = new ComponentDescriptor[ contentIds.size() ];
        int index = 0;
        for( Iterator i=contentIds.iterator(); i.hasNext(); index++ ) {
            res[index] = (ComponentDescriptor)descriptors.get( i.next() );
        }
        return res;
    }

    public BackgroundPainter getBackgroundPainter() {
        BackgroundPainter painter = null;
        if( null != backgroundPainterClassName ) {
            ClassLoader loader = (ClassLoader)Lookup.getDefault().lookup( ClassLoader.class );
            if( null == loader )
                loader = ClassLoader.getSystemClassLoader();
            try {
                Class clazz = Class.forName(backgroundPainterClassName, true, loader );
                painter = (BackgroundPainter)clazz.newInstance();
            } catch( Throwable e ) {
                ErrorManager.getDefault().notify( ErrorManager.INFORMATIONAL, e );
            }
        }
        return painter;
    }
    
        
        
    public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException {
        if( qName.equals( ELEM_ROOT ) ) {
            
            //check version
            String version = attributes.getValue( "version" ); // NOI18N
            if( !CURRENT_VERSION.equals( version ) ) {
                throw new SAXParseException( "Unsupported file version", null ); // NOI18N
            }
            
        } else if( qName.equals( ELEM_BACKGROUND ) ) {
            
            //background painter
            backgroundPainterClassName = attributes.getValue( "class" );
            
        } else if( qName.equals( ELEM_PANEL ) ) {
            
            //new panel
            String id = attributes.getValue( "id" ); // NOI18N
            if( null == id )
                throw new SAXParseException( "Required attribute missing: id", null ); // NOI18N
            String className = attributes.getValue( "class" ); // NOI18N
            String method = attributes.getValue( "method" ); // NOI18N
            if( null == className && method == null)
                throw new SAXParseException( "Either class or method attribute must be specified", null ); // NOI18N
            
            ComponentDescriptor descriptor = (ComponentDescriptor)descriptors.get( id );
            if( null == descriptor ) {
                descriptor = new DefaultComponentDescriptor( id );
                descriptors.put( id, descriptor );
            }
            
            descriptor.setClassName( className );
            descriptor.setMethod( method );
            
            currentDescriptor = descriptor;
            
            if( !contentIds.contains( id ) )
                contentIds.add( id );
            
        } else if( qName.equals( ELEM_CONSTRAINTS ) ) {
            
            //panel constraints
            if( null == currentDescriptor ) {
                throw new SAXParseException( "Unexpected element: " + ELEM_CONSTRAINTS, null ); // NOI18N
            }
            
            currentDescriptor.setConstraints( parseConstraints( attributes, currentDescriptor.getConstraints() ) );
        }
    }

    public void endElement(String uri, String localName, String qName) throws SAXException {
        if( qName.equals( ELEM_PANEL ) ) {
            currentDescriptor = null;
        }
    }
    
    private GridBagConstraints parseConstraints( Attributes attrs, GridBagConstraints defaults ) {
        GridBagConstraints res = defaults;
        if( null == res )
            res = new GridBagConstraints();
        
        res.gridx = parseInt( attrs, "gridx", 0, 100, res.gridx ); // NOI18N
        res.gridy = parseInt( attrs, "gridy", 0, 100, res.gridy ); // NOI18N

        res.gridwidth = parseInt( attrs, "gridwidth", 0, 100, res.gridwidth ); // NOI18N
        res.gridheight = parseInt( attrs, "gridheight", 0, 100, res.gridheight ); // NOI18N

        res.weightx = parseDouble( attrs, "weightx", 0.0, 1.0, res.weightx ); // NOI18N
        res.weighty = parseDouble( attrs, "weighty", 0.0, 1.0, res.weighty ); // NOI18N

        res.insets.left = parseInt( attrs, "insetsleft", 0, 1000, res.insets.left ); // NOI18N
        res.insets.right = parseInt( attrs, "insetsright", 0, 1000, res.insets.right ); // NOI18N
        res.insets.top = parseInt( attrs, "insetstop", 0, 1000, res.insets.top ); // NOI18N
        res.insets.bottom = parseInt( attrs, "insetsbottom", 0, 1000, res.insets.bottom ); // NOI18N

        String anchor = attrs.getValue( "anchor" ); // NOI18N
        if( null != anchor ) {
            anchor = anchor.toUpperCase();
            Integer intValue = (Integer)anchors.get( anchor );
            if( null != intValue ) {
                res.anchor = intValue.intValue();
            }
        }

        String fill = attrs.getValue( "fill" ); // NOI18N
        if( null != fill ) {
            fill = fill.toUpperCase();
            Integer intValue = (Integer)fills.get( fill );
            if( null != intValue ) {
                res.fill = intValue.intValue();
            }
        }

        res.ipadx = parseInt( attrs, "padx", 0, 100, res.ipadx ); // NOI18N
        res.ipady = parseInt( attrs, "pady", 0, 100, res.ipady ); // NOI18N

        return res;
    }

    private int parseInt( Attributes attrs, String name, int min, int max, int defaultValue ) {
        int res = defaultValue;
        String str = attrs.getValue( name );
        if( null != str ) {
            try {
                res = Integer.parseInt( str );
            } catch( NumberFormatException nfE ) {
                // ignore
            }
            if( res < min )
                res = min;
            if( res > max )
                res = max;
        }
        return res;
    }

    private double parseDouble( Attributes attrs, String name, double min, double max, double defaultValue ) {
        double res = defaultValue;
        String str = attrs.getValue( name );
        if( null != str ) {
            try {
                res = Double.parseDouble( str );
            } catch( NumberFormatException nfE ) {
                // ignore
            }
            if( res < min )
                res = min;
            if( res > max )
                res = max;
        }
        return res;
    }
}
