/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.editor.ext.html;

import java.io.PrintStream;
import java.net.URL;
import org.netbeans.editor.TokenContext;
import org.netbeans.editor.TokenContextPath;
import org.netbeans.editor.TokenID;
import org.netbeans.editor.TokenItem;
import org.netbeans.editor.ext.html.HTMLSyntax;
import org.netbeans.junit.NbTestCase;

import org.openide.ErrorManager;

/** Basic html syntax parser tests.
 *
 * @author  Marek Fukala
 */
public class HTMLSyntaxTest extends NbTestCase {
    
    //it's static since the junit creates a new instance of this class for each test method
    private static HTMLSyntax syntax = new HTMLSyntax();
    
    public HTMLSyntaxTest() {
        super("htmlsyntaxtest");
    }
    
    public void setUp() {
        //print out a header to the ref file
        getRef().println("'token image' [offset, length]; tokenID name; tokenID id; token category name; <list of token context names>\n--------------------\n");
    }
    
    public void tearDown() {
        compareReferenceFiles();
    }
    
    //test methods -----------
    
    public void testHtml() {
        dumpTokensForContent("<html>\n<body>\n<h1>hello</h1>\n</body>\n</html>");
    }
    
    public void testErrorInTag() {
        dumpTokensForContent("<html@><head #/></html>");
    }
    
    public void testAttrs() {
        dumpTokensForContent("<meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\">");
    }
    
    public void testErrorInAttrs() {
        dumpTokensForContent("<meta 2 http-equiv=\"Content-Type\" content@=\"text/html; charset=UTF-8\">");
    }
    
    public void testMultilineAttrs() {
        dumpTokensForContent("<meta \n    http-equiv=\"Content-Type\" \n    content=\"text/html; charset=UTF-8\">");
    }
    
    public void testSGMLEscape() {
        dumpTokensForContent("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">");
    }
    
    public void testSGMLEscapeOnMoreLines() {
        dumpTokensForContent("<!DOCTYPE HTML \nPUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\"\n>");
    }

    public void testHTMLComment() {
        dumpTokensForContent("<html>\n<!-- this is a commnent -->\n</html>");
    }
    
    public void testHTMLMulitilineComment() {
        dumpTokensForContent("<html>\n<!-- this \nis \na \ncommnent \n-->\n</html>");
    }
    
    public void testEmbededCSS() {
         dumpTokensForContent("<style type=\"text/css\">\n"+
            "#search {height:100%;}\n"+
            "#topmodule {align:right;}\n"+
            "</style>\n");
    }
    
    public void testBug53102() {
        dumpTokensForContent("<html>\n<head />\n<he");
    }
     
    public void testBugWrongJsptagType() {
        dumpTokensForContent("\n<a >\n");
    }
    
    public void testPlainText() {
        dumpTokensForContent("this is just a text");
    }
    
    public void testEntityReference() {
        dumpTokensForContent("<html>\n &amp; \n </html>");
    }
    
    public void testNumericReference() {
        dumpTokensForContent("<html>\n &#345; \n </html>");
    }
    
    //helper methods -----------
    
    private void dumpTokensForContent(String content) {
        loadContentToSyntax(content);
        dumpTokensData(getRef()); //print output to reference stream
    }
    
    private void dumpTokensData(PrintStream out) {
        TokenID tokenID = null;
        char[] buffer = syntax.getBuffer();
        String tokenImage = null;
        TokenContextPath tcp = null;
        do {
            //acquire all token relevant data
            tokenID = syntax.nextToken();
            
            if( tokenID == null ) break;
            
            tokenImage = new String(buffer, syntax.getTokenOffset(), syntax.getTokenLength());
            tcp = syntax.getTokenContextPath();
            
            //print it
            out.print("'" + SyntaxUtils.normalize(tokenImage) + "' ["+syntax.getTokenOffset() + ", " + syntax.getTokenLength() + "]; " + tokenID.getName() + "; " + tokenID.getNumericID() + "; "+ (tokenID.getCategory() != null ? tokenID.getCategory().getName() : "-") + "; ");
            SyntaxUtils.dumpTokenContextPath(tcp, out);
            out.println();
            
        }
        while(true);
    }
    
    private void loadContentToSyntax(String content) {
        //load syntax - scan the whole buffer - the buffer is last one
        char[] buffer = content.toCharArray();
        syntax.load(null, buffer, 0, buffer.length, true, -1);
    }
    
}
