/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.editor.settings.storage;

import java.awt.Color;
import java.awt.event.KeyEvent;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import javax.swing.KeyStroke;
import javax.swing.text.AttributeSet;
import javax.swing.text.DefaultEditorKit;
import javax.swing.text.SimpleAttributeSet;
import javax.swing.text.StyleConstants;
import org.netbeans.api.editor.mimelookup.MimeLookup;
import org.netbeans.api.editor.settings.FontColorSettings;
import org.netbeans.api.editor.settings.KeyBindingSettings;
import org.netbeans.api.editor.settings.MultiKeyBinding;
import org.netbeans.junit.NbTestCase;
import org.netbeans.modules.editor.settings.storage.api.EditorSettings;
import org.netbeans.modules.editor.settings.storage.api.FontColorSettingsFactory;
import org.netbeans.modules.editor.settings.storage.api.KeyBindingSettingsFactory;
import org.openide.util.Lookup;
import org.openide.util.LookupEvent;
import org.openide.util.LookupListener;

/** Settings object found via MimeLookup should be immutable.
 *  Covering issue #62675
 * 
 *  @author Martin Roskanin
 */
public class EditorSettingsStorageImmutabilityTest extends EditorSettingsStorageTestBase{

    public EditorSettingsStorageImmutabilityTest(String testName) {
        super(testName);
    }
    
    protected void setUp() throws Exception {
        super.setUp();
    
        EditorTestLookup.setLookup(
                getWorkDir(),
            new URL[] {
                getClass().getClassLoader().getResource(
                        "org/netbeans/modules/java/editor/resources/layer.xml"),
                getClass().getClassLoader().getResource(
                        "org/netbeans/modules/editor/resources/layer.xml"),
                getClass().getClassLoader().getResource(
                        "org/netbeans/modules/defaults/mf-layer.xml")
            },
            new Object[] {},
            getClass().getClassLoader()
        );
    }


    public void testColoringsImmutability() throws IOException{
        EditorSettings editorSettings = EditorSettings.getDefault ();        
        MimeLookup mimelookup = MimeLookup.getMimeLookup("text/x-java");

        // gather old color
        FontColorSettings oldFcs = (FontColorSettings) mimelookup.lookup(FontColorSettings.class);
        AttributeSet set = oldFcs.getTokenFontColors("java-keywords");
        Color oldBackColor = (Color)set.getAttribute(StyleConstants.Foreground);
        
        // perform change
        String types[] = new String[]{"text/x-java"};
        FontColorSettingsFactory fcsImpl = 
                EditorSettingsImpl.getDefault().getFontColorSettings(types);
        SimpleAttributeSet a = new SimpleAttributeSet();
        a.addAttribute (
            StyleConstants.NameAttribute, 
            "java-keywords"
        );
        Color setColor = (oldBackColor != Color.RED) ? Color.RED : Color.GREEN;
        a.addAttribute(StyleConstants.Foreground, setColor);
        assertTrue(oldBackColor != setColor);
        setSetting("text/x-java", "java-keywords", a);
        
        // gather new color
        FontColorSettings newFcs = (FontColorSettings) mimelookup.lookup(FontColorSettings.class);
        set = newFcs.getTokenFontColors("java-keywords"); //NOI18N
        Color newBackColor = (Color)set.getAttribute(StyleConstants.Foreground);
        assertFalse(newBackColor.equals(oldBackColor));
        
        // check old color
        AttributeSet oldSet = oldFcs.getTokenFontColors("java-keywords");
        Color oldBackColor2 = (Color)oldSet.getAttribute(StyleConstants.Foreground);
        // oldBackColor should be immutable - the same value as oldBackColor2
        assertNotNull(oldBackColor2);
        assertTrue(oldBackColor2.equals(oldBackColor));
    }
    

    public void testKeyBindingsImmutability() throws IOException{
        EditorSettings editorSettings = EditorSettings.getDefault ();        
        MimeLookup mimelookup = MimeLookup.getMimeLookup("text/base");
        
        // gather old kb
        KeyBindingSettings oldKbs = (KeyBindingSettings) mimelookup.lookup(KeyBindingSettings.class);
        List oldKbList = oldKbs.getKeyBindings();
        MultiKeyBinding oldMkb = getMkbForAction(DefaultEditorKit.copyAction, oldKbList);

        // set new value
        String types[] = new String[]{"text/base"};
        KeyBindingSettingsFactory kbsImpl = 
                EditorSettingsImpl.getDefault().getKeyBindingSettings(types);
        List keybs = new ArrayList();
        KeyStroke ks = KeyStroke.getKeyStroke(KeyEvent.VK_K, KeyEvent.CTRL_DOWN_MASK | KeyEvent.SHIFT_DOWN_MASK);
        assertTrue(oldMkb.getKeyStrokeCount() == 1);
        assertFalse(ks.equals(oldMkb.getKeyStroke(0)));
        MultiKeyBinding newMkb = new MultiKeyBinding(ks, DefaultEditorKit.copyAction);
        keybs.add(newMkb);
        // changing settings - foreground of java-keywords will be different
        kbsImpl.setKeyBindings(
            editorSettings.getCurrentFontColorProfile (),
            keybs
        );
        
        // gather new kb
        KeyBindingSettings newKbs = (KeyBindingSettings) mimelookup.lookup(KeyBindingSettings.class);
        List newKbList = newKbs.getKeyBindings();
        MultiKeyBinding newMkb2 = getMkbForAction(DefaultEditorKit.copyAction, newKbList);
        assertFalse(newMkb2.equals(oldMkb));
        
        // check old kb
        List oldKbList2 = oldKbs.getKeyBindings();
        MultiKeyBinding oldMkb2 = getMkbForAction(DefaultEditorKit.copyAction, oldKbList2);
        // oldMkb should be immutable - the same value as oldMkb2
        assertNotNull(oldMkb2);
        assertTrue(oldMkb2.equals(oldMkb));
    }

    
    
    public void testColoringsTemplateImmutability() throws IOException{    
        EditorSettings editorSettings = EditorSettings.getDefault ();        
        MimeLookup mimelookup = MimeLookup.getMimeLookup("text/x-java");
        Lookup.Result result = mimelookup.lookup(
                new Lookup.Template(FontColorSettings.class));
        Collection col = result.allInstances();
        
        // Expecting just one instance of FontColorSettings.class
        assertTrue(col.size() == 1);
        
        LookupListener listener = new LookupListener(){
            public void resultChanged(LookupEvent ev){
                resultChangedCount[0]++;
            }
        };
        
        resultChangedCount[0] = 0;
        result.addLookupListener(listener);
        
        FontColorSettings fcsOne = (FontColorSettings)col.iterator().next();
        assertTrue(fcsOne!=null);
        AttributeSet set = fcsOne.getTokenFontColors("java-keywords");
        Color oldBackColor = (Color)set.getAttribute(StyleConstants.Foreground);
        
        //perform change
        String types[] = new String[]{"text/x-java"};
        FontColorSettingsFactory fcsImpl = 
                EditorSettingsImpl.getDefault().getFontColorSettings(types);
        SimpleAttributeSet a = new SimpleAttributeSet();
        a.addAttribute (
            StyleConstants.NameAttribute, 
            "java-keywords"
        );
        Color setColor = (oldBackColor != Color.RED) ? Color.RED : Color.GREEN;
        a.addAttribute(StyleConstants.Foreground, setColor);
        assertTrue(oldBackColor != setColor);
        setSetting("text/x-java", "java-keywords", a);          
        
        checkResultChange(1);

        result.removeLookupListener(listener);
        resultChangedCount[0] = 0;
        
        col = result.allInstances();

        // we are expecting just one instance of FontColorSettings.class
        assertTrue(col.size() == 1);
        
        FontColorSettings fcsTwo = (FontColorSettings)col.iterator().next();
        assertTrue(fcsTwo!=null);
        set = fcsTwo.getTokenFontColors("java-keywords");
        Color oldBackColor2 = (Color)set.getAttribute(StyleConstants.Foreground);

        // ensure there is a different color found via lookup.template after the change
        assertFalse(oldBackColor2.equals(oldBackColor));
        
        // checking immutability
        set = fcsOne.getTokenFontColors("java-keywords");
        Color oldBackColor3 = (Color)set.getAttribute(StyleConstants.Foreground);
        assertTrue(oldBackColor3.equals(oldBackColor));
    }
    
    public void testKeybindingsTemplateImmutability() throws IOException{    
        EditorSettings editorSettings = EditorSettings.getDefault ();        
        MimeLookup mimelookup = MimeLookup.getMimeLookup("text/base");
        Lookup.Result result = mimelookup.lookup(
                new Lookup.Template(KeyBindingSettings.class));
        Collection col = result.allInstances();
        
        // we are expecting just one instance of KeyBindingSettings.class
        assertTrue(col.size() == 1);
        
        LookupListener listener = new LookupListener(){
            public void resultChanged(LookupEvent ev){
                resultChangedCount[0]++;
            }
        };
        
        resultChangedCount[0] = 0;
        result.addLookupListener(listener);
        
        KeyBindingSettings oldKbs = (KeyBindingSettings)col.iterator().next();
        List oldKbList = oldKbs.getKeyBindings();
        MultiKeyBinding oldMkb = getMkbForAction(DefaultEditorKit.copyAction, oldKbList);

        String types[] = new String[]{"text/base"};
        KeyBindingSettingsFactory kbsImpl = 
                EditorSettingsImpl.getDefault().getKeyBindingSettings(types);
        
        List keybs = new ArrayList();
        KeyStroke ks = KeyStroke.getKeyStroke(KeyEvent.VK_L, KeyEvent.CTRL_DOWN_MASK | KeyEvent.SHIFT_DOWN_MASK);
        assertTrue(oldMkb.getKeyStrokeCount() == 1);
        assertFalse(ks.equals(oldMkb.getKeyStroke(0)));
        MultiKeyBinding newMkb = new MultiKeyBinding(ks, DefaultEditorKit.copyAction);
        keybs.add(newMkb);
        // changing settings - foreground of java-keywords will be different
        kbsImpl.setKeyBindings(
            editorSettings.getCurrentFontColorProfile (),
            keybs
        );

        // check firing
        checkResultChange(1);

        result.removeLookupListener(listener);
        resultChangedCount[0] = 0;

        col = result.allInstances();

        // we are expecting just one instance of FontColorSettings.class
        assertTrue(col.size() == 1);
        
        KeyBindingSettings newKbs = (KeyBindingSettings)col.iterator().next();
        List newKbList = newKbs.getKeyBindings();
        MultiKeyBinding newMkb2 = getMkbForAction(DefaultEditorKit.copyAction, newKbList);
        
        // ensure there is a different color found via lookup.template after the change
        assertFalse(newMkb2.equals(oldMkb));
        
        // checking immutability
        List oldKbList2 = oldKbs.getKeyBindings();
        MultiKeyBinding oldMkb2 = getMkbForAction(DefaultEditorKit.copyAction, oldKbList2);
        assertTrue(oldMkb2.equals(oldMkb));
    }

    
    private static MultiKeyBinding getMkbForAction(String actionName, List kbList){
        int cnt = kbList.size();
        for (int i = 0; i < cnt; i++){
            MultiKeyBinding mkb = (MultiKeyBinding)kbList.get(i);
            if (actionName.equals(mkb.getActionName())){
                return mkb;
            }
        }
        assertTrue("Shouldn\'t return null:"+actionName, false);
        return null;
    }
    
}
