 /*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.editor.settings.storage;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.swing.KeyStroke;
import org.netbeans.api.editor.settings.KeyBindingSettings;
import org.netbeans.api.editor.settings.MultiKeyBinding;
import org.netbeans.modules.editor.settings.storage.api.EditorSettings;
import org.netbeans.modules.editor.settings.storage.api.KeyBindingSettingsFactory;
import org.netbeans.spi.editor.mimelookup.MimeLookupInitializer;
import org.openide.ErrorManager;
import org.openide.filesystems.FileChangeAdapter;
import org.openide.filesystems.FileEvent;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;


/**
 * KeyBindings settings are represented by List of keybindings.
 * The List contains the instances of {@link MultiKeyBinding}.
 * <br>
 * Instances of this class should be retrieved from the {@link org.netbeans.api.editor.mimelookup.MimeLookup}
 * for a given mime-type.
 * <br>
 * <font color="red">This class must NOT be extended by any API clients</font>
 *
 * @author Jan Jancura
 */
public class KeyBindingSettingsImpl extends KeyBindingSettingsFactory 
implements MimeLookupInitializerImpl.Factory {

    private String[]                mimeTypes;
    private PropertyChangeSupport   pcs;
    private Map                     keyMaps = new HashMap ();
    private KeyBindingSettingsFactory      baseKBS;
    private Listener                listener;
    
    
    /**
     * Construction prohibited for API clients.
     */
    KeyBindingSettingsImpl (String[] mimeTypes) {
        this.mimeTypes = mimeTypes;
        pcs = new PropertyChangeSupport (this);
        
        // init logging
        String myClassName = KeyBindingSettingsImpl.class.getName ();
        if (System.getProperty (myClassName) != null) {
            isLoggable = true;
            if (!"true".equals (System.getProperty (myClassName)))
                logActionName = System.getProperty (myClassName);
            return;
        }
        if (mimeTypes.length != 1) return;
        String propertyValue = System.getProperty 
           (myClassName + '.' + mimeTypes [0]);
        if (propertyValue == null) return;
        isLoggable = true;
        logActionName = propertyValue;
    }
    
    private boolean init = false;
    private void init () {
        if (init) return;
        init = true;
        if (mimeTypes.length != 1 || 
            !mimeTypes [0].equals ("text/base")
        )
            baseKBS = getEditorSettingsImpl ().getKeyBindingSettings (
                new String[] {"text/base"}
            );
        listener = new Listener (this, mimeTypes, baseKBS, getEditorSettingsImpl ());
    }
    
    /**
     * Gets the keybindings list, where items are instances of {@link MultiKeyBinding}
     *
     * @return List of {@link MultiKeyBinding}
     */
    public List /*<MultiKeyBinding>*/ getKeyBindings () {
        List result = getKeyBindings (
            getEditorSettingsImpl ().getCurrentKeyMapProfile ()
        );
        return result;
    }
    
    /**
     * Gets the keybindings list, where items are instances of {@link MultiKeyBinding}
     *
     * @return List of {@link MultiKeyBinding}
     */
    public List /*<MultiKeyBinding>*/ getKeyBindings (String profile) {
        init ();
        
        // 1) get real profile
	String ss = getEditorSettingsImpl ().getInternalKeymapProfile (profile);
        if (ss == null) ss = profile;
        
        List result = new ArrayList ();
        if (!keyMaps.containsKey (profile)) {
            synchronized (this) {

                // 2) load original profile for this mimeType
                //       Map (List (KeyStroke) > MultiKeyBinding)
                Map defaults = new HashMap (getDefaults (profile));

                // 3) load & apply modifications to defaults
                Object[] s = KeyMapsStorage.loadKeyMaps 
                    (mimeTypes, profile, EditorSettingsImpl.KEYBINDING_FILE_NAME);
                Map shortcuts = (Map) s [0]; // Map (List (KeyStroke) > MultiKeyBinding)
                Set removedShortcuts = (Set) s [1]; // Set (List (KeyStroke))
                Iterator it = removedShortcuts.iterator ();
                while (it.hasNext ())
                    defaults.remove (it.next ());
                defaults.putAll (shortcuts);

                List localShortcuts = new ArrayList (defaults.values ());
                keyMaps.put (profile, localShortcuts);
                result.addAll (localShortcuts);
            }
	} else
            result.addAll ((List) keyMaps.get (profile));
        log ("getKeyBindings", result);

        // 4) add global editor shortcuts
        if (baseKBS != null) {
            List baseShortcuts = baseKBS.getKeyBindings (profile);
            log ("", Collections.EMPTY_LIST);
            result.addAll (baseShortcuts);
        }
        
	return Collections.unmodifiableList (result);
    }
    
    /**
     * Returns default keybindings list for given keymap name, where items 
     * are instances of {@link MultiKeyBinding}.
     *
     * @return List of {@link MultiKeyBinding}
     */
    public List getKeyBindingDefaults (String profile) {
        return Collections.unmodifiableList (new ArrayList (getDefaults (profile).values ()));
    }
    
    /**
     * Gets the keybindings list, where items are instances of 
     * {@link MultiKeyBinding}.
     *
     * @return List of {@link MultiKeyBinding}
     */
    public synchronized void setKeyBindings (
        String profile, 
        List/*<MultiKeyBinding>*/ keyBindings
    ) {
        log ("setKeyBindings", keyBindings);

        init ();
        if (keyBindings == null) {
            // 1) delete user changes / user profile
            keyMaps.remove (profile);
            KeyMapsStorage.deleteProfile (
                mimeTypes, 
                profile, 
                EditorSettingsImpl.KEYBINDING_FILE_NAME
            );
            return;
        }
        keyMaps.put (profile, keyBindings);

        // 1) convert keyBindings: List (MultiKeyBinding) to 
        //            m: Map (List (KeyStroke) > MultiKeyBinding).
        Map m = new HashMap ();
        Iterator it = keyBindings.iterator ();
        while (it.hasNext ()) {
            MultiKeyBinding mkb = (MultiKeyBinding) it.next ();
            m.put (mkb.getKeyStrokeList (), mkb);
        }

        // 2) compute removed shortcuts & remove unchanged maappings from m
        Map defaults = getDefaults (profile); // Map (List (KeyStroke) > MultiKeyBinding)
        Set removed = new HashSet ();
        it = defaults.keySet ().iterator ();
        while (it.hasNext ()) {
            List shortcut = (List) it.next ();
            MultiKeyBinding mkb2 = (MultiKeyBinding) defaults.get (shortcut);
            if (!m.containsKey (shortcut)) {
                removed.add (shortcut);
            } else {
                MultiKeyBinding mkb1 = (MultiKeyBinding) m.get (shortcut);
                if (mkb1.getActionName ().equals (mkb2.getActionName ()))
                    m.remove (shortcut);
            }
        }

        log ("  changed:", m.values ());
        log ("  removed:", removed);
        log ("", Collections.EMPTY_LIST);

        // 3) save diff & removed
        listener.removeListeners ();
        KeyMapsStorage.saveKeyMaps (
            mimeTypes, 
            profile, 
            EditorSettingsImpl.KEYBINDING_FILE_NAME,
            m.values (),
            removed
        );
        listener.addListeners ();
        pcs.firePropertyChange (null, null, null);
    }
    
    /**
     * PropertyChangeListener registration.
     *
     * @param l a PropertyChangeListener to be registerred
     */
    public void addPropertyChangeListener (PropertyChangeListener l) {
        pcs.addPropertyChangeListener (l);
    }
    
    /**
     * PropertyChangeListener registration.
     *
     * @param l a PropertyChangeListener to be unregisterred
     */
    public void removePropertyChangeListener (PropertyChangeListener l) {
        pcs.removePropertyChangeListener (l);
    }    
    
    // other methods ...........................................................
    
    private EditorSettingsImpl editorSettingsImpl;
    
    private EditorSettingsImpl getEditorSettingsImpl () {
	if (editorSettingsImpl == null) {
	    editorSettingsImpl = (EditorSettingsImpl) EditorSettings.
                getDefault ();
	}
	return editorSettingsImpl;
    }
    
    // Map (String (profile) > Map (String (shortcut) > MultiKeyBinding)).
    private Map defaults = new HashMap ();
    
    /**
     * Returns default shortcut set for given profile. Returns empty map for 
     * custom (user defined) profiles.
     *
     * @return Map (List (KeyStroke) > MultiKeyBinding)
     */
    private Map getDefaults (String profile) {
        if (!defaults.containsKey (profile)) {
            FileObject defaultFo = Utils.getFileObject (
                mimeTypes, 
                "NetBeans".equals (profile) ? null : profile, 
                "Defaults/" + EditorSettingsImpl.KEYBINDING_FILE_NAME
            );
            if (defaultFo != null) {
                defaults.put (profile, (Map) (
                    ((Object[]) KeyMapsStorage.loadKeyMaps (defaultFo)) [0]
                ));
            } else
                defaults.put (profile, Collections.EMPTY_MAP);
        }
        return (Map) defaults.get (profile);
    }

    /**
     * External change.
     */
    private void refresh () {
        keyMaps = new HashMap ();
        log ("refresh", Collections.EMPTY_SET);
        pcs.firePropertyChange (null, null, null);
    }
    
    private boolean isLoggable = false;
    private String logActionName = null;
    
    private void log (String text, Collection keymap) {
        if (!isLoggable) return;
        if (!text.equals ("")) {
            if (mimeTypes.length == 1) text += " " + mimeTypes [0];
            text += " " + getEditorSettingsImpl ().getCurrentKeyMapProfile ();
        }
        if (keymap == null) {
            System.out.println (text + " : null");
            return;
        }
        System.out.println (text);
        Iterator it = keymap.iterator ();
        while (it.hasNext ()) {
            Object mkb = it.next ();
            if (logActionName == null || !(mkb instanceof MultiKeyBinding))
                System.out.println ("  " + mkb);
            else
            if (mkb instanceof MultiKeyBinding &&
                logActionName.equals (((MultiKeyBinding) mkb).getActionName ())
            )
                System.out.println ("  " + mkb);
        }
    }

    public Object createInstance() {
        List keyB = getKeyBindings();
        return new Immutable(new ArrayList(keyB));
    }

    
    private static class Listener extends FileChangeAdapter 
    implements PropertyChangeListener {
        
        private WeakReference           wr;
        /** /Editor/mimetype/currentProfile/ folder*/
        private FileObject              fo;
        private String[]                mimeTypes;
        private KeyBindingSettingsFactory      baseKBS;
        private EditorSettings          editorSettings;
        
        Listener (
            KeyBindingSettingsImpl      kb,
            String[]                    mimeTypes,
            KeyBindingSettingsFactory          baseKBS,
            EditorSettings              editorSettings
        ) {
            this.mimeTypes = mimeTypes;
            this.editorSettings = editorSettings;
            this.baseKBS = baseKBS;
            addListeners ();
            wr = new WeakReference (kb);
        }
        
        private KeyBindingSettingsImpl getSettings () {
            KeyBindingSettingsImpl r = (KeyBindingSettingsImpl) wr.get ();
            if (r != null) return r;
            removeListeners ();
            return null;
        }
        
        private void addListeners () {
            editorSettings.addPropertyChangeListener (
                EditorSettings.PROP_CURRENT_KEY_MAP_PROFILE,
                this
            );
            if (baseKBS != null)
                baseKBS.addPropertyChangeListener (this);
            setFolderListener ();
        }
        
        private void removeListeners () {
            fo.removeFileChangeListener (this);
            if (baseKBS != null)
                baseKBS.removePropertyChangeListener (this);
            editorSettings.removePropertyChangeListener (
                EditorSettings.PROP_CURRENT_KEY_MAP_PROFILE,
                this
            );
        }
        
        public void propertyChange (PropertyChangeEvent evt) {
            KeyBindingSettingsImpl r = getSettings ();
            if (r == null) return;
            if (EditorSettings.PROP_CURRENT_KEY_MAP_PROFILE.equals (
                evt.getPropertyName ()
            ))
                setFolderListener ();
            r.log ("refresh2", Collections.EMPTY_SET);
            r.pcs.firePropertyChange (null, null, null);
        }
        
        public void fileDataCreated (FileEvent fe) {
        }

        public void fileChanged (FileEvent fe) {
            KeyBindingSettingsImpl r = getSettings ();
            if (r == null) return;
            if (fe.getFile ().getNameExt ().equals 
                  (EditorSettingsImpl.KEYBINDING_FILE_NAME)
            )
                r.refresh ();
        }

        public void fileDeleted (FileEvent fe) {
            KeyBindingSettingsImpl r = getSettings ();
            if (r == null) return;
            if (fe.getFile ().getNameExt ().equals 
                    (EditorSettingsImpl.KEYBINDING_FILE_NAME)
            )
                r.refresh ();
        }
        
        private void setFolderListener () {
            if (fo != null) fo.removeFileChangeListener (this);
            String profile = editorSettings.getCurrentKeyMapProfile ();
            if (profile.equals ("NetBeans")) profile = null;
            fo = Utils.createFileObject (mimeTypes, profile, null);
            fo.addFileChangeListener (this);
            //log.log (log.INFORMATIONAL, "add listener to:" + fo);
        }
    }
    
    private static final class Immutable extends KeyBindingSettings {
        private List keyBindings;
        
        public Immutable(List keyBindings) {
            this.keyBindings = keyBindings;
        }
        
        public List getKeyBindings() {
            return Collections.unmodifiableList(keyBindings);
        }
    }
    
}
