 /*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
*/

package org.netbeans.modules.editor.settings.storage;

import java.awt.Color;
import java.awt.Font;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import javax.swing.UIManager;
import javax.swing.text.AttributeSet;
import javax.swing.text.StyleConstants;
import org.netbeans.api.editor.settings.EditorStyleConstants;
import org.netbeans.api.editor.settings.FontColorSettings;
import org.netbeans.modules.editor.settings.storage.api.EditorSettings;
import org.netbeans.modules.editor.settings.storage.api.FontColorSettingsFactory;
import org.netbeans.modules.editor.settings.storage.*;
import org.openide.ErrorManager;


/**
 *
 * @author Jan Jancura
 */
public class FontColorSettingsImpl extends FontColorSettingsFactory 
implements MimeLookupInitializerImpl.Factory {

    private String[]                mimeTypes;
    private String                  mimeType;
    private PropertyChangeSupport   pcs;
    private String                  test;
    private ErrorManager            logger;
    private static final String DEFAULT = "default"; //NOI18N

    
    /**
     * Construction prohibited for API clients.
     */
     FontColorSettingsImpl (String[] mimeTypes) {
        if (mimeTypes.length > 0 && mimeTypes [0].startsWith ("test")) { //NOI18N
            String[] newMT = new String [mimeTypes.length];
            System.arraycopy (mimeTypes, 0, newMT, 0, newMT.length);
            int i = mimeTypes [0].indexOf ('_');
            test = newMT [0].substring (0, i);
            newMT [0] = newMT [0].substring (i + 1);
            this.mimeTypes = newMT;
        } else
            this.mimeTypes = mimeTypes;
        mimeType = getMimeType ();
        logger = ErrorManager.getDefault ().getInstance 
            (FontColorSettingsImpl.class.getName () + '.' + 
                (test == null ? mimeType : test + "_" + mimeType)); //NOI18N
        
        pcs = new PropertyChangeSupport (this);
	new Listener (this);
    }

    
    // FontColorSettings impl ..................................................
    
    /**
     * Gets the font and colors. 
     * 
     * @param settingName font and colors setting name
     *
     * @return AttributeSet describing the font and colors. 
     */
    public AttributeSet getFontColors (String settingName) {
        if (settingName.equals (DEFAULT))
            // default highlighting is internally managed as syntax coloring
            return getTokenFontColors (settingName);
        
        // resolve profile name
	String profile = test != null ? 
            test : getEditorColoringImpl ().getCurrentFontColorProfile ();
        
        // delegate to EditorSettings
	AttributeSet as = (AttributeSet) getEditorColoringImpl ().
            getHighlightings (profile).get (settingName);
        if (as == null) {
            logger.log ("highlighting2 " + settingName + " : null");
            return null;
        }
        AttributeSet result = resolveDefaults (as, profile);
        logger.log ("highlighting3 " + settingName + " : " + result);
        return result;
    }
    
    /**
     * Gets the token font and colors. 
     * 
     * @param tokenName token name
     *
     * @return AttributeSet describing the font and colors
     */
    public AttributeSet getTokenFontColors (String tokenName) {
        
        // 1) resolve profile name
	String profile = null;
	String profileOriginal = null;
        if (test != null) 
            profileOriginal = profile = test;
        else {
            profile = getEditorColoringImpl ().getCurrentFontColorProfile ();
            profileOriginal = getEditorColoringImpl ().getInternalFontColorProfile (profile);
        }

        // read coloring & resolve defaults
	Map m = getColorings (profileOriginal);
	if (m == null) {
            logger.log ("syntax1 " + tokenName + " : null");
            return null;
        }
	AttributeSet as = (AttributeSet) m.get (tokenName);
	if (as == null) {
            if (tokenName.equals (DEFAULT)) {
                as = new SimpleAttributeSet ();
                ((SimpleAttributeSet) as).addAttribute (
                    EditorStyleConstants.Default,
                    DEFAULT
                );
            } else {
                logger.log ("syntax2 " + tokenName + " : null");
                return null;
            }
        }
        
	SimpleAttributeSet result = resolveDefaults (as, profile);
        if (tokenName.equals (DEFAULT)) {
            // fix default params.
            if (result.getAttribute (StyleConstants.Foreground) == null)
                result.addAttribute (
                    StyleConstants.Foreground, 
                    Color.black
                );
            if (result.getAttribute (StyleConstants.Background) == null)
                result.addAttribute (
                    StyleConstants.Background, 
                    Color.white
                );
            if (result.getAttribute (StyleConstants.FontFamily) == null)
                result.addAttribute (
                    StyleConstants.FontFamily, 
                    getDefaultSystemFont ().getName ()
                );
            if (result.getAttribute (StyleConstants.FontSize) == null)
                result.addAttribute (
                    StyleConstants.FontSize, 
                    new Integer (getDefaultSystemFont ().getSize ())
                );
            logger.log ("syntax3 " + tokenName + " : " + result);
            return result;
        }
        
        // workarround
        if ( (!result.isDefined (StyleConstants.FontFamily)) &&
             ( result.isDefined (StyleConstants.FontSize) ||
               result.isDefined (StyleConstants.Bold) ||
               result.isDefined (StyleConstants.Italic)
             )
        ) {
            AttributeSet defaultCategory = getTokenFontColors (DEFAULT);
            result.addAttribute (
                StyleConstants.FontFamily,
                defaultCategory.getAttribute (StyleConstants.FontFamily)
            );
            if (!result.isDefined (StyleConstants.FontSize)) {
                result.addAttribute (
                    StyleConstants.FontSize,
                    defaultCategory.getAttribute (StyleConstants.FontSize)
                );
            }
        }
        logger.log ("syntax4 " + tokenName + " : " + result);
        return result;
    }

    /**
     * Gets all token font and colors for given profile or null, if 
     * profile does not exists. 
     * 
     * @param profile the name of profile
     *
     * @return AttributeSet describing the font and colors or null, if 
     *                      profile does not exists
     */
    public Collection /*<AttributeSet>*/ getAllFontColors (String profile) {
        
        // 1) resolve profile name
	profile = test != null ? 
            test : getEditorColoringImpl ().getInternalFontColorProfile (profile);
        
        // 2) read colorings
	Map m = getColorings (profile);
        if (m == null) return Collections.EMPTY_LIST;
	return Collections.unmodifiableCollection (m.values ());
    }
    
    /**
     * Gets default values for all font & colors for given profile, or null
     * if profile does not exist or if it does not have any defaults. 
     * 
     * @param profile the name of profile
     *
     * @return AttributeSet describing the font and colors or null, if 
     *                      profile does not exists
     */
    public Collection /*<AttributeSet>*/ getAllFontColorDefaults (String profile) {
        
	// 1) resolve profile name
	profile = test != null ? 
            test : getEditorColoringImpl ().getInternalFontColorProfile (profile);
        
        // 2) read defaults
	Map m = getDefaults (profile);
	if (m == null) return Collections.EMPTY_LIST;
	return Collections.unmodifiableCollection (m.values ());
    }
    
    /**
     * Sets all token font and colors for given profile. 
     * 
     * @param profile the name of profile
     * @param fontColors new colorings
     */
    public void setAllFontColors (
        String profile,
        Collection /*<AttributeSet>*/ fontColors
    ) {
        // 1) resolve profile name
        logger.log ("set " + profile);
        profile = test != null ? 
            test : getEditorColoringImpl ().getInternalFontColorProfile (profile);
        
        if (fontColors == null) {
            // 2) remove coloring / revert to defaults
            ColoringStorage.deleteColorings
                (mimeTypes, profile, "coloring.xml"); //NOI18N
            colorings.remove (profile);
            pcs.firePropertyChange (FontColorSettings.PROP_FONT_COLORS, null, null);
            return;
        }
        
        // 2) save coloring
        if (test == null)
            ColoringStorage.saveColorings 
                (mimeTypes, profile, "coloring.xml", fontColors); //NOI18N
        
        // 3) cache coloring
	Map m = getColorings (profile);
	if (m == null) return;
        Iterator it = fontColors.iterator ();
        boolean changed = false;
        while (it.hasNext ()) {
            AttributeSet as = (AttributeSet) it.next ();
            String name = (String) as.getAttribute 
                (StyleConstants.NameAttribute);
            if ((!changed) &&!as.equals (m.get (name)))
                changed = true;
            m.put (name, as);
            logger.log ("set " + name + " : " + as);
        }
        
        if (changed){
            pcs.firePropertyChange (FontColorSettings.PROP_FONT_COLORS, null, null);
        }
    }
    
    /**
     * PropertyChangeListener registration.
     *
     * @param l a PropertyChangeListener to be registerred
     */
    public void addPropertyChangeListener (PropertyChangeListener l) {
        pcs.addPropertyChangeListener (l);
    }
    
    /**
     * PropertyChangeListener registration.
     *
     * @param l a PropertyChangeListener to be unregisterred
     */
    public void removePropertyChangeListener (PropertyChangeListener l) {
        pcs.removePropertyChangeListener (l);
    } 
    
    
    // other methods ...........................................................
    
    private EditorSettingsImpl editorSettingsImpl;
    
    private EditorSettingsImpl getEditorColoringImpl () {
	if (editorSettingsImpl == null)
	    editorSettingsImpl = (EditorSettingsImpl) 
                EditorSettings.getDefault ();
	return editorSettingsImpl;
    }
    
    private Map colorings = new HashMap ();
    
    private Map getColorings (String profile) {
	if (!colorings.containsKey (profile)) {
            Map m = ColoringStorage.loadColorings (
                mimeTypes, 
                test != null ? "NetBeans" : profile,                  // NOI18N
                "coloring.xml",                                       // NOI18N
                false
            );
            if (m == null)
                m = ColoringStorage.loadColorings (
                    mimeTypes, 
                    "NetBeans",                                       // NOI18N
                    "coloring.xml",                                   // NOI18N
                    false
                );
            colorings.put (profile, m);
	}
	return (Map) colorings.get (profile);
    }
    
    private Map defaults = new HashMap ();
    
    private Map getDefaults (String profile) {
	if (!defaults.containsKey (profile)) {
            Map m = ColoringStorage.loadColorings 
                (mimeTypes, profile, "coloring.xml", true);           // NOI18N
            defaults.put (profile, m);
	}
	return (Map) defaults.get (profile);
    }
    
    private String getMimeType () {
        int i, k = mimeTypes.length;
        if (k == 0) return ""; //NOI18N
        if (k == 1) return mimeTypes [0];
        StringBuffer result = new StringBuffer (mimeTypes [0]);
        for (i = 1; i < k; i++)
            result.append ('+').append (mimeTypes [i]);
        return result.toString ();
    }
    
    private SimpleAttributeSet resolveDefaults (AttributeSet as, String profile) {
	SimpleAttributeSet sas = new SimpleAttributeSet (as);
	resolveDefaults (sas, sas, StyleConstants.Background, profile);
	resolveDefaults (sas, sas, StyleConstants.Foreground, profile);
	resolveDefaults (sas, sas, StyleConstants.Underline, profile);
	resolveDefaults (sas, sas, StyleConstants.StrikeThrough, profile);
	resolveDefaults (sas, sas, EditorStyleConstants.WaveUnderlineColor, profile);
	resolveDefaults (sas, sas, StyleConstants.Bold, profile);
	resolveDefaults (sas, sas, StyleConstants.Italic, profile);
	resolveDefaults (sas, sas, StyleConstants.FontFamily, profile);
	resolveDefaults (sas, sas, StyleConstants.FontSize, profile);
	return sas;
    }
    
    private void resolveDefaults (
	SimpleAttributeSet	originalAS, 
	AttributeSet		currentAS, 
	Object			name,
	String			profile
    ) {
	Object value = currentAS.getAttribute (name);
	if (value != null) {
	    originalAS.addAttribute (name, value);
	    return;
	}
	String defaultCategory = (String) currentAS.getAttribute 
	    (EditorStyleConstants.Default);
	if (defaultCategory == null) return;
	if (!defaultCategory.equals 
		(currentAS.getAttribute (StyleConstants.NameAttribute))
        ) {
	    Iterator it = getAllFontColors (profile).iterator ();
	    while (it.hasNext ()) {
		AttributeSet as1 = (AttributeSet) it.next ();
		if (defaultCategory.equals 
			(as1.getAttribute (StyleConstants.NameAttribute))
		)
		    resolveDefaults (originalAS, as1, name, profile);
	    }
	}
	
        Collection defaults = getEditorColoringImpl ().getDefaultFontColors (
            test == null ? profile : test
        );
        if (defaults == null) return;
	Iterator it = defaults.iterator ();
	while (it.hasNext ()) {
	    AttributeSet as1 = (AttributeSet) it.next ();
	    if (defaultCategory.equals 
		    (as1.getAttribute (StyleConstants.NameAttribute))
	    )
		resolveDefaults (originalAS, as1, name, profile);
	}
    }
    
    private static Font defaultSystemFont;
    private static Font getDefaultSystemFont () {
        if (defaultSystemFont == null) {
            Integer defaultFontSize = (Integer) UIManager.get 
                ("customFontSize");                                   // NOI18N
            int size = defaultFontSize != null ? 
                defaultFontSize.intValue () :
                UIManager.getFont ("TextField.font").getSize ();      // NOI18N
            if (size < 12) size = 12;
            defaultSystemFont = new Font (
                "Monospaced",                                         // NOI18N
                Font.PLAIN,
                size
            );
        }
        return defaultSystemFont;
    }

    public Object createInstance() {
        String profile = test != null ?
            test : getEditorColoringImpl().getCurrentFontColorProfile();

        // 1) resolve profile name
        String profileOriginal = null;
        if (test != null)
            profileOriginal = profile = test;
        else {
            profileOriginal = getEditorColoringImpl().getInternalFontColorProfile(profile);
        }
        
        Map highlightings = getEditorColoringImpl().getHighlightings(profile);
        if (highlightings != null) {
            highlightings = new HashMap(highlightings);
        } else {
            highlightings = Collections.EMPTY_MAP;
        }
        Map colorings = getColorings(profileOriginal);
        if (colorings != null) {
            colorings = new HashMap(colorings);
        }
    Collection defFC = getEditorColoringImpl().getDefaultFontColors(profile);
        if (defFC != null) {
            defFC = new ArrayList(defFC);
        }
        Collection allFC = getAllFontColors(profile);
        if (allFC != null) {
            allFC = new ArrayList(allFC);
        }
        return new Immutable(logger, test, profile, highlightings, colorings, defFC, allFC);
    }
    
    
    // innerclasses ............................................................
    
    private static class Listener implements PropertyChangeListener {
        private WeakReference       fontColorSettings;
        private EditorSettingsImpl  editorSettings;
        private String              test;
        
        Listener (FontColorSettingsImpl fontColorSettings) {
            this.fontColorSettings = new WeakReference (fontColorSettings);
            editorSettings = fontColorSettings.getEditorColoringImpl ();
            editorSettings.addPropertyChangeListener 
                (EditorSettings.PROP_CURRENT_FONT_COLOR_PROFILE, this);
            test = fontColorSettings.test;
            if (test != null)
                editorSettings.addPropertyChangeListener 
                    (test, this);
            else {
                editorSettings.addPropertyChangeListener 
                    (EditorSettings.PROP_DEFAULT_FONT_COLORS, this);
                editorSettings.addPropertyChangeListener 
                    (EditorSettings.PROP_EDITOR_FONT_COLORS, this);
            }
        }
        
        private FontColorSettingsImpl getFontColorSettings () {
            FontColorSettingsImpl r = (FontColorSettingsImpl) 
                fontColorSettings.get ();
            if (r != null) return r;
            editorSettings.removePropertyChangeListener 
                (EditorSettings.PROP_CURRENT_FONT_COLOR_PROFILE, this);
            if (test != null)
                editorSettings.removePropertyChangeListener 
                    (test, this);
            else {
                editorSettings.removePropertyChangeListener 
                    (EditorSettings.PROP_DEFAULT_FONT_COLORS, this);
                editorSettings.removePropertyChangeListener 
                    (EditorSettings.PROP_EDITOR_FONT_COLORS, this);
            }
            return null;
        }
        
        public void propertyChange (PropertyChangeEvent evt) {
            FontColorSettingsImpl fontColorSettings = getFontColorSettings ();
            fontColorSettings.pcs.firePropertyChange (FontColorSettings.PROP_FONT_COLORS, null, null);
        }
    }

    private static final class Immutable extends FontColorSettings {
        private final ErrorManager logger;
        private final String test;
        private final String profile;
        private final Map highlightings;
        private final Map colorings;
        private final Collection defFC;
        private final Collection allFC;
        
        public Immutable(ErrorManager logger, String test, String profile, Map highlightings, Map colorings, Collection defFC, Collection allFC) {
            this.logger = logger;
            this.test = test;
            this.profile = profile;
            this.highlightings = highlightings;
            this.colorings = colorings;
            this.defFC = defFC;
            this.allFC = allFC;
        }
        
        // FontColorSettings impl ..................................................
        
        /**
         * Gets the font and colors.
         *
         * @param settingName font and colors setting name
         *
         * @return AttributeSet describing the font and colors.
         */
        public AttributeSet getFontColors(String settingName) {
            if (settingName.equals(DEFAULT))
                // default highlighting is internally managed as syntax coloring
                return getTokenFontColors(settingName);
            
            // resolve profile name
            
            // delegate to EditorSettings
            AttributeSet as = (AttributeSet)highlightings.get(settingName);
            if (as == null) {
                logger.log("highlighting2 " + settingName + " : null");
                return null;
            }
            AttributeSet result = resolveDefaults(as, profile);
            logger.log("highlighting3 " + settingName + " : " + result);
            return result;
        }
        
        /**
         * Gets the token font and colors.
         *
         * @param tokenName token name
         *
         * @return AttributeSet describing the font and colors
         */
        public AttributeSet getTokenFontColors(String tokenName) {
            
            
            // read coloring & resolve defaults
            Map m = colorings;
            if (m == null) {
                logger.log("syntax1 " + tokenName + " : null");
                return null;
            }
            AttributeSet as = (AttributeSet) m.get(tokenName);
            if (as == null) {
                if (tokenName.equals(DEFAULT)) {
                    as = new SimpleAttributeSet();
                    ((SimpleAttributeSet) as).addAttribute(
                        EditorStyleConstants.Default,
                        DEFAULT 
                        );
                } else {
                    logger.log("syntax2 " + tokenName + " : null");
                    return null;
                }
            }
            
            SimpleAttributeSet result = resolveDefaults(as, profile);
            if (tokenName.equals(DEFAULT)) {
                
                // fix default params.
                if (result.getAttribute(StyleConstants.Foreground) == null)
                    result.addAttribute(
                        StyleConstants.Foreground,
                        Color.black
                        );
                if (result.getAttribute(StyleConstants.Background) == null)
                    result.addAttribute(
                        StyleConstants.Background,
                        Color.white
                        );
                if (result.getAttribute(StyleConstants.FontFamily) == null)
                    result.addAttribute(
                        StyleConstants.FontFamily,
                        getDefaultSystemFont().getName()
                        );
                if (result.getAttribute(StyleConstants.FontSize) == null)
                    result.addAttribute(
                        StyleConstants.FontSize,
                        new Integer(getDefaultSystemFont().getSize())
                        );
                logger.log("syntax3 " + tokenName + " : " + result);
                return result;
            }
            
            // workarround
            if ( (!result.isDefined(StyleConstants.FontFamily)) &&
                ( result.isDefined(StyleConstants.FontSize) ||
                result.isDefined(StyleConstants.Bold) ||
                result.isDefined(StyleConstants.Italic)
                )
                ) {
                AttributeSet defaultCategory = getTokenFontColors(DEFAULT);
                result.addAttribute(
                    StyleConstants.FontFamily,
                    defaultCategory.getAttribute(StyleConstants.FontFamily)
                    );
                if (!result.isDefined(StyleConstants.FontSize)) {
                    result.addAttribute(
                        StyleConstants.FontSize,
                        defaultCategory.getAttribute(StyleConstants.FontSize)
                        );
                }
            }
            logger.log("syntax4 " + tokenName + " : " + result);
            return result;
        }
        private SimpleAttributeSet resolveDefaults(AttributeSet as, String profile) {
            SimpleAttributeSet sas = new SimpleAttributeSet(as);
            resolveDefaults(sas, sas, StyleConstants.Background, profile);
            resolveDefaults(sas, sas, StyleConstants.Foreground, profile);
            resolveDefaults(sas, sas, StyleConstants.Underline, profile);
            resolveDefaults(sas, sas, StyleConstants.StrikeThrough, profile);
            resolveDefaults(sas, sas, EditorStyleConstants.WaveUnderlineColor, profile);
            resolveDefaults(sas, sas, StyleConstants.Bold, profile);
            resolveDefaults(sas, sas, StyleConstants.Italic, profile);
            resolveDefaults(sas, sas, StyleConstants.FontFamily, profile);
            resolveDefaults(sas, sas, StyleConstants.FontSize, profile);
            return sas;
        }
    
        private void resolveDefaults(
            SimpleAttributeSet	originalAS,
            AttributeSet		currentAS,
            Object			name,
            String			profile
        ) {
            Object value = currentAS.getAttribute(name);
            if (value != null) {
                originalAS.addAttribute(name, value);
                return;
            }
            String defaultCategory = (String) currentAS.getAttribute
                (EditorStyleConstants.Default);
            if (defaultCategory == null) return;
            if (!defaultCategory.equals
                (currentAS.getAttribute(StyleConstants.NameAttribute))
                ) {
                Iterator it = allFC.iterator();
                while (it.hasNext()) {
                    AttributeSet as1 = (AttributeSet) it.next();
                    if (defaultCategory.equals
                        (as1.getAttribute(StyleConstants.NameAttribute))
                        )
                        resolveDefaults(originalAS, as1, name, profile);
                }
            }
            
            Collection defaults = defFC;
            if (defaults == null) return;
            Iterator it = defaults.iterator();
            while (it.hasNext()) {
                AttributeSet as1 = (AttributeSet) it.next();
                if (defaultCategory.equals
                    (as1.getAttribute(StyleConstants.NameAttribute))
                    )
                    resolveDefaults(originalAS, as1, name, profile);
            }
        }
        
    }

    
}
