/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.editor.settings.storage;

import java.awt.Color;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import javax.swing.text.AttributeSet;
import javax.swing.text.StyleConstants;
import org.netbeans.api.editor.settings.EditorStyleConstants;
import org.netbeans.modules.editor.settings.storage.XMLStorage.Attribs;
import org.openide.ErrorManager;

import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileSystem;
import org.openide.filesystems.Repository;
import org.openide.util.NbBundle;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;


/**
 * This class contains support static methods for loading / saving and 
 * translating coloring (fontsColors.xml) files. It calls XMLStorage utilities.
 *
 * @author Jan Jancura
 */
class ColoringStorage {

    
    // load ....................................................................
    
    static Map loadColorings (
        String[] mimeTypes, 
        String profile, 
        String fileName,
        boolean defaults   // read default values only
    ) {
        // 1) load colorings
        FileSystem fs = Repository.getDefault ().getDefaultFileSystem ();
        String folderName = Utils.getFileName (mimeTypes, profile, null);
        FileObject defaultFo = fs.findResource (
            Utils.getFileName (mimeTypes, profile, "Defaults/" + fileName)
        );
        FileObject fo = defaults ? null : fs.findResource (folderName + "/" + fileName);
        if (fo == null) fo = defaultFo;
        if (fo == null) return null;
        List l = (List) XMLStorage.load (fo, new ColoringsReader ());

        // 2) translate names of categories
        FileObject basicName = fs.findResource (
             Utils.getFileName (mimeTypes, "NetBeans", "Defaults/" + fileName)
        );
        String bundleName = basicName != null ? 
            (String) basicName.getAttribute 
                ("SystemFileSystem.localizingBundle") : null;
        ResourceBundle bundle = null;
        if (bundleName != null)
            bundle = NbBundle.getBundle (bundleName);
        Map m = new HashMap ();
        Iterator it = l.iterator ();
        while (it.hasNext()) {
            SimpleAttributeSet as = (SimpleAttributeSet) it.next ();
            String name = (String) as.getAttribute (StyleConstants.NameAttribute);
            String displayName = name;
            if (bundle != null) {
                try {
                    displayName = bundle.getString (name);
                } catch (MissingResourceException ex) {
                    System.out.println("ColoringStorage.not localized: " + basicName + " : " + bundleName + " : " + name);
                }
            }
            as.addAttribute (EditorStyleConstants.DisplayName, displayName);
            m.put (name, as);
        }
        return m;
    }    

    private static class ColoringsReader extends XMLStorage.Handler {
        private List colorings = new ArrayList ();
        
        Object getResult () {
            return colorings;
        }
        
        public void startElement (
            String uri, 
            String localName,
            String name, 
            Attributes attributes
        ) throws SAXException {
            try {
                if (name.equals ("fontscolors")) {
                } else
                if (name.equals ("fontcolor")) {
                    SimpleAttributeSet a = new SimpleAttributeSet ();
                    a.addAttribute (
                        StyleConstants.NameAttribute, 
                        attributes.getValue ("name")
                    );
                    if (attributes.getValue ("bgColor") != null)
                        a.addAttribute (
                            StyleConstants.Background, 
                            XMLStorage.stringToColor (attributes.getValue ("bgColor"))
                        );
                    if (attributes.getValue ("foreColor") != null)
                        a.addAttribute (
                            StyleConstants.Foreground, 
                            XMLStorage.stringToColor (attributes.getValue ("foreColor"))
                        );
                    if (attributes.getValue ("underline") != null)
                        a.addAttribute (
                            StyleConstants.Underline, 
                            XMLStorage.stringToColor (attributes.getValue ("underline"))
                        );
                    if (attributes.getValue ("strikeThrough") != null)
                        a.addAttribute (
                            StyleConstants.StrikeThrough, 
                            XMLStorage.stringToColor (attributes.getValue ("strikeThrough"))
                        );
                    if (attributes.getValue ("waveUnderlined") != null)
                        a.addAttribute (
                            EditorStyleConstants.WaveUnderlineColor, 
                            XMLStorage.stringToColor (attributes.getValue ("waveUnderlined"))
                        );
                    if (attributes.getValue ("default") != null)
                        a.addAttribute (
                            EditorStyleConstants.Default, 
                            (String) attributes.getValue ("default")
                        );
                    colorings.add (a);
                } else
                if (name.equals ("font")) {
                    SimpleAttributeSet a = (SimpleAttributeSet) colorings.get 
                        (colorings.size () - 1);
                    if (attributes.getValue ("name") != null)
                        a.addAttribute (
                            StyleConstants.FontFamily,
                            attributes.getValue ("name")
                        );
                    if (attributes.getValue ("size") != null)
                        try {
                            a.addAttribute (
                                StyleConstants.FontSize,
                                Integer.decode (attributes.getValue ("size"))
                            );
                        } catch (NumberFormatException ex) {
                            ex.printStackTrace ();
                        }
                    if (attributes.getValue ("style") != null) {
                        a.addAttribute (
                            StyleConstants.Bold,
                            Boolean.valueOf (attributes.getValue ("style").indexOf ("bold") >= 0)
                        );
                        a.addAttribute (
                            StyleConstants.Italic,
                            Boolean.valueOf (attributes.getValue ("style").indexOf ("italic") >= 0)
                        );
                    }
                }
            } catch (Exception ex) {
                ErrorManager.getDefault ().notify (ex);
            }
        }
        
        public InputSource resolveEntity (String pubid, String sysid) {
            return new InputSource (
		new java.io.ByteArrayInputStream (new byte [0])
	    );
        }
    }    

    
    // save colorings ..........................................................
    
     static void deleteColorings (
        String[] mimeTypes, 
        String profile,
        String fileName
    ) {
        Utils.deleteFileObject (mimeTypes, profile, fileName);
    }
    
    static void saveColorings (
        String[] mimeTypes, 
        String profile,
        String fileName,
        Collection colorings
    ) {
        FileObject fo = Utils.createFileObject (mimeTypes, profile, fileName);
        //S ystem.out.println("\nColoringStorage.saveColorings " + fo);
        saveColorings (fo, colorings);
    }
    
    private static void saveColorings (FileObject fo, Collection colorings) {
        final StringBuffer sb = XMLStorage.generateHeader ();
        XMLStorage.generateFolderStart (sb, "fontscolors", null, "");
        Iterator it = colorings.iterator ();
        while (it.hasNext ()) {
            AttributeSet category = (AttributeSet) it.next ();
            Attribs attributes = new Attribs (true);
            attributes.add (
		"name", 
		(String) category.getAttribute (StyleConstants.NameAttribute)
	    );
	    if (category.isDefined (StyleConstants.Foreground))
		attributes.add (
		    "foreColor", 
		    XMLStorage.colorToString (
			(Color) category.getAttribute (StyleConstants.Foreground)
		    )
		);
	    if (category.isDefined (StyleConstants.Background))
		attributes.add (
		    "bgColor", 
		    XMLStorage.colorToString (
			(Color) category.getAttribute (StyleConstants.Background)
		    )
		);
	    if (category.isDefined (StyleConstants.StrikeThrough))
		attributes.add (
		    "strikeThrough", 
		    XMLStorage.colorToString (
			(Color) category.getAttribute (StyleConstants.StrikeThrough)
		    )
		);
	    if (category.isDefined (EditorStyleConstants.WaveUnderlineColor))
		attributes.add (
		    "waveUnderlined", 
		    XMLStorage.colorToString (
			(Color) category.getAttribute (EditorStyleConstants.WaveUnderlineColor)
		    )
		);
	    if (category.isDefined (StyleConstants.Underline))
		attributes.add (
		    "underline", 
		    XMLStorage.colorToString (
			(Color) category.getAttribute (StyleConstants.Underline)
		    )
		);
	    if (category.isDefined (EditorStyleConstants.Default))
                attributes.add (
		    "default", 
		    (String) category.getAttribute (EditorStyleConstants.Default)
		);
	    if ( category.isDefined (StyleConstants.FontFamily) ||
                 category.isDefined (StyleConstants.FontSize) ||
                 category.isDefined (StyleConstants.Bold) ||
                 category.isDefined (StyleConstants.Italic)
            ) {
		XMLStorage.generateFolderStart (sb, "fontcolor", attributes, "    ");
		attributes = new Attribs (true);
                if (category.isDefined (StyleConstants.FontFamily))
                    attributes.add (
                        "name", 
                        (String) category.getAttribute (StyleConstants.FontFamily)
                    );
                if (category.isDefined (StyleConstants.FontSize))
                    attributes.add (
                        "size", 
                        "" + category.getAttribute (StyleConstants.FontSize)
                    );
                if (category.isDefined (StyleConstants.Bold) ||
                    category.isDefined (StyleConstants.Italic)
                ) {
                    Boolean bold = Boolean.FALSE, italic = Boolean.FALSE;
                    if (category.isDefined (StyleConstants.Bold))
                        bold = (Boolean) category.getAttribute (StyleConstants.Bold);
                    if (category.isDefined (StyleConstants.Italic))
                        italic = (Boolean) category.getAttribute (StyleConstants.Italic);
                    attributes.add ("style", 
                        bold.booleanValue () ?
                            (italic.booleanValue () ?
                                "bold+italic" : 
                                "bold") :
                            (italic.booleanValue () ?
                                "italic" : "plain")
                    );
                }
		XMLStorage.generateLeaf (sb, "font", attributes, "        ");
		XMLStorage.generateFolderEnd (sb, "fontcolor", "    ");
	    } else
		XMLStorage.generateLeaf (sb, "fontcolor", attributes, "    ");
        }
        XMLStorage.generateFolderEnd (sb, "fontscolors", "");
        XMLStorage.save (fo, new String (sb));
    }
}
