/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.options.keymap;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.IOException;
import java.io.InputStream;
import java.lang.ref.WeakReference;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Vector;
import javax.swing.KeyStroke;
import javax.swing.text.EditorKit;
import javax.swing.text.Keymap;
import javax.swing.text.TextAction;

import org.netbeans.api.editor.settings.MultiKeyBinding;
import org.netbeans.editor.BaseKit;
import org.netbeans.editor.ext.ExtKit;
import org.netbeans.modules.editor.settings.storage.api.EditorSettings;
import org.netbeans.modules.editor.settings.storage.api.KeyBindingSettingsFactory;
import org.openide.ErrorManager;

import org.openide.util.Lookup;
import org.openide.util.Utilities;
import org.openide.xml.XMLUtil;


/**
 *
 * @author Jan Jancura
 */
public class KeymapModel {
    
    private LayersBridge        layersBridge = new LayersBridge ();
    private EditorBridge        editorBridge = new EditorBridge ();
    private ErrorManager        log = ErrorManager.getDefault ().getInstance 
                                    (KeymapModel.class.getName ());
                                    
    // actions .................................................................
    
    public Set /*<String>*/ getActionCategories () {
        Set result = new HashSet ();
        result.addAll (layersBridge.getActions ().keySet ());
        result.addAll (editorBridge.getActions ().keySet ());
        return Collections.unmodifiableSet (result);
    }
    
    /** Map (String (category name) > Set (ActionImpl)). */
    private Map categoryToActions = new HashMap ();
    
    /**
     * Returns List (ActionImpl) of all global and editor actions.
     */
    public Set getActions (String category) {
        if (!categoryToActions.containsKey (category)) {
            Set actions = new HashSet ();
            Set s = (Set) layersBridge.getActions ().get (category);
            if (s != null) actions.addAll (s);
            s = (Set) editorBridge.getActions ().get (category);
            if (s != null)
                actions = mergeActions (s, actions);
            categoryToActions.put (category, actions);
        }
        return (Set) categoryToActions.get (category);
    }

    /**
     * Clear action caches.
     */
    public void refreshActions () {
        categoryToActions = new HashMap ();
        editorBridge.refreshActions ();
    }
    
    // keymaps .................................................................
    
    public String getCurrentProfile () {
        return editorBridge.getCurrentProfile ();
    }
    
    public void setCurrentProfile (String profile) {
        //layersBridge.setCurrentProfile (profile); !!!!!!!!!!!!!
        editorBridge.setCurrentProfile (profile);
    }
    
    public List getProfiles () {
        return layersBridge.getProfiles ();
    }
    
    public boolean isCustomProfile (String profile) {
        return editorBridge.isCustomProfile (profile);
    }
    
    /** Map (String (profile) > Map (ActionImpl > Set (String (shortcut AS-M)))). */
    private Map keyMaps = new HashMap ();
    
    /**
     * Returns Map (ActionImpl > Set (String (shortcut))).
     */
    public Map getKeymap (String profile) {
        if (!keyMaps.containsKey (profile)) {
            keyMaps.put (
                profile, 
                mergeShortcuts (
                    (Map) editorBridge.readKeymap (profile),
                    (Map) layersBridge.getKeymap (profile)
                )
            );
        }
        return (Map) keyMaps.get (profile);
    }
    
    /** Map (String (keymap name) > Map (ActionImpl > Set (String (shortcut AS-M)))). */
    private Map keyMapDefaults = new HashMap ();
    
    /**
     * Returns Map (ActionImpl > Set (String (shortcut))).
     */
    public Map getKeymapDefaults (String profile) {
        if (!keyMapDefaults.containsKey (profile)) {
            keyMapDefaults.put (
                profile, 
                mergeShortcuts (
                    (Map) editorBridge.readKeymap (profile),
                    (Map) layersBridge.getKeymap (profile)
                )
            );
        }
        return (Map) keyMapDefaults.get (profile);
    }
    
    public void deleteProfile (String profile) {
        layersBridge.deleteProfile (profile);
        editorBridge.deleteProfile (profile);
    }
    
    /**
     * Defines new shortcuts for some actions in given keymap.
     * Map (ActionImpl > Set (String (shortcut AS-M P)).
     */
    public void changeKeymap (String profile, Map actionToShortcuts) {
        log ("changeKeymap.actionToShortcuts", actionToShortcuts.entrySet ());
        
        // 1) mix changes with current keymap and put them to cached current shortcuts
        Map m = new HashMap (getKeymap (profile));
        m.putAll (actionToShortcuts);
        keyMaps.put (profile, m);
        log ("changeKeymap.m", m.entrySet ());
        
        layersBridge.saveKeymap (profile, m);
        editorBridge.saveKeymap (profile, m);
    }
    
    
    // private methods .........................................................
    
    private void log (String name, Collection items) {
        if (!log.isLoggable (log.INFORMATIONAL)) return;
        log.log (
            log.INFORMATIONAL, 
            name
        );
        Iterator it = items.iterator ();
        while (it.hasNext ()) {
            Object item = it.next ();
            log.log (log.INFORMATIONAL, "  " + item);
        }
    }
    
    private Map sharedActions = new HashMap ();
    
    /**
     * Merges editor actions and layers actions. Creates CompoundAction for
     * actions like Copy, registerred to both contexts.
     */
    private Set mergeActions (
        Collection editorActions,
        Collection layersActions 
    ) {
        Set result = new HashSet ();
        Map idToAction = new HashMap ();
        if (editorActions != null) {
            Iterator it = editorActions.iterator ();
            while (it.hasNext ()) {
                ActionImpl action = (ActionImpl) it.next ();
                String id = action.getDelegatingActionId ();
                if (id != null)
                    idToAction.put (id, action);
                else
                    result.add (action);
            }
        }
        
        if (layersActions != null) {
            Iterator it = layersActions.iterator ();
            while (it.hasNext ()) {
                ActionImpl layersAction = (ActionImpl) it.next ();
                String id = layersAction.getId ();
                if (!idToAction.containsKey (id))
                    result.add (layersAction);
                else {
                    ActionImpl editorAction = (ActionImpl) idToAction.
                        remove (id);
                    CompoundAction compoundAction = new CompoundAction (
                        editorAction,                           // editor action
                        layersAction                            // layers action
                    );
                    result.add (compoundAction);
                    sharedActions.put (editorAction, compoundAction);
                    sharedActions.put (layersAction, compoundAction);
                }
            }
        }
        
        result.addAll (idToAction.values ());
        
        return result;
    }
    
    /**
     * Merges editor actions and layers actions. Creates CompoundAction for
     * actions like Copy, registerred to both contexts.
     */
    private Map mergeShortcuts (
        Map editorActions,
        Map layersActions 
    ) {
        Map result = new HashMap ();
        Iterator it = editorActions.keySet ().iterator ();
        while (it.hasNext ()) {
            ActionImpl action = (ActionImpl) it.next ();
            Set shortcuts = (Set) editorActions.get (action);
            if (sharedActions.containsKey (action))
                action = (CompoundAction) sharedActions.get (action);
            result.put (action, shortcuts);
        }
        it = layersActions.keySet ().iterator ();
        while (it.hasNext ()) {
            ActionImpl action = (ActionImpl) it.next ();
            Set shortcuts = (Set) layersActions.get (action);
            if (sharedActions.containsKey (action))
                action = (CompoundAction) sharedActions.get (action);
            result.put (action, shortcuts);
        }
        return result;
    }

    {
        // HACK - loads all actions. othervise during second open of Options
        // Dialog (after cancel) map of sharedActions is not initialized.
        Iterator it = getActionCategories ().iterator ();
        while (it.hasNext ())
            getActions ((String) it.next ());
    }
}
