/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.editor.mimelookup;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.WeakHashMap;
import org.netbeans.spi.editor.mimelookup.InstanceProvider;
import org.netbeans.spi.editor.mimelookup.MimeLookupInitializer;
import org.openide.filesystems.FileChangeAdapter;
import org.openide.filesystems.FileEvent;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import org.openide.filesystems.Repository;
import org.openide.loaders.DataFolder;
import org.openide.loaders.DataObject;
import org.openide.loaders.DataObjectNotFoundException;
import org.openide.loaders.FolderLookup;
import org.openide.util.Lookup;
import org.openide.util.LookupEvent;
import org.openide.util.LookupListener;


/** 
 *
 *  @author Martin Roskanin
 */
public class LayerFolderObjectsProvider {
    
    private static final boolean debugFoldersListening = 
        Boolean.getBoolean("netbeans.debug.editor.mimelookup.mimefolders.listening"); // NOI18N
    
    private String suffixFolder;
    private Class clazz;
    private InstanceProvider instanceProvider;
    private Map mimePath2folderObjectsResults = new HashMap();    
    
    public LayerFolderObjectsProvider(){
    }
    
    /**
     *   @param suffixFolder name of the suffixFolder or null, if the searching class is just
     *          under <mime-type> folder directly
     *   @param clazz class of searched object
     */
    public LayerFolderObjectsProvider(String suffixFolder, Class clazz, InstanceProvider instanceProvider){
        this.suffixFolder = suffixFolder;
        this.clazz = clazz;
        this.instanceProvider = instanceProvider;
    }
    
    public Class getClazz(){
        return clazz;
    }
    
    private DataFolder getDataFolder(FileObject fo){
        if (fo == null){
            return null;
        }
        try {
            DataObject dobj = DataObject.find(fo);
            if ((dobj instanceof DataFolder)) {
                return (DataFolder)dobj;
            }
        } catch (DataObjectNotFoundException e) {
        }
        return null;
    }
    
    private Lookup.Result getLookupResult(String mimePath[]){
        String s = mimePath2String(mimePath, true, true);
        FileObject fo = getExistingParentFolder(s);//getLayerFolder(s);
        
        DataFolder dataFolder = getDataFolder(fo);
        if (dataFolder == null){
            return null;
        }
        List inheritedFolders = new ArrayList();
        inheritedFolders = getInheritedFolders(mimePath);
        Lookup lookup = new CompositeLayerFolderLookup(instanceProvider, inheritedFolders);
        Lookup.Result res = lookup.lookup(new Lookup.Template(clazz));
        return res;
    }
    
    public Lookup.Result folderObjects(String[] mimePath) {
        String s = mimePath2String(mimePath, true, true);
        synchronized (mimePath2folderObjectsResults){
            Lookup.Result result = (Lookup.Result) mimePath2folderObjectsResults.get(s);
            if (result != null){
                return result;
            }
            
            // Layer folder need not be created yet. Let's create temporary
            // lookup result, that will listen on creation of desired layer folder and
            // fire resultChanged if folder will appear. If mimePath contains
            // "+" sign it means, it is a compound mime type => special
            // DelegatingLookupResult will be created - CompoundDelegatingLookupResult
            result = (s.indexOf("+") > 0) ? // NOI18N
                (Lookup.Result) new CompoundDelegatingLookupResult(mimePath) :
                (Lookup.Result) new DelegatingLookupResult(mimePath);
            mimePath2folderObjectsResults.put(s, result);
            return result;
        }
    }
        
    private List/*<DataFolder>*/ getInheritedFolders(String mimePath[]){
        List list = new ArrayList();
        for (int i = 0; i<mimePath.length; i++){
            String mimePathLocal[] = new String[mimePath.length - i];
            System.arraycopy(mimePath, 0, mimePathLocal, 0, mimePathLocal.length);
            String s = mimePath2String(mimePathLocal, true, true);
            DataFolder dataFolder = getDataFolder(getLayerFolder(s));
            
            if (dataFolder!=null){
                list.add(dataFolder);
            }
            
        }
        FileObject layerFolder = getLayerFolder(mimePath2String(new String[0], true, true));
        DataFolder dataFolder = getDataFolder(layerFolder);
        if (dataFolder != null){
            list.add(dataFolder);
        }
        
        return list;
    }
    
    private FileObject getLayerFolder(String path){
        FileObject parentFolder = Repository.getDefault().
                getDefaultFileSystem().findResource(path); // NOI18N

        if (parentFolder == null || parentFolder.isData()
                || !parentFolder.isValid() || parentFolder.isVirtual()) {
            return null; //invalid folder
        }
        
        return parentFolder;
    }
    
    private String mimePath2String(String[] mimePath, boolean prefixEditorsFolder, boolean appendLookupFolder){
        StringBuffer sb = new StringBuffer();
        if (prefixEditorsFolder){
            sb.append("Editors/"); //NOI18N
        }
        for (int i=0; i<mimePath.length; i++){
            sb.append(mimePath[i]);
            if (i < mimePath.length-1){
                sb.append("/"); //NOI18N
            }
        }
        if (appendLookupFolder && suffixFolder != null){
            String sep = (mimePath.length > 0) ? "/" : ""; //NOI18N
            sb.append(sep+suffixFolder); //NOI18N
        }
        return sb.toString();
    }

    private FileObject getExistingParentFolder(String mimePath){
        StringTokenizer tokenizer = new StringTokenizer(mimePath, "/"); //NOI18N
        int countTokens = tokenizer.countTokens();
        String tokens[] = new String[countTokens];
        for (int i = 0; i<countTokens; i++){
            if (i > 0){
                tokens[i] = tokens[i-1] + "/" + tokenizer.nextToken(); //NOI18N
            } else {
                tokens[i] = tokenizer.nextToken();
            }
        }
        for (int i = tokens.length -1; i>=0; i--){
            FileObject folder = getLayerFolder(tokens[i]);
            if (folder != null){
                return folder;
            }
        }
        return null;
    }
    
    /** DelegatingLookupResult works in dual mode. <p>
     *  1. If the mime type folder is not created in xml layer, it will act as 
     *     an empty result, that will return empty Collection for allInstances, allItems, etc. <p>
     *  2. If the mime folder will be created then the delegator will be set up and the 
     *     DelegatingLookupResult will operate as standard Lookup.Result over FodlerLookup.
     */
    private class DelegatingLookupResult extends Lookup.Result implements ParentFolderNotificator{
        private String[] mimePath;
        private Lookup.Result delegator;
//        private final List tempListeners = new ArrayList(); // List<LookupListener>
        private ParentFolderListener parentFolderListener;
        private final String path;
        private String lastAttachedListenerPath = null;
        private WeakHashMap listeners = new WeakHashMap(); //<delegator-itsListenersInList>
        
        public DelegatingLookupResult(String[] mimePath){
            
            this.path = mimePath2String(mimePath, true, true);
            this.delegator = getLookupResult(mimePath);
            this.mimePath = mimePath;
            attachFolderListener(path);
        }

        public void folderCreated(FileObject fo){
            fo = getExistingParentFolder(path);
            Lookup.Result oldDelegator = delegator;
            delegator = getLookupResult(mimePath);
            if (debugFoldersListening){
                System.out.println("    *** delegator created for:"+fo); //NOI18N
            }

            List list = (List)listeners.get(oldDelegator);
            if (list!=null){
                for (int i=0; i<list.size(); i++){
                    LookupListener l = (LookupListener)list.get(i);
                    oldDelegator.removeLookupListener(l);                        
                    delegator.addLookupListener(l);
                }
            }
            listeners.remove(oldDelegator);
            listeners.put(delegator, list);

            oldDelegator = null;
            
            //notify about this change
            fireResultChanged(new LookupEvent (this));
            
            // reattaching the listener
            attachFolderListener(path);
        }
        
        public void folderDeleted(FileObject fo){
            Lookup.Result oldDelegator = delegator;
            delegator = getLookupResult(mimePath);
            
            List list = (List)listeners.get(oldDelegator);
            if (list!=null){
                for (int i=0; i<list.size(); i++){
                    LookupListener l = (LookupListener)list.get(i);
                    oldDelegator.removeLookupListener(l);
                    delegator.addLookupListener(l);
                }
            }
            listeners.remove(oldDelegator);
            listeners.put(delegator, list);
            
            oldDelegator = null;
            
            //notify about this change
            fireResultChanged(new LookupEvent(this));
            
            if (debugFoldersListening){
                System.out.println("    *** delegator disposed, acting as FAKE lookup result"); //NOI18N
            }
            
            // reattaching the listener
            attachFolderListener(path);
        }

        private void fireResultChanged(LookupEvent evt){
            if (delegator!=null){
                List list = (List)listeners.get(delegator);
                if (list!=null){
                    for (int i=0; i<list.size(); i++){
                        LookupListener l = (LookupListener)list.get(i);
                        l.resultChanged(evt);
                    }
                }
            }
        }
        
        private synchronized void attachFolderListener(String path){
            // first deattach
            if (lastAttachedListenerPath != null){
                FileObject lastAttachedListener = getLayerFolder(lastAttachedListenerPath);
                if (lastAttachedListener != null){
                    if (debugFoldersListening){
                        System.out.println("remove PREVIOUSLY ATTACHED and now not needed parentFolderListener from:"+lastAttachedListener); //NOI18N
                    }
                    lastAttachedListener.removeFileChangeListener(parentFolderListener);
                }
            }
            
            FileObject existingParent = getExistingParentFolder(path);
            if (existingParent == null){
                return;
            }
            parentFolderListener = new ParentFolderListener(this, existingParent.getPath());
            if (debugFoldersListening){
                System.out.println("");
                System.out.println("attaching parentFolderListener to existingParent:"+existingParent); //NOI18N
                System.out.println("for global path:"+path); // NOI18N
            }
            
            existingParent.addFileChangeListener(
                    FileUtil.weakFileChangeListener(
                    parentFolderListener, existingParent)
                    );
            lastAttachedListenerPath = existingParent.getPath();
        }
        
        public synchronized void addLookupListener(LookupListener l) {
            delegator.addLookupListener(l);
            List list = (List)listeners.get(delegator);
            if (list==null){
                list = new ArrayList();
            }
            list.add(l);
            listeners.remove(delegator);
            listeners.put(delegator, list);
        }
        
        public synchronized void removeLookupListener(LookupListener l) {
            delegator.removeLookupListener(l);
            List list = (List)listeners.get(delegator);
            if (list!=null){
                list.remove(l);
            }
            listeners.remove(delegator);
            listeners.put(delegator, list);
        }

        public Collection allInstances() {
            if (delegator != null){
                return delegator.allInstances();
            }
            
            return Collections.EMPTY_LIST;
        }

        public Collection allItems() {
            if (delegator != null){
                return delegator.allItems();
            }
            
            return Collections.EMPTY_LIST;
        }

        public Set allClasses() {
            if (delegator != null){
                return delegator.allClasses();
            }
            
            return Collections.EMPTY_SET;
        }
        
    }

    private interface ParentFolderNotificator{
        void folderCreated(FileObject fo);
        void folderDeleted(FileObject fo);
    }
    
    private class ParentFolderListener extends FileChangeAdapter{
        
        ParentFolderNotificator notificator;
        String path;
        
        public ParentFolderListener(ParentFolderNotificator notificator, String path){
            this.notificator = notificator;
            this.path = path;
        }
        
        public void fileDeleted(FileEvent fe) {
            if (debugFoldersListening) {
                System.out.println(""); //NOI18N
                System.out.println("ParentFolderListener.fileDeleted:"+ //NOI18N
                        fe.getFile());
            }
            
            // check if a folder was removed
            FileObject removedFolder = fe.getFile();
            if (removedFolder == null || !removedFolder.isFolder()){
                // not a folder, ignore
                if (debugFoldersListening) {
                    System.out.println("    - not a folder, ignore"); //NOI18N
                }
                return;
            }
            
            if (path.equals(removedFolder.getPath())){
                if (debugFoldersListening) {
                    System.out.println("    - folder on which we were listening has been removed:"+path); //NOI18N
                }
                notificator.folderDeleted(removedFolder);
            }
        }
        
        public void fileFolderCreated(FileEvent fe) {
            if (debugFoldersListening) {
                System.out.println(""); //NOI18N
                System.out.println("ParentFolderListener.fileFolderCreated:"+fe.getFile()); //NOI18N
            }
            
            FileObject createdFolder = fe.getFile();
            // check if a folder was created
            if (createdFolder == null || !createdFolder.isFolder()){
                // not a folder, ignore
                if (debugFoldersListening) {
                    System.out.println("    - not a folder, ignore"); //NOI18N
                }
                return;
            }
            
            notificator.folderCreated(createdFolder);
        }        
        
    }

    private class CompoundDelegatingLookupResult extends Lookup.Result {
        private String mimePath[];
        private List/*<DelegatingLookupResult>*/ delegators = new ArrayList();
        
        public CompoundDelegatingLookupResult(String[] mimePath){
            this.mimePath = mimePath;
            List paths = getPaths(mimePath);
            for (int i=0; i<paths.size(); i++){
                String s[] = (String[])paths.get(i);
                this.delegators.add(new DelegatingLookupResult(s));    
            }
        }
        
        public synchronized void addLookupListener(LookupListener l){
            Iterator it = delegators.iterator();
            while (it.hasNext()){
                DelegatingLookupResult dlr = (DelegatingLookupResult) it.next();
                dlr.addLookupListener(l);
            }
        }

        public synchronized void removeLookupListener(LookupListener l){
            Iterator it = delegators.iterator();
            while (it.hasNext()){
                DelegatingLookupResult dlr = (DelegatingLookupResult) it.next();
                dlr.removeLookupListener(l);
            }
        }

        public synchronized java.util.Collection allInstances(){
            
            // gather all paths from delegators
            List inheritedFolders = new ArrayList();
            Iterator it = delegators.iterator();
            while (it.hasNext()){
                DelegatingLookupResult dlr = (DelegatingLookupResult) it.next();
                DataFolder dataFolder = getDataFolder(getExistingParentFolder(dlr.path));
                if (dataFolder == null){
                    continue;
                }

                List folders = getInheritedFolders(dlr.mimePath);
                for (int i=0; i<folders.size(); i++){
                    Object folder = folders.get(i);
                    if (!inheritedFolders.contains(folder)){
                        inheritedFolders.add(folder);
                    }
                }
            }
            
            Lookup lookup = new CompositeLayerFolderLookup(instanceProvider, inheritedFolders);
            Lookup.Result res = lookup.lookup(new Lookup.Template(clazz));
            return res.allInstances();
        }
        
        public List getPaths(String path[]){
            List ret = new ArrayList();
            ret.add(new String[0]);
            
            for (int i = 0; i<path.length; i++){
                String forkName = null;
                int plusIndex = path[i].indexOf("+"); //NOI18N
                int slashIndex = path[i].indexOf("/"); //NOI18N
                if (plusIndex > 0){
                    String prefix = path[i].substring(0, slashIndex+1);
                    String suffix = path[i].substring(plusIndex+1);
                    if (suffix!=null && "xml".equals(suffix.toLowerCase())){ //NOI18N
                        prefix = "text/"; //NOI18N
                    }
                    forkName = prefix + suffix;
                }
                
                
                List tempList = new ArrayList();
                for (int j = 0; j<ret.size(); j++){
                    String oldPath[] = (String[])ret.get(j);
                    int oldPathLength = oldPath.length;
                    String newPath[] = new String[oldPathLength+1];
                    System.arraycopy(oldPath, 0, newPath, 0, oldPathLength);
                    newPath[oldPathLength] = path[i];
                    tempList.add(newPath);
                }
                
                
                List tempListFork = new ArrayList();
                if (forkName != null){
                    for (int j = 0; j<ret.size(); j++){
                        String oldPath[] = (String[])ret.get(j);
                        int oldPathLength = oldPath.length;
                        String newPath[] = new String[oldPathLength+1];
                        System.arraycopy(oldPath, 0, newPath, 0, oldPathLength);
                        newPath[oldPathLength] = forkName;
                        tempListFork.add(newPath);
                    }
                }
                
                ret = tempList;
                ret.addAll(tempListFork);
            }
            return ret;
        }
        

    }
}
