/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.lib.editor.hyperlink;

import java.awt.Color;
import java.awt.Cursor;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import javax.swing.SwingUtilities;
import javax.swing.text.BadLocationException;
import javax.swing.text.Document;
import javax.swing.text.EditorKit;
import javax.swing.text.JTextComponent;
import javax.swing.text.Position;
import org.netbeans.editor.*;
import org.netbeans.lib.editor.hyperlink.spi.HyperlinkProvider;
import org.openide.ErrorManager;

/**
 *
 * @author Jan Lahoda
 */
public class HyperlinkOperation implements MouseListener, MouseMotionListener, PropertyChangeListener, KeyListener {

    private static Cursor HAND_CURSOR = null;
    
    private JTextComponent component;
    private Document       currentDocument;
    private HyperlinkLayer layer;
    private String         mimeType;
    private Cursor         oldComponentsMouseCursor;
    private boolean        hyperlinkUp;
    private boolean        listenersSetUp;

    private boolean        hyperlinkEnabled;
    private int            actionKeyMask;
    
    public static HyperlinkOperation create(JTextComponent component, String mimeType) {
        return new HyperlinkOperation(component, mimeType);
    }
    
    private static synchronized Cursor getHandCursor() {
        if (HAND_CURSOR == null)
            HAND_CURSOR = Cursor.getPredefinedCursor(Cursor.HAND_CURSOR);
        
        return HAND_CURSOR;
    }
    
    /** Creates a new instance of HoveringImpl */
    private HyperlinkOperation(JTextComponent component, String mimeType) {
        this.component = component;
        this.mimeType  = mimeType;
        this.oldComponentsMouseCursor = null;
        this.hyperlinkUp = false;
        this.listenersSetUp = false;
        
        readSettings();
        
        if (hyperlinkEnabled) {
            //Fix for #57409:
            EditorUI ui = Utilities.getEditorUI(component);
            
            layer = new HyperlinkLayer();
            
            if (ui != null)
                ui.addLayer(layer, 1100);
            
            component.addPropertyChangeListener("document", this); // NOI18N
        }
    }
    
    private void documentUpdated() {
        if (!hyperlinkEnabled)
            return ;
        
        currentDocument = component.getDocument();
        
        if (currentDocument instanceof BaseDocument) {
            if (!listenersSetUp) {
                component.addMouseListener(this);
                component.addMouseMotionListener(this);
                component.addKeyListener(this);
                listenersSetUp = true;
            }
        }
    }
    
    private void readSettings() {
        String hyperlinkActivationKeyPropertyValue = System.getProperty("org.netbeans.lib.editor.hyperlink.HyperlinkOperation.activationKey");
        
        if (hyperlinkActivationKeyPropertyValue != null) {
            if ("off".equals(hyperlinkActivationKeyPropertyValue)) { // NOI18N
                this.hyperlinkEnabled = false;
                this.actionKeyMask = (-1);
            } else {
                this.hyperlinkEnabled = true;
                this.actionKeyMask = (-1);
                
                for (int cntr = 0; cntr < hyperlinkActivationKeyPropertyValue.length(); cntr++) {
                    int localMask = 0;
                    
                    switch (hyperlinkActivationKeyPropertyValue.charAt(cntr)) {
                        case 'S': localMask = InputEvent.SHIFT_DOWN_MASK; break;
                        case 'C': localMask = InputEvent.CTRL_DOWN_MASK;  break;
                        case 'A': localMask = InputEvent.ALT_DOWN_MASK;   break;
                        case 'M': localMask = InputEvent.META_DOWN_MASK;  break;
                        default:
                            ErrorManager.getDefault().log(ErrorManager.WARNING, "Incorrect value of org.netbeans.lib.editor.hyperlink.HyperlinkOperation.activationKey property (only letters CSAM are allowed): " + hyperlinkActivationKeyPropertyValue.charAt(cntr));
                    }
                    
                    if (localMask == 0) {
                        //some problem, ignore
                        this.actionKeyMask = (-1);
                        break;
                    }
                    
                    if (this.actionKeyMask == (-1))
                        this.actionKeyMask = localMask;
                    else
                        this.actionKeyMask |= localMask;
                }
                
                if (this.actionKeyMask == (-1)) {
                    ErrorManager.getDefault().log(ErrorManager.WARNING, "Some problem with property org.netbeans.lib.editor.hyperlink.HyperlinkOperation.activationKey, more information might be given above. Falling back to the default behaviour.");
                } else {
                    return;
                }
            }
        }
        
        this.hyperlinkEnabled = true;
        
        Object activation = Settings.getValue(Utilities.getKitClass(component), SettingsNames.HYPERLINK_ACTIVATION_MODIFIERS);
        
        if (activation != null && activation instanceof Integer) {
            this.actionKeyMask = ((Integer) activation).intValue();
        } else {
            this.actionKeyMask = InputEvent.CTRL_DOWN_MASK;
        }
    }
    
    public void mouseMoved(MouseEvent e) {
        if (isHyperlinkEvent(e)) {
            int position = component.viewToModel(e.getPoint());
            
            if (position < 0) {
                unHyperlink(true);
                
                return ;
            }
            
            performHyperlinking(position);
        } else {
            unHyperlink(true);
        }
    }
    
    public void mouseDragged(MouseEvent e) {
        //ignored
    }
    
    private boolean isHyperlinkEvent(InputEvent e) {
        return ((e.getModifiers() | e.getModifiersEx()) & actionKeyMask) == actionKeyMask;
    }
    
    private void performHyperlinking(int position) {
        HyperlinkProvider provider = findProvider(position);
        
        if (provider != null) {
            int[] offsets = provider.getHyperlinkSpan(component.getDocument(), position);
            
            if (offsets != null)
                makeHyperlink(offsets[0], offsets[1]);
        } else {
            unHyperlink(true);
        }
    }
    
    private void performAction(int position) {
        HyperlinkProvider provider = findProvider(position);
        
        if (provider != null) {
            unHyperlink(true);
            
            //make sure the position is correct and the JumpList works:
            component.getCaret().setDot(position);
            JumpList.checkAddEntry(component, position);
            
            provider.performClickAction(component.getDocument(), position);
        }
    }
    
    private HyperlinkProvider findProvider(int position) {
        Object mimeTypeObj = component.getDocument().getProperty("mimeType");  //NOI18N
        String mimeType;
        
        if (mimeTypeObj instanceof String)
            mimeType = (String) mimeTypeObj;
        else {
            mimeType = this.mimeType;
        }
        
        List/*<HyperlinkProvider>*/ providers = HyperlinkProviderManager.getDefault().getHyperlinkProviders(mimeType);
        
        for (Iterator/*<HyperlinkProvider>*/ i = providers.iterator(); i.hasNext(); ) {
            HyperlinkProvider provider = (HyperlinkProvider) i.next();
            
            if (provider.isHyperlinkPoint(component.getDocument(), position)) {
                return provider;
            }
        }
        
        return null;
    }
    
    private synchronized void makeHyperlink(final int start, final int end) {
        boolean makeCursorSnapshot = true;
        
        if (hyperlinkUp) {
            unHyperlink(false);
            makeCursorSnapshot = false;
        }
        
        try {
            Position startPos = layer.hyperlinkStart = component.getDocument().createPosition(start);
            Position   endPos = layer.hyperlinkEnd   = component.getDocument().createPosition(end);
            
            hyperlinkUp = true;
            
            damageRange(startPos, endPos);
            
            if (makeCursorSnapshot) {
                if (component.isCursorSet()) {
                    oldComponentsMouseCursor = component.getCursor();
                } else {
                    oldComponentsMouseCursor = null;
                }
                component.setCursor(getHandCursor());
            }
        } catch (BadLocationException e) {
            ErrorManager.getDefault().notify(e);
            unHyperlink(false);
        }
    }
    
    private void damageRange(final Position start, final Position end) {
        if (start != null && end != null) {
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    component.getDocument().render(new Runnable() {
                        public void run() {
                            int startIndex = start.getOffset();
                            int endIndex = end.getOffset();
                            
                            component.getUI().damageRange(component, startIndex, endIndex);
                        }
                    });
                }
            });
        }
    }
    private synchronized void unHyperlink(boolean removeCursor) {
        if (!hyperlinkUp)
            return ;
        
        final Position start = layer.hyperlinkStart;
        final Position end   = layer.hyperlinkEnd;
        
        layer.hyperlinkStart = null;
        layer.hyperlinkEnd   = null;
        
        damageRange(start, end);
        
        if (removeCursor) {
            if (component.isCursorSet() && component.getCursor() == getHandCursor()) {
                component.setCursor(oldComponentsMouseCursor);
            }
            oldComponentsMouseCursor = null;
        }
        
        hyperlinkUp = false;
    }
    
    public void propertyChange(PropertyChangeEvent evt) {
        if (currentDocument != component.getDocument())
            documentUpdated();
    }
    
    public void keyTyped(KeyEvent e) {
        //ignored
    }

    public void keyReleased(KeyEvent e) {
        if ((e.getModifiers() & actionKeyMask) == 0)
            unHyperlink(true);
    }

    public void keyPressed(KeyEvent e) {
       //ignored
    }

    public void mouseReleased(MouseEvent e) {
        //ignored
    }

    public void mousePressed(MouseEvent e) {
        //ignored
    }

    public void mouseExited(MouseEvent e) {
        //ignored
    }

    public void mouseEntered(MouseEvent e) {
        //ignored
    }

    public void mouseClicked(MouseEvent e) {
        if (isHyperlinkEvent(e) && !e.isPopupTrigger() && e.getClickCount() == 1 && e.getButton() == MouseEvent.BUTTON1) {
            int position = component.viewToModel(e.getPoint());
            
            if (position < 0) {
                return ;
            }
            
            performAction(position);
        }
    }
    
    private static class HyperlinkLayer extends DrawLayer.AbstractLayer {
        
        private Position hyperlinkStart = null;
        private Position hyperlinkEnd = null;
        
        public static final String NAME = "hyperlink-layer"; // NOI18N
        
        public static final int VISIBILITY = 1050;
        
        private boolean initialized = false;
        
        public HyperlinkLayer() {
            super(NAME);
            
            this.initialized = false;
        }
        
        private void checkDocument(Document doc) {
        }
        
        public boolean extendsEOL() {
            return true;
        }
        
        public synchronized void init(final DrawContext ctx) {
            if (!initialized) {
                Document doc = ctx.getEditorUI().getDocument();
                
                initialized = true;
                checkDocument(doc);
            }
            
            if (isActive())
                setNextActivityChangeOffset(hyperlinkStart.getOffset());
        }
        
        private boolean isActive() {
            return hyperlinkStart != null && hyperlinkEnd != null;
        }
        
        public boolean isActive(DrawContext ctx, MarkFactory.DrawMark mark) {
            return isActive();
        }
        
        private boolean isIn(int offset) {
            return offset >= hyperlinkStart.getOffset() && offset < hyperlinkEnd.getOffset();
        }
        
        private static Coloring hoverColoring = new Coloring(null, 0, Color.BLUE, null, Color.BLUE, null);
        
        public void updateContext(DrawContext ctx) {
            if (!isActive())
                return ;
            
            int currentOffset = ctx.getFragmentOffset();
            
            if (isIn(currentOffset)) {
                hoverColoring.apply(ctx);
                
                if (isIn(currentOffset + ctx.getFragmentLength()))
                    setNextActivityChangeOffset(currentOffset + ctx.getFragmentLength());
            }
        }
        
    }
        
}
