/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.editor.ext;

import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.io.IOException;
import java.util.List;
import java.util.Iterator;
import javax.swing.Action;
import javax.swing.JPopupMenu;
import javax.swing.JMenuItem;
import javax.swing.KeyStroke;
import javax.swing.text.Caret;
import javax.swing.text.Keymap;
import javax.swing.text.JTextComponent;
import javax.swing.text.TextAction;
import javax.swing.text.BadLocationException;
import org.netbeans.editor.ActionFactory;
import org.netbeans.editor.BaseKit;
import org.netbeans.editor.EditorUI;
import org.netbeans.editor.BaseAction;
import org.netbeans.editor.Utilities;
import org.netbeans.editor.Settings;
import org.netbeans.editor.SettingsUtil;
import org.netbeans.editor.BaseDocument;
import org.netbeans.editor.SyntaxSupport;
import org.netbeans.editor.SettingsChangeEvent;
import org.netbeans.editor.Formatter;
import org.netbeans.editor.ext.JDCPopupPanel;
import javax.swing.SwingUtilities;
import org.openide.util.NbBundle;


/**
* Extended kit offering advanced functionality
*
* @author Miloslav Metelka
* @version 1.00
*/
public class ExtKit extends BaseKit {

    /** This action is searched and executed when the popup menu should
    * be displayed to build the popup menu.
    */
    public static final String buildPopupMenuAction = "build-popup-menu"; // NOI18N

    /** Show popup menu.
    */
    public static final String showPopupMenuAction = "show-popup-menu"; // NOI18N

    /** This action is searched and executed when the tool-tip should
    * be displayed by tool-tip support to build the tool-tip.
    */
    public static final String buildToolTipAction = "build-tool-tip"; // NOI18N

    /** Open find dialog action - this action is defined in view package, but
    * its name is defined here for clarity
    */
    public static final String findAction = "find"; // NOI18N

    /** Open replace dialog action - this action is defined in view package, but
    * its name is defined here for clarity
    */
    public static final String replaceAction = "replace"; // NOI18N

    /** Open goto dialog action - this action is defined in view package, but
    * its name is defined here for clarity
    */
    public static final String gotoAction = "goto"; // NOI18N

    /** Goto declaration depending on the context under the caret */
    public static final String gotoDeclarationAction = "goto-declaration"; // NOI18N

    /** Goto source depending on the context under the caret */
    public static final String gotoSourceAction = "goto-source"; // NOI18N

    public static final String gotoSuperImplementationAction = "goto-super-implementation"; // NOI18N

    /** Goto help depending on the context under the caret */
    public static final String gotoHelpAction = "goto-help"; // NOI18N

    /** Match brace */
    public static final String matchBraceAction = "match-brace"; // NOI18N

    /** Select the text to the matching bracket */
    public static final String selectionMatchBraceAction = "selection-match-brace"; // NOI18N

    /** Toggle the case for the first character of the word under caret */
    public static final String toggleCaseIdentifierBeginAction = "toggle-case-identifier-begin"; // NOI18N

    /** Advanced code selection technique
     * @deprecated this action name is not actively used by ExtKit and will be removed in future releases.
     */
    public static final String codeSelectAction = "code-select"; // NOI18N

    /** Action used when escape is pressed. By default it hides popup-menu
     * @deprecated this action name is not actively used by ExtKit and will be removed in future releases.
     */
    public static final String escapeAction = "escape"; // NOI18N

    /** Find the completion help and show it in the completion pane. */
    public static final String completionShowAction = "completion-show"; // NOI18N

    /** Show documentation popup panel */
    public static final String documentationShowAction = "documentation-show"; // NOI18N
    
    /** Show completion tooltip */
    public static final String completionTooltipShowAction = "tooltip-show"; // NOI18N

    /** Comment out the selected block */
    public static final String commentAction = "comment"; // NOI18N

    /** Uncomment the selected block */
    public static final String uncommentAction = "uncomment"; // NOI18N
    
    /** Toggle the toolbar */
    public static final String toggleToolbarAction = "toggle-toolbar"; // NOI18N
   
    /** Trimmed text for go to submenu*/
    public static final String TRIMMED_TEXT = "trimmed-text";    //NOI18N
    
    /** Shared suport for find and replace dialogs */
    private static FindDialogSupport findDialogSupport;
    
    private static FindAction findActionDef = new FindAction();
    private static ReplaceAction replaceActionDef = new ReplaceAction();
    private static GotoAction gotoActionDef = new GotoAction();


    /** Whether editor popup menu creation should be dumped to console */
    private static final boolean debugPopupMenu
            = Boolean.getBoolean("netbeans.debug.editor.popup.menu"); // NOI18N
    
    public ExtKit() {
    }

    /** Create caret to navigate through document */
    public Caret createCaret() {
        return new ExtCaret();
    }

    public SyntaxSupport createSyntaxSupport(BaseDocument doc) {
        return new ExtSyntaxSupport(doc);
    }

    public Completion createCompletion(ExtEditorUI extEditorUI) {
        return null;
    }
    
    public CompletionJavaDoc createCompletionJavaDoc(ExtEditorUI extEditorUI) {
        return null;
    }
    
    protected EditorUI createEditorUI() {
        return new ExtEditorUI();
    }

    protected Action[] createActions() {
        Action[] extActions = new Action[] {
            new BuildPopupMenuAction(),
            new ShowPopupMenuAction(),
            new BuildToolTipAction(),
            findActionDef,
            replaceActionDef,
            gotoActionDef,
            //new GotoDeclarationAction(), the instance will be created in extending editor... i.e. JavaKit
            new ToggleCaseIdentifierBeginAction(),
            new MatchBraceAction(matchBraceAction, false),
            new MatchBraceAction(selectionMatchBraceAction, true),
//            new CodeSelectAction(),   // the actionPerformed is empty and so I'm removing the action from the list
//            new EscapeAction(),
            new ExtDefaultKeyTypedAction(),
            new CompletionShowAction(),
            new DocumentationShowAction(),
            new CompletionTooltipShowAction()
        };
        return TextAction.augmentList(super.createActions(), extActions);
    }
    
    /**
     * Action that is localized in org.netbeans.editor package.
     * <br/>
     * <code>BaseKit.class</code> is used as a bundle class.
     */
    private abstract static class BaseKitLocalizedAction extends BaseAction {
        
        public BaseKitLocalizedAction(String name) {
            super(name);
        }
        
        public BaseKitLocalizedAction(String name, int updateMask) {
            super(name, updateMask);
        }
        
        protected Class getShortDescriptionBundleClass() {
            return BaseKit.class;
        }
        
    }

    /** Called before the popup menu is shown to possibly rebuild
    * the popup menu.
    */
    public static class BuildPopupMenuAction extends BaseKitLocalizedAction {

        static final long serialVersionUID =4257043398248915291L;

        public BuildPopupMenuAction() {
            super(buildPopupMenuAction, NO_RECORDING);
            putValue(BaseAction.NO_KEYBINDING, Boolean.TRUE);
        }

        public void actionPerformed(ActionEvent evt, JTextComponent target) {
            if (target != null) {
                if (debugPopupMenu) {
                    /*DEBUG*/System.err.println("POPUP CREATION START <<<<<"); // NOI18N
                }
                JPopupMenu pm = buildPopupMenu(target);
                if (debugPopupMenu) {
                    /*DEBUG*/System.err.println("POPUP CREATION END >>>>>"); // NOI18N
                }
                ExtUtilities.getExtEditorUI(target).setPopupMenu(pm);
            }
        }
        
        protected JPopupMenu createPopupMenu(JTextComponent target) {
            return new JPopupMenu();
        }

        protected JPopupMenu buildPopupMenu(JTextComponent target) {
            JPopupMenu pm = createPopupMenu(target);

            EditorUI ui = Utilities.getEditorUI(target);
            List l = (List)Settings.getValue(Utilities.getKitClass(target),
                (ui == null || ui.hasExtComponent())
                    ? ExtSettingsNames.POPUP_MENU_ACTION_NAME_LIST
                    : ExtSettingsNames.DIALOG_POPUP_MENU_ACTION_NAME_LIST
            );

            if (l != null) {
                Iterator i = l.iterator();
                while (i.hasNext()) {
                    String an = (String)i.next();
                    addAction(target, pm, an);
                }
            }
            return pm;
        }

        /** Add the action to the popup menu. This method is called
         * for each action-name found in the action-name-list. It should
         * add the appopriate menu item to the popup menu.
         * @param target target component for which the menu is being
         *  constructed.
         * @param popupMenu popup menu to which this method should add
         *  the item corresponding to the action-name.
         * @param actionName name of the action to add. The real action
         *  can be retrieved from the kit by calling <tt>getActionByName()</tt>.
         */
        protected void addAction(JTextComponent target, JPopupMenu popupMenu,
        String actionName) {
            Action a = Utilities.getKit(target).getActionByName(actionName);
            if (a != null) {
                JMenuItem item = null;
                if (a instanceof BaseAction) {
                    item = ((BaseAction)a).getPopupMenuItem(target);
                }
                if (item == null) {
                    String itemText = getItemText(target, actionName, a);
                    if (itemText != null) {
                        item = new JMenuItem(itemText);
                        item.addActionListener(a);
                        // Try to get the accelerator
                        Keymap km = target.getKeymap();
                        if (km != null) {
                            KeyStroke[] keys = km.getKeyStrokesForAction(a);
                            if (keys != null && keys.length > 0) {
                                item.setAccelerator(keys[0]);
                            }
                        }
                        item.setEnabled(a.isEnabled());
                        Object helpID = a.getValue ("helpID"); // NOI18N
                        if (helpID != null && (helpID instanceof String))
                            item.putClientProperty ("HelpID", helpID); // NOI18N
                    }
                }

                if (item != null) {
                    debugPopupMenuItem(item, a);
                    popupMenu.add(item);
                }

            } else if (actionName == null){ // action-name is null, add the separator
                if (popupMenu.getComponentCount()>0){
                    debugPopupMenuItem(null, null);
                    popupMenu.addSeparator();
                }
            }
        }
        
        protected final void debugPopupMenuItem(JMenuItem item, Action action) {
            if (debugPopupMenu) {
                StringBuffer sb = new StringBuffer("POPUP: "); // NOI18N
                if (item != null) {
                    sb.append('"');
                    sb.append(item.getText());
                    sb.append('"');;
                    if (!item.isVisible()) {
                        sb.append(", INVISIBLE"); // NOI18N
                    }
                    if (action != null) {
                        sb.append(", action="); // NOI18N
                        sb.append(action.getClass().getName());
                    }
                    
                } else { // null item means separator
                    sb.append("--Separator--"); // NOI18N
                }

                /*DEBUG*/System.err.println(sb.toString());
            }

        }

        protected String getItemText(JTextComponent target, String actionName, Action a) {
            String itemText;
            if (a instanceof BaseAction) {
                itemText = ((BaseAction)a).getPopupMenuText(target);
            } else {
                itemText = actionName;
            }
            return itemText;
        }

    }

    /** Show the popup menu.
    */
    public static class ShowPopupMenuAction extends BaseKitLocalizedAction {

        static final long serialVersionUID =4257043398248915291L;

        public ShowPopupMenuAction() {
            super(showPopupMenuAction, NO_RECORDING);
        }

        public void actionPerformed(ActionEvent evt, JTextComponent target) {
            if (target != null) {
                try {
                    int dotPos = target.getCaret().getDot();
                    Rectangle r = target.getUI().modelToView(target, dotPos);
                    if (r != null) {
                        ExtEditorUI eui = ExtUtilities.getExtEditorUI(target);
                        if (eui != null){
                            eui.showPopupMenu(r.x, r.y + r.height);
                        }
                    }
                } catch (BadLocationException e) {
                    target.getToolkit().beep();
                }
            }
        }

    }

    public static class BuildToolTipAction extends BaseAction {

        static final long serialVersionUID =-2701131863705941250L;

        public BuildToolTipAction() {
            super(buildToolTipAction, NO_RECORDING);
            putValue(BaseAction.NO_KEYBINDING, Boolean.TRUE);
            putValue(Action.SHORT_DESCRIPTION, ""); // No explicit description NOI18N
        }

        protected String buildText(JTextComponent target) {
            ToolTipSupport tts = ExtUtilities.getExtEditorUI(target).getToolTipSupport();
            return  (tts != null)
                ? target.getToolTipText(tts.getLastMouseEvent())
                : target.getToolTipText();
        }

        public void actionPerformed(ActionEvent evt, JTextComponent target) {
            if (target != null) {
                ToolTipSupport tts = ExtUtilities.getExtEditorUI(target).getToolTipSupport();
                if (tts != null) {
                    tts.setToolTipText(buildText(target));
                }
            }
        }

    }

    public static class FindAction extends BaseKitLocalizedAction {

        static final long serialVersionUID =719554648887497427L;

        public FindAction() {
            super(findAction, ABBREV_RESET
                  | MAGIC_POSITION_RESET | UNDO_MERGE_RESET | NO_RECORDING);
            putValue(BaseAction.ICON_RESOURCE_PROPERTY,
                    "org/netbeans/modules/editor/resources/find"); //NOI18N
        }

        public FindDialogSupport getSupport() {
            return FindDialogSupport.getFindDialogSupport();
        }

        public void actionPerformed(ActionEvent evt, JTextComponent target) {
            if (target != null) {
                getSupport().showFindDialog(new KeyEventBlocker(target, false));
            }
        }

    }

    public static class ReplaceAction extends BaseKitLocalizedAction {

        static final long serialVersionUID =1828017436079834384L;

        public ReplaceAction() {
            super(replaceAction, ABBREV_RESET
                  | MAGIC_POSITION_RESET | UNDO_MERGE_RESET | NO_RECORDING);
        }

        public FindDialogSupport getSupport() {
            return FindDialogSupport.getFindDialogSupport();
        }

        public void actionPerformed(ActionEvent evt, JTextComponent target) {
            if (target != null) {
                // make KeyEventBlocker to discard the first key typed event (Ctrl-H)
                // because it is mapped to backspace in the replace dialog
                getSupport().showReplaceDialog(new KeyEventBlocker(target, true));
            }
        }

    }

    public static class GotoAction extends BaseKitLocalizedAction {

        static final long serialVersionUID =8425585413146373256L;

        public GotoAction() {
            super(gotoAction, ABBREV_RESET
                  | MAGIC_POSITION_RESET | UNDO_MERGE_RESET);
            String name = NbBundle.getBundle(BaseKit.class).getString("goto_trimmed");
            putValue(TRIMMED_TEXT, name);
            putValue(POPUP_MENU_TEXT, name);
        }


        /** This method is called by the dialog support
        * to translate the line offset to the document position. This
        * can be changed for example for the diff operations.
        * @param doc document to operate over
        * @param lineOffset the line offset to convert to position
        * @return document offset that corresponds to the row-start
        *  of the line with the line-number equal to (lineOffset + 1).
        */
        protected int getOffsetFromLine(BaseDocument doc, int lineOffset) {
            return Utilities.getRowStartFromLineOffset(doc, lineOffset);
        }

        public void actionPerformed(ActionEvent evt, JTextComponent target) {
            if (target != null) {
                new GotoDialogSupport().showGotoDialog(new KeyEventBlocker(target, false));
            }
        }

    }

    /** Action to go to the declaration of the variable under the caret.
    */
    public static class GotoDeclarationAction extends BaseKitLocalizedAction {

        static final long serialVersionUID =-6440495023918097760L;

        public GotoDeclarationAction() {
            super(gotoDeclarationAction,
                  ABBREV_RESET | MAGIC_POSITION_RESET | UNDO_MERGE_RESET
                  | SAVE_POSITION
                 );
            String name = NbBundle.getBundle(BaseKit.class).getString("goto-declaration-trimmed");
            putValue(TRIMMED_TEXT, name);  //NOI18N            
            putValue(POPUP_MENU_TEXT, name);  //NOI18N            
        }

        public boolean gotoDeclaration(JTextComponent target) {
            try {
                Caret caret = target.getCaret();
                int dotPos = caret.getDot();
                BaseDocument doc = (BaseDocument)target.getDocument();
                int[] idBlk = Utilities.getIdentifierBlock(doc, dotPos);
                ExtSyntaxSupport extSup = (ExtSyntaxSupport)doc.getSyntaxSupport();
                if (idBlk != null) {
                    int decPos = extSup.findDeclarationPosition(doc.getText(idBlk), idBlk[1]);
                    if (decPos >= 0) {
                        caret.setDot(decPos);
                        return true;
                    }
                }
            } catch (BadLocationException e) {
            }
            return false;
        }

        public void actionPerformed(ActionEvent evt, JTextComponent target) {
            if (target != null) {
                gotoDeclaration(target); // try to go to the declaration position
            }
        }
    }

    public static class ToggleCaseIdentifierBeginAction extends BaseKitLocalizedAction {

        static final long serialVersionUID =584392193824931979L;

        ToggleCaseIdentifierBeginAction() {
            super(toggleCaseIdentifierBeginAction, ABBREV_RESET
                  | MAGIC_POSITION_RESET | UNDO_MERGE_RESET | WORD_MATCH_RESET);
        }

        public void actionPerformed(ActionEvent evt, JTextComponent target) {
            if (target != null) {
                if (!target.isEditable() || !target.isEnabled()) {
                    target.getToolkit().beep();
                    return;
                }

                try {
                    Caret caret = target.getCaret();
                    BaseDocument doc = (BaseDocument)target.getDocument();
                    int[] idBlk = Utilities.getIdentifierBlock(doc, caret.getDot());
                    if (idBlk != null) {
                        Utilities.changeCase(doc, idBlk[0], 1, Utilities.CASE_SWITCH);
                    }
                } catch (BadLocationException e) {
                    target.getToolkit().beep();
                }
            }
        }
    }

    public static class MatchBraceAction extends BaseKitLocalizedAction {

        boolean select;

        static final long serialVersionUID =-184887499045886231L;

        public MatchBraceAction(String name, boolean select) {
            super(name, ABBREV_RESET
                  | MAGIC_POSITION_RESET | UNDO_MERGE_RESET);
            this.select = select;
        }

        public void actionPerformed(ActionEvent evt, JTextComponent target) {
            if (target != null) {
                try {
                    Caret caret = target.getCaret();
                    BaseDocument doc = Utilities.getDocument(target);
                    int dotPos = caret.getDot();
                    ExtSyntaxSupport sup = (ExtSyntaxSupport)doc.getSyntaxSupport();
                    if (dotPos > 0) {
                        int[] matchBlk = sup.findMatchingBlock(dotPos - 1, false);
                        if (matchBlk != null) {
                            if (select) {
                                caret.moveDot(matchBlk[1]);
                            } else {
                                caret.setDot(matchBlk[1]);
                            }
                        }
                    }
                } catch (BadLocationException e) {
                    target.getToolkit().beep();
                }
            }
        }
    }

    /**
     * @deprecated this action is deprecated and will be removed in future releases.
     */
    public static class CodeSelectAction extends BaseKitLocalizedAction {

        static final long serialVersionUID =4033474080778585860L;

        public CodeSelectAction() {
            super(codeSelectAction);
        }

        public void actionPerformed(ActionEvent evt, JTextComponent target) {
/*            if (target != null) {
                BaseDocument doc = (BaseDocument)target.getDocument();
                SyntaxSupport sup = doc.getSyntaxSupport();
                Caret caret = target.getCaret();
                try {
                    int bracketPos = sup.findUnmatchedBracket(caret.getDot(), sup.getRightBrackets());
                    if (bracketPos >= 0) {
                        caret.setDot(bracketPos);
                        while (true) {
                          int bolPos = Utilities.getRowStart(doc, bracketPos);
                          boolean isWSC = sup.isCommentOrWhitespace(bolPos, bracketPos);
                          if (isWSC) { // get previous line end
                            
                          }
                        }
                    }
                } catch (BadLocationException e) {
                    target.getToolkit().beep();
                }
            }
*/
        }
    }

    /** Prefix maker adds the prefix before the identifier under cursor.
    * The prefix is not added if it's already present. The prefix to be
    * added is specified in the constructor of the action together
    * with the prefix group. If there's already any prefix from the prefix
    * group at the begining of the identifier, that prefix is replaced
    * by the actual prefix.
    */
    public static class PrefixMakerAction extends BaseKitLocalizedAction {

        static final long serialVersionUID =-2305157963664484920L;

        private String prefix;

        private String[] prefixGroup;

        public PrefixMakerAction(String name, String prefix, String[] prefixGroup) {
            super(name);
            this.prefix = prefix;
            this.prefixGroup = prefixGroup;
            
            // [PENDING] This should be done in a better way
            String iconRes = null;
            if ("get".equals(prefix)) { // NOI18N
                iconRes = "org/netbeans/modules/editor/resources/var_get.gif"; // NOI18N
            } else if ("set".equals(prefix)) { // NOI18N
                iconRes = "org/netbeans/modules/editor/resources/var_set.gif"; // NOI18N
            } else if ("is".equals(prefix)) { // NOI18N
                iconRes = "org/netbeans/modules/editor/resources/var_is.gif"; // NOI18N
            }
            if (iconRes != null) {
                putValue(BaseAction.ICON_RESOURCE_PROPERTY, iconRes);
            }
        }

        public void actionPerformed(ActionEvent evt, JTextComponent target) {
            if (target != null) {
                if (!target.isEditable() || !target.isEnabled()) {
                    target.getToolkit().beep();
                    return;
                }

                BaseDocument doc = (BaseDocument)target.getDocument();
                int dotPos = target.getCaret().getDot();
                try {
                    // look for identifier around caret
                    int[] block = org.netbeans.editor.Utilities.getIdentifierBlock(doc, dotPos);

                    // If there is no identifier around, warn user
                    if (block == null) {
                        target.getToolkit().beep();
                        return;
                    }

                    // Get the identifier to operate on
                    String identifier = doc.getText(block[0], block[1] - block[0]);

                    // Handle the case we already have the work done - e.g. if we got called over 'getValue'
                    if (identifier.startsWith(prefix) && Character.isUpperCase(identifier.charAt(prefix.length()))) return;

                    // Handle the case we have other type of known xEr: eg isRunning -> getRunning
                    for (int i=0; i<prefixGroup.length; i++) {
                        String actPref = prefixGroup[i];
                        if (identifier.startsWith(actPref)
                                && identifier.length() > actPref.length()
                                && Character.isUpperCase(identifier.charAt(actPref.length()))
                           ) {
                            doc.remove(block[0], actPref.length());
                            doc.insertString(block[0], prefix, null);
                            return;
                        }
                    }

                    // Upcase the first letter
                    Utilities.changeCase(doc, block[0], 1, Utilities.CASE_UPPER);
                    // Prepend the prefix before it
                    doc.insertString(block[0], prefix, null);
                } catch (BadLocationException e) {
                    target.getToolkit().beep();
                }
            }
        }
    }

    public static class CommentAction extends BaseKitLocalizedAction {

        static final long serialVersionUID =-1422954906554289179L;

        private String lineCommentString;

        public CommentAction(String lineCommentString) {
            super(commentAction);
            this.lineCommentString = lineCommentString;
            putValue(BaseAction.ICON_RESOURCE_PROPERTY,
                "org/netbeans/modules/editor/resources/comment.png"); // NOI18N
        }

        public void actionPerformed(ActionEvent evt, JTextComponent target) {
            if (target != null) {
                if (!target.isEditable() || !target.isEnabled()) {
                    target.getToolkit().beep();
                    return;
                }
                Caret caret = target.getCaret();
                BaseDocument doc = (BaseDocument)target.getDocument();
                try {
                    if (caret.isSelectionVisible()) {
                        int startPos = Utilities.getRowStart(doc, target.getSelectionStart());
                        int endPos = target.getSelectionEnd();
                        doc.atomicLock();
                        try {

                            if (endPos > 0 && Utilities.getRowStart(doc, endPos) == endPos) {
                                endPos--;
                            }

                            int pos = startPos;
                            for (int lineCnt = Utilities.getRowCount(doc, startPos, endPos);
                                    lineCnt > 0; lineCnt--
                                ) {
                                doc.insertString(pos, lineCommentString, null); // NOI18N
                                pos = Utilities.getRowStart(doc, pos, +1);
                            }

                        } finally {
                            doc.atomicUnlock();
                        }
                    } else { // selection not visible
                        doc.insertString(Utilities.getRowStart(doc, target.getSelectionStart()),
                                         lineCommentString, null); // NOI18N
                    }
                } catch (BadLocationException e) {
                    target.getToolkit().beep();
                }
            }
        }

    }

    public static class UncommentAction extends BaseKitLocalizedAction {

        static final long serialVersionUID =-7005758666529862034L;

        private String lineCommentString;

        public UncommentAction(String lineCommentString) {
            super(uncommentAction);
            this.lineCommentString = lineCommentString;
            putValue(BaseAction.ICON_RESOURCE_PROPERTY,
                "org/netbeans/modules/editor/resources/uncomment.png"); // NOI18N
        }

        public void actionPerformed(ActionEvent evt, JTextComponent target) {
            if (target != null) {
                if (!target.isEditable() || !target.isEnabled()) {
                    target.getToolkit().beep();
                    return;
                }
                Caret caret = target.getCaret();
                BaseDocument doc = (BaseDocument)target.getDocument();
                int commentLen = lineCommentString.length();
                try {
                    if (caret.isSelectionVisible()) {
                        int startPos = Utilities.getRowFirstNonWhite(doc, target.getSelectionStart());
                        int endPos = target.getSelectionEnd();
                        doc.atomicLock();
                        try {

                            if (endPos > 0 && Utilities.getRowStart(doc, endPos) == endPos) {
                                endPos--;
                            }

                            int pos = startPos;
                            for (int lineCnt = Utilities.getRowCount(doc, startPos, endPos);
                                    lineCnt > 0; lineCnt--
                                ) {
                                if (Utilities.getRowEnd(doc, pos) - pos >= commentLen
                                        && doc.getText(pos, commentLen).equals(lineCommentString)
                                   ) {
                                    doc.remove(pos, commentLen);
                                }
                                pos = Utilities.getRowFirstNonWhite(doc, Utilities.getRowStart(doc, pos, +1));
                            }

                        } finally {
                            doc.atomicUnlock();
                        }
                    } else { // selection not visible
                        int pos = Utilities.getRowFirstNonWhite(doc, caret.getDot());
                        if (Utilities.getRowEnd(doc, pos) - pos >= commentLen
                                && doc.getText(pos, commentLen).equals(lineCommentString) // NOI18N
                           ) {
                            doc.remove(pos, commentLen);
                        }
                    }
                } catch (BadLocationException e) {
                    target.getToolkit().beep();
                }
            }
        }

    }



    /** Executed when the Escape key is pressed. By default it hides
    * the popup menu if visible.
    */
    public static class EscapeAction extends BaseKitLocalizedAction {

        public EscapeAction() {
            super(escapeAction);
        }

        public void actionPerformed(ActionEvent evt, JTextComponent target) {
            if (target != null) {
                ExtUtilities.getExtEditorUI(target).hidePopupMenu();
            }
        }
    }


    // Completion customized actions
    public static class ExtDefaultKeyTypedAction extends DefaultKeyTypedAction {

        static final long serialVersionUID =5273032708909044812L;

        public void actionPerformed(ActionEvent evt, JTextComponent target) {
            String cmd = evt.getActionCommand();
            int mod = evt.getModifiers();

            // Dirty fix for Completion shortcut on Unix !!!
            if (cmd != null && cmd.equals(" ") && (mod == ActionEvent.CTRL_MASK)) { // NOI18N
                // Ctrl + SPACE
            } else {
                Caret caret = target.getCaret();
                if (caret instanceof ExtCaret) {
                    ((ExtCaret)caret).requestMatchBraceUpdateSync(); // synced bracket update
                }
                super.actionPerformed(evt, target);
            }

            if ((target != null) && (evt != null)) {
                if ((cmd != null) && (cmd.length() == 1) &&
                        ((mod & ActionEvent.ALT_MASK) == 0
                         && (mod & ActionEvent.CTRL_MASK) == 0)
                   ) {
                    // Check whether char that should reindent the line was inserted
                    checkIndentHotChars(target, cmd);

                    // Check the completion
                    checkCompletion(target, cmd);
                }
            }
        }

        /** Check the characters that should cause reindenting the line. */
        protected void checkIndentHotChars(JTextComponent target, String typedText) {
            BaseDocument doc = Utilities.getDocument(target);
            if (doc != null) {
                Caret caret = target.getCaret();
                Formatter f = doc.getFormatter();
                if (f instanceof ExtFormatter) {
                    ExtFormatter ef = (ExtFormatter)f;
                    int[] fmtBlk = ef.getReformatBlock(target, typedText);

                    if (fmtBlk != null) {
                        try {
                            fmtBlk[0] = Utilities.getRowStart(doc, fmtBlk[0]);
                            fmtBlk[1] = Utilities.getRowEnd(doc, fmtBlk[1]);

                            //this was the of #18922, that causes the bug #20198
                            //ef.reformat(doc, fmtBlk[0], fmtBlk[1]);

                            //bugfix of the bug #20198. Bug #18922 is fixed too as well as #6968
                            ef.reformat(doc, fmtBlk[0], fmtBlk[1], true);
                            
                        } catch (BadLocationException e) {
                        } catch (IOException e) {
                        }
                    }
                }
            }
        }


        /** Check and possibly popup, hide or refresh the completion */
        protected void checkCompletion(JTextComponent target, String typedText) {
            Completion completion = ExtUtilities.getCompletion(target);

            BaseDocument doc = (BaseDocument)target.getDocument();
            ExtSyntaxSupport extSup = (ExtSyntaxSupport)doc.getSyntaxSupport();
            
            if (completion != null && typedText.length() > 0) {
                if( !completion.isPaneVisible() ) {
                    if (completion.isAutoPopupEnabled()) {
                        int result = extSup.checkCompletion( target, typedText, false );
                        if ( result == ExtSyntaxSupport.COMPLETION_POPUP ) {
                            completion.popup(true);
                        } else if ( result == ExtSyntaxSupport.COMPLETION_CANCEL ) {
                            completion.cancelRequest();
                        }
                    }
                } else {
                    int result = extSup.checkCompletion( target, typedText, true );
                    switch( result ) {
                        case ExtSyntaxSupport.COMPLETION_HIDE:
                            completion.setPaneVisible(false);
                            break;
                        case ExtSyntaxSupport.COMPLETION_REFRESH:
                            completion.refresh(false);
                            break;
                        case ExtSyntaxSupport.COMPLETION_POST_REFRESH:
                            completion.refresh(true);
                            break;
                    }
                }
            }
        }
    }

    public static class CompletionShowAction extends BaseKitLocalizedAction {

        static final long serialVersionUID =1050644925893851146L;

        public CompletionShowAction() {
            super(completionShowAction);
        }

        public void actionPerformed(ActionEvent evt, JTextComponent target) {
        }

    }

    public static class DocumentationShowAction extends BaseKitLocalizedAction {

        public DocumentationShowAction() {
            super(documentationShowAction);
        }

        public void actionPerformed(ActionEvent evt, JTextComponent target) {
        }

    }

    public static class CompletionTooltipShowAction extends BaseKitLocalizedAction {

        public CompletionTooltipShowAction() {
            super(completionTooltipShowAction);
        }

        public void actionPerformed(ActionEvent evt, JTextComponent target) {
        }

    }

  public static class ExtDeleteCharAction extends DeleteCharAction {

    public ExtDeleteCharAction(String nm, boolean nextChar) {
      super(nm, nextChar);
    }
    
  }


}
