/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.editor;

import java.awt.Frame;
import java.awt.Insets;
import java.awt.Rectangle;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeEvent;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.swing.text.JTextComponent;
import javax.swing.text.BadLocationException;
import javax.swing.text.Caret;
import javax.swing.text.Document;
import javax.swing.text.Position;
import org.netbeans.editor.DocumentFinder.FindReplaceResult;
import org.openide.util.NbBundle;

/**
* Find management
*
* @author Miloslav Metelka
* @version 1.00
*/

public class FindSupport {

    private static final String FOUND_LOCALE = "find-found"; // NOI18N
    private static final String NOT_FOUND_LOCALE = "find-not-found"; // NOI18N
    private static final String WRAP_START_LOCALE = "find-wrap-start"; // NOI18N
    private static final String WRAP_END_LOCALE = "find-wrap-end"; // NOI18N
    private static final String WRAP_BLOCK_START_LOCALE = "find-block-wrap-start"; // NOI18N
    private static final String WRAP_BLOCK_END_LOCALE = "find-block-wrap-end"; // NOI18N
    private static final String ITEMS_REPLACED_LOCALE = "find-items-replaced"; // NOI18N
    public static final String REVERT_MAP = "revert-map"; // NOI18N

    private static final String SEARCH_BLOCK_START="search-block-start"; //NOI18N
    private static final String SEARCH_BLOCK_END="search-block-end"; //NOI18N
    
    public static final String FIND_HISTORY_PROP = "find-history-prop"; //NOI18N
    public static final String FIND_HISTORY_CHANGED_PROP = "find-history-changed-prop"; //NOI18N
    
    /** Shared instance of FindSupport class */
    static FindSupport findSupport;

    /** Find properties */
    private Map findProps;

    /** Support for firing change events */
    WeakPropertyChangeSupport changeSupport
    = new WeakPropertyChangeSupport();
    
    SearchPatternWrapper lastSelected;
    List historyList;/*<SearchPatternWrapper>*/

    private FindSupport() {
        // prevent instance creation
    }

    /** Get shared instance of find support */
    public static FindSupport getFindSupport() {
        if (findSupport == null) {
            findSupport = new FindSupport();
        }
        return findSupport;
    }

    public Map getDefaultFindProperties() {
        HashMap props = new HashMap();
        Class kitClass = BaseKit.class;
        props.put(SettingsNames.FIND_WHAT, Settings.getValue(
                      kitClass, SettingsNames.FIND_WHAT));
        props.put(SettingsNames.FIND_REPLACE_WITH, Settings.getValue(
                      kitClass, SettingsNames.FIND_REPLACE_WITH));
        props.put(SettingsNames.FIND_HIGHLIGHT_SEARCH, Settings.getValue(
                      kitClass, SettingsNames.FIND_HIGHLIGHT_SEARCH));
        props.put(SettingsNames.FIND_INC_SEARCH, Settings.getValue(
                      kitClass, SettingsNames.FIND_INC_SEARCH));
        props.put(SettingsNames.FIND_BACKWARD_SEARCH, Settings.getValue(
                      kitClass, SettingsNames.FIND_BACKWARD_SEARCH));
        props.put(SettingsNames.FIND_WRAP_SEARCH, Settings.getValue(
                      kitClass, SettingsNames.FIND_WRAP_SEARCH));
        props.put(SettingsNames.FIND_MATCH_CASE, Settings.getValue(
                      kitClass, SettingsNames.FIND_MATCH_CASE));
        props.put(SettingsNames.FIND_SMART_CASE, Settings.getValue(
                      kitClass, SettingsNames.FIND_SMART_CASE));
        props.put(SettingsNames.FIND_WHOLE_WORDS, Settings.getValue(
                      kitClass, SettingsNames.FIND_WHOLE_WORDS));
        props.put(SettingsNames.FIND_REG_EXP, Settings.getValue(
                      kitClass, SettingsNames.FIND_REG_EXP));
        props.put(SettingsNames.FIND_HISTORY, Settings.getValue(
                      kitClass, SettingsNames.FIND_HISTORY));

        return props;
    }

    private int getBlockEndOffset(){
        Position pos = (Position) getFindProperties().get(SettingsNames.FIND_BLOCK_SEARCH_END);
        return (pos != null) ? pos.getOffset() : -1;
    }
    
    public Map getFindProperties() {
        if (findProps == null) {
            findProps = getDefaultFindProperties();
        }
        return findProps;
    }

    /** Get find property with specified name */
    public Object getFindProperty(String name) {
        return getFindProperties().get(name);
    }

    private Map getValidFindProperties(Map props) {
        return (props != null) ? props : getFindProperties();
    }

    int[] getBlocks(int[] blocks, BaseDocument doc,
                    int startPos, int endPos) throws BadLocationException {
        Map props = getValidFindProperties(null);
        
        Boolean b = (Boolean)props.get(SettingsNames.FIND_BLOCK_SEARCH);
        boolean blockSearch = (b != null && b.booleanValue());
        Integer i = (Integer) props.get(SettingsNames.FIND_BLOCK_SEARCH_START);
        int blockSearchStart = (i != null) ? i.intValue() : -1;
        int blockSearchEnd = getBlockEndOffset();

        if (blockSearch && blockSearchStart>-1 && blockSearchEnd >0){
            if (endPos>=blockSearchStart && startPos <=blockSearchEnd){
                startPos = Math.max(blockSearchStart, startPos);
                endPos = Math.min(blockSearchEnd, endPos);
            }else{
                return blocks;
            }
        }
        return DocumentFinder.findBlocks(doc, startPos, endPos, props, blocks);
    }

    /** Get find property without performing initialization
    * of find properties. This is useful for example for base document
    * when it wants to query whether it should do highlight search.
    */
    Object getPropertyNoInit(String name) {
        if (findProps == null) {
            return null;
        } else {
            return getFindProperty(name);
        }
    }

    /** Set find property with specified name and fire change.
    */
    public void putFindProperty(String name, Object newValue) {
        Object oldValue = getFindProperty(name);
        if ((oldValue == null && newValue == null)
                || (oldValue != null && oldValue.equals(newValue))
           ) {
            return;
        }
        if (newValue != null) {
            getFindProperties().put(name, newValue);
        } else {
            getFindProperties().remove(name);
        }
        firePropertyChange(name, oldValue, newValue);
    }

    /** Add/replace properties from some other map
    * to current find properties. If the added properties
    * are different than the original ones,
    * the property change is fired.
    */
    public void putFindProperties(Map propsToAdd) {
        if (!getFindProperties().equals(propsToAdd)) {
            getFindProperties().putAll(propsToAdd);
            firePropertyChange(null, null, null);
        }
    }
    
    public void setBlockSearchHighlight(int startSelection, int endSelection){
        JTextComponent c = Utilities.getLastActiveComponent();
        if (c==null) return;
        EditorUI editorUI = ((BaseTextUI)c.getUI()).getEditorUI();
        DrawLayerFactory.BlockSearchLayer blockLayer
        = (DrawLayerFactory.BlockSearchLayer)editorUI.findLayer(
              DrawLayerFactory.BLOCK_SEARCH_LAYER_NAME);
        Boolean b = (Boolean)getFindProperties().get(SettingsNames.FIND_BACKWARD_SEARCH);
        boolean back = (b != null && b.booleanValue());
        
        if (startSelection >= endSelection){
            if (blockLayer != null) {
                if (blockLayer.isEnabled()) {
                    blockLayer.setEnabled(false);
                    try {
                        editorUI.repaintBlock(blockLayer.getOffset(), blockLayer.getOffset()+blockLayer.getLength());
                    } catch (BadLocationException e) {
                        Utilities.annotateLoggable(e);
                    }
                }
            }
        }else{
            //init layer
            if (blockLayer == null) {
                blockLayer = new DrawLayerFactory.BlockSearchLayer();
                if (!editorUI.addLayer(blockLayer,
                    DrawLayerFactory.BLOCK_SEARCH_LAYER_VISIBILITY)
                ) {
                    return; // couldn't add layer
                }
            } else {
                if (blockLayer.isEnabled()) {
                    blockLayer.setEnabled(false);
                    try {
                        editorUI.repaintOffset(blockLayer.getOffset());
                    } catch (BadLocationException e) {
                        Utilities.annotateLoggable(e);
                    }
                }
            }
            
            blockLayer.setEnabled(true);
            blockLayer.setArea(startSelection, endSelection-startSelection);
            try {
                editorUI.repaintBlock(startSelection, endSelection);
            } catch (BadLocationException e) {
                Utilities.annotateLoggable(e);
                return;
            }
            c.getCaret().setDot(back ? endSelection : startSelection);
        }
    }
    
    public boolean incSearch(Map props, int caretPos) {
        props = getValidFindProperties(props);

        // if regexp terminate incSearch
        Boolean b = (Boolean)props.get(SettingsNames.FIND_REG_EXP);
        if (b !=null && b.booleanValue()){
            return false;
        }
        
        b = (Boolean)props.get(SettingsNames.FIND_INC_SEARCH);
        if (b != null && b.booleanValue()) { // inc search enabled
            JTextComponent c = Utilities.getLastActiveComponent();
            if (c != null && c.getDocument() instanceof BaseDocument) {
                BaseDocument doc = (BaseDocument)c.getDocument();
                b = (Boolean)props.get(SettingsNames.FIND_BACKWARD_SEARCH);
                boolean back = (b != null && b.booleanValue());
                b = (Boolean)props.get(SettingsNames.FIND_BLOCK_SEARCH);
                boolean blockSearch = (b != null && b.booleanValue());
                Integer i = (Integer) props.get(SettingsNames.FIND_BLOCK_SEARCH_START);
                int blockSearchStart = (i != null) ? i.intValue() : -1;
                
                Position endPos = (Position) props.get(SettingsNames.FIND_BLOCK_SEARCH_END);
                int blockSearchEnd = (endPos != null) ? endPos.getOffset() : -1;
                int endOffset = (back) ? 0 : -1;
                int pos;
                try {
                    int start = (blockSearch && blockSearchStart > -1) ? blockSearchStart : 0;
                    int end = (blockSearch && blockSearchEnd > 0) ? blockSearchEnd : -1;
                    if (start>0 && end == -1) return false;
                    int findRet[] = findInBlock(c, caretPos, 
                        start, 
                        end, 
                        props, false);
                            
                    if (findRet == null) {
                        incSearchReset();
                        return false;
                    }
                    pos = findRet[0];
                } catch (BadLocationException e) {
                    Utilities.annotateLoggable(e);
                    return false;
                }
                
                // possibly create incSearch layer
                BaseTextUI ui = (BaseTextUI)c.getUI();
                EditorUI editorUI = ui.getEditorUI();
                DrawLayerFactory.IncSearchLayer incLayer
                = (DrawLayerFactory.IncSearchLayer)editorUI.findLayer(
                      DrawLayerFactory.INC_SEARCH_LAYER_NAME);
                if (incLayer == null) {
                    incLayer = new DrawLayerFactory.IncSearchLayer();
                    if (!editorUI.addLayer(incLayer,
                        DrawLayerFactory.INC_SEARCH_LAYER_VISIBILITY)
                    ) {
                        return false; // couldn't add layer
                    }
                } else {
                    if (incLayer.isEnabled()) {
                        incLayer.setEnabled(false);
                        try {
                            editorUI.repaintOffset(incLayer.getOffset());
                        } catch (BadLocationException e) {
                            Utilities.annotateLoggable(e);
                        }
                    }
                }

                if (pos >= 0) {
                    String s = (String)props.get(SettingsNames.FIND_WHAT);
                    int len = (s != null) ? s.length() : 0;
                    if (len > 0) {
                        if (c.getSelectionEnd() > c.getSelectionStart()){
                            c.select(caretPos, caretPos);
                        }
                        incLayer.setInversion(!blockSearch);
                        incLayer.setEnabled(true);
                        incLayer.setArea(pos, len);
                        // reset higlighting
                        Map defaultProps = getValidFindProperties(null);
                        String findWhatDef = (String)defaultProps.get(SettingsNames.FIND_WHAT);
                        if (findWhatDef!=null && findWhatDef.length()>0){
                            defaultProps.put(SettingsNames.FIND_WHAT, ""); //NOI18N
                            editorUI.getComponent().repaint();
                        }
                        try {
                            editorUI.repaintOffset(pos);
                            ensureVisible(c, pos, pos);
                            
                        } catch (BadLocationException e) {
                            Utilities.annotateLoggable(e);
                        }
                        return true;
                    }
                } else { // string not found
                    // !!!          ((BaseCaret)c.getCaret()).dispatchUpdate();
                }
               
            }
        } else { // inc search not enabled
            incSearchReset();
        }
        return false;
    }

    public void incSearchReset() {
        JTextComponent c = Utilities.getLastActiveComponent();
        if (c==null) return; //  #19558 bugfix. Editor window has been closed => c==null
        EditorUI editorUI = ((BaseTextUI)c.getUI()).getEditorUI();
        DrawLayerFactory.IncSearchLayer incLayer
        = (DrawLayerFactory.IncSearchLayer)editorUI.findLayer(
              DrawLayerFactory.INC_SEARCH_LAYER_NAME);
        if (incLayer != null) {
            if (incLayer.isEnabled()) {
                incLayer.setEnabled(false);
                try {
                    editorUI.repaintOffset(incLayer.getOffset());
                } catch (BadLocationException e) {
                    Utilities.annotateLoggable(e);
                }
            }
        }
    }

    
    private boolean isBackSearch(Map props, boolean oppositeDir) {
        Boolean b = (Boolean)props.get(SettingsNames.FIND_BACKWARD_SEARCH);
        boolean back = (b != null && b.booleanValue());
        if (oppositeDir) {
            back = !back;
        }
        return back;
    }

    private void selectText(JTextComponent c, int start, int end, boolean back){
        Caret caret = c.getCaret();
        ensureVisible(c, start, end);
        if (back) {
            caret.setDot(end);
            caret.moveDot(start);
        } else { // forward direction
            caret.setDot(start);
            caret.moveDot(end);
        }
    }
    
    private void ensureVisible(JTextComponent c, int startOffset, int endOffset) {
        Class kitClass = Utilities.getKitClass(c);
        if (kitClass == null) {
            kitClass = BaseKit.class;
        }
        ensureVisible(c, startOffset, endOffset, 
                (Insets)Settings.getValue(kitClass, SettingsNames.SCROLL_FIND_INSETS));
    }
    /**
     * Ensure that the given region will be visible in the view
     * with the appropriate find insets.
     */
    private void ensureVisible(JTextComponent c, int startOffset, int endOffset, Insets extraInsets) {
        try {
            Rectangle startBounds = c.modelToView(startOffset);
            Rectangle endBounds = c.modelToView(endOffset);
            if (startBounds != null && endBounds != null) {
                startBounds.add(endBounds);
                if (extraInsets != null) {
                    Rectangle visibleBounds = c.getVisibleRect();
                    int extraTop = (extraInsets.top < 0)
                        ? -extraInsets.top * visibleBounds.height / 100 // percentage
                        : extraInsets.top * endBounds.height; // line count
                    startBounds.y -= extraTop;
                    startBounds.height += extraTop;
                    startBounds.height += (extraInsets.bottom < 0)
                        ? -extraInsets.bottom * visibleBounds.height / 100 // percentage
                        : extraInsets.bottom * endBounds.height; // line count
                    int extraLeft = (extraInsets.left < 0)
                        ? -extraInsets.left * visibleBounds.width / 100 // percentage
                        : extraInsets.left * endBounds.width; // char count
                    startBounds.x -= extraLeft;
                    startBounds.width += extraLeft;
                    startBounds.width += (extraInsets.right < 0)
                        ? -extraInsets.right * visibleBounds.width / 100 // percentage
                        : extraInsets.right * endBounds.width; // char count
                }
                c.scrollRectToVisible(startBounds);
            }
        } catch (BadLocationException e) {
            // do not scroll
        }
    }
    
    private FindReplaceResult findReplaceImpl(String replaceExp, Map props, boolean oppositeDir){
        incSearchReset();
        props = getValidFindProperties(props);
        boolean back = isBackSearch(props, oppositeDir);
        JTextComponent c = Utilities.getLastActiveComponent();
        Object findWhat = props.get(SettingsNames.FIND_WHAT);
        if (findWhat == null) { // nothing to search for
            return null;
        }

        String exp = "'" + findWhat + "' "; // NOI18N
        if (c != null) {
            Utilities.clearStatusText(c);
            Caret caret = c.getCaret();
            int dotPos = caret.getDot();
            if (findWhat.equals(c.getSelectedText())) {
                Object dp = props.get(SettingsNames.FIND_BACKWARD_SEARCH);
                boolean direction = (dp != null) ? ((Boolean)dp).booleanValue() : false;
                
                if (dotPos == (oppositeDir ^ direction ? c.getSelectionEnd() : c.getSelectionStart()))
                    dotPos += (oppositeDir ^ direction ? -1 : 1);
            }
            
            Boolean b = (Boolean)props.get(SettingsNames.FIND_BLOCK_SEARCH);
            boolean blockSearch = (b != null && b.booleanValue());
            Integer i = (Integer) props.get(SettingsNames.FIND_BLOCK_SEARCH_START);
            int blockSearchStart = (i != null) ? i.intValue() : -1;
            int blockSearchEnd = getBlockEndOffset();

            try {
                FindReplaceResult result = findReplaceInBlock(replaceExp, c, dotPos, 
                        (blockSearch && blockSearchStart > -1) ? blockSearchStart : 0, 
                        (blockSearch && blockSearchEnd > 0) ? blockSearchEnd : -1, 
                        props, oppositeDir);
                int[] blk = null; 
                if (result != null){
                    blk = result.getFoundPositions();
                }
                if (blk != null) {
                    selectText(c, blk[0], blk[1], back);
                    JumpList.checkAddEntry();
                    String msg = exp + NbBundle.getBundle(BaseKit.class).getString(FOUND_LOCALE)
                                 + ' ' + Utilities.debugPosition((BaseDocument)c.getDocument(), blk[0]);
                    if (blk[2] == 1) { // wrap was done
                        msg += "; "; // NOI18N
                        if (blockSearch && blockSearchEnd>0 && blockSearchStart >-1){
                            msg += back ? NbBundle.getBundle(BaseKit.class).getString(WRAP_BLOCK_END_LOCALE)
                                   : NbBundle.getBundle(BaseKit.class).getString(WRAP_BLOCK_START_LOCALE);
                        }else{
                            msg += back ? NbBundle.getBundle(BaseKit.class).getString(WRAP_END_LOCALE)
                                   : NbBundle.getBundle(BaseKit.class).getString(WRAP_START_LOCALE);
                        }
                        Utilities.setStatusBoldText(c, msg);
                        c.getToolkit().beep();
                    } else {
                        Utilities.setStatusText(c, msg);
                    }
                    return result;
                } else { // not found
                    Utilities.setStatusBoldText(c, exp + NbBundle.getBundle(BaseKit.class).getString(
                                                    NOT_FOUND_LOCALE));
                    // issue 14189 - selection was not removed
                    c.getCaret().setDot(c.getCaret().getDot());
                }
            } catch (BadLocationException e) {
                Utilities.annotateLoggable(e);
            }
        }
        return null;
    }
    
    /** Find the text from the caret position.
    * @param props search properties
    * @param oppositeDir whether search in opposite direction
    */
    public boolean find(Map props, boolean oppositeDir) {
        FindReplaceResult result = findReplaceImpl(null, props, oppositeDir);
        return (result != null);
    }

    private FindReplaceResult findReplaceInBlock(String replaceExp, JTextComponent c, int startPos, int blockStartPos,
                             int blockEndPos, Map props, boolean oppositeDir) throws BadLocationException {
        if (c != null) {
            props = getValidFindProperties(props);
            BaseDocument doc = (BaseDocument)c.getDocument();
            int pos = -1;
            boolean wrapDone = false;
            String replaced = null;

            boolean back = isBackSearch(props, oppositeDir);
            Boolean b = (Boolean)props.get(SettingsNames.FIND_WRAP_SEARCH);
            boolean wrap = (b != null && b.booleanValue());
            int docLen = doc.getLength();
            if (blockEndPos == -1) {
                blockEndPos = docLen;
            }

            int retFind[];
            while (true) {
                //pos = doc.find(sf, startPos, back ? blockStartPos : blockEndPos);
                int off1 = startPos;
                int off2 = back ? blockStartPos : blockEndPos;
                FindReplaceResult result = DocumentFinder.findReplaceResult(replaceExp, doc, Math.min(off1, off2), Math.max(off1, off2), 
                       props, oppositeDir );
                if (result == null){
                    return null;
                }
                retFind = result.getFoundPositions();
                replaced = result.getReplacedString();
                if (retFind == null){
                    break;
                }
                pos = retFind[0];
                
                if (pos != -1) {
                    break;
                }

                if (wrap) {
                    if (back) {
                        //Bug #20552 the wrap search check whole document
                        //instead of just the remaining not-searched part to be
                        //able to find expressions with the cursor in it

                        //blockStartPos = startPos;
                        startPos = blockEndPos;
                    } else {
                        //blockEndPos = startPos;
                        startPos = blockStartPos;
                    }
                    wrapDone = true;
                    wrap = false; // only one loop
                } else { // no wrap set
                    break;
                }

            }

            if (pos != -1) {
                int[] ret = new int[3];
                ret[0] = pos;
                ret[1] = retFind[1];
                ret[2] = wrapDone ? 1 : 0;
                return new FindReplaceResult(ret, replaced);
            }
        }
        return null;
    }
    
    /** Find the searched expression
    * @param startPos position from which to search. It must be inside the block.
    * @param blockStartPos starting position of the block. It must
    *   be valid position greater or equal than zero. It must be lower than
    *   or equal to blockEndPos (except blockEndPos=-1).
    * @param blockEndPos ending position of the block. It can be -1 for the end
    *   of document. It must be greater or equal than blockStartPos (except blockEndPos=-1).
    * @param props search properties
    * @param oppositeDir whether search in opposite direction
    * @param displayWrap whether display messages about the wrapping
    * @return either null when nothing was found or integer array with three members
    *    ret[0] - starting position of the found string
    *    ret[1] - ending position of the found string
    *    ret[2] - 1 or 0 when wrap was or wasn't performed in order to find the string 
    */
    public int[] findInBlock(JTextComponent c, int startPos, int blockStartPos,
                             int blockEndPos, Map props, boolean oppositeDir) throws BadLocationException {
        FindReplaceResult result = findReplaceInBlock(null, c, startPos, blockStartPos,
                             blockEndPos, props, oppositeDir);
        return result == null ? null : result.getFoundPositions();
    }

    public boolean replace(Map props, boolean oppositeDir)
    throws BadLocationException {
        incSearchReset();
        props = getValidFindProperties(props);
        Boolean b = (Boolean)props.get(SettingsNames.FIND_BACKWARD_SEARCH);
        boolean back = (b != null && b.booleanValue());
        if (oppositeDir) {
            back = !back;
        }

        b = (Boolean)props.get(SettingsNames.FIND_BLOCK_SEARCH);
        boolean blockSearch = (b != null && b.booleanValue());
        Integer i = (Integer) props.get(SettingsNames.FIND_BLOCK_SEARCH_START);
        int blockSearchStart = (i != null) ? i.intValue() : -1;
        int blockSearchEnd = getBlockEndOffset();

        JTextComponent c = Utilities.getLastActiveComponent();
        if (c != null) {
            String s = (String)props.get(SettingsNames.FIND_REPLACE_WITH);
            Caret caret = c.getCaret();
            if (caret.isSelectionVisible()){
                int dotPos = caret.getDot();
                Object dp = props.get(SettingsNames.FIND_BACKWARD_SEARCH);
                boolean direction = (dp != null) ? ((Boolean)dp).booleanValue() : false;
                dotPos = (oppositeDir ^ direction ? c.getSelectionEnd() : c.getSelectionStart());
                c.setCaretPosition(dotPos);
            }
            
            FindReplaceResult result = findReplaceImpl(s, props, oppositeDir);
            if (result!=null){
                s  = result.getReplacedString();
            } else {
                return false;
            }

            BaseDocument doc = (BaseDocument)c.getDocument();
            int startPos = c.getSelectionStart();
            int len = c.getSelectionEnd() - startPos;
            doc.atomicLock();
            try {
                if (len > 0) {
                    doc.remove(startPos, len);
                }
                if (s != null && s.length() > 0) {
                    doc.insertString(startPos, s, null);
                }
            } finally {
                doc.atomicUnlock();
                if (blockSearch){
                    setBlockSearchHighlight(blockSearchStart, getBlockEndOffset());
                }
            }
            
            // adjust caret pos after replace operation
            int adjustedCaretPos = (back || s == null) ? startPos : startPos + s.length();
            caret.setDot(adjustedCaretPos);
            
        }
        
        return true;
    }

    public void replaceAll(Map props) {
        incSearchReset();
        JTextComponent c = Utilities.getLastActiveComponent();
        BaseDocument doc = (BaseDocument)c.getDocument();
        int maxCnt = doc.getLength();
        int replacedCnt = 0;
        int totalCnt = 0;

        props = getValidFindProperties(props);
        props = new HashMap(props);
        String replaceWithOriginal = (String)props.get(SettingsNames.FIND_REPLACE_WITH);
        
        Boolean b = (Boolean)props.get(SettingsNames.FIND_BLOCK_SEARCH);
        boolean blockSearch = (b != null && b.booleanValue());
        b = (Boolean)props.get(SettingsNames.FIND_WRAP_SEARCH);
        boolean wrapSearch = (b != null && b.booleanValue());
        b = (Boolean)props.get(SettingsNames.FIND_BACKWARD_SEARCH);
        boolean backSearch = (b != null && b.booleanValue());
        
        if (wrapSearch){
            props.put(SettingsNames.FIND_WRAP_SEARCH, Boolean.FALSE);
            props.put(SettingsNames.FIND_BACKWARD_SEARCH, Boolean.FALSE);
            firePropertyChange(null, null, null); 
        }
        
        Integer i = (Integer) props.get(SettingsNames.FIND_BLOCK_SEARCH_START);
        int blockSearchStart = (i != null) ? i.intValue() : -1;
        int blockSearchEnd = getBlockEndOffset();

        if (c != null) {
            doc.atomicLock();
            try {
                int startPosWholeSearch = 0;
                int endPosWholeSearch = -1;
                int caretPos = c.getCaret().getDot();

                if (!wrapSearch){
                    if (backSearch){
                        startPosWholeSearch = 0;
                        endPosWholeSearch = caretPos;
                    }else{
                        startPosWholeSearch = caretPos;
                        endPosWholeSearch = -1;
                    }
                }
                
                int actualPos = wrapSearch ? 0 : c.getCaret().getDot();
                
                int pos = (blockSearch && blockSearchStart > -1) ? ( backSearch ? blockSearchEnd : blockSearchStart) : actualPos; // actual position
                
                while (true) {
                    blockSearchEnd = getBlockEndOffset();
                    FindReplaceResult result = findReplaceInBlock(replaceWithOriginal, c, pos, 
                            (blockSearch && blockSearchStart > -1) ? blockSearchStart : startPosWholeSearch, 
                            (blockSearch && blockSearchEnd > 0) ? blockSearchEnd : endPosWholeSearch, 
                            props, false);
                    if (result == null){
                        break;
                    }
                    int[] blk = result.getFoundPositions();
                    String replaceWith = result.getReplacedString();
                    if (blk == null) {
                        break;
                    }
                    totalCnt++;
                    int len = blk[1] - blk[0];
                    boolean skip = false; // cannot remove (because of guarded block)?
                    try {
                        doc.remove(blk[0], len);
                    } catch (GuardedException e) {
                        // replace in guarded block
                        skip = true;
                    }
                    if (skip) {
                        pos = blk[0] + len;

                    } else { // can and will insert the new string
                        if (replaceWith != null && replaceWith.length() > 0) {
                            doc.insertString(blk[0], replaceWith, null);
                        }
                        pos = blk[0] + ((replaceWith != null) ? replaceWith.length() : 0);
                        replacedCnt++;
                    }
                }
                
                // Display message about replacement
                if (totalCnt == 0){
                    Object findWhat = props.get(SettingsNames.FIND_WHAT);
                    String exp = "'' "; //NOI18N
                    if (findWhat != null) { // nothing to search for
                        exp = "'" + findWhat + "' "; // NOI18N
                    }
                    Utilities.setStatusBoldText(c, exp + NbBundle.getBundle(BaseKit.class).getString(
                                NOT_FOUND_LOCALE));
                }else{
                    MessageFormat fmt = new MessageFormat(
                                            NbBundle.getBundle(BaseKit.class).getString(ITEMS_REPLACED_LOCALE));
                    String msg = fmt.format(new Object[] { new Integer(replacedCnt), new Integer(totalCnt) });
                    Utilities.setStatusText(c, msg);
                }

            } catch (BadLocationException e) {
                e.printStackTrace();
            } finally {
                doc.atomicUnlock();
                if (blockSearch){
                    setBlockSearchHighlight(blockSearchStart, getBlockEndOffset());
                }
            }
        }
    }

    /** Get position of wrap mark for some document */
    public int getWrapSearchMarkPos(BaseDocument doc) {
        Mark mark = (Mark)doc.getProperty(BaseDocument.WRAP_SEARCH_MARK_PROP);
        try {
            return (mark != null) ? mark.getOffset() : doc.getLength();
        } catch (InvalidMarkException e) {
            throw new RuntimeException(); // shouldn't happen
        }
    }

    /** Set new position of wrap mark for some document */
    public void setWrapSearchMarkPos(BaseDocument doc, int pos) {
        //!!!
    }

    /** Add weak listener to listen to change of any property. The caller must
    * hold the listener object in some instance variable to prevent it
    * from being garbage collected.
    */
    public void addPropertyChangeListener(PropertyChangeListener l) {
        changeSupport.addPropertyChangeListener(l);
    }

    public synchronized void addPropertyChangeListener(String findPropertyName,
            PropertyChangeListener l) {
        changeSupport.addPropertyChangeListener(findPropertyName, l);
    }

    /** Remove listener for changes in properties */
    public void removePropertyChangeListener(PropertyChangeListener l) {
        changeSupport.removePropertyChangeListener(l);
    }

    void firePropertyChange(String settingName, Object oldValue, Object newValue) {
        changeSupport.firePropertyChange(this, settingName, oldValue, newValue);
    }

    public void setHistory(List/*<SearchPatternWrapper>*/ spwList){
        this.historyList = new ArrayList(spwList);
        firePropertyChange(FIND_HISTORY_CHANGED_PROP,null,null);
    }
    
    public List/*<SearchPatternWrapper>*/ getHistory(){
        return historyList;
    }
    
    public void setLastSelected(SearchPatternWrapper spw){
        this.lastSelected = spw;
        Map props = getFindProperties();
        if (spw == null) return;
        props.put(SettingsNames.FIND_WHAT, spw.getSearchExpression());
        props.put(SettingsNames.FIND_MATCH_CASE, Boolean.valueOf(spw.isMatchCase()));
        props.put(SettingsNames.FIND_REG_EXP, Boolean.valueOf(spw.isRegExp()));
        props.put(SettingsNames.FIND_WHOLE_WORDS, Boolean.valueOf(spw.isWholeWords()));
    }
    
    public SearchPatternWrapper getLastSelected(){
        return lastSelected;
    }
    
    public void addToHistory(SearchPatternWrapper spw){
        if (spw == null) return;
        firePropertyChange(FIND_HISTORY_PROP, null, spw);
    }
    
    public static class SearchPatternWrapper{
        private String searchExpression;
        private boolean wholeWords;
        private boolean matchCase;
        private boolean regExp;
        
        public SearchPatternWrapper(String searchExpression, boolean wholeWords,
            boolean matchCase, boolean regExp){
            this.searchExpression = searchExpression;
            this.wholeWords = wholeWords;
            this.matchCase = matchCase;
            this.regExp = regExp;
        }
        
        /** @return searchExpression */
        public String getSearchExpression(){
            return searchExpression;
        }

        /** @return true if the wholeWords parameter was used during search performing */
        public boolean isWholeWords(){
            return wholeWords;
        }

        /** @return true if the matchCase parameter was used during search performing */
        public boolean isMatchCase(){
            return matchCase;
        }

        /** @return true if the regExp parameter was used during search performing */
        public boolean isRegExp(){
            return regExp;
        }
        
        public boolean equals(Object obj){
            if (!(obj instanceof SearchPatternWrapper)){
                return false;
            }
            SearchPatternWrapper sp = (SearchPatternWrapper)obj;
            return (this.searchExpression.equals(sp.getSearchExpression()) &&
                    this.wholeWords == sp.isWholeWords() &&
                    this.matchCase == sp.isMatchCase() &&
                    this.regExp == sp.isRegExp());
        }

        public int hashCode() {
            int result = 17;
            result = 37*result + (this.wholeWords ? 1:0);
            result = 37*result + (this.matchCase ? 1:0);
            result = 37*result + (this.regExp ? 1:0);
            result = 37*result + this.searchExpression.hashCode();
            return result;
        }
        
        public String toString(){
            StringBuffer sb = new StringBuffer("[SearchPatternWrapper:]\nsearchExpression:"+searchExpression);//NOI18N
            sb.append('\n');
            sb.append("wholeWords:");//NOI18N
            sb.append(wholeWords);
            sb.append('\n');
            sb.append("matchCase:");//NOI18N
            sb.append(matchCase);
            sb.append('\n');
            sb.append("regExp:");//NOI18N
            sb.append(regExp);
            return  sb.toString();
        }
    }
}
