/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.editor.hints.spi;

import java.util.ArrayList;
import java.util.List;
import javax.swing.text.Position;
import org.openide.filesystems.FileObject;

/**
 * Represents a set of changes made by a hint.  Note that the start/end 
 * parameters refer to offsets that should be <i>selected</i>, not to 
 * the general offsets of the modified element.
 * <p>
 * In other words, a change generated by a hint can generate things like
 * variable names;  ChangeInfo provides a list of generated code which the
 * user might want to modify/replace with their own text.  The start/end offsets
 * are offsets into the file in question, which determine selection start/end
 * in the editor.
 * <p>
 * This class provides for a list of changes, anticipating &quot;live template&quot;
 * support (where the user confirms each generated element).  Currently only
 * the first change provided is used.
 * 
 * @author Tim Boudreau
 */
public final class ChangeInfo {
    private List changes = null;
    
    /** Create an instance of ChangeInfo prepopulated with a 
     * single change */
    public ChangeInfo(FileObject fileObject, Position start, Position end) {
        add (fileObject, start, end);
    }
    
    public ChangeInfo(Position start, Position end) {
        add (null, start, end);
    }
    
    
    public ChangeInfo() {
    }
    
    public final int size() {
        return changes != null ? changes.size() : 0;
    }
    
    public final void add (FileObject fileObject, Position start, Position end) {
        if (changes == null) {
            changes = new ArrayList(5);
        }
        changes.add (new ChangeImpl (fileObject, start, end));
    }
    
    public final Change get (int i) {
        if (changes == null) {
            throw new ArrayIndexOutOfBoundsException ("No changes"); // NOI18N
        }
        return (Change) changes.get(i);
    }
    
    public String toString() {
        int size = size();
        if (size == 0) {
            return "Empty ChangeInfo"; // NOI18N
        } else {
            StringBuffer sb = new StringBuffer (100);
            sb.append ("ChangeInfo ["); // NOI18N
            for (int i=0; i < size; i++) {
                sb.append (get(i));
                if (i != size-1) {
                    sb.append (',');
                }
            }
            sb.append ("]"); // NOI18N
            return sb.toString();
        }
    }
    
    /**
     * Interface representing a single caret-positioning or user-modifiable
     * change.
     */
    public static interface Change {
        public Position getStart();
        public Position getEnd();
        public FileObject getFileObject();
    }
    
    private static final class ChangeImpl implements Change {
        Position start;
        Position end;
        FileObject fileObject;
        
        ChangeImpl(FileObject fileObject, Position start, Position end) {
            this.fileObject = fileObject;
            this.start = start;
            this.end = end;
        }
        
        public Position getStart() {
            return start;
        }
        
        public Position getEnd() {
            return end;
        }
        
        public FileObject getFileObject() {
            return fileObject;
        }
        
        public String toString() {
            return "Change from " + start.getOffset() + " to " + end.getOffset() + " in " + fileObject; // NOI18N
        }
    }
    
}
