/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.editor.hints;

import java.awt.Rectangle;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.lang.ref.Reference;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import javax.swing.SwingUtilities;
import javax.swing.event.CaretEvent;
import javax.swing.event.CaretListener;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.text.BadLocationException;
import javax.swing.text.JTextComponent;
import org.netbeans.api.editor.mimelookup.MimeLookup;
import org.netbeans.editor.BaseDocument;
import org.netbeans.editor.BaseKit;
import org.netbeans.editor.Registry;
import org.netbeans.editor.Utilities;
import org.netbeans.modules.editor.hints.spi.Hint;
import org.netbeans.modules.editor.hints.spi.HintsProvider;
import org.openide.ErrorManager;
import org.openide.util.Lookup;
import org.openide.util.RequestProcessor;


/**
 * @author Jan Lahoda
 * @author leon chiver
 */
public final class HintsOperator implements CaretListener, ChangeListener, 
        PropertyChangeListener, ComponentListener, FocusListener, KeyListener {
    private static ErrorManager ERR = ErrorManager.getDefault().getInstance("org.netbeans.modules.hints"); // NOI18N
    private static RequestProcessor HINTS_REQUEST_PROCESSOR = new RequestProcessor("Hints RP", 1); // NOI18N
    
    private Reference             componentRef;
    private RequestProcessor.Task hintTask;
    private RequestProcessor.Task hintPopupTask;
    private List                  hintsProviders;
    private int                   hintsType;
    private int                   lastOffset;
    private int                   lineOffset;
    private int                   lastLineOffset;
    
    private HintsUI ui = new HintsUI();
    
    /** Creates a new instance of HintsOperator */
    private HintsOperator() {
        this.componentRef = null;
        this.hintTask = HINTS_REQUEST_PROCESSOR.create(new HintPopupTaskImpl(false));
        this.hintPopupTask = HINTS_REQUEST_PROCESSOR.create(new HintPopupTaskImpl(true));
        this.lastOffset = -1;
        
        this.hintTask.setPriority(Thread.MIN_PRIORITY);
        
        Registry.addChangeListener(this);
    }
    
    private static HintsOperator instance = new HintsOperator();
    
    public static HintsOperator getDefault() {
        return instance;
    }
    
    public synchronized void caretUpdate(CaretEvent e) {
	JTextComponent component = getComponent();
	if (component == null) {
	    return;
	}

        if (!component.isFocusOwner()) {
            return;
        }
        lastOffset = component.getCaretPosition();
        // Hint should be hidden when the caret row changes
        BaseDocument doc = Utilities.getDocument(component);
        if (doc != null) {
            try {
                lineOffset = Utilities.getLineOffset(doc, lastOffset);
                if (lineOffset != lastLineOffset && ui.isActive()) {
                    ui.setHints (null, null, false);
                }
                lastLineOffset = lineOffset;
            } catch (BadLocationException ex) {
                // Ignore it
            }
        }
        hintTask.schedule(200);
    }
    
    private void setCurrentHints(final List currentHints, final int hintsType, final boolean showPopup) {
	JTextComponent component = getComponent();
        if (component == null || !component.hasFocus()) {
            return;
        }
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
		JTextComponent component = getComponent();
		if (component == null) {
		    return;
		}
		
                //this method is going to take the read lock on the document
                //(through the Utilities.getRowStart at least)
                //But, the lock on this cannot be held BEFORE the document lock,
                //put it all into a render section:
                component.getDocument().render(new Runnable() {
                    public void run() {
                        synchronized (HintsOperator.this) {
			    JTextComponent component = getComponent();
			    if (component == null) {
				return;
			    }
			    
                            HintsOperator.this.hintsType = hintsType;
                            if (currentHints != null && currentHints.size() > 0) {
                                ui.setHints(currentHints, component, showPopup);
                            } else {
                                ui.setHints(null, null, false);
                            }
                        }
                    }
                });
            }
        });
    }
    
    public void componentHidden(ComponentEvent e) {
        ui.setHints(null, null, false);
    }
    
    public void componentShown(ComponentEvent e) {
        // Do nothing
    }
    
    public void componentMoved(ComponentEvent e) {
        ui.setHints(null, null, false);
        hintTask.schedule(500);
    }
    
    public void componentResized(ComponentEvent e) {
        componentMoved (e);
    }
    
    public void keyReleased(KeyEvent e) {
        // Do nothing
    }
    
    public void keyPressed(KeyEvent e) {
        if (e.getKeyCode() == KeyEvent.VK_ENTER && e.getModifiersEx() == KeyEvent.ALT_DOWN_MASK
                && !ui.isActive()) {
            if (!hintTask.isFinished()) {
                hintTask.cancel();
            }
            if (!hintPopupTask.isFinished()) {
                hintPopupTask.cancel();
            }
            hintPopupTask.schedule(0);
            e.consume();
        }
    }
    
    public void keyTyped(KeyEvent e) {
        // Do nothing
    }
    
    private class HintPopupTaskImpl implements Runnable {
        
        private boolean showPopup;

        private Throwable lastCatched;
        
        public HintPopupTaskImpl(boolean showPopup) {
            this.showPopup = showPopup;
        }
        
        public void run() {
            //TODO: task id number, to assure that long-running tasks do not provide data in place of a later task.
            int position = -1;
            JTextComponent component = null;
            
            synchronized (HintsOperator.this) {
                position = lastOffset;
                component = HintsOperator.this.getComponent();
            }
            
            // Are hints already shown?
            if (showPopup && ui.isPopupActive()) {
                return;
            } 
            
            if (component == null || position == (-1)) {
                return ;
            }
            
            // Is the position visible?
            try {
                Rectangle r = component.modelToView(position);
                Rectangle visible = component.getVisibleRect();
                if (r == null || visible == null) {
                    return;
                }
                if (r.y <= visible.y) {
                    return;
                }
                if (r.y + r.height > visible.y + visible.height) {
                    return;
                }
                if (r.x < visible.x) {
                    return;
                }
            } catch (BadLocationException ex) {
                // Should not happen
            }
            
            List result = new ArrayList();
            
            if (ERR.isLoggable(ErrorManager.INFORMATIONAL)) {
                ERR.log("hintsProviders = " + hintsProviders );
            }
            
            for (Iterator i = hintsProviders.iterator(); i.hasNext(); ) {
                HintsProvider provider = (HintsProvider) i.next();
                
                if (ERR.isLoggable(ErrorManager.INFORMATIONAL)) {
                    ERR.log("provider = " + provider );
                    ERR.log("result = " + result );
                }
                try {
                    result.addAll(provider.getHints(component.getDocument(), position));
                } catch (Throwable e) {
                    // prevent cyclic exception when computing hint.
                    if (!equalsException(lastCatched, e)) {
                        lastCatched = e;
                        ErrorManager.getDefault().notify(ErrorManager.EXCEPTION, e);
                    }
                }
                
                if (ERR.isLoggable(ErrorManager.INFORMATIONAL)) {
                    ERR.log("result = " + result );
                }
            }
            
            int hintsType = Hint.SUGGESTION;
            
            for (Iterator i = result.iterator(); i.hasNext(); ) {
                Hint h = (Hint) i.next();
                
                if (h.getType() < hintsType) {
                    hintsType = h.getType();
                }
            }
            
            setCurrentHints(result, hintsType, showPopup);
        }

        private boolean equalsException(Throwable e1, Throwable e2) {
            if (e1 == null) {
                return false;
            }
            return Arrays.equals(e1.getStackTrace(), e2.getStackTrace());
        }
        
    }
    
    public void stateChanged(ChangeEvent e) {
        JTextComponent active = Registry.getMostActiveComponent();
        
        if (ui.getComponent() != active) {
            ui.setHints(null, null, false);
            unregisterFromComponent();
            registerNewComponent(active);
        }
    }
    
    private synchronized void unregisterFromComponent() {
	JTextComponent component = getComponent();
	if (component == null) {
	    return;
	}

        if (component != null) {
            component.removeCaretListener(this);
            component.removeComponentListener(this);
            component.removeFocusListener(this);
            component.removeKeyListener(this);
        }
        removeListenersFromProviders();
        componentRef = null;
    }
    
    private synchronized void registerNewComponent(JTextComponent c) {
        if (c == null)
            return ;
        
        hintTask.cancel();
        this.componentRef = new WeakReference(c);
        this.lastOffset = c.getCaretPosition();
        
        gatherProviders();
        
        addListenersToProviders();
        
        c.addCaretListener(this);
        c.addComponentListener(this);
        c.addFocusListener(this);
        c.addKeyListener(this);
        hintTask.schedule(500);
    }
    
    private JTextComponent getComponent() {
	return (componentRef != null)
	    ? (JTextComponent)componentRef.get()
	    : null;
    }

    private void gatherProviders() {
        BaseKit kit = Utilities.getKit(getComponent());
        
        if (kit == null) {
            hintsProviders = Collections.EMPTY_LIST;
            return ;
        }
        Lookup lookup = MimeLookup.getMimeLookup(kit.getContentType());
        Lookup.Result result = lookup.lookup(new Lookup.Template(HintsProvider.class));

        hintsProviders = new ArrayList(result.allInstances());
    }
    
    private void addListenersToProviders() {
        if (hintsProviders == null)
            return ;
        
        for (Iterator i = hintsProviders.iterator(); i.hasNext(); ) {
            HintsProvider provider = (HintsProvider) i.next();
            
            provider.addPropertyChangeListener(this);
        }
    }

    private void removeListenersFromProviders() {
        if (hintsProviders == null)
            return ;
        
        for (Iterator i = hintsProviders.iterator(); i.hasNext(); ) {
            HintsProvider provider = (HintsProvider) i.next();
            
            provider.removePropertyChangeListener(this);
        }
    }

    public void propertyChange(PropertyChangeEvent evt) {
	JTextComponent component = getComponent();
	
        if (component != null && (evt.getNewValue() == component || evt.getNewValue() == component.getDocument())) {
            // some of the providers notify us that the list of hints should be updated:
            hintTask.schedule(100);
        }
    }
    
    public void focusGained (FocusEvent fe) {
        if (!ui.isActive()) {
            hintTask.schedule(500);
        }
    }
    
    public void focusLost (FocusEvent fe) {
        if (!hintTask.isFinished()) {
            hintTask.cancel();
        }
        if (!hintPopupTask.isFinished()) {
            hintPopupTask.cancel();
        }
        if (!ui.isKnownComponent(fe.getOppositeComponent())) {
            ui.setHints (null, null, false);
            lineOffset = -1;
            lastOffset = -2;
            lastLineOffset = -2;
        }
    }
}
