/* Copyright (c) 2000  Kevin Sullivan <nite@gis.net>
 *
 * Please refer to the COPYRIGHT file for more information.
 */

#include <signal.h>
#include <sys/time.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <ncurses.h>

#include "defines.h"
#include "colors.h"
#include "timer.h"
#include "winio.h"
#include "cmds.h"
#include "lists.h"

/* a list of scheduled events. Each carries a time when it is supposed
   to be performed, then deleted. */
ntimer_t *timers;

/* do any time events that are due */
void tevent(void)
{
  time_t t = time(NULL);
  ntimer_t *cur;  /* last was called n. What is it for? */
  
  for (cur=timers;cur;)
  {
    if (cur->t <= t)
    {
      pfunc(cur);
      deltimer(cur);
      cur = timers;  
      /* this is inelegant: after deleting the element, we have to
	 start from the beginning. But note that pfunc could have side
	 effects, thus it may not in general work to save cur->next
	 before deleting cur.  */
    }
    else
    {
      cur = cur->next;
    }
  }
}

/* print the current timers */
void timerlist(WINDOW *win)
{
  ntimer_t *cur;
  int i, t, min, sec;
  time_t ct = time(NULL);
  
  if (!timers) {
    wp(win, "No timers set\n");
    drw(win);
    return;
  }

  for (cur=timers,i=0;cur;cur=cur->next,i++)
  {
    t = cur->t-ct;
    min = t/60;
    sec = t%60;
    wp(win, ""BRIGHT(BLUE)"* (in %i:%02i): "WHITE"%s\n", min, sec, cur->desc);
  }
  
  drw(win);
}

/* perform the function scheduled in t. */
void pfunc(ntimer_t *t)
{
  t->callback(t->data);
}

#if 0
/* perform the function scheduled in t (old version). */
void pfunc_old(ntimer_t *t)
{
  int i, c;
  char dbuf[512];
  
  memset(dbuf, 0, sizeof(dbuf));
  
  strcpy(dbuf+1, t->d);
  dbuf[0] = '/';
  
  /* remove any top-level curly braces */
  /* what is the purpose of this? -PS */
  for (i=0,c=0;dbuf[i];i++)
  {
    if (c < 0)
      c = 0;
    if (dbuf[i] == '{' && !c)
    {
      dbuf[i] = ' ';
      c++;
    }
    else if (dbuf[i] == '{')
      c++;
    else if (dbuf[i] == '}' && c == 1)
    {
      dbuf[i] = ' ';
      c--;
    }
    else if (dbuf[i] == '}')
      c--;
  }
  
  lpbrk = 0;   /* clear previous loop interrupt request */
  parseout(t->s, dbuf, t->win);
}
#endif

/* add a timer to the list. Return pointer to element in timer list */
ntimer_t *addtimer(int sec, void callback(void *data), void *data, char *desc)
{
  ntimer_t *cur;
  time_t t = time(NULL);
  
  /* create new timerlist entry */
  cur = (ntimer_t *)malloc(sizeof(ntimer_t));
  cur->t = t+sec;
  cur->callback = callback;
  cur->data = data;
  cur->desc = strdup(desc);
  cur->next = NULL;
  
  /* add it to the list */
  list_append(ntimer_t, timers, cur);
  
  return((void *)cur);
}

void deltimer(ntimer_t *t)
{
  ntimer_t *cur;
  
  if (!t)
    return;

  list_unlink_cond(ntimer_t, timers, cur, cur==(ntimer_t *)t);
  
  if (!cur)
    return;
  
  free(cur->data);
  free(cur->desc);
  free(cur);
}
