/* Copyright (c) 2000  Kevin Sullivan <nite@gis.net>
 *
 * Please refer to the COPYRIGHT file for more information.
 */

#ifndef MCURSES
#include <ncurses.h>
#endif
#include <dlfcn.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <errno.h>
#include <unistd.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <time.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <ctype.h>

#include "getopt.h"
#include "defines.h"
#include "colors.h"
#include "codes.h"
#include "lists.h"
#include "handlers.h"
#include "alias.h"
#include "nap.h"
#include "sscr.h"
#include "event.h"
#include "winio.h"
#include "timer.h"
#include "scheck.h"
#include "irc.h"
#include "cmds.h"
#include "mp3s.h"

#ifdef USERCMDS
  #include "usercmds.h"
#endif

#ifdef MCURSES
  #include "wind.h"
#endif


extern info_t info;
extern scrls_t *mscroll, *scur;
extern chans_t *chanl, *curchan, *recent;
extern int tind, ircsock;
extern int noprint, ircmode, lpbrk;
extern int ipcs[2];
extern char *mnick;
extern hotlist_t *hlist;
extern WINDOW *winput, *wchan;
extern void *hnd;
extern alias_t *alhead;
extern handler_t *hndhead;
extern int quit_now, quit_after_transfers, upsocks, downsocks;

/* names of the different connection types, 0-10 */
const char *conns[] = { "Unknown", "14.4", "28.8", "33.6", "56k", "ISDN-64K",
                  "ISDN-128K", "Cable", "DSL", "T1", "T3" };
/* how many simultaneous uploads we allow by default, for each
   connection type */
int ups[] = { 3, 2, 3, 3, 4, 5, 5, 7, 7, 9, 9 };

int fl, wmode = 0;
unsigned char *dwi=NULL;
char *tbuf = NULL;
int tin = 0;
char *ctbuf;
upload_t *up;              /* the upload list */
download_t *down = NULL;   /* the download list */
scount_t scount;           /* libraries / songs / gigs */
ssearch_t *search=NULL;    /* search results */
int srch=0;                /* set while a search is in progress (and
                              thus the state of the search result list
                              is not well-defined) */
int quit_after_transfers = 0; /* set by /tquit and checked by tevent() */



in_nap_cmd_t in[] = {
	 { F_SRET, ssret },
	 { F_SEND, ssend },
	 { F_RBROWSE, srbrowse },
	 { F_DBROWSE, sdbrowse },
	 { F_SGET, sget },
	 { F_NGET, snget },
	 { F_NACC, snacc },
	 { F_RQLIMIT, sqlimit },
	 { F_FREQ, sfreq },
	 { F_SSF, ssf },
	 { F_COUNT, sscount },
	 { F_SWHOIS, swhois },
	 { F_SOFF, soff },
	 { F_UON, suon },
	 { F_UOFF, suoff },
	 { F_NOTICE, snotice },
	 { F_RY, sry },
	 { F_SJOIN, sjoin },
	 { F_SPART, spart },
	 { F_PART, smpart },
	 { F_TOPIC, stopic },
	 { F_SAID, ssay },
	 { F_NCHAN, snchan },
	 { F_JCHAN, sjchan },
	 { F_MNAME, snend },
	 { F_MNAME2, snend },
	 { F_TELL, stell },
	 { F_USER, suser },
	 { F_SOP, sop },
	 { F_PCHANGE, spchange },
	 { F_BPORT, sbport },
	 { F_ANNOUNCE, sannounce },
	 { F_SBANLIST, sbanlist },
	 { F_SBANLISTU, sbanlist },
	 { SERVER_PING, ssping },
	 { CLIENT_PING, scping },
	 { CLIENT_PONG, scpong },
	 { CLIENT_REDIRECT, sredir },
	 { CLIENT_CYCLE, scycle },
	 { F_SCLIST, sclist },
	 { CHANNEL_ENTRY2, sclist2 },
	 { F_NAMES, suser },
	 { F_SBLOCKLIST, sblocklist },
	 { IGNORE_LIST, signoreend },
	 { IGNORE_ENTRY, signorelist },
	 { IGNORE_ADD, signoreadd },
	 { IGNORE_REMOVE, signoreremove },
	 { IGNORE_UNKNOWN, signoreunknown },
	 { IGNORE_EXISTS, signoreexists },
	 { IGNORE_CLEAR, signoreclear },
	 { IGNORE_FAIL, signorefail },
	 { CHANNEL_BAN_ENTRY, scbanlist },
	 { NOTIFY_UNKNOWN, snerr },
	 { NOTIFY_EXISTS, snadd },
	 { CHAN_EMOTE, sme },
	 { 0x0, NULL },
};

out_nap_cmd_t out[] = {
          { "about", 1, dabout, NULL },
          { "alias", 1, dalias, "<name> <args> - Creates an alias" },
          { "aliaslist", 1, daliaslist, "- Returns a list of aliases created" },
          { "announce", 0, dannounce, "<msg> - Broadcasts a message to all users" },
	  { "ban", 0, dban, "<user/IP> - Bans the specified user or IP" },
          { "banlist", 0, dbanlist, "- Prints a list of the current bans on the server" },
          { "block", 0, dblock, "<IP> [reason] - Blocks the specified IP" },
          { "blocklist", 0, dblocklist, "- Gives a list of current blocked" },
          { "break", 1, dbreak, "- Breaks out of a loop" },
          { "browse", 0, dbrowse, "<user> - Browses user's files" },
          { "cban", 0, dcban, "<user> [reason] - Bans a user from a channel" },
          { "cbanlist", 0, dcbanlist, "- Returns a list of banned users in a channel" },
          { "chupload", 1, dchupload, "<path> - Changes your upload path (still need to /rebuild to update your files)" },
          { "clear", 1, dclear, "- Clears your screen buffer" },
          { "clearalias", 1, dclearalias, "- Clears all aliases" },
          { "clearhandler", 1, dclearhandler, "- Clears all handlers" },
          { "clist", 0, dclist, "- Gets a list of channels" },
          { "clist2", 0, dclist2, "- Gets a list of channels (includes user created)" },
          { "cloak", 0, dcloak, "- Cloaks yourself" },
          { "conf", 0, dsetconf, NULL },
          { "cunban", 0, dcunban, "<user> [reason] - Unbans a user from a channel" },
          { "ddown", 0, dddown, "<number or range> - Deletes downloads by number as returned from /pdown" },
          { "dec", 1, ddec, "- Decreases the variable by one" },
          { "debug", 1, ddebug, NULL },
          { "disconnect", 0, ddisconnect, "- Disconnects you from the server" },
          { "dns", 1, ddns, "<host/IP> - Attempts to resolve the specified address" },
          { "done", 1, ddone, "- Ends an alias" },
          { "dup", 0, ddup, "<number or range> - Deletes uploads by number as returned from /pup" },
          { "echo", 1, decho, "<text> - Echos text to the screen" },
          { "eval", 1, deval, "<name> - Returns the value of a variable" },
          { "exec", 1, dexec, "[-o] <command> - Executes a command from a shell and redirects the input to the client" },
	  { "finger", 0, dwhois, "<user> - Gets information on the specified user" },
	  { "force", 0, dforce, "<number or range> - Forces download of queued items, overriding download limit" },
	  { "g", 0, dg, "<number or range> - Gets file by number as returned from /search" },
	  { "get", 0, dg, "<number or range>... - Gets file by number as returned from /search" },
          { "gusers", 0, dgusers, "- Gets a global list of users" },
          { "handler", 1, dhandler, "<code> <args> - Adds a handler" },
          { "handlerlist", 1, dhandlerlist, "- Returns a list of handlers created" },
          { "help", 0, dhelp, "<command> - Returns help on the specified command" },
          { "hotlist", 0, dhotlist, "- Lists users on your hotlist that are on" },
          { "if", 1, dif, "(<val> <op> <val>) <cmd> - Compares two values" },
          { "ignore", 0, dignore, "<user> - Ignores a user" },
          { "ignoreclear", 0, dignoreclear, "- Clears your ignore list" },
          { "ignorelist", 0, dignorelist, "- Lists ignored users" },
          { "inc", 1, dinc, "<var> - Increases the variable by 1" },
          { "irc", 0, dirc, NULL },
	  { "join", 0, djoin, "<chan> - Joins the specified channel" },
          { "kick", 0, dkick, "<user> [reason] - Kicks a user from a channel" },
          { "kickall", 0, dkickall, "<user> [reason] - Kicks a user from all channels you and the user are in" },
	  { "kill", 0, dkill, "<user> - Kills the specified user" },
	  { "lastlog", 1, dlastlog, "<str> - Returns all occurences of \"str\" that have been said or printed" },
          { "loadalias", 1, dloadalias, "[filename] - Loads a list of aliases from a file" },
          { "loadchannels", 1, dloadchannels, "[filename] - Reads channels from a filename and joins them" },
          { "loadconfig", 1, dloadconfig, "[filename] - Loads a list of settings from a filename" },
          { "loadhandler", 1, dloadhandler, "<filename> - Loads a list of handlers from a filename" },
          { "me", 0, dme, "<string> - Does an emotion" },
	  { "msg", 0, dtell, "<user> <msg> - Sends the user the message specified" },
          { "muzzle", 0, dmuzzle, "<user> <msg> - Muzzles the user with the specified message" },
          { "names", 0, dusers, "<channel> - Gets a list of channel users" },
          { "news", 1, dnews, "- Checks for any news on the client" },
          { "noprint", 1, dnoprint, "- Stops the client from echoing anything until the command returns" },
          { "notify", 0, dnotify, "<user> - Attempts to add a user to your hotlist" },
	  { "opsay", 0, dopsay, "<msg> - Broadcasts a message to all moderators/admins/elite" },
	  { "part", 0, dpart, "[chan/user] - Parts the specified or current channel or query" },
          { "pchans", 1, dpchans, "- Shows which channels you are on" },
	  { "pdown", 0, dpdown, "- Gives a listing of your current downloads" },
          { "ping", 0, dping, "<user> - Pings a user" },
          { "psocks", 1, dpsocks, "- Print the socket list (for debugging purposes)" },
	  { "pup", 0, dpup, "- Gives a listing of your current uploads" },
	  { "purge", 0, dpurge, "- Removes all stopped items from upload and download lists" },
	  { "purgedown", 0, dpurgedown, "- Removes all stopped items from download list" },
	  { "purgeup", 0, dpurgeup, "- Removes all stopped items from upload list" },
	  { "pvars", 0, dpvars, "- Prints the values of all variables currently set" },
          { "query", 0, dquery, "<user> - Queries a user" },
	  { "q", 0, dquit, "- Closes the program" }, 
	  { "quit", 0, dquit, "- Closes the program" }, 
	  { "rebuild", 1, drebuild, "- Rebuilds your library unconditionally. See also /update." },
          { "reconnect", 0, dreconnect, "- Reconnects you to the server" },
          { "reload", 0, dreloadconf, NULL },
#ifdef USERCMDS
          { "reloadm", 1, dreload, "- Reloads the user command module" },
#endif /* USERCMDS */
          { "results", 1, dresults, "- Switches to the search results screen" },
	  { "retry", 0, dretry, "<number or range> - Puts stopped downloads back in the download queue" },
	  { "retryall", 0, dretry, " - Puts all stopped downloads back in the download queue" },
          { "savealias", 1, dsavealias, "[filename] - Saves current aliases" },
          { "savechannels", 1, dsavechannels, "[filename] - Saves current channels to a filename" },
          { "saveconfig", 1, dsaveconfig, "[filename] - Saves current settings to a filename" },
          { "savehandler", 1, dsavehandler, "<filename> - Saves current handlers to a filename" },
          { "say", 0, dsay, "<msg> - Sends msg to the current channel" },
	  { "search", 0, dsearch, "[-b>bitrate] [-c>conn] [-mmaxres] <query> - Searches the napster database" },
	  { "serv", 0, dserv, "- Returns the current IP and port of the server you're connected to" },
	  { "server", 0, dserver, "<IP:port> - Connects to the specificed server and port" },
          { "set", 1, dset, "<name> <value> - Sets a variable to the specified value" },
	  { "setdataport", 0, ddatap, "<user> <port> - Sets a user's data port" },
          { "setlevel", 0, dsetlevel, "<channel> <level> - ?" },
          { "setlinespeed", 0, dlspeed, "<user> <speed> - Changes a user's linespeed" },
          { "setpassword", 0, duserpass, "<user> <password> - Sets a user's password" },
	  { "setuserlevel", 0, dlevel, "<user> <level> - Changes a user's userlevel" },
          { "sraw", 0, dsraw, NULL },
          { "stop", 1, dstop, "- Returns from the current command and stops all processing on it" },
          { "sver", 0, dsver, "- Returns the server version" },
	  { "tell", 0, dtell, "<user> <msg> - Sends the user the message specified" },
	  { "timer", 1, dtimer, "<min:sec> <cmd> - Initiates a timer to execute in the specified time" },
	  { "tlist", 1, dtlist, "- Prints out a list of the current timers" },
	  { "topic", 0, dtopic, "<channel> <topic> - Changes a channel's topic" },
          { "tquit", 0, dtquit, "- Quits when all remaining transfers have completed. Can be canceled with /unquit." },
          { "unalias", 1, dunalias, "<name> - Removes an alias" },
	  { "unban", 0, dunban, "<IP> - Unbans the specified IP" },
          { "unblock", 0, dunblock, "<IP> - Unblocks the specified IP" },
          { "unhandler", 1, dunhandler, "<code> - Removes a handler" },
          { "unignore", 0, dunignore, "<user> - Unignores a user" },
          { "unmuzzle", 0, dunmuzzle, "<user> - Unmuzzles the user" },
          { "unnotify", 0, dunnotify, "<user> - Removes a user from your hotlist" },
	  { "unquit", 0, dunquit, "- Cancels the effect of /tquit" },
          { "unset", 1, dunset, "<name> - Unsets a variable" },
          { "update", 1, dupdate, "- Rebuilds your library if necessary. See also /rebuild." },
          { "while", 1, dwhile, "(<val> <op> <val>) cmd - Keeps executing cmd while the comparison is true" },
	  { "whois", 0, dwhois, "<user> - Gets information on the specified user" },
          { "window", 1, dwindow, "- Enables/disables window mode" },
          { "wstats", 1, dwstats, NULL },
	  { NULL, 0, NULL, NULL },
};


/* destructively replaces double quotes by single quotes - returns the
   same string */
char *fixquotes(char *s) {
  char *p = s;
  while (*p) {
    if (*p == '\"')
      *p = '\'';
    p++;
  }
  return s;
}

void upchan(chans_t *n)
{
  int i;
  chans_t *cur=NULL;
  
  tind = 0;
  
  if (!n || !chanl)
    return;
  
  if (curchan)
    cur = curchan->next;
  if (!cur)
    cur = chanl;
  for (i=0;;cur=cur->next)
  {
    if (i == 2)
      return;
    if (!cur)
      cur=chanl;
    if (!strcmp(cur->nm, n->nm))
      i++;
    else
      break;
  }
  curchan = cur;
  
  return;
}

int
in_cksum(u_short *addr, int len)
{
	register int nleft = len;
	register u_short *w = addr;
	register int sum = 0;
	u_short answer = 0;

	/*
	 * Our algorithm is simple, using a 32 bit accumulator (sum), we add
	 * sequential 16 bit words to it, and at the end, fold back all the
	 * carry bits from the top 16 bits into the lower 16 bits.
	 */
	while (nleft > 1)  {
		sum += *w++;
		nleft -= 2;
	}

	/* mop up an odd byte, if necessary */
	if (nleft == 1) {
		*(u_char *)(&answer) = *(u_char *)w ;
		sum += answer;
	}

	/* add back carry outs from top 16 bits to low 16 bits */
	sum = (sum >> 16) + (sum & 0xffff);	/* add hi 16 to low 16 */
	sum += (sum >> 16);			/* add carry */
	answer = ~sum;				/* truncate to 16 bits */
	return(answer);
}

void
tvsub(register struct timeval *out, register struct timeval *in)
{
	if ((out->tv_usec -= in->tv_usec) < 0) {
	  --out->tv_sec;
	  out->tv_usec += 1000000;
	}
	out->tv_sec -= in->tv_sec;
}

/* receive ping response */
int icmpin(WINDOW *win, sock_t *m)
{
  int s = m->fd;
  ssearch_t *cur;
  char buf[256];
  struct sockaddr_in frm;
  int frmlen = sizeof(frm);
  struct timeval tv;
  
  gettimeofday(&tv, NULL);
  
  memset(buf, 0, sizeof(buf));
  recvfrom(s, buf, sizeof(buf), 0, (struct sockaddr *)&frm, &frmlen);
  
  for (cur=search; cur!=NULL; cur=cur->next)
  {
    if (cur->nip == frm.sin_addr.s_addr && !timerisset(&cur->r))
      memcpy(&cur->r, &tv, sizeof(tv));
  }
  
  return(1);
}

char **form_toks(char *buf, int *cnt)
{
  int i, j, k, l = 0, c=0;
  char **ret;
  
  ret = (char **)malloc(4096);
  ret[0] = NULL;
  
  for (j=0,i=0;buf[j];i++)
  {
    while (buf[j] == ' ')
      j++;
    if (!buf[j])
      break;
    ret[i] = (char *)malloc(2048);
    memset(ret[i], 0, 2048);
    if (buf[j] == '\\')
    {
      for (k=0,j++;buf[j]!=' '&&buf[j]&&k!=2048;k++,j++)
        ret[i][k] = buf[j];
      ret[i][k] = 0;
    }
    else if (buf[j] == '\'')
    {
      j++;
      while (buf[j] && buf[j] == ' ')
        j++;
      for (k=0;buf[j]!='\''&&buf[j]&&k!=2048;k++,j++)
        ret[i][k] = buf[j];
      ret[i][k--] = 0;
      while (k >= 0 && ret[i][k] == ' ')
        ret[i][k--] = 0;
      j++;
    }
    else if (buf[j] == '\"')
    {
      for (k=0,j++;buf[j]!='\"'&&buf[j]&&k!=2048;k++,j++)
        ret[i][k] = buf[j];
      ret[i][k] = 0;
      j++;
    }
    else if (buf[j] == '(')
    {
      for (k=0,c=0;buf[j]&&k!=2047;k++,j++)
      {
        if (buf[j] == ')' && c == 1)
          break;
        if (c < 0)
          c = 0;
        if (buf[j] == '(')
          c++;
        else if (buf[j] == ')')
          c--;
        ret[i][k] = buf[j];
      }
      if (buf[j])
      {
        ret[i][k] = ')';
        ret[i][k+1] = 0;
        j++;
      }
    }
    else if (buf[j] == '{')
    {
      for (k=0,c=0;buf[j]&&k!=2047;k++,j++)
      {
        if (buf[j] == '}' && c == 1)
          break;
        if (c < 0)
          c = 0;
        if (buf[j] == '{')
          c++;
        else if (buf[j] == '}')
          c--;
        ret[i][k] = buf[j];
      }
      if (buf[j])
      {
        ret[i][k] = '}';
        ret[i][k+1] = 0;
        j++;
      }
    }
    else
    {
      for (k=0;buf[j]!=' '&&buf[j]&&k!=2048;k++,j++)
        ret[i][k] = buf[j];
      ret[i][k] = 0;
    }
    ret[i] = (char *)realloc(ret[i], strlen(ret[i])+1);
    l+=strlen(ret[i])+1;
  }
  
  ret[i] = NULL;
  
  *cnt = i;
  
  return(ret);
}

/* break up the contents of a server packet (or remote client packet)
 * into an array of tokens */
char **form_tokso(char *buf, int *cnt)
{
  int i, j, k, l = 0;
  char **ret;
  
  ret = (char **)malloc(4096);
  ret[0] = NULL;
  
  for (j=0,i=0;buf[j];i++)
  {
    while (buf[j] == ' ')
      j++;
    if (!buf[j])
      break;
    ret[i] = (char *)malloc(2048);
    memset(ret[i], 0, 2048);
    if (buf[j] == '\"')
    {
      /* tokens in double quotes (e.g filenames) get copied without the
       * double quotes */
      for (k=0,j++;buf[j]!='\"'&&buf[j]&&k!=2048;k++,j++)
        ret[i][k] = buf[j];
      ret[i][k] = 0;
      j++;
    }
    else
    {
      for (k=0;buf[j]!=' '&&buf[j]&&k!=2048;k++,j++)
        ret[i][k] = buf[j];
      ret[i][k] = 0;
    }
    ret[i] = (char *)realloc(ret[i], strlen(ret[i])+1);
    l+=strlen(ret[i])+1;
  }
  
  ret[i] = NULL;
  
/*  ret = (char **)realloc(ret, l+sizeof(char *)); */
  
  *cnt = i;
  
  return(ret);
}

int parsein(int s, unsigned short op, char *buf, WINDOW *win)
{
  int i, cnt, r=0;
  char **tok, *b, *t=NULL;
  handler_t *cur;
  
  for (cur=hndhead;cur;cur=cur->next)
  {
    if (op == cur->op)
    {
      msprintf(&t, "%i %s", op, buf);
      tok = form_toks(t, &cnt);
      free(t);
      t = NULL;
      b = dohandler(cur, tok, cnt);
      for (i=0;b[i]=='\n'||b[i]==' ';i++);
      msprintf(&t, "/%s", b+i);
      free(b);
      r = parseout(s, t, win);
      free(t);
      for (i=0;i<cnt;i++)
        free(tok[i]);
      free(tok);
      break;
    }
  }
  
  if (r == 2)
    return(1);
  
  if (noprint)
    noprint = 2;
  
#ifdef USERCMDS
  if (strlen(buf))
  {
    tok = form_toks(buf, &cnt);
    r = userhandler(s, op, tok, cnt, buf, win);
    for (i=0;i<cnt;i++)
      free(tok[i]);
    free(tok);
  }
  else
    r = userhandler(s, op, NULL, 0, NULL, win);
  if (r == 2)
    return(1);
#endif /* USERCMDS */
  
  for (i=0;;i++)
  {
    if (in[i].func == NULL)
    {
      if (nvar("debug") == 1)
      {
        wp(win, ""RED"UNKNOWN OP: (0x%x - %i) |%s|"WHITE"\n", op, op, buf);
        drw(win);
      }
      return(1);
    }
    
    if (in[i].op == op)
    {
      if (strlen(buf))
      {
        tok = form_tokso(buf, &cnt);
        r = in[i].func(s, buf, tok, cnt, win);
        for (i=0;i<cnt;i++)
          free(tok[i]);
        free(tok);
      }
      else
        r = in[i].func(s, NULL, NULL, 0, win);
      return(r);
    }
  }
}

char **fxv(char **in, int cnt, int *rcnt)
{
  int i, k, j, n=cnt, c=0;
  char **ret;
  
  ret = (char **)malloc(4096);
  
  for (i=0,k=0,*rcnt=0;in[i];i++,k++)
  {
    if (*in[i] == '$' && strchr(in[i], '('))
    {
      ret[k] = (char *)malloc(512);
      ret[k+1] = NULL;
      memset(ret[k], 0, 512);
      do
      {
        if (!in[i])
          break;
        for (j=0;in[i][j];j++)
        {
          if (c < 0)
            c = 0;
          if (in[i][j] == '(')
            c++;
          else if (in[i][j] == ')')
            c--;
        }
        strcat(ret[k], in[i++]);
        strcat(ret[k], " ");
      } while (c);
      
      ret[k][strlen(ret[k])-1] = 0;
      ret[k] = (char *)realloc(ret[k], strlen(ret[k])+1);
      if (!in[i])
      {
        k++;
        break;
      }
      else
        i--;
    }
    else
    {
      ret[k] = strdup(in[i]);
      ret[k+1] = NULL;
    }
  }
  
  if (ret[k])
    k++;
  ret[k] = NULL;
  *rcnt = k;
  
  for (i=0;i<n;i++)
    free(in[i]);
  free(in);
  
  return(ret);
}

/* The functions parseout and parsenout, as well as the O_NAP_FUNC
 * functions (command handlers), use their return values to control
 * the control flow. The return values observe the following
 * convention: If the return value is
 *
 * 1: continue execution normally,
 *
 * 2: skip the remaining commands. If inside a loop, continue
 * with the next loop iteration. (This is like "continue" in C).
 * 
 * -4: break out of the most recent loop, then continue
 * normally. (This is like "break" in C).
 *
 * -3: stop execution and unwind to the top level.
 *
 * Here, "top level" means the enclosing call of parseout, normally
 * from winio.c:input(), but sometimes from timer.c or irc.c:inap.
 *
 * 2 is returned by dstop().
 * -4 is returned by dbreak().
 * -3 is returned by: ddone(), dquit(), dwhile() if the loop was
 * interrupted by a SIGINT, and sometimes by commands which produce
 * a fatal error (although this is not consistently done).
 *
 * Previously, the return values 0, -1, and -2 also existed, but they
 * were used in buggy ways and have been removed. In particular,
 * dquit() now uses a global variable to signal its intent to quit
 * nap, rather than doing it through a return value. -PS
 *
 **/

/* parseout: parse and execute the (user) command in buf. If it's an
   alias, expand it, otherwise let parseoutn handle it (or its various
   subparts individually?) */
int parseout(int s, char *buf, WINDOW *win)
{
  char **tok, *st, t[2048], *b, *p, nb[2048];
  alias_t *cur;
  int i, j=0, k, cnt, c = 0, rn, sw=0, js=0, x, y, f=0, bc = 0;
  
  if (*buf == '/')
  {
    b = strdup(buf+1);
    sw = 1;
  }
  else
    b = strdup(buf);
  
  while (1)
  {
    if (!b[j])
      break;
    
    if (sw && !bc)
    {
      memset(t, 0, sizeof(t));
      while (b[j] && b[j] == ' ')
        j++;
      for (k=0,c=0,f=j;b[j];j++)
      {
        if (b[j] == '{')
          c++;
        else if (b[j] == '}')
          c--;
        else if ((b[j] == '|' || b[j] == '\n' || !b[j]) && !c)
          break;

        t[k] = b[j];
        k++;
        if (c < 0)
          c = 0;
      }
    }
    else
    {
      strcpy(t, b);
      j = strlen(t);
    }
    
    if (*t == '#')
    {
      if (b[j])
        j++;
      continue;
    }
    
    tok = form_toks(t, &cnt);
    if (tok[0] && *tok[0] == '#')
    {
      if (b[j])
        j++;
      for (i=0;i<cnt;i++)
        free(tok[i]);
      free(tok);
      continue;
    }
    
    if (*tok && *tok[0] == '.')
    {
      bc = 1;
      strcpy(tok[0], tok[0]+1);
      tok[0][strlen(tok[0])] = 0;
      strcpy(t, t+1);
      t[strlen(t)] = 0;
      f++;
    }
    
    for (cur=alhead;;cur=cur->next)
    {
      if (!cur || !tok[0])
      {
        if (sw && tok[0] && !bc)
        {
          js = 0;
          tok = fxv(tok, cnt, &cnt);
          for (i=0;i<cnt;i++)
          {
            if (*tok[i] == '$' && tok[i][1] && tok[i][1] != '+')
            {
              p = dovars(tok[i]+1);
              if (p)
              {
                free(tok[i]);
                tok[i] = p;
                js = 1;
              }
            }
          } /* for i */
          for (i=0; i<cnt; i++)
          {
            if (*tok[i] == '$' && tok[i][1] == '+' && !tok[i][2])
            {
              if (i > 0 && tok[i-1] && tok[i+1])
              {
                for (x=1;tok[i-x]&&!(*tok[i-x]);x++);
                for (y=1;tok[i+y]&&!(*tok[i+y]);y++);
                if (!tok[i-x])
                  x--;
                if (!tok[i+y])
                  y--;
                tok[i] = (char *)realloc(tok[i], strlen(tok[i-x])+strlen(tok[i+y])+1);
                if (strcmp(tok[i-x], "$+"))
                  strcpy(tok[i], tok[i-x]);
                else
                  *tok[i] = 0;
                if (strcmp(tok[i+y], "$+"))
                  strcat(tok[i], tok[i+y]);
                *tok[i-x] = 0;
                *tok[i+y] = 0;
              }
              else
                *tok[i] = 0;
              js = 1;
            }
          } /* for i */
          if (js)
          {
            memset(nb, 0, sizeof(nb));
            for (i=0;tok[i];i++)
            {
              if (*tok[i])
              {
                strcat(nb, tok[i]);
                strcat(nb, " ");
              }
            }
            nb[strlen(nb)-1] = 0;
            p = ins(b, nb, j-strlen(t), j);
            j = f;
            free(b);
            b = p;
            break;
          }
        } /* if (sw && tok[0] && !bc) */
        if (b[j])
          j++;
        break;
      }
      if (!strcasecmp(tok[0], cur->nm))
      {
        st = doalias(cur, tok, cnt); /* doalias just does the substitution */
        if (!st)
        {
          wp(win, "%s* Error: insufficient parameters%s\n", RED, WHITE);
          drw(win);
          free(b);
          free(st);
          for (i=0;i<cnt;i++)
            free(tok[i]);
          free(tok);
          return(1);
        }
        p = ins(b, st, j-strlen(t), j);
        free(b);
        b = p;
        free(st);
        j = f;
        sw = 1;
        break;
      } /* if */
    } /* for (cur=alhead;;cur=cur->next) */
    
    for (i=0;i<cnt;i++)
      free(tok[i]);
    free(tok);
    
    if (j != f)
    {
      for (i=f,k=0;i<j;i++,k++)
        t[k] = b[i];
      t[k] = 0;
      if (sw)
      {
        if (t[strlen(t)-1] == '|' || t[strlen(t)-1] == '\n')
          t[strlen(t)-1] = 0;
        for (i=strlen(t)-1;t[i]==' '&&i>=0;i--)
          t[i] = 0;
      }
      rn = parseoutn(s, t, win);
      if (rn == -3 || rn == -4 || rn == 2 ) { /* see block comment above */
        free(b);
        return(rn);
      }
      if (bc == 1)
        bc = 0;
    } /* if (j != f) */
  } /* while (1) */
  
  free(b);
  return(1);
}

/* parseoutn: parse and execute a (user) command, but do not do alias
   expansion. Try whether it's a usercmd, irc stuff, or a regular
   built-in command from out[]. */
int parseoutn(int s, char *buf, WINDOW *win)
{
  int i, cnt, r = 0;
  char **tok;
  unsigned char *t = buf, f=0;
  
  tok = form_toks(t, &cnt);
  if (!tok[0])
  {
    for (i=0;i<cnt;i++)
      free(tok[i]);
    free(tok);
    return(1);
  }
  
#ifdef USERCMDS
  /* usercmd returns 0 on failure (i.e. no such command), else
     whatever the implemented function returns */
  r = usercmd(s, tok, cnt, t, win);
  if (r == 2)
  {
    for (i=0;i<cnt;i++)
      free(tok[i]);
    free(tok);
    return(1);
  }
#endif /* USERCMDS */
  
  if (ircmode && strcasecmp(tok[0], "server"))
  {
    if (!ircsock)
      return(1);
    if (!checkouts(ircsock, t, tok, cnt, win))
      f = 1;
    else
    {
      for (i=0;i<cnt;i++)
        free(tok[i]);
      free(tok);
      return(1);
    }
  }
  
  for (i=0;;i++)
  {
    if (out[i].func == NULL)
    {
      if (!r && !f)
      {
        wp(win, "%s* Unknown command: %s. Try /help.%s\n", RED, tok[0], WHITE);
        drw(win);
	return(-3);
      }
      else if (f)
        ssock(ircsock, "%s\n", t);
      for (i=0;i<cnt;i++)
        free(tok[i]);
      free(tok);
      return(1);
    }
    
    if ((!f || out[i].a) && !strcasecmp(tok[0], out[i].nm))
    {
      r = out[i].func(s, t, tok, cnt, win);
      for (i=0;i<cnt;i++)
        free(tok[i]);
      free(tok);
      return(r);
    }
  }
 
  return(r);
}

/* return a unique connection name for an upload or a download
   connection. Upload connections must start with "u" and download
   connections with "d" (the functions in scheck.c rely on this).
   Never use the same name twice. The result of this function is a
   pointer to a static string, which will be overwritten with the next
   call. t==1 for uploads, t==0 for downloads. */
char *gnum(int t)
{
  static int count[2] = { 0, 0 };
  static const char *template[2] = {"d %i", "u %i"};
  static char buf[10];

  sprintf(buf, template[t], count[t]++);

  return buf;
}

/* return the part of the filename with all the (unix or dos)
   directory components removed. Note: we return a pointer into the
   original string */
char *basename(char *fn) {
  char *p;

  p = strrchr(fn, '/');
  if (!p)
    p = strrchr(fn, '\\');
  if (!p)
    p = fn;
  else 
    p++;

  return p;
}


/* skip n space-delimited tokens, and return a copy of the remainder
   of the string. */
char *cstr(char *str, int n)
{
  int i=0, c;
  
  if (ctbuf)
    free(ctbuf);
  
  while (str[i] == ' ' && str[i])
    i++;
    
  if (!str[i])
  {
    ctbuf = strdup("");
    return(ctbuf);
  }
  
  for (c=0;str[i]&&c!=n;i++)
  {
    if (str[i] == ' ')
    {
      c++;
      while (str[i] == ' ' && str[i])
        i++;
      if (c == n)
        break;
      if (!str[i])
        break;
    }
  }
  
  if (!str[i])
    ctbuf = strdup("");
  else
    ctbuf = strdup(str+i);
  
  return(ctbuf);
}

char *cspstr(char **tok, int cnt, int beg)
{
  int i;
  
  if (ctbuf)
    free(ctbuf);
  
  if (beg > cnt)
  {
    ctbuf = strdup("");
    return(ctbuf);
  }
  
  ctbuf = (char *)malloc(4096);
  memset(ctbuf, 0, 4096);
  
  for (i=beg;i<cnt;i++)
  {
    strcat(ctbuf, tok[i]);
    strcat(ctbuf, " ");
  }
  
  ctbuf[strlen(ctbuf)-1] = 0;
  ctbuf = (char *)realloc(ctbuf, strlen(ctbuf)+1);
  
  return(ctbuf);
}

/* delete an upload from the upload list. Frees all the resources
   associated with this upload, including deleting any timer set for
   it. Also delete the item's socket, if any. Return -1 if item was
   not in list, else 0. */
int dupload(upload_t *task) 
{ 
  list_unlink(upload_t, up, task);

  if (!task) {  /* ?? not found */
    return(-1);
  }

  free(task->nick);
  free(task->rfn);
  free(task->fn);
  free(task->lfn);
  if (task->state == CONNECTING || task->state == CONNECTING1 || task->state == IN_PROGRESS) {
    if (task->sk) 
      delsock(task->sk->fd);
  }
  if (task->state == IN_PROGRESS) {
    if (task->f)
      fclose(task->f);
  }    
  free(task);
  return(0);
}

/* delete an element from the download task lists. Also close this
   download's file (if any), move the file to the appropriate place or
   delete turds. Also delete the item's socket, if any. Return -1 if
   item was not in list, 0 on success */
int ddownload(WINDOW *win, download_t *task)
{
  list_unlink(download_t, down, task);

  if (!task)  /* ?? task was not in the list */
    return(-1);

  if (task->state == IN_PROGRESS) {
    interrupt_download(win, task);
    if (task->sk)
      delsock(task->sk->fd);
  }

  switch (task->state) {

  case CONNECTING:
    if (task->sk)
      delsock(task->sk->fd);
    /* fall through to next case*/

  case WAITING:
    free(task->check);
    /* fall through to next case */
    
  default:
    free(task->nick);
    free(task->rfn);
    free(task->fn);
    free(task);
  }

  return(0);
}

/* move a task from IN_PROGRESS to either INCOMPLETE or COMPLETE,
   depending on which is the case. Free all resources that would be
   inappropriate for a stopped task. I.e., close the download's file
   (if any), move the file to the appropriate place or delete
   turds. However, do not delete the item's socket. Return 1 if an
   incomplete file was saved, 2 if it was a turd, 3 if a complete file
   was saved. */
int interrupt_download(WINDOW *win, download_t *task)
{
  int turdsize;
  int ret = 0;

  rqueued_hook(task->nick);

  if (task->f) {
    fclose(task->f);
    turdsize = getval("turdsize") ? nvar("turdsize") : TURDSIZE;
    
    if (task->pos == task->size) {      /* completed download */
      move_to_downloaddir(win, task);
      task->state = COMPLETE;
      task->d_time = time(0);
      ret = 3;
    } else if ((ssize_t)(task->pos) <= turdsize) {   /* turd */
      remove_turd(win, task, turdsize);
      task->state = INCOMPLETE;
      task->d_time = time(0);
      ret = 2;
    } else {                      /* prepare incomplete for later resume */
      mark_incomplete(win, task);
      task->state = INCOMPLETE;
      task->d_time = time(0);
      ret = 1;
    }
    }
  free(task->lfn);
  free(task->check);
  
  return(ret);
}


/* move a completed download from the incomplete directory to the
   download directory, renaming it as appropriate. Do this silently
   unless there is an error */
void move_to_downloaddir(WINDOW *win, download_t *task) {
  char *path, *newfullname, *newname;
  int n, r;
  char *q;
  struct stat st;

  /* figure out the filename to move it to. */
  path = getval("download");
  if (!path) {
    path = ".";
  }
  path = home_file(path);
  if (*path && path[strlen(path)-1]=='/') {
    /* this is purely for cosmetic reasons */
    path[strlen(path)-1]=0;
  }
  
  newname = strdup(task->fn);
  newfullname = NULL;
  msprintf(&newfullname, "%s/%s", path, task->fn);

  /* check if the file already exists. */
  r = stat(newfullname, &st);
  if (r==0) {
    char *base, *suffix;
    base = strdup(task->fn);
    q = strrchr(base, '.');
    if (q) {
      suffix = strdup(q);
      *q = 0;
    } else {
      suffix = strdup("");
    }
    for (n=1; n<=100; n++) {
      /* the file exists, we must choose a different name */
      msprintf(&newname, "%s-%d%s", base, n, suffix);
      msprintf(&newfullname, "%s/%s", path, newname);
      r = stat(newfullname, &st);
      if (r!=0)
	break;
    }
    if (r==0) {
      wp(win, ""RED"* Error: ran out of temporary filenames. Unable to move \"%s\" to download directory."WHITE"\n", task->lfn);
      drw(win);
      free(newfullname);
      free(newname);
      newfullname=NULL;
      newname=NULL;
    } else {
      wp(win, "* File \"%s\" exists. Saving new file as \"%s\" instead.\n", task->fn, newname);
      drw(win);
    }
    free(base);
    free(suffix);
  }
  free(path);

  if (newfullname) {
    r = rename(task->lfn, newfullname);
    if (r==-1) {
      wp(win, ""RED"* Error: could not move \"%s\" to \"%s\": %s"WHITE"\n", task->lfn, newfullname, sys_errlist[errno]);
      drw(win);
    }
  }
  
  free(newfullname);
  free(newname);
}

void remove_turd(WINDOW *win, download_t *task, int turdsize) {
  int r;
  r = unlink(task->lfn);
  if (r==-1) {
    wp(win, ""RED"* Failed to remove turd \"%s\": %s"WHITE"\n",
       task->lfn, sys_errlist[errno]);
    drw(win);
  } else {
    wp(win, "* Removed turd \"%s\" (%d bytes)\n", task->fn, task->pos);
    drw(win);
  }
}

/* add some information to an incomplete file, so that we can later do
   a proper "resume" on that file (if that is ever implemented). The
   idea is to append to the end of the file one or several MD5 hashes.
   If a remote client's (valid) MD5 hash was reported by the server,
   we add that. If the file is long enough to calculate our own MD5
   hashes, then we add them too - two of them in fact, a hash of the
   first 299008 bytes, and one of the first 300032 bytes, since it is
   unclear which one a remote client is going to be using. (We add
   these hashes so that we don't have to do it each time we are
   looking for a match for a potential resume */

/* some care needs to be taken because most OpenNap servers do not
   send valid hashes; rather they send the string 0000...(32 times) as
   the "checksum". Of course one should not base a resume on such a
   hash. */

/* In the future, when downloading a file, we will check if any of the
   hashes in the incomplete directory match, and if so, treat it as a
   resume, if not, not. In the far future, we'll also be able to
   request a resume for a specific incomplete file and (gasp) maybe
   even have an autoresume command. */
void mark_incomplete(WINDOW *win, download_t *task) {
  mhdr_t  *m;

  if (task->pos >= 300032) {
    /* we have a chance to calculate a valid md5 ourselves. */
    m = gethdr(task->lfn, 292);
    if (m->sz1 >= 300032) {
      tagincompletefile(task->lfn, m->check);
    }
    if (m->sz1 >= 299008) {
      free(m);
      m = gethdr(task->lfn, 293);
      tagincompletefile(task->lfn, m->check);
    }
    free(m);
  }
  if (task->check && strncmp(task->check, "00000000", 8)) {
    /* have a valid checksum from remote client */
    tagincompletefile(task->lfn, task->check);
  }
}

/* attach a tag at the end of a file which encapsulates the given
 * string. We use this to attach one or several md5 checksums to
 * incomplete files, for convenient reference when resuming later. 
 *
 * The format of the tag is:
 * < INCTAG_START size 0 string 0 size INCTAG_END >,
 * where INCTAG_START and INCTAG_END are defined in defines.h, and
 * size is a 4-byte unsigned integer in big-endian byte order equal to
 * the size of the entire tag.  Note that the string is terminated by
 * 0 on both ends, and we give the size twice; this way, the tag is
 * conveniently scannable both forward and backward.
 **/

void tagincompletefile(char *fn, char *str) {
  int size;
  char sz[4];
  FILE *f;

  size = strlen(str) + strlen(INCTAG_START) + strlen(INCTAG_END) + 10;
  sz[0] = (size >> 24) & 0xff;
  sz[1] = (size >> 16) & 0xff;
  sz[2] = (size >> 8) & 0xff;
  sz[3] = (size >> 0) & 0xff;
  
  f = fopen(fn, "ab");
  if (!f) {
    return;  /* fail silently, since there is not much we can do */
  }
  fputs(INCTAG_START, f);
  fwrite(sz, 4, 1, f);
  fputc(0, f);
  fputs(str, f);
  fputc(0, f);
  fwrite(sz, 4, 1, f);
  fputs(INCTAG_END, f);
 
  fclose(f);
}

void adduser(chans_t *c, char *nm, int scount, unsigned char conn)
{
  user_t *cur, *cur1 = NULL;
  
  if (!c)
    return;
  
  if (!c->users)
  {
    c->users = (user_t *)malloc(sizeof(user_t));
    cur = c->users;
  }
  else
  {
    for (cur=c->users;cur;cur=cur->next)
    {
      if (!strcasecmp(cur->nm, nm))
        return;
      cur1 = cur;
    }
    cur = (user_t *)malloc(sizeof(user_t));
    cur1->next = cur;
  }
  
  cur->nm = strdup(nm);
  cur->addr = NULL;
  cur->conn = conn;
  cur->scount = scount;
  cur->flag = 0;
  cur->next = NULL;
}

/* delete a user from a channel */
void deluser(chans_t *c, char *nm)
{
  user_t *cur, *cur1 = NULL;
  
  for (cur=c->users;;cur=cur->next)
  {
    if (!cur)
      return;
    if (!strcasecmp(cur->nm, nm))
      break;
    cur1 = cur;
  }
  
  if (cur1)
    cur1->next = cur->next;
  else if (cur->next)
    c->users = cur->next;
  else
    c->users = NULL;
  
  free(cur->nm);
  if (cur->addr)
    free(cur->addr);
  free(cur);
}

/* find user in a channel or return NULL if not found */
user_t *finduser(chans_t *c, char *nm)
{
  user_t *cur;

  if (!nm)
    return(NULL);

  for (cur=c->users;;cur=cur->next)
  {
    if (!cur)
      return(NULL);
    if (!strcasecmp(cur->nm, nm))
      return(cur);
  }
}

/* a callback function for timer to run a user command */
void timed_command(void *data) {
  char *cmd = (char *)data;
  int s;
  sock_t *sk;
  
  wp(wchan, ""BRIGHT(BLUE)"* Timed event: "WHITE"%s\n", cmd);
  drw(wchan);
  
  sk = findsock("server");
  s = sk ? sk->fd : -1;

  parseout(s, cmd, wchan);
}

/* ---------------------------------------------------------------------- */
/* begin server commands (react to messages sent by the server) */

/* 0x193 <channel> <nick> <text> [SERVER]: public message */
/* a message arrived. Color code it based on whether it was from us,
   and whether it is on the current channel or some other channel. */
I_NAP_FUNC(ssay)
{
  char *hilit;

  if (!(recent = findchan(chanl, tok[0])))
    return(1);  /* ?? we don't seem to be on that channel */

  /* highlight our own messages in MAGENTA, all others in BLUE */
  if (!strcasecmp(info.user, tok[1])) {
    hilit = MAGENTA;
  } else {
    hilit = BLUE;
  }

  if (recent == curchan || wmode)
  {
    wp(win, ""BOLD"%s<%s%s"BOLD"%s>%s %s\n", hilit, WHITE, tok[1], hilit, WHITE, str+strlen(tok[1])+strlen(tok[0])+2);
  }
  else
  {
    wp(win, "%s<%s%s%s:%s%s%s>%s %s\n", hilit, WHITE, tok[1], DARK, WHITE, tok[0], hilit, WHITE, str+strlen(tok[1])+strlen(tok[0])+2);
  }
  drw(win);
  recent = NULL;
  
  return(1);
}

I_NAP_FUNC(stell)
{
  char *autoreply;
  static char *lastreply = NULL;  /* nick of last user we autoreplied to */

  recent = findquery(chanl, tok[0]);

  wp(win, "%s* [%s%s%s]%s %s\n", BRIGHT(GREEN), WHITE, tok[0], BRIGHT(GREEN), WHITE, str+strlen(tok[0])+1);
  drw(win);
  recent = NULL;
    
  if ((autoreply = getval("autoreply")) != NULL) {
    /* do not autoreply to the same user twice in a row, to discourage
       infinite chats between two autoreplying clients */
    if (!(lastreply && !strcmp(lastreply, tok[0]))) {
      autoreply = strdup(autoreply);
      sendpack(s, F_TELL, "%s %s", tok[0], fixquotes(autoreply));
      wp(win, "%s* --> (%s%s%s)%s %s\n", GREEN, WHITE, tok[0], GREEN, WHITE, autoreply);
      drw(win);
      free(autoreply);
      free(lastreply);
      lastreply = strdup(tok[0]);
    }
  }

  return(1);
}

/* 406 (0x196) join message (user has joined channel) [SERVER]
   <channel> <user> <sharing> <link-type> */
I_NAP_FUNC(sjoin)
{
  if (!(recent = findchan(chanl, tok[0])))
    return(1);
  if (!strcasecmp(tok[1], info.user))
    wp(win, "\n");
  wp(win, "%s* %s (%s%s%s) [%ssharing %i songs%s] has joined %s.%s\n", BRIGHT(GREEN), tok[1], WHITE, atoi(tok[3])>10?"Unknown":conns[atoi(tok[3])], BRIGHT(GREEN), WHITE, atoi(tok[2]), BRIGHT(GREEN), tok[0], WHITE);
  drw(win);
  adduser(recent, tok[1], atoi(tok[2]), atoi(tok[3]));
  recent = NULL;
  
  return(1);
}

/* 407 (0x197)     user parted channel [SERVER]
   <channel> <nick> <sharing> <linespeed>  */
I_NAP_FUNC(spart)
{
  if (!(recent = findchan(chanl, tok[0])))
    return(1);
  wp(win, "%s* %s (%s%s%s) [%ssharing %i songs%s] has left %s.%s\n", GREEN, tok[1], WHITE, conns[atoi(tok[3])], GREEN, WHITE, atoi(tok[2]), GREEN, tok[0], WHITE);
  drw(win);
  deluser(recent, tok[1]);
  recent = NULL;
  
  return(1);
}

I_NAP_FUNC(stopic)
{
  if (!(recent = findchan(chanl, tok[0])))
    return(1);
  wp(win, "%s* Topic for %s: %s%s\n", YELLOW, tok[0], str+strlen(tok[0])+1, WHITE);
  drw(win);
  
  if (recent->topic)
    free(recent->topic);
  recent->topic = strdup(str+strlen(tok[0])+1);
  if (recent == curchan)
    tind = 0;
  recent = NULL;
  
  return(1);
}

I_NAP_FUNC(sjchan)
{
  chans_t *cur, *cur1=NULL;

  if (!chanl)
  {
    chanl = (chans_t *)malloc(sizeof(chans_t));
    cur = chanl;
  }
  else
  {
    for (cur=chanl;cur!=NULL;cur=cur->next)
      cur1 = cur;
    cur = (chans_t *)malloc(sizeof(chans_t));
    cur1->next = cur;
  }
  
  cur->nm = strdup(tok[0]);
  cur->users = NULL;
  cur->q = 0;
  cur->topic = NULL;
  cur->l = 0;
  cur->flag = 0;
  cur->key = NULL;
  cur->p = 0;
  cur->next = NULL;
  curchan = cur;
  
  tin = -1;

  return(1);
}

/* 0x194 (404) server error message */
I_NAP_FUNC(snchan)
{
  if (dwi && ((!strcasecmp(tok[0], "information") && !strcasecmp(tok[2], dwi)) || (!strcasecmp(tok[0], "user") && !strcasecmp(tok[1], dwi))))
  {
    wp(win, "* Unable to resolve %s\n", dwi);
    drw(win);
    free(dwi);
    dwi = NULL;
  }

  wp(win, "%s* Server: %s%s\n", RED, str, WHITE);
  drw(win);
  
  if (!strcmp(str, "You are now cloaked.") || !strcmp(str, "You are cloaked."))
    cloaked = 1;
  else if (!strcmp(str, "You are no longer cloaked."))
    cloaked = 0;
  return(1);
}

I_NAP_FUNC(snotice)
{
  wp(win, "%s* %s%s\n", YELLOW, (str)?str:"", WHITE);
  drw(win);
  return(1);
}

I_NAP_FUNC(sscount)
{
  scount.libraries = atoi(tok[0]);
  scount.songs = atoi(tok[1]);
  scount.gigs = atoi(tok[2]);
  return(1);
}

/* 604 (0x25c)	whois response [SERVER]
 *
 *	<nick> "<user-level>" <time> "<channels>" "<status>" <shared>
 *	<downloads> <uploads> <link-type> "<client-info>" [ <total
 *	downloads> <total_uploads> <ip> <connecting port> <data port>
 *	<email> ] 
 */
I_NAP_FUNC(swhois)
{
  int hr, min, sec, t;
  struct hostent *d1, *d2;
  struct sockaddr_in in;
  
  if (dwi && !strcasecmp(dwi, tok[0]))
  {
    free(dwi);
    dwi = NULL;
    if (num < 13)
      return(1);
    if (fork())
      return(1);
      
    d1 = gethostbyname(tok[12]);
    if (!d1)
    {
      ssock(ipcs[1], "* Unable to resolve %s\n", tok[0]);
      exit(1);
    }
    
    memcpy(&in.sin_addr, d1->h_addr, d1->h_length);
    d2 = gethostbyaddr((char *)&in.sin_addr, d1->h_length, AF_INET);
    if (!d2)
    {
      ssock(ipcs[1], "* %s is %s\n", tok[0], inet_ntoa(in.sin_addr));
      exit(1);
    }
     
    ssock(ipcs[1], "* %s is %s [%s]\n", tok[0], d2->h_name, inet_ntoa(in.sin_addr));
    
    exit(1);
  }
  
  t = atoi(tok[2]);
  min = t/60;
  hr = min/60;
  sec = t%60;
  
  while (min >= 60)
    min-=60;

  wp(win, "%s ______________________%s\n", DARK, WHITE);
  wp(win, "%s|%s User: %s\n", DARK, WHITE, tok[0]);
  if (!strcasecmp(tok[0], "Ignitor"))
    wp(win, "%s|%s Class: Coder (%s)\n", DARK, WHITE, tok[1]);
  else if (!strcasecmp(tok[0], "nYtr0"))
    wp(win, "%s|%s Class: MoM (%s)\n", DARK, WHITE, tok[1]);
  else
    wp(win, "%s|%s Class: %s\n", DARK, WHITE, tok[1]);
  wp(win, "%s|%s Line: %s\n", DARK, WHITE, conns[atoi(tok[8])]);
  wp(win, "%s|%s Time: %i h %i m %i s\n", DARK, WHITE, hr, min, sec);
  wp(win, "%s|%s Channels: %s\n", DARK, WHITE, tok[3]);
  wp(win, "%s|%s Status: %s\n", DARK, WHITE, tok[4]);
  wp(win, "%s|%s Shared: %s\n", DARK, WHITE, tok[5]);
  wp(win, "%s|%s Client: %s\n", DARK, WHITE, tok[9]);
  wp(win, "%s|%s %s Downloading, %s Uploading\n", DARK, WHITE, tok[6], tok[7]);
  if (num > 10)
  {
    wp(win, "%s|%s %s Downloads, %s Uploads\n", DARK, WHITE, tok[10], tok[11]);
    wp(win, "%s|%s Address: %s\n", DARK, WHITE, tok[12]);
    wp(win, "%s|%s Data port: %s\n", DARK, WHITE, tok[14]);
  }
  drw(win);
  
  return(1);
}

I_NAP_FUNC(soff)
{
  time_t pt;
  char *t;
  
  if (dwi && !strcasecmp(dwi, tok[0]))
  {
    wp(win, "* Unable to resolve %s\n", tok[0]);
    drw(win);
    free(dwi);
    dwi = NULL;
  }
  
  pt = atol(tok[2]);
  t = ctime(&pt);
  t[strlen(t)-1] = '\0';
  t[strlen(t)] = 0;
  
  wp(win, "%s ______________________%s\n", DARK, WHITE);
  wp(win, "%s|%s User: %s\n", DARK, WHITE, tok[0]);
  if (!strcasecmp(tok[0], "Ignitor"))
    wp(win, "%s|%s Class: Coder (%s)\n", DARK, WHITE, tok[1]);
  else if (!strcasecmp(tok[0], "nYtr0"))
    wp(win, "%s|%s Class: MoM (%s)\n", DARK, WHITE, tok[1]);
  else
    wp(win, "%s|%s Class: %s\n", DARK, WHITE, tok[1]);
  wp(win, "%s|%s Last On: %s\n", DARK, WHITE, t);
  wp(win, "%s|%s Status: Offline\n", DARK, WHITE);
  drw(win);
  
  return(1);
}

/* 620 (0x26c) remote client queue limit reached:
 * <nick> "<filename>" <filesize> <digit> */
I_NAP_FUNC(sqlimit)
{
  download_t *task;

  /* find matching REQUESTED or RRQUEUED item in the download list */
  list_find(task, down, !strcasecmp(task->nick, tok[0]) && !strcasecmp(task->rfn, tok[1]) && (task->state == REQUESTED || task->state == RRQUEUED));

  if (!task) {  /* we didn't request this item, so we don't care */
    return(1);
  }

  if (task->state == REQUESTED) {
    wp(win, "* Remotely queued \"%s\" from %s.\n", task->fn, task->nick, tok[3]);
    drw(win);
  }

  task->state = RQUEUED;
  task->r_time = time(0);

  return(1);
}

/* 206 (0xce) get error: <nick> "<filename>"  */
I_NAP_FUNC(snget)
{
  download_t *task;
  char *fn;

  /* find matching REQUESTED or RRQUEUED item in the download list */
  list_find(task, down, !strcasecmp(task->nick, tok[0]) && !strcasecmp(task->rfn, tok[1]) && (task->state == REQUESTED || task->state == RRQUEUED));

  if (!task) {  /* we didn't request this item, so we don't care */
    return(1);
  }

  /* calculate short filename */
  fn = basename(tok[1]);

  wp(win, "%s* Unable to get \"%s\" from %s (206 get error)%s\n", RED, \
      fn, tok[0], WHITE);
  drw(win);
  
  task->state = FAILED;
  task->d_time = time(0);

  return(1);
}

/* 609 (0x261) accept failed: <nick> "<filename>" */
I_NAP_FUNC(snacc)
{
  download_t *task;
  char *fn;

  /* find matching REQUESTED or RRQUEUED item in the download list */
  list_find(task, down, !strcasecmp(task->nick, tok[0]) && !strcasecmp(task->rfn, tok[1]) && (task->state == REQUESTED || task->state == RRQUEUED));

  if (!task) {  /* we didn't request this item, so we don't care */
    return(1);
  }

  /* calculate short filename */
  fn = basename(tok[1]);

  wp(win, "%s* Unable to get \"%s\" from %s (609 accept failed)%s\n", RED, \
      fn, tok[0], WHITE);
  drw(win);
  
  task->state = FAILED;
  task->d_time = time(0);

  return(1);
}

I_NAP_FUNC(nothing)
{
  return(1);
}

/* 401 (0x191)	part channel [CLIENT, SERVER]
   <channel-name> */
I_NAP_FUNC(smpart)
{
  chans_t *pt, *cur, *cur1=NULL;

  pt = findchan(chanl, tok[0]);
  
  if (!pt)
    return(1);
  
  wp(win, "%s* Left channel %s%s\n", GREEN, pt->nm, WHITE);
  drw(win);
  
  for (cur=chanl;cur!=pt;cur=cur->next)
    cur1 = cur;
  
  if (cur1)
    cur1->next = cur->next;
  else if (cur->next)
    chanl = cur->next;
  else
  {
    chanl = NULL;
    curchan = NULL;
  }
  if (curchan && !strcmp(pt->nm, curchan->nm))
    upchan(curchan);
  
  while (pt->users)
    deluser(pt, pt->users->nm);

  if (pt->topic)
    free(pt->topic);
  free(pt->nm);
  if (pt->key)
    free(pt->key);
  free(pt);
  
  if (wmode)
  {
    dscr(win);
    drw(win);
  }
  
  return(1);
}

/* 408 (0x198)    channel user list entry [SERVER] 
   <channel> <user> <sharing> <link-type> */
/* 825 (0x339) user list entry [SERVER]
   <channel> <user> <files shared> <speed> */
I_NAP_FUNC(suser)
{
  int n, sp;
  const char *c;
  
  if (num < 4)
    return(1);
  
  n = 16-strlen(tok[1]);
  sp = atoi(tok[3]);
  
  if (sp <= 3)
    c = RED;
  else if (sp <= 6)
    c = BRIGHT(YELLOW);
  else
    c = GREEN;
  
  recent = findchan(chanl, tok[0]);
  
  adduser(recent, tok[1], atoi(tok[2]), sp);
  
  if (tin == -1)
  {
    tin = 0;
    wp(win, "%s* Users:%s\n", YELLOW, WHITE);
  }

  if (tin == 4)
  {
    tin = 0;
    wp(win, "\n");
  }
  wp(win, "%s[%s%s", c, WHITE, tok[1]);
  for (;n>=0;n--)
    wp(win, " ");
  wp(win, "%s]%s", c, WHITE);
  
  recent = NULL;
  
  tin++;
  
  return(1);
}

/* server sends a 0x33e packet to end a list of users in a channel */
I_NAP_FUNC(snend)
{
  tin = -1;
  wp(win, "\n");
  drw(win);
  
  return(1);
}

/* 607 (0x25f) upload request: <nick> "<filename>" <speed>. Note: not
   all servers send the <speed>.  */
I_NAP_FUNC(sfreq)
{
  char *lfn, *fn;
  upload_t *task;
  int i, r;

  lfn = strdup(tok[1]);
  for (i=0;i<strlen(lfn);i++)
    if (lfn[i] == '\\')
      lfn[i] = '/';
  fn = basename(lfn);
  
  /* check all the different reasons we might reject this upload */

  /* first check that we are really sharing the requested file. This
     check was missing from previous versions of nap (pre 1.4.4-ps5).
     This was a major security hole, since it would allow a remote
     client to download any file, whether it was shared or not */

  if (!isshared(lfn, info.shared_filename)) {
    sendpack(s, F_GFR, "%s", str);  /* see comment below */
    wp(win, "%s* File \"%s\" was requested by %s, but is not shared!%s\n", RED, lfn, tok[0], WHITE);
    drw(win);
    free(lfn);
    return(1);
  }

  /* if /tquit is in effect, don't accept any new uploads */
  if (quit_after_transfers) {
    sendpack(s, F_GFR, "%s", str);
    wp(win, ""RED"* Not accepting new upload of \"%s\" to %s because /tquit is in effect"WHITE"\n", fn, tok[0]);
    drw(win);
    free(lfn);
    return(1);
  }

  list_find(task, up, !strcasecmp(task->nick, tok[0]) && !strcasecmp(task->rfn, tok[1]) && !STOPPED(task->state));

  if (task != NULL)
  {
    /* we are already uploading the file to nick */
    /* send a F_QLIMIT or send nothing ? F_QLIMIT would be
     * inappropriate.  If the download is not yet IN_PROGRESS, there
     * is no harm in sending an accept to the server (but not adding a
     * duplicate to the list) - maybe the remote user already deleted
     * the first download (which we would not know). -PS */
    if (task->state != IN_PROGRESS) {
      sendpack(s, F_GFR, "%s", str);
    } else {
      wp(win, ""RED"* Already uploading \"%s\" to %s (duplicate request)"WHITE"\n", fn, tok[0]);
      drw(win);
    }
    free(lfn);
    return(1);
  }

  /* see if global or per-user upload limit exceeded */
  r = upload_limit_reached(tok[0]);
  if (r) {
    /* sendpack(s, F_GFR, "%s", str); */ /* why is this here ? */ 
    /* apparently the official napster server does not forward QLIMIT
       packets to the requestor, thus keeping them waiting
       forever. However, this is too bad, I don't think we should use
       F_GFR here. Do the opennap servers react differently? ### check
       this another time. */
    sendpack(s, F_QLIMIT, "%s \"%s\" %i", tok[0], tok[1], r==2 ? nvar("maxuploads") : nvar("maxupuser"));
    if (nvar("showtoomanyuploads")==1) {
      wp(win, "%s* Too many uploads %s(can't send \"%s\" to %s)%s\n", RED, r==2 ? "" : "for this user ", tok[1], tok[0], WHITE);
      drw(win);
    }
    free(lfn);
    return(1);
  }

  /* otherwise, accept the upload. Create an item in the upload list */
  task = (upload_t *)malloc(sizeof(upload_t));
  task->state = WAITING;
  task->nick = strdup(tok[0]);
  task->rfn = strdup(tok[1]);
  task->fn = strdup(fn);
  task->lfn = lfn;  /* already allocated with strdup */
  task->linespeed = num>=3 ? atoi(tok[2]) : 0;
  task->c_time = time(0);

  /* add it to the upload list */
  list_append(upload_t, up, task);

  /* sendpack(s, F_SX, "%s", task->nm); */ /* why send a link speed query? */
  /* accept upload request: <nick> "<filename>" */
  sendpack(s, F_GFR, "%s", str); 

  return(1);
}


/* link speed response: <nick> <linespeed> */
I_NAP_FUNC(sry)
{
  return(1);
}

/* 501 (0x1f5) alternate download ack: 
 * <nick> <ip> <port> "<filename>" <md5> <speed>
 * we are being asked to UPLOAD a file to a remote client (presumably
 * because we are firewalled and cannot accept incoming connections) */
I_NAP_FUNC(ssf)
{
  upload_t *task;
  sock_t *sk=NULL;
  int k;
  struct sockaddr_in dst;
  
  /* find the corresponding upload */
  list_find(task, up, task->state == WAITING && !strcasecmp(task->nick, tok[0]) && !strcasecmp(task->rfn, tok[3]));

  if (!task) { 
    /* we were not waiting for this upload. Ignore silently. */
    /* note: if in the future, we choose to accept 501 without prior 607, we
       must then also do the security checks; check the file is actually
       shared, /tquit is not active, etc */
    return(1);
  }
  
  /* contact remote client */

  /* note: the integer which represents the IP address is sent by the
   * napster protocol in "twisted" form. For instance, 1.0.0.0 = 1,
   * 0.1.0.0 = 256, etc.  If this integer is stored in its native
   * format on a given host, it will end up representing the IP
   * address in *non-host* byte order. On little-endian machines, such
   * as x86 and Alpha, this happens to be the correct network
   * byteorder (=big-endian), but on big-endian machines, it is
   * not. Thus, the correct thing to do is to first swap the byte
   * order to convert from *non-host* to *host* byte order, then apply
   * htonl(3) to convert from *host* to *network* byte order. This
   * causes some unnecessary work on little-endian machines, but
   * to heck with that. -PS */

  dst.sin_addr.s_addr = htonl(swapl(strtoul(tok[1], (char **)NULL, 10)));
  dst.sin_port = htons(atoi(tok[2]));
  dst.sin_family = AF_INET;
  
  k = socket(AF_INET, SOCK_STREAM, 0);
  
  /* we connect in a child process, so as not to hang */

  if (fork()==0) {     /* child process */
    connect(k, (struct sockaddr *)&dst, sizeof(dst));
    exit(1);
  }
  
  /* parent process */

  sk = addsock(k, gnum(1), S_R, initsend);

  /* update task */
  task->state = CONNECTING;
  task->sk = sk;

  /* connect task to socket */
  
  sk->utask = task;
  
  return(1);
}

/* 0xcc: got a download ack from the server. It is a string of the form:
 * <nick> <ip> <port> "<filename>" <md5> <linespeed>
 *   0      1     2        3          4       5       <-- token numbers */
I_NAP_FUNC(sget)
{
  char *fn;
  download_t *task;
  int rport;
  int r, k;
  sock_t *sk;

  /* find matching REQUESTED or RRQUEUED item in the download list */
  list_find(task, down, !strcasecmp(task->nick, tok[0]) && !strcasecmp(task->rfn, tok[3]) && (task->state == REQUESTED || task->state == RRQUEUED));

  if (!task)  /* we received an ack for an item we had not requested. */
    return(1);  /* ignore it silently. */
  
  /* calculate short filename */
  fn = basename(tok[3]);
  
  rport = atoi(tok[2]);
  
  if (!rport && !info.port)
  {
    wp(win, "%s* Unable to get \"%s\" from %s because both parties are " \
        "firewalled%s\n", RED, fn, tok[0], WHITE);
    drw(win);
    task->state = FAILED;
    task->d_time = time(0);
    return(1);
  }

  task->check = strdup(tok[4]);
  task->linespeed = atoi(tok[5]);
  task->addr.sin_family = AF_INET;
  task->addr.sin_port = htons(rport);
  task->addr.sin_addr.s_addr = htonl(swapl(strtoul(tok[1], 0, 10)));

  if (!rport)   /* firewalled client, send push request and go WAITING */
  {
    sendpack(s, F_DSF, "%s \"%s\"", tok[0], tok[3]);
    task->state = WAITING;
    return(1);
  }

  /* else connect to the remote client and go CONNECTING */

  /* sendpack(s, F_SX, "%s", tok[0]); */ /* why send a link speed query? */

  k = socket(AF_INET, SOCK_STREAM, 0);
    
  if (fork()==0)             /* child process */
  {
    r = connect(k, (struct sockaddr *)&task->addr, sizeof(task->addr));
    /* if there is an error, we do not need to handle it here; 
       it will be handled in initget. */
    exit(1);
  }
  
  /* parent process */

  if (task->state == RRQUEUED) {
    wp(win, "* Getting \"%s\" from %s\n", task->fn, task->nick);
    drw(win);
  }

  sk = addsock(k, gnum(0), S_R, initget);
  
  sk->dtask = task;
  task->sk = sk;
  task->state = CONNECTING;
  
  return(1);
}

/* Opens a new file in the incomplete directory || download directory
   || cwd whose name is based on fn.  The file is renamed if a file
   of the exact name already exists. Returns the file *f and its name
   *lfn. Returns 0 on success, -1 on failure with errno set, or -2 on
   failure if we ran out of alternative filenames. */

int opendownloadfile(char *fn, FILE **f, char **lfn) {
  char *path;
  int r, n;
  struct stat st;
  char *suffix = getval("incompletesuffix");

  if (!suffix) {
    suffix = ".incomplete";
  }

  path = getval("incomplete");
  if (!path) {
    path = getval("download");
  }
  if (!path) {
    path = ".";
  }
  path = home_file(path);
  if (*path && path[strlen(path)-1]=='/') {  /* cosmetic */
    path[strlen(path)-1]=0;
  }
 
  *lfn = NULL;
  msprintf(lfn, "%s/%s%s", path, fn, suffix);

  /* choose a different name if this one already exists. */
  r = stat(*lfn, &st);
  if (r==0) {
    for (n=1; n<=100; n++) {
      msprintf(lfn, "%s/%s-%d%s", path, fn, n, suffix);
      r = stat(*lfn, &st);
      if (r!=0) 
	break;
    }
  }
  free(path);

  if (r==0) {
    free(*lfn);
    *lfn = NULL;
    *f = NULL;
    return(-2);
  }
	
  *f = fopen(*lfn, "w");
  if (!*f) 
  {
    free(*lfn);
    *lfn = NULL;
    *f = NULL;   /* redundant */
    return(-1);
  }  
  
  return(0);
}

/* 209 (0xd1) <user> <speed>: user signon [SERVER]. server is
   notifying client that a user in their hotlist, <user>, has signed
   on the server with link <speed>
 */
I_NAP_FUNC(suon)
{
  hotlist_t *elt;
  
  list_find(elt, hlist, !strcmp(elt->nm, tok[0]));

  if (elt)
  {
    elt->conn = atoi(tok[1]);
    elt->on = 1;
    wp(win, "%s* %s (%s) is on napster%s\n", BRIGHT(GREEN), tok[0], conns[atoi(tok[1])], WHITE);
    drw(win);
  }
  
  return(1);
}

/* 210 (0xd2) <user>: user signoff [SERVER]. server is notifying
   client that a user on their hotlist, <user>, has signed off the
   server.

   this message is also sent by the server when the client attempts to
   browse a nonexistent client.  [why don't they just use 404 for
   this? -ed]
*/
I_NAP_FUNC(suoff)
{
  hotlist_t *elt;

  list_find(elt, hlist, !strcmp(elt->nm, tok[0]));
  
  if (elt && elt->on)
  {
    elt->on = 0;
    wp(win, "%s* %s has left napster%s\n", GREEN, tok[0], WHITE);
    drw(win);
  }
  
  return(1);
}

/* 201 (0xc9) search response:
 * "<filename>" <md5> <size> <bitrate> <frequency> <length> <nick> <ip>
 *  <link-type> [weight] */
I_NAP_FUNC(ssret)
{
  ssearch_t *elt;
  char *t;
  
  if (num < 9)
    return(1);
  
  /* calculate short filename */
  t = basename(tok[0]);
  
  /* create a new search list element */
  elt = (ssearch_t *)malloc(sizeof(ssearch_t));
  
  elt->song = strdup(t);
  elt->fn = strdup(tok[0]);
  elt->sz = atoi(tok[2]);
  elt->brate = atoi(tok[3]);
  elt->freq = atoi(tok[4]);
  elt->time = atoi(tok[5]);
  elt->cmp = strdup(tok[6]);
  elt->nip = htonl(swapl(strtoul(tok[7], (char **)NULL, 10)));
  elt->conn = atoi(tok[8]);
  if (elt->conn > 10 || elt->conn < 0)
    elt->conn = 0;
  timerclear(&elt->s);
  timerclear(&elt->r);
  elt->ping = -2;
  elt->next = NULL;
  
  /* and add it to the list. Note that it is faster to order the list
     at the end, rather than keeping it ordered here. */

  list_prepend(search, elt);

  return(1);
}

/* 202 (0xca) end of search response from server */
I_NAP_FUNC(ssend)
{
  ssearch_t *cur, *cur1;
  char *desc;
  char *d;
  sock_t *sk;
  struct icmphdr *icmp;
  struct sockaddr_in dst;
  
  sk = findsock("icmp");  /* icmp is the socket for ping results, 
			     if they were requested */
  
  if (!sk) {
    showresults(win, 1);
    return(1);
  }

  if (!search) {
    delsock(sk->fd);
    
    showresults(win, 1);
    return(1);
  }

  /* send out pings */
  for (cur=search; cur; cur=cur->next) {
    if (timerisset(&cur->s))
      continue;
    
    dst.sin_addr.s_addr = cur->nip; 
    dst.sin_family = AF_INET;
    
    d = (char *)malloc(64);
    
    /* the following code will not work on big-endian machines. */
    memset(d, 0, 64);
    icmp = (struct icmphdr *)d;  
    icmp->type = ICMP_ECHO;
    icmp->code = 0;
    icmp->un.echo.id = (getpid()&0xffff);
    icmp->un.echo.sequence = 0;
    icmp->checksum = in_cksum((u_short *)icmp, 64);
    
    sendto(sk->fd, d, 64, 0, (struct sockaddr *)&dst, sizeof(dst));
    gettimeofday(&cur->s, NULL);
    free(d);
    
    for (cur1=search;cur1;cur1=cur1->next)
      if (cur1->nip == cur->nip)
	memcpy(&cur1->s, &cur->s, sizeof(cur->s));
  }
  
  wp(win, "* Waiting for ping results...\n");
  drw(win);
  
  /* set timer for 3 seconds */
  desc = strdup("[waiting for ping results]");
  addtimer(3, timed_pingresults, (void *)NULL, desc);
  return (1);
}

void timed_pingresults(void *dummy) {
  sock_t *sk;
  ssearch_t *cur;
  struct timeval tv;

  sk = findsock("icmp");
  if (sk)
    delsock(sk->fd);
  
  /* compute ping times (cur->r - cur->s) */
  for (cur=search;cur;cur=cur->next) {
    if (timerisset(&cur->r)) {
      memcpy(&tv, &cur->r, sizeof(tv));
      tvsub(&tv, &cur->s);
      cur->ping = (tv.tv_sec*10000+(tv.tv_usec/100))/10;
      timerclear(&tv);
    }
    else
      cur->ping = -1;
  }
  
  showresults(wchan, 2);
}

/* diplay search results, either on the main screen or on the result
   screen. LEVEL is 0 for browse, 1 for regular search, and 2 for
   search with pings. */
void showresults(WINDOW *win, int level) {
  int min, sec, i;
  float mb;
  ssearch_t *a, *b, *cur;
  
  /* sort the list, by ping times if we have them, else connection
     speed, or by filename if this is a browse. A ping time of -1 is
     considered "infinitely large" */

  switch (level) {
  case 2: /* display pings and connection speed, sort by pings first */
    list_mergesort(ssearch_t, search, a, b, 
		   (a->ping == b->ping && a->conn >= b->conn) ||
		   (a->ping != -1 && (b->ping == -1 || a->ping < b->ping)));
    break;
  case 1: /* no pings; sort by connection speed */
    list_mergesort(ssearch_t, search, a, b, a->conn >= b->conn);
    break;
  case 0: default: /* browse; sort by filename */
    list_mergesort(ssearch_t, search, a, b, strcmp(a->song, b->song) <= 0);
    break;
  }

  srch = 0;
  
  /* show search result on main screen or result screen, depending
     on whether "noresultscreen" was selected. */
  if (nvar("noresultscreen")==1) 
  {
    if (!search) {
      wp(win, "* Received 0 %s results\n", level==0 ? "browse" : "search");
      drw(win);
      return;
    }

    wp(win, ""BOLD"#"WHITE" | "BOLD"Song"WHITE" | Bitrate | Frequency | "BOLD"Length"WHITE" | "BOLD"Size"WHITE" | User%s%s\n", level>=1 ? " | "BOLD"Speed"WHITE"" : "" , level>=2 ? " | Ping" : "");
    wp(win, "-----------------------------------------------------%s%s\n", level>=1 ? "--------" : "", level>=2 ? "-------" : "");
    i=1;
    list_forall (cur, search) {
      min = cur->time/60;
      sec = cur->time%60;
      mb = ((float)cur->sz)/1048576.0;
      
      wp(win, ""BOLD"%i. %s"WHITE" %ibps %ihz "BOLD"%i:%02i %.02fmb"WHITE" %s", i, (fl)?cur->fn:cur->song, cur->brate, cur->freq, min, sec, mb, cur->cmp);

      if (level>=1) {
	wp(win, " "BOLD"%s"WHITE"", conns[cur->conn]);
      }
      if (level>=2 && cur->ping == -1)
	wp(win, " N/A");
      else if (level>=2)
	wp(win, " %i", cur->ping);

      wp(win, "\n");
      i++;
    }
    drw(win);
  }
  else 
  {
    list_length(ssearch_t, search, i);
    wp(win, i==1 ? "* Received %d %s result\n" : "* Received %d %s results\n",
       i, level==0 ? "browse" : "search");
    drw(win);
    if (i>0)
      switchtoscreen(RESULT_SCREEN);
    plist();
  }
} 
     

/* received a browse response from server:
 * <nick> "<filename>" <md5> <size> <bitrate> <frequency> <time> 
 *   0        1          2     3       4          5         6    */
I_NAP_FUNC(srbrowse)
{
  ssearch_t *cur, *cur1;
  char *t;
  
  if (num < 7)
    return(1);
  
  if (search == NULL)
  {
    search = (ssearch_t *)malloc(sizeof(ssearch_t));
    cur = search;
  }
  else
  {
    for (cur1=NULL,cur=search;cur!=NULL;cur=cur->next)
      cur1 = cur;
    cur = (ssearch_t *)malloc(sizeof(ssearch_t));
    cur1->next = cur;
  }
  
  t = basename(tok[1]);
  
  cur->ping = -2;
  cur->song = strdup(t);
  cur->fn = strdup(tok[1]);
  cur->sz = atoi(tok[3]);
  cur->brate = atoi(tok[4]);
  cur->freq = atoi(tok[5]);
  cur->time = atoi(tok[6]);
  cur->cmp = strdup(tok[0]);
  cur->conn = -1;
  cur->next = NULL;
  
  return(1);
}

/* received an "end of browse list" packet from server:
 * <nick> [ip] */
I_NAP_FUNC(sdbrowse)
{
  showresults(win, 0);
  return(1);
}

I_NAP_FUNC(sop)
{
  wp(win, "%s!%s%s%s!%s %s\n", GREEN, WHITE, tok[0], GREEN, WHITE, str+strlen(tok[0])+1);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(spchange)
{
  wp(win, "%s* Your data port was changed to %s%s\n", RED, tok[0], WHITE);
  
  closefserv();
  info.port = initfserv(tok[0]);
  if (info.port == -1) {
    wp(win, "%sWarning: could not open port %s: %s\n", tok[0], sys_errlist[errno]);
    info.port = 0;
  }

  drw(win);
  return(1);
}

I_NAP_FUNC(sbport)
{
  wp(win, "%s* Your dataport is misconfigured%s\n", RED, WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(sannounce)
{
  wp(win, "%s[%s%s%s]%s %s\n", RED, WHITE, tok[0], RED, WHITE, str+strlen(tok[0])+1);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(sbanlist)
{
  wp(win, "%s* Banned: %s%s\n", BRIGHT(BLUE), str, WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(ssping)
{
  sendpack(s, SERVER_PING, NULL);
  
  return(1);
}

I_NAP_FUNC(scping)
{
  wp(win, "%s* Received PING from %s%s\n", RED, tok[0], WHITE);
  drw(win);
  sendpack(s, CLIENT_PONG, "%s", str);
  
  return(1);
}

I_NAP_FUNC(scpong)
{
  wp(win, "%s* Received PONG from %s%s\n", RED, tok[0], WHITE);
  drw(win);

  return(1);
}

I_NAP_FUNC(sredir)
{
  int t;
  sock_t *sk;
  chans_t *cur;
  char *t1=NULL;
  int c;

  wp(win, "%s* You've been redirected%s\n", RED, WHITE);
  drw(win);
  
  msprintf(&t1, "%s:%s", tok[0], tok[1]);
  
  wp(win, "Connecting to %s...\n", t1);
  drw(win);

  t = conn(t1, PORT);
  if (t == -1)
  {
    drw(win);
    free(t1);
    return(1);
  }
  free(t1);
  wp(win, "Logging in...\n");
  drw(win);
  c = info.conn ? atoi(info.conn) : 0;
  if (c < 0 || c > 10)
    c = 0;
  if (login(t, info.user, info.pass, info.port, CLIENT, c, info.email) == -1)
  {
    drw(win);
    close(t);
    return(1);
  }
  
  sk = findsock("server");
  if (sk)
    delsock(sk->fd);
  sk = addsock(t, "server", S_R, inserv);
  
  curchan=NULL;
  
  if (chanl)
  {
    for (cur=chanl;;cur=cur->next)
      if (!cur->q)
        sendpack(sk->fd, F_JOIN, "%s", cur->nm);
    delnapchans();
  }
  
  upchan(chanl);
  
  return(1);
}

I_NAP_FUNC(scycle)
{
  int t;
  sock_t *sk;
  chans_t *cur;
  int c;

  wp(win, "%s* You've been cycled%s\n", RED, WHITE);
  drw(win);
  
  wp(win, "Connecting to %s...\n", tok[0]);
  drw(win);
  t = conn(tok[0], PORT);
  if (t == -1)
  {
    drw(win);
    return(1);
  }
  wp(win, "Logging in...\n");
  drw(win);
  c = info.conn ? atoi(info.conn) : 0;
  if (c < 0 || c > 10)
    c = 0;
  if (login(t, info.user, info.pass, info.port, CLIENT, c, info.email) == -1)
  {
    drw(win);
    close(t);
    return(1);
  }
  
  sk = findsock("server");
  if (sk)
    delsock(sk->fd);
  sk = addsock(t, "server", S_R, inserv);
  
  curchan=NULL;
  
  if (chanl)
  {
    for (cur=chanl;;cur=cur->next)
      if (!cur->q)
        sendpack(sk->fd, F_JOIN, "%s", cur->nm);
    delnapchans();
  }
  
  upchan(chanl);
  
  return(1);
}

I_NAP_FUNC(sclist)
{
  wp(win, "%s%s%s - %s - %s\n", BOLD, tok[0], WHITE, tok[1], str+strlen(tok[0])+strlen(tok[1])+2);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(sclist2)
{
  wp(win, "%s%s%s - %s - %s\n", BOLD, tok[0], WHITE, tok[1], tok[5] /* str+strlen(tok[0])+strlen(tok[1])+strlen(tok[2])+strlen(tok[3])+strlen(tok[4])+5 */ );
  drw(win);
  
  return(1);
}

I_NAP_FUNC(sblocklist)
{
  wp(win, "%s* Blocked: %s%s\n", BRIGHT(BLUE), str, WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(signorelist)
{
  wp(win, "%s* Ignored: %s%s\n", BRIGHT(BLUE), str, WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(signoreend)
{
  wp(win, "%s* Total users ignored: %s%s\n", BRIGHT(BLUE), str, WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(signoreadd)
{
  wp(win, "%s* Added %s to the ignore list%s\n", BRIGHT(BLUE), str, WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(signoreremove)
{
  wp(win, "%s* Removed %s from the ignore list%s\n", BRIGHT(BLUE), str, WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(signoreunknown)
{
  wp(win, "%s* %s is not ignored or does not exist%s\n", RED, str, WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(signoreexists)
{
  wp(win, "%s* %s is already on your ignore list%s\n", RED, str, WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(signoreclear)
{
  wp(win, "%s* Cleared %s users from your ignore list%s\n", BRIGHT(BLUE), str, WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(signorefail)
{
  wp(win, "%s* Ignoring %s failed!%s\n", RED, str, WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(scbanlist)
{
  wp(win, "%s* Channel ban: %s%s\n", BRIGHT(BLUE), str, WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(snerr)
{
  wp(win, "%s* Server: Could not add %s to your hotlist%s\n", RED, tok[0], WHITE);
  drw(win);
  
  return(1);
}

I_NAP_FUNC(snadd)
{
  /* the server is telling us that it successfully added somebody to
     our hotlist. Since we keep our own copy of our hotlist, do we
     really care? */
    
  return(1);
}

I_NAP_FUNC(sme)
{
  if (!(recent = findchan(chanl, tok[0])))
    return(1);
  if (!strcasecmp(curchan->nm, tok[0]) || wmode)
  {
    wp(win, "%s* %s%s %s\n", CYAN, tok[1], WHITE, tok[2]);
    drw(win);
  }
  else
  {
    wp(win, "%s* %s%s/%s%s%s %s\n", CYAN, tok[1], WHITE, CYAN, tok[0], WHITE, tok[2]);
    drw(win);
  }
  recent = NULL;
  
  return(1);
}

/* end server commands */
/* ---------------------------------------------------------------------- */
/* begin user commands */

O_NAP_FUNC(dquit)
{
  int n, m;
  download_t *dtask;
  upload_t *utask;

  if (num > 1 && !strcasecmp(tok[1], "yes")) {
    quit_now = 1;
    return(-3); /* jump out of all nested loops */
  }
  list_count(down, n, dtask, !STOPPED(dtask->state));
  list_count(up, m, utask, !STOPPED(utask->state));

  if (m || n)
  {
    wp(win, ""RED"* There are %d uploads and %d downloads pending. Use \"/quit yes\" if you want to quit immediately, and \"/tquit\" if you want to quit after pending transfers have finished."WHITE"\n", m, n);
    drw(win);
    return(1);
  }
  quit_now = 1;
  return(-3);   /* return to top level */
}

/* conditional quit: quit when all pending transfers have been completed */
O_NAP_FUNC(dtquit)
{
  int n, m;
  download_t *dtask;
  upload_t *utask;

  quit_after_transfers = 1;

  list_count(down, n, dtask, !STOPPED(dtask->state));
  list_count(up, m, utask, !STOPPED(utask->state));

  wp(win, ""RED"* Scheduled to quit after %d uploads and %d downloads have completed."WHITE"\n", m, n);
  drw(win);

  /* note that a tquit does not cause execution of a macro to be aborted. */
  return(1);
}

O_NAP_FUNC(dunquit) 
{
  quit_after_transfers = 0; 
  
  wp(win, "* Effect of /tquit was canceled\n");
  drw(win);
  return(1);
}

O_NAP_FUNC(djoin)
{
  chans_t *t;

  if (tok[1] == NULL)
  {
    wp(win, "%s* You didn't enter a channel name%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  t = findchan(chanl, tok[1]);
  if (t && !t->p)
  {
    curchan = t;
    return(1);
  }
                  
  sendpack(s, F_JOIN, "%s", tok[1]);
  
  return(1);
}

O_NAP_FUNC(dpart)
{
  chans_t *pt, *cur, *cur1=NULL;

  if (num == 1)
    pt = curchan;
  else
    pt = findchan(chanl, tok[1]);
  
  if (!pt && num > 1)
  {
    wp(win, "%s* You are not on %s channel %s\n", RED, tok[1], WHITE);
    drw(win);
    return(1);
  }
  else if (!pt)
  {
    wp(win, "%s* You are not on a channel %s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (!pt->q)
  {
    sendpack(s, F_PART, "%s", pt->nm);
    pt->p |= 1;
    return(1);
  }
  else if (pt->q == 2)
  {
    ssock(ircsock, "PART %s\n", pt->nm);
    pt->p |= 1;
    return(1);
  }

  wp(win, "%s* Ended query with %s%s\n", GREEN, pt->nm, WHITE);
  drw(win);
  
  for (cur=chanl;cur!=pt;cur=cur->next)
    cur1 = cur;
  
  if (cur1)
    cur1->next = cur->next;
  else if (cur->next)
    chanl = cur->next;
  else
  {
    chanl = NULL;
    curchan = NULL;
  }
  if (curchan && !strcmp(pt->nm, curchan->nm))
    upchan(curchan);

  if (pt->topic)
    free(pt->topic);
  free(pt->nm);
  free(pt);
  
  if (wmode)
  {
    dscr(win);
    drw(win);
  }
  
  return(1);
}

void usage(WINDOW *win, char *cmd) 
{
  int i;
  char *help, *p;

  for (i=0; out[i].nm; i++) {
    if (!strcasecmp(out[i].nm, cmd)) {
      break;
    }
  }
  if (!out[i].nm || !out[i].help) {
    wp(win, ""RED"* /%s: Invalid usage (no help available)"WHITE"\n", cmd);
  } else {
    help = strdup(out[i].help);
    p = strchr(help, '-');
    if (p) {
      *p = 0;
    }
    wp(win, ""RED"* Usage: /%s %s"WHITE"\n", cmd, help);
    free(help);
  }
  drw(win);
}

O_NAP_FUNC(dtell)
{
  char *msg;
  if (num < 3)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  msg = fixquotes(cstr(str, 2));
  sendpack(s, F_TELL, "%s %s", tok[1], msg);
  recent = findquery(chanl, tok[1]);
  wp(win, "%s* --> (%s%s%s)%s %s\n", GREEN, WHITE, tok[1], GREEN, WHITE, msg);
  drw(win);

  recent = NULL;
  return(1);
}

O_NAP_FUNC(dwhois)
{
  if (num < 2)
    sendpack(s, F_WHOIS, "%s", info.user);
  else
    sendpack(s, F_WHOIS, "%s", tok[1]);
  return(1);
}

/* parses a range of non-negative numbers, e.g. "1,3-4,5" or "1 3-4
   5". The first time this is called, RANGE should be a string
   representing the range of numbers to be parsed. Each subsequent
   call should pass RANGE=NULL. Each call will return the next
   available integer, or -1 when done. */
int parse_range(char *range) {
  static char *ran=NULL;  /* next subrange */
  static int i=0;         /* next number to return */
  static int e=-1;         /* last number in current subrange to return */

  if (range) {
    ran = strtok(range, ", ");
    i = 0;
    e = -1;
  }
  while (1) {
    if (i<=e) {
      i++;
      return i-1;
    }
    if (!ran)
      return -1;
    if (*ran && strchr(ran+1, '-')) /* assume token is a subrange */
      sscanf(ran, "%i-%i", &i, &e);
    else                            /* token is a single number */
      i = e = atoi(ran);
    if (i<0) 
      i=0;
    ran = strtok(NULL, ", ");
  }
}

/* get files by number e.g. "/g 1,3-4,5" or "/g 1 3-4 5" */
O_NAP_FUNC(dg)
{
  ssearch_t *cur=search;
  int n;
  int r;
  
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  if (!search)
  {
    wp(win, "%s* No search performed yet%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  for (n = parse_range(cstr(str, 1)); n != -1; n = parse_range(NULL)) {
    list_nth(cur, search, n-1);
    if (cur == NULL) {   /* no matching item */
      wp(win, ""RED"* Error: no search result numbered %d"WHITE"\n", n);
      drw(win);
      return(1); /* skip rest of range */
    }
    r = requestdownload(s, cur);
    
    switch(r) {
    case 0: 
      wp(win, "* Getting \"%s\" from %s\n", cur->song, cur->cmp);
      break;
    case 1: case 2:
      wp(win, "* Queued download of \"%s\" from %s\n", cur->song, cur->cmp);
      break;
    case -1: 
      wp(win, "* Already getting \"%s\" from %s\n", cur->song, cur->cmp);
      break;
    case -2:
      wp(win, "* Download of \"%s\" from %s is already queued\n", cur->song, cur->cmp);
      break;
    case -3:
      wp(win, "* You can't get \"%s\" from yourself\n", cur->song);
      break;
    default:
      break;
    }
    drw(win);
  }
  
  drw(win);
  
  return(1);
}

/* request to download the given search item. Returns: 0 on success,
   -1 if we are already getting this item, -2 if the item is already
   queued, -3 if we can't get this item because we are the user
   offering it, 1 if this item was queued because the per-user
   download limit was reached, 2 if it was queued because the overall
   download limit was reached. */

int requestdownload(int s, ssearch_t *sitem) {
  download_t *elt, *task;
  int r;

  /* see if the user offering this item is us */
  if (!strcasecmp(sitem->cmp, info.user)) {
    return -3;
  }

  /* see if we're already getting this item */

  list_find(elt, down, !strcasecmp(elt->nick, sitem->cmp) && !strcasecmp(elt->fn, sitem->song) && !STOPPED(elt->state));
  
  if (elt) {
    if (elt->state == QUEUED || elt->state == RQUEUED)
      return -2;
    else
      return -1;
  }

  /* check whether we will queue or request this item */
  r = download_limit_reached(sitem->cmp);


  /* create the item for the download list */
  task = (download_t *)malloc(sizeof(download_t));
  task->nick = strdup(sitem->cmp);
  task->rfn = strdup(sitem->fn);
  task->fn = strdup(sitem->song);

  /* add it to the list */
  list_append(download_t, down, task);

  if (r) {
    task->state = QUEUED;
    return r;
  } else {
    sendpack(s, F_DGET, "%s \"%s\"", task->nick, task->rfn);
    task->state = REQUESTED;
    task->c_time = time(0);
    return 0;
  }
}

/* check whether the download limit for this nick is currently
   reached. Return 0 if not reached, 2 if overall limit reached, 1 if
   per-user limit reached */
int download_limit_reached(char *nick) {
  download_t *elt;
  int maxd = nvar("maxdownloads");
  int maxdu = nvar("maxdownuser");
  int n;

  /* see if the overall download limit is reached */
  list_count(down, n, elt, ACTIVE(elt->state));
  if (maxd >= 0 && n >= maxd) {
    return 2;
  }

  /* see if the per-user download limit is reached */
  list_count(down, n, elt, !strcasecmp(elt->nick, nick) && ACTIVE(elt->state));
  if (maxdu >= 0 && n >= maxdu) {
    return 1;
  }
  return 0;
}

/* check whether the upload limit for this nick is currently
   reached. Return 0 if not reached, 2 if overall limit reached, 1 if
   per-user limit reached */
int upload_limit_reached(char *nick) {
  upload_t *elt;
  int maxu = nvar("maxuploads");
  int maxuu = nvar("maxupuser");
  int n;

  /* see if the overall upload limit is reached */
  list_count(up, n, elt, ACTIVE(elt->state));
  if (maxu >= 0 && n >= maxu) {
    return 2;
  }

  /* see if the per-user upload limit is reached */
  list_count(up, n, elt, !strcasecmp(elt->nick, nick) && ACTIVE(elt->state));
  if (maxuu >= 0 && n >= maxuu) {
    return 1;
  }
  return 0;
}

/* "/force <range>": force queued items to be downloaded, overriding
   any download limit, or cause a stopped item to be downloaded. */
O_NAP_FUNC(dforce)
{
  download_t *task;
  int n;

  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  for (n=parse_range(cstr(str, 1)); n!=-1; n=parse_range(NULL)) {

    /* find the n-th element in the download list */
    list_nth(task, down, n-1);
    
    if (!task) {
      wp(win, "* Download number %d does not exist\n", n);
      drw(win);
      continue;
    }
    
    if (task->state == RRQUEUED) {
      /* we've already requested this item from the server, so this is
	 just a change of label and verbosity */
      task->state = REQUESTED;
      wp(win, "* Getting \"%s\" from %s\n", task->fn, task->nick);
      drw(win); 
    }

    if (ACTIVE(task->state)) {
      wp(win, "* Already getting \"%s\" from %s\n", task->fn, task->nick);
      drw(win);
      continue;
    }
    
    /* otherwise, go ahead request the download */
    sendpack(s, F_DGET, "%s \"%s\"", task->nick, task->rfn);
    task->state = REQUESTED;
    task->c_time = time(0);
    wp(win, "* Getting \"%s\" from %s\n", task->fn, task->nick);
    drw(win);
  }
  return(1);
}

/* "/retry <range>": put stopped items back in the queue. This only
   applies to items that are FAILED, INCOMPLETE, or TIMED_OUT */

O_NAP_FUNC(dretry)
{
  download_t *task;
  int n;

  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  for (n=parse_range(cstr(str, 1)); n!=-1; n=parse_range(NULL)) {

    /* find the n-th element in the download list */
    list_nth(task, down, n-1);
    
    if (!task) {
      wp(win, "* Download number %d does not exist\n", n);
      drw(win);
      continue;
    }
    
    if (ACTIVE(task->state)) {
      wp(win, "* Already getting \"%s\" from %s\n", task->fn, task->nick);
      drw(win);
      continue;
    }
    
    if (task->state == QUEUED || task->state == RQUEUED) {
      wp(win, "* Download \"%s\" from %s is already queued\n", task->fn, task->nick);
      drw(win);
      continue;
    }

    if (task->state == COMPLETE) {
      wp(win, "* Download \"%s\" from %s is already finished\n", task->fn, task->nick);
      drw(win);
      continue;
    }
    
    /* otherwise, go ahead put it in the queue or request it */
    if (download_limit_reached(task->nick)) {
      task->state = QUEUED;
      wp(win, "* Queued \"%s\" from %s\n", task->fn, task->nick);
      drw(win);
    } else {
      sendpack(s, F_DGET, "%s \"%s\"", task->nick, task->rfn);
      task->state = REQUESTED;
      task->c_time = time(0);
      wp(win, "* Getting \"%s\" from %s\n", task->fn, task->nick);
      drw(win);
    }
  }
  return(1);
}

/* "/retryall: put all stopped items back in the queue. This only
   applies to items that are FAILED, INCOMPLETE, or TIMED_OUT */

O_NAP_FUNC(dretryall)
{
  download_t *task;

  list_forall(task, down) {
    
    if (ACTIVE(task->state) || task->state == QUEUED 
	|| task->state == RQUEUED || task->state == COMPLETE) {
      continue;
    }
    
    /* otherwise, go ahead put it in the queue or request it */
    if (download_limit_reached(task->nick)) {
      task->state = QUEUED;
      wp(win, "* Queued \"%s\" from %s\n", task->fn, task->nick);
      drw(win);
    } else {
      sendpack(s, F_DGET, "%s \"%s\"", task->nick, task->rfn);
      task->state = REQUESTED;
      task->c_time = time(0);
      wp(win, "* Getting \"%s\" from %s\n", task->fn, task->nick);
      drw(win);
    }
  }
  return(1);
}

O_NAP_FUNC(dpup)
{
  int count, c, d, p, max;
  upload_t *task;
  float kb, rate;
  int tm;

  wp(win, "* Uploads:\n");
  drw(win);

  count = 0;
  c = d = 0;
  list_forall (task, up) {
    count++;
    wp(win, ""BOLD"%i."WHITE" ", count);
    switch (task->state) {
    case WAITING:
      c++;
      wp(win, "Waiting | %s | "BOLD"%s"WHITE"\n", task->nick, task->fn);
      break;
    case CONNECTING: case CONNECTING1: /* note: this state always firewalled */
      c++;
      wp(win, "Connecting | %s (firewalled) | "BOLD"%s"WHITE"\n", task->nick, task->fn);
      break;
    case IN_PROGRESS:
      c++;
      p = (int)(100*(float)task->pos/(float)task->size);
      kb = (float)(task->pos - task->bsz)/1024;
      tm = time(0) - task->p_time;
      rate = tm ? kb / tm : 0.0;
      wp(win, "Sending | %s | "BOLD"%s"WHITE" at %.2f k/s (%i%%)\n", task->nick, task->fn, rate, p); 
      break;
    case FAILED:
      d++;
      wp(win, "Failed | %s | "BOLD"%s"WHITE"\n", task->nick, task->fn);
      break;
    case INCOMPLETE:
      d++;
      wp(win, "Interrupted | %s | "BOLD"%s"WHITE"\n", task->nick, task->fn);
      break;
    case COMPLETE:
      d++;
      wp(win, "Finished | %s | "BOLD"%s"WHITE"\n", task->nick, task->fn);
      break;
    case TIMED_OUT:
      d++;
      wp(win, "Timed out | %s | "BOLD"%s"WHITE"\n", task->nick, task->fn);
      break;
    default:
      break;
    }
  }
  max = nvar("maxuploads");
  if (max>=0) {
    wp(win, ""BOLD"%i uploading (limit %i), %i stopped"WHITE"\n", c, max, d);
  } else {
    wp(win, ""BOLD"%i uploading, %i stopped"WHITE"\n", c, d);
  }
  drw(win);
  return(1);
}

/* print download list and download queue, numbering the items in the order
 * they appear in the lists. */
O_NAP_FUNC(dpdown)
{
  int count, c, q, d, p, max;
  download_t *task;
  float kb, rate;
  int tm;

  wp(win, "* Downloads:\n");
  drw(win);

  count = 0;
  c = q = d = 0;
  list_forall (task, down) {
    count++;
    wp(win, ""BOLD"%i."WHITE" ", count);
    switch (task->state) {
    case QUEUED:
      q++;
      wp(win, "Queued | %s | "BOLD"%s"WHITE"\n", task->nick, task->fn);
      break;
    case RQUEUED: case RRQUEUED:
      q++;
      wp(win, "Remotely queued | %s | "BOLD"%s"WHITE"\n", task->nick, task->fn);
      break;
    case REQUESTED:
      c++;
      wp(win, "Requested | %s | "BOLD"%s"WHITE"\n", task->nick, task->fn);
      break;
    case WAITING:
      c++;
      wp(win, "Waiting | %s%s | "BOLD"%s"WHITE"\n", task->nick, task->addr.sin_port ? "" : " (firewalled)", task->fn);
      break;
    case CONNECTING:
      c++;
      wp(win, "Connecting | %s%s | "BOLD"%s"WHITE"\n", task->nick, task->addr.sin_port ? "" : " (firewalled)", task->fn);
      break;
    case IN_PROGRESS:
      c++;
      p = (int)(100*(float)task->pos/(float)task->size);
      kb = (float)(task->pos - task->bsz)/1024;
      tm = time(0) - task->p_time;
      rate = tm ? kb / tm : 0.0;
      wp(win, "Getting | %s%s | "BOLD"%s"WHITE" at %.2f k/s (%i%%)\n", task->nick, task->addr.sin_port ? "" : " (firewalled)", task->fn, rate, p); 
      break;
    case FAILED:
      d++;
      wp(win, "Failed | %s | "BOLD"%s"WHITE"\n", task->nick, task->fn);
      break;
    case INCOMPLETE:
      d++;
      wp(win, "Interrupted | %s | "BOLD"%s"WHITE"\n", task->nick, task->fn);
      break;
    case COMPLETE:
      d++;
      wp(win, "Finished | %s | "BOLD"%s"WHITE"\n", task->nick, task->fn);
      break;
    case TIMED_OUT:
      d++;
      wp(win, "Timed out | %s | "BOLD"%s"WHITE"\n", task->nick, task->fn);
      break;
    default:
      break;
    }
  }
  max = nvar("maxdownloads");
  if (max>=0) {
    wp(win, ""BOLD"%i downloading (limit %i), %i queued, %i stopped"WHITE"\n", c, max, q, d);
  } else {
    wp(win, ""BOLD"%i downloading, %i queued, %i stopped"WHITE"\n", c, q, d);
  }
  drw(win);
  return(1);
}

O_NAP_FUNC(dsearch)
{
  ssearch_t *cur, *cur1;
  char brate[256], freq[256], speed[256], size[256];
  int k, noping, c, max, t, l;
  struct sockaddr_in me;
  
  if (s==-1)
  {
    wp(win, ""RED"* Not connected to server"WHITE"\n");
    drw(win);
    return(1);
  }
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  if (srch)
  {
    wp(win, "%s* Another search is in progress%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  srch = 1;
  
  for (cur=search,cur1=cur;cur1;cur=cur1)
  {
    cur1 = cur->next;
    free(cur->song);
    free(cur->fn);
    free(cur->cmp);
    free(cur);
  }
  search = NULL;
  
  *brate = *freq = *speed = *size = 0;

  opterr = 0;
  max = 250;
  optind = 0;
  fl = 0;
  l = 0;
  noping = 0;

  while ((c = getopt(num, tok, "lfpb:c:r:s:m:")) != -1)
  {
    switch (c)
    {
      case 'f':
        fl = 1;
        break;
      case 'l':
        l = 1;
        break;
      case 'p':
        noping = 1;
        break;
      case 'b':
	t = 0;
        if (*optarg == '=')
          strcpy(brate, "BITRATE \"EQUAL TO\" \"");
        else if (*optarg == '<')
          strcpy(brate, "BITRATE \"AT BEST\" \"");
        else if (*optarg == '>')
          strcpy(brate, "BITRATE \"AT LEAST\" \"");
        else
        {
          strcpy(brate, "BITRATE \"AT LEAST\" \"");
          t = 1;
        }
        if (t)
          strcat(brate, optarg);
        else
          strcat(brate, optarg+1);
        strcat(brate, "\"");
        break;
      case 'c':
	t = 0;
        if (*optarg == '=')
          strcpy(speed, "LINESPEED \"EQUAL TO\" ");
        else if (*optarg == '<')
          strcpy(speed, "LINESPEED \"AT BEST\" ");
        else if (*optarg == '>')
          strcpy(speed, "LINESPEED \"AT LEAST\" ");
        else
        {
          strcpy(speed, "LINESPEED \"AT LEAST\" ");
          t = 1;
        }
        if (t)
          strcat(speed, optarg);
        else
          strcat(speed, optarg+1);
        break;
      case 'r':
	t = 0;
        if (*optarg == '=')
          strcpy(freq, "FREQ \"EQUAL TO\" \"");
        else if (*optarg == '<')
          strcpy(freq, "FREQ \"AT BEST\" \"");
        else if (*optarg == '>')
          strcpy(freq, "FREQ \"AT LEAST\" \"");
        else
        {
          strcpy(freq, "FREQ \"AT LEAST\" \"");
          t = 1;
        }
        if (t)
          strcat(freq, optarg);
        else
          strcat(freq, optarg+1);
        strcat(freq, "\"");
        break;
      case 's':
	t = 0;
	if (*optarg == '=')
	  strcpy(size, "SIZE \"EQUAL TO\" \"");
	else if (*optarg == '<')
	  strcpy(size, "SIZE \"AT BEST\" \"");
	else if (*optarg == '>')
	  strcpy(size, "SIZE \"AT LEAST\" \"");
        else
	  {
            strcpy(size, "SIZE \"AT LEAST\" \"");
            t = 1;
	  }
	if (t)
	  strcat(size, optarg);
        else
	  strcat(size, optarg+1);
	strcat(size, "\"");
	break;
      case 'm':
	max = atoi(optarg);
        break;
      default:
        break;
    }
  }
  
  sendpack(s, F_SEARCH, "FILENAME CONTAINS \"%s\" MAX_RESULTS %i %s %s %s %s %s", cstr(str, optind), max, l?"LOCAL_ONLY":"", speed, brate, freq, size);
  
  wp(win, "* Searching for \"%s\"...\n", cstr(str, optind));
  drw(win);
  
  if (noping)
    return(1);
  
  /* socket fails (returns -1) with errno=EPERM if you don't have root
   * privileges, so the "icmp" sock_t never gets created and the ping
   * times are not calculated in ssend() */
  k = socket(AF_INET, SOCK_RAW, IPPROTO_ICMP);
  if (k == -1)
    return(1);
  
  me.sin_addr.s_addr = INADDR_ANY;
  me.sin_family = AF_INET;
  
  if (bind(k, (struct sockaddr *)&me, sizeof(me)) == -1)
  {
    close(k);
    return(1);
  }
  
  addsock(k, "icmp", S_R, icmpin);
  
  return(1);
}

O_NAP_FUNC(dpvars)
{
  wp(win, "* User variables:\n");
  drw(win);

  printsets(win);
  return(1);
}

O_NAP_FUNC(dkill)
{
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  if (num == 2)
    sendpack(s, F_KILL, "%s", tok[1]);
  else
    sendpack(s, F_KILL, "%s \"%s\"", tok[1], cstr(str, 2));
  
  return(1);
}

O_NAP_FUNC(dban)
{
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  if (num == 2)
    sendpack(s, F_BAN, "%s", tok[1]);
  else
    sendpack(s, F_BAN, "%s \"%s\"", tok[1], cstr(str, 2));
  
  return(1);
}

O_NAP_FUNC(dunban)
{
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  if (num == 2)
    sendpack(s, F_UNBAN, "%s", tok[1]);
  else
    sendpack(s, F_UNBAN, "%s \"%s\"", tok[1], cstr(str, 2));
  
  return(1);
}

O_NAP_FUNC(ddatap)
{
  if (num < 3)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  sendpack(s, F_DATAP, "%s %i", tok[1], atoi(tok[2]));
  
  return(1);
}

O_NAP_FUNC(dtopic)
{
  chans_t *cur;

  if (num == 1)
  {
    if (!curchan)
    {
      wp(win, "%s* Error: You're not on channel%s\n", RED, WHITE);
      drw(win);
      return(1);
    }
    wp(win, "%s* Topic for %s: %s%s\n", YELLOW, curchan->nm, curchan->topic, WHITE);
    drw(win);
    return(1);
  }
  else if (num == 2)
  {
    cur = findchan(chanl, tok[1]);
    if (!cur)
    {
      wp(win, "%s* Error: Not on channel %s%s\n", RED, tok[1], WHITE);
      drw(win);
      return(1);
    }
    wp(win, "%s* Topic for %s: %s%s\n", YELLOW, cur->nm, cur->topic, WHITE);
    drw(win);
    return(1);
  }
      
  sendpack(s, F_TOPIC, "%s", cstr(str, 1));
  
  return(1);
}

O_NAP_FUNC(dlevel)
{
  if (num < 3)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  sendpack(s, F_LEVEL, "%s", cstr(str, 1));
  
  return(1);
}

O_NAP_FUNC(dopsay)
{
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  sendpack(s, F_SOP, "%s", cstr(str, 1));
  
  return(1);
}

O_NAP_FUNC(ddns)
{
  struct hostent *d1, *d2;
  struct sockaddr_in in;
  
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  wp(win, "* Resolving %s...\n", tok[1]);
  drw(win);

  if (!strchr(tok[1], '.'))
  {
    dwi = strdup(tok[1]);
    sendpack(s, F_WHOIS, "%s", dwi);
    return(1);
  }
  
  if (fork())
    return(1);
  
  d1 = gethostbyname(tok[1]);
  if (!d1)
  {
    ssock(ipcs[1], "* Unable to resolve %s\n", tok[1]);
    exit(1);
  }
  
  memcpy(&in.sin_addr, d1->h_addr, d1->h_length);
  d2 = gethostbyaddr((char *)&in.sin_addr, d1->h_length, AF_INET);
  if (!d2)
  {
    ssock(ipcs[1], "* %s is %s\n", tok[1], inet_ntoa(in.sin_addr));
    exit(1);
  }
  
  ssock(ipcs[1], "* %s is %s [%s]\n", tok[1], d2->h_name, inet_ntoa(in.sin_addr));
  
  exit(1);
}

O_NAP_FUNC(dannounce)
{
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  sendpack(s, F_ANNOUNCE, "%s", cstr(str, 1));
  
  return(1);
}

O_NAP_FUNC(dpchans)
{
  chans_t *cur;
  
  wp(win, "* Channels:\n");
  drw(win);

  if (!chanl)
    wp(win, "No channels\n");
  else if (curchan)
    wp(win, "Current: %s\n", curchan->nm);
  else
    wp(win, "Current: None\n");
  for (cur=chanl;cur!=NULL;cur=cur->next)
    wp(win, "%s\n", cur->nm);
  
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dbanlist)
{
  wp(win, "* Banned users:\n");
  drw(win);

  sendpack(s, F_DBANLIST, "%s", cstr(str, 1));
  
  return(1);
}

O_NAP_FUNC(dlspeed)
{
  if (num < 3)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  sendpack(s, F_LSCHANGE, "%s %s", tok[1], tok[2]);
  
  return(1);
}

O_NAP_FUNC(dpsocks)
{
  wp(win, "* Sockets:\n");
  drw(win);

  psocks(win);
  
  return(1);
}

/* /dup <range> - delete an entry from the upload list */
O_NAP_FUNC(ddup)
{
  upload_t *task;
  int count, n;
  
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  /* as we are deleting items, their relative numbering also changes.
     To avoid strange effects, we first number all the items in the
     list, and then delete them by number, rather than by position */

  count = 0;
  list_forall(task, up) {
    count++;
    task->tmp = count;
  }
  
  for (n=parse_range(cstr(str, 1)); n!=-1; n=parse_range(NULL)) {
  
    /* find the n-th element in the upload list */
    list_find(task, up, task->tmp == n);

    if (!task && 0<n && n<=count) {
      /* double delete of the same item; ignore */
    } else if (!task) {
      wp(win, "* Upload number %d does not exist\n", n);
      drw(win);
    } else {
      wp(win, "* Deleted upload of \"%s\" to %s\n", task->fn, task->nick);
      drw(win);
      dupload(task);
    }
  }
  return(1);
}

/* /ddown <range> - delete one or more items from the download list or
 * the download queue.  Items are (implicitly) numbered according to
 * their position in the list.  Use /pdown to view the numbering. */
O_NAP_FUNC(dddown)
{
  download_t *task;
  int count, n; 
  
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  /* as we are deleting items, their relative numbering also changes.
     To avoid strange effects, we first number all the items in the
     list, and then delete them by number, rather than by position */

  count = 0;
  list_forall(task, down) {
    count++;
    task->tmp = count;
  }

  for (n=parse_range(cstr(str, 1)); n!=-1; n=parse_range(NULL)) {
  
    /* find the n-th element in the download list */
    list_find(task, down, task->tmp == n);

    if (!task && 0<n && n<=count) {
      /* double delete of the same item; ignore */
    } else if (!task) {
      wp(win, "* Download number %d does not exist\n", n);
      drw(win);
    } else {
      wp(win, "* Deleted download of \"%s\" from %s\n", task->fn, task->nick);
      drw(win);
      ddownload(win, task);
    }
  }
  return(1);
}

/* /purgedown - remove all stopped items from the download list */
O_NAP_FUNC(dpurgedown)
{
  int c;

  c = purgedown(win);
  
  wp(win, "* Purged %d items from the download list\n", c);
  drw(win);

  return(1);
}

/* /purgeup - remove all stopped items from the upload list */
O_NAP_FUNC(dpurgeup)
{
  int c;

  c = purgeup();
  
  wp(win, "* Purged %d items from the upload list\n", c);
  drw(win);

  return(1);
}

/* /purgeup - remove all stopped items from the upload and download lists */
O_NAP_FUNC(dpurge)
{
  int u, d;

  u = purgeup();
  d = purgedown(win);

  wp(win, "* Purged %d uploads and %d downloads\n", u, d);
  drw(win);

  return(1);
}

/* remove all stopped items from the download list. Return number of
   items deleted. */
int purgedown(WINDOW *win)
{
  download_t *task;
  int c=0;

  task = down;
  while (task) {
    if (STOPPED(task->state)) {
      c++;
      ddownload(win, task);
      task=down;
      continue;
    }
    task = task->next;
  }
  return(c);
}

/* remove all stopped items from the upload list. Return number of
   items deleted.  */
int purgeup(void)
{
  upload_t *task;
  int c=0;

  task = up;
  while (task) {
    if (STOPPED(task->state)) {
      c++;
      dupload(task);
      task=up;
      continue;
    }
    task = task->next;
  }
  return(c);
}

O_NAP_FUNC(dmuzzle)
{
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  if (num == 2)
    sendpack(s, F_MUZZLE, "%s", tok[1]);
  else
    sendpack(s, F_MUZZLE, "%s \"%s\"", tok[1], cstr(str, 2));
  
  return(1);
}

O_NAP_FUNC(dunmuzzle)
{
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  if (num == 2)
    sendpack(s, F_UNMUZZLE, "%s", tok[1]);
  else
    sendpack(s, F_UNMUZZLE, "%s \"%s\"", tok[1], cstr(str, 2));
  
  return(1);
}

O_NAP_FUNC(ddisconnect)
{
  sock_t *sk;

  curchan=NULL;
  
  if (chanl)
    delnapchans();
  
  upchan(chanl);
  
  sk = findsock("server");
  if (sk)
    delsock(sk->fd);
  
  wp(win, "%s* Disconnected from server%s\n", RED, WHITE);
  drw(win);

  return(1);
}

#ifdef USERCMDS
O_NAP_FUNC(dreload)
{
  libunload(&hnd);
  if (libload(USERLIB) == -1)
  {
    wp(win, "%s* Error loading %s%s\n", RED, dlerror(), WHITE);
    drw(win);
  }
  
  return(1);
}
#endif /* USERCMDS */

O_NAP_FUNC(dreconnect)
{
  int t;
  sock_t *sk;
  chans_t *cur;
  char *t1 = NULL;
  char *srv;
  char *serverlist_dup;
  int c;

  wp(win, "Getting best host...\n");
  drw(win);

  serverlist_dup = strdup(info.serverlist);
  srv = strtok(serverlist_dup, ";");

  for (;  ; srv = strtok(NULL, ";")) {
    if (srv==NULL && info.reconnect) {  /* loop endlessly */
      sleep(1);
      free(serverlist_dup);
      serverlist_dup = strdup(info.serverlist);
      srv = strtok(serverlist_dup, ";");
    } else if (srv==NULL) {
      free(serverlist_dup);
      wp(win, ""RED"* Failed to connect to a server"WHITE"\n");
      drw(win);
      return(1);
    }
    srv = strip(srv);   /* remove whitespace on both ends */
    wp(win, "Trying %s\n", srv);
    t = conn(srv, PORT);
    if (t == -1)
      {
	drw(win);
	continue;
      }
    
    sk = findsock("server");
    if (sk) {
      close(sk->fd);
      delsock(sk->fd);
    }
    wp(win, "Logging in...\n");
    drw(win);
    c = info.conn ? atoi(info.conn) : 0;
    if (c < 0 || c > 10)
      c = 0;
    if (login(t, info.user, info.pass, info.port, CLIENT, c, info.email) == -1)
      {
	drw(win);
	close(t);
	continue;
      }
    break;
  }
  free(serverlist_dup);

  {
    struct sockaddr_in dst;
    int frmlen = sizeof(dst);
    if (!getpeername(t, (struct sockaddr *)&dst, &frmlen))
      wp(win, "* Connected to %s:%i\n", inet_ntoa(dst.sin_addr), ntohs(dst.sin_port));
  }

  sk = addsock(t, "server", S_R, inserv);
  
  t1 = glistn(info.user);
  checkhotlist(t, t1);
  free(t1);
  
  curchan = NULL;
  
  if (chanl)
  {
    for (cur=chanl;cur;cur=cur->next)
      if (!cur->q)
        sendpack(sk->fd, F_JOIN, "%s", cur->nm);
    delnapchans();
  }
  
  upchan(chanl);
  
  return(1);
}

O_NAP_FUNC(dbrowse)
{
  ssearch_t *cur, *cur1;
  
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  if (srch)
  {
    wp(win, "%s* Another search is in progress%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  for (cur=search,cur1=cur;cur1!=NULL;cur=cur1)
  {
    cur1 = cur->next;
    free(cur->song);
    free(cur->fn);
    free(cur->cmp);
    free(cur);
  }
  search = NULL;
  
  sendpack(s, F_BROWSE, "%s", tok[1]);
  
  wp(win, "* Browsing %s...\n", tok[1]);
  drw(win);
  
  srch = 1;
  
  return(1);
}

O_NAP_FUNC(dhelp)
{
  int i, j, l;
  char buf[64];
  char *cmd;
  
  if (num < 2)
  {
    wp(win, "%s* Commands:%s\n", MAGENTA, WHITE);
    for (i=0,j=0;out[i].nm;i++,j++)
    {
      memset(buf, 0, sizeof(buf));
      if (j != 3)
      {
        for (l=COLS/4-strlen(out[i].nm);l>=0;l--)
          buf[l] = ' ';
      }
      wp(win, "%s%s%s%s", MAGENTA, out[i].nm, WHITE, buf);
      if (j == 3)
      {
        wp(win, "\n");
        j = -1;
      }
    }
    if (j)
      wp(win, "\n");

    wp(win, "%s* To get help on a specific command, type /help <command>%s\n",
       MAGENTA, WHITE);

    drw(win);
    
    return(1);
  }
      
  cmd = tok[1];
  if (cmd[0] == '/')
    cmd++;
      
  for (i=0;;i++)
  {
    if (!out[i].nm)
    {
      wp(win, "%s* Could not find help on command: %s%s\n", RED, cmd, WHITE);
      drw(win);
      return(1);
    }
    if (!strcasecmp(out[i].nm, cmd))
    {
      if (!out[i].help)
      {
        wp(win, "%s* No help available on %s%s\n", MAGENTA, out[i].nm, WHITE);
        drw(win);
        return(1);
      }
      wp(win, "%s* Help on %s:%s\n", MAGENTA, out[i].nm, WHITE);
      wp(win, "%s/%s %s%s\n", MAGENTA, out[i].nm, out[i].help, WHITE);
      drw(win);
      return(1);
    }
  }
}

O_NAP_FUNC(dping)
{
  sendpack(s, CLIENT_PING, "%s", cstr(str, 1));
  
  return(1);
};

O_NAP_FUNC(duserpass)
{
  sendpack(s, SET_USER_PASSWORD, "%s", cstr(str, 1));
  
  return(1);
}

O_NAP_FUNC(dreloadconf)
{
  sendpack(s, SERVER_RELOAD_CONFIG, "%s", cstr(str, 1));

  return(1);
}

O_NAP_FUNC(dsver)
{
  sendpack(s, SERVER_VERSION, "%s", cstr(str, 1));
  
  return(1);
}

O_NAP_FUNC(dchanclear)
{
  sendpack(s, CHANNEL_CLEAR, "%s", cstr(str, 1));
  
  return(1);
}

O_NAP_FUNC(dsetlevel)
{
  sendpack(s, CHANNEL_SETLEVEL, "%s", cstr(str, 1));
  
  return(1);
}

O_NAP_FUNC(dme)
{
  if (!curchan)
  {
    wp(win, "%sYou're not on a channel%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  sendpack(s, CHAN_EMOTE, "%s \"%s\"", curchan->nm, cstr(str, 1));
  
  return(1);
}

O_NAP_FUNC(dusers)
{
  if (num < 2)
  {
    if (!curchan)
    {
      wp(win, "%s* You're not on a channel%s\n", RED, WHITE);
      drw(win);
      return(1);
    }
    sendpack(s, CHANNEL_USERLIST, "%s", curchan->nm);
  }
  else
    sendpack(s, CHANNEL_USERLIST, "%s", cstr(str, 1));
  
  return(1);
}

O_NAP_FUNC(dgusers)
{
  sendpack(s, SERVER_USERLIST, "%s", cstr(str, 1));
  
  return(1);
}

O_NAP_FUNC(dsetconf)
{
  sendpack(s, SERVER_SET_CONFIG, "%s", cstr(str, 1));
  
  return(1);
}

O_NAP_FUNC(dclist)
{
  sendpack(s, F_CLIST, NULL);
  
  wp(win, ""BOLD"Channel"WHITE" | Users | Topic\n");
  wp(win, "-----------------------\n");
  
  return(1);
}

O_NAP_FUNC(dclist2)
{
  sendpack(s, CHANNEL_LIST2, NULL);
  
  wp(win, ""BOLD"Channel"WHITE" | Users | Topic\n");
  wp(win, "-----------------------\n");
  
  return(1);
}

O_NAP_FUNC(dclear)
{
  scrls_t *cur, *cur1;
  
  for (cur=mscroll,cur1=cur;;cur=cur1)
  {
    if (!cur)
      break;
    if (!cur->ddd)
    {
      cur1 = cur->next;
      continue;
    }
    if (wmode && cur->chan && cur->chan != curchan)
    {
      cur1 = cur->next;
      continue;
    }
    if (cur->prev)
      cur->prev->next = cur->next;
    cur1 = cur->next;
    if (cur1)
      cur1->prev = cur->prev;
    if (cur == mscroll)
      mscroll = cur1;
    free(cur);
  }
  
  scur = NULL;
  dscr(win);
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dnews)
{
  int r;

  wp(win, "* Retrieving news...\n");
  drw(win);

  if (!fork()) {
    r = checknv(ipcs[1], 0);
    
    if (r==-1) {
      ssock(ipcs[1], ""RED"* Could not access news: %s"WHITE"\n", sys_errlist[errno]);
    } else if (r==-2) {
      ssock(ipcs[1], ""RED"* Could not access news: %s"WHITE"\n", hstrerror(h_errno));
    } else if (r==0) {
      ssock(ipcs[1], "* No news\n");
    }
    exit(1);
  }
  return(1);
}

O_NAP_FUNC(dsraw)
{
  if (num < 2) {
    usage(win, tok[0]);
    return(-3);
  }
  sendpack(s, atoi(tok[1]), "%s", cstr(str, 2));
  
  return(1);
}

O_NAP_FUNC(dquery)
{
  chans_t *cur, *cur1=NULL;
  
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  if ((cur = findchan(chanl, tok[1])))
  {
    curchan = cur;
    return(1);
  }

  if (!chanl)
  {
    chanl = (chans_t *)malloc(sizeof(chans_t));
    cur = chanl;
  }
  else
  {
    for (cur=chanl;cur!=NULL;cur=cur->next)
      cur1 = cur;
    cur = (chans_t *)malloc(sizeof(chans_t));
    cur1->next = cur;
  }
  
  cur->nm = strdup(tok[1]);
  cur->q = 1;
  cur->topic = NULL;
  cur->p = 0;
  cur->key = NULL;
  cur->next = NULL;
  cur->users = NULL;
  curchan = cur;
  
  recent = cur;
  wp(win, "%s* Querying %s%s\n", GREEN, cur->nm, WHITE);
  drw(win);
  recent = NULL;
  
  return(1);
}

O_NAP_FUNC(dcloak)
{
  sendpack(s, F_CLOAK, NULL);
  
  return(1);
}

O_NAP_FUNC(dblocklist)
{
  wp(win, "* Blocked users:\n");
  drw(win);

  sendpack(s, F_BLOCKLIST, NULL);
  
  return(1);
}

O_NAP_FUNC(dblock)
{
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  if (num == 2)
    sendpack(s, F_BLOCK, "%s", tok[1]);
  else
    sendpack(s, F_BLOCK, "%s \"%s\"", tok[1], cstr(str, 2));
  
  return(1);
}

O_NAP_FUNC(dunblock)
{
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  if (num == 2)
    sendpack(s, F_UNBLOCK, "%s", tok[1]);
  else
    sendpack(s, F_UNBLOCK, "%s \"%s\"", tok[1], cstr(str, 2));
  
  return(1);
}

O_NAP_FUNC(dwindow)
{
  if (!wmode)
  {
    wp(win, "%s* Window mode enabled%s\n", GREEN, WHITE);
    wmode = 1;
  }
  else
  {
    wp(win, "%s* Window mode disabled%s\n", GREEN, WHITE);
    wmode = 0;
  }
  
  dscr(win);
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dignore)
{
  if (num < 2) {
    usage(win, tok[0]);
    return(-3);
  }
    
  sendpack(s, IGNORE_ADD, "%s", tok[1]);
  
  return(1);
}

O_NAP_FUNC(dunignore)
{
  if (num < 2) {
    usage(win, tok[0]);
    return(-3);
  }

  sendpack(s, IGNORE_REMOVE, "%s", tok[1]);
  
  return(1);
}

O_NAP_FUNC(dignoreclear)
{
  sendpack(s, IGNORE_CLEAR, NULL);
  
  return(1);
}

O_NAP_FUNC(dignorelist)
{
  wp(win, "* Ignored users:\n");
  drw(win);

  sendpack(s, IGNORE_LIST, NULL);
  
  return(1);
}

O_NAP_FUNC(dalias)
{
  alias_t *cur, *old;
  
  if (num < 3)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  /* unlink (any) old alias of that name */
  list_unlink_cond(alias_t, alhead, old, !strcasecmp(old->nm, tok[1]));
  
  /* and add the new one */

  cur = (alias_t *)malloc(sizeof(alias_t));
  cur->nm = strdup(tok[1]);
  cur->args = strdup(cstr(str, 2));
  list_append(alias_t, alhead, cur);
  
  if (old) {
    free(old->nm);
    free(old->args);
    free(old);
    wp(win, ""BRIGHT(BLUE)"* Updated alias \"%s\""WHITE"\n", tok[1]);
  } else {
    wp(win, ""BRIGHT(BLUE)"* Added alias \"%s\""WHITE"\n", tok[1]);
  }
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dunalias)
{
  alias_t *cur;
  
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  list_unlink_cond(alias_t, alhead, cur, !strcasecmp(cur->nm, tok[1]));
  if (!cur)
  {
    wp(win, ""BRIGHT(BLUE)"* No such alias \"%s\""WHITE"\n", tok[1]);
  }
  else
  {
    free(cur->nm);
    free(cur->args);
    free(cur);
    
    wp(win, "%s* Removed alias \"%s\"%s\n", BRIGHT(BLUE), tok[1], WHITE);
  }
  drw(win);

  return(1);
}

O_NAP_FUNC(dhandler)
{
  handler_t *cur, *cur1=NULL;
  
  if (num < 3)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  if (!hndhead)
  {
    cur = (handler_t *)malloc(sizeof(handler_t));
    hndhead = cur;
  }
  else
  {
    for (cur=hndhead;;cur=cur->next)
    {
      if (!cur)
        break;
      if (cur->op == atoi(tok[1]))
      {
        wp(win, "%s* Error: a handler by that name already exists%s\n", RED, WHITE);
        drw(win);
        return(1);
      }
      cur1 = cur;
    }
    cur = (handler_t *)malloc(sizeof(handler_t));
    cur1->next = cur;
  }
  
  cur->op = atoi(tok[1]);
  cur->args = strdup(cspstr(tok, num, 2));
  cur->next = NULL;
  
  wp(win, "%s* Added handler for code (%i)%s\n", BRIGHT(BLUE), cur->op, WHITE);
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dunhandler)
{
  handler_t *cur, *cur1=NULL;
  
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  for (cur=hndhead;;cur=cur->next)
  {
    if (!cur)
    {
      wp(win, "%s* Error: handler could not be found%s\n", RED, WHITE);
      drw(win);
      return(1);
    }
    if (cur->op == atoi(tok[1]))
      break;
    cur1 = cur;
  }
  
  if (cur1)
    cur1->next = cur->next;
  else if (cur->next)
    hndhead = cur->next;
  else
    hndhead = NULL;
  free(cur->args);
  free(cur);
  
  wp(win, "%s* Removed handler for code (%i)%s\n", BRIGHT(BLUE), atoi(tok[1]), WHITE);
  drw(win);
  
  return(1);
}

O_NAP_FUNC(daliaslist)
{
  alias_t *cur;
  
  wp(win, "* Aliases:\n");
  drw(win);

  if (!alhead)
  {
    wp(win, "No aliases\n");
    drw(win);
  }
  
  for (cur=alhead;cur;cur=cur->next)
  {
    if (*cur->args == '{' && cur->args[1] == '\n')
      wp(win, "%s%s ->%s\n%s\n", BRIGHT(BLUE), cur->nm, WHITE, cur->args);
    else
      wp(win, "%s%s ->%s %s\n", BRIGHT(BLUE), cur->nm, WHITE, cur->args);
    drw(win);
  }
  
  return(1);
}

O_NAP_FUNC(dhandlerlist)
{
  handler_t *cur;
  
  if (!hndhead)
  {
    wp(win, "%s* No handlers%s\n", BRIGHT(BLUE), WHITE);
    drw(win);
  }
  
  for (cur=hndhead;cur;cur=cur->next)
  {
    if (*cur->args == '{' && cur->args[1] == '\n')
      wp(win, "%s* Handler: (%i) ->%s\n%s\n", BRIGHT(BLUE), cur->op, WHITE, cur->args);
    else
      wp(win, "%s* Handler: (%i) ->%s %s\n", BRIGHT(BLUE), cur->op, WHITE, cur->args);
    drw(win);
  }
  
  return(1);
}

O_NAP_FUNC(dkick)
{
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  if (!curchan)
  {
    wp(win, "%s* Not on a channel%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (num == 2)
    sendpack(s, CHANNEL_KICK, "%s %s", curchan->nm, tok[1]);
  else
    sendpack(s, CHANNEL_KICK, "%s %s \"%s\"", curchan->nm, tok[1], cstr(str, 2));

  return(1);
}

O_NAP_FUNC(dcbanlist)
{
  if (!curchan)
  {
    wp(win, "%s* Not on a channel%s\n", RED, WHITE);
    drw(win);
    return(1);
  }

  wp(win, "* Banned users on %s:\n", curchan->nm);
  drw(win);

  sendpack(s, CHANNEL_BAN_LIST, "%s", curchan->nm);
  
  return(1);
}

O_NAP_FUNC(dcban)
{
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  if (!curchan)
  {
    wp(win, "%s* Not on a channel%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (num == 2)
    sendpack(s, CHANNEL_BAN_ADD, "%s %s", curchan->nm, tok[1]);
  else
    sendpack(s, CHANNEL_BAN_ADD, "%s %s \"%s\"", curchan->nm, tok[1], cstr(str, 2));

  return(1);
}

O_NAP_FUNC(dcunban)
{
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  if (!curchan)
  {
    wp(win, "%s* Not on a channel%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  sendpack(s, CHANNEL_BAN_REMOVE, "%s %s", curchan->nm, tok[1]);

  return(1);
}

O_NAP_FUNC(dloadalias)
{
  char *filename;

  if (num < 2) {
    filename = home_file(ALIASFILE);
  } else {
    filename = strdup(tok[1]);
  }
  
  if (loadaliases(filename) == -1) {
    wp(win, "%s* Error loading aliases from %s: %s%s\n", RED, filename, sys_errlist[errno], WHITE);
  } else {
    wp(win, "%s* Successfully loaded aliases from %s%s\n", BRIGHT(BLUE), filename, WHITE);
  }
  drw(win);
  
  free(filename);
  return(1);
}

O_NAP_FUNC(dsavealias)
{
  char *filename;

  if (num < 2) {
    filename = home_file(ALIASFILE);
  } else {
    filename = strdup(tok[1]);
  }
  
  if (savealiases(filename) == -1) {
    wp(win, "%s* Error saving aliases to %s: %s%s\n", RED, filename, sys_errlist[errno], WHITE);
  } else {
    wp(win, "%s* Successfully saved aliases to %s%s\n", BRIGHT(BLUE), filename, WHITE);
  }
  drw(win);
  
  free(filename);
  return(1);
}

O_NAP_FUNC(dclearalias)
{
  alias_t *cur, *cur1=NULL;
  
  for (cur=alhead;;cur=cur1)
  {
    if (!cur)
    {
      if (!cur1)
        alhead = NULL;
      break;
    }
    cur1 = cur->next;
    free(cur->nm);
    free(cur->args);
    free(cur);
  }
  
  wp(win, "%s* Cleared all aliases%s\n", BRIGHT(BLUE), WHITE);
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dloadhandler)
{
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  if (loadhandlers(tok[1]) == -1)
  {
    wp(win, "%s* Error loading handlers%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  wp(win, "%s* Successfully loaded handlers%s\n", BRIGHT(BLUE), WHITE);
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dsavehandler)
{
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  if (savehandlers(tok[1]) == -1)
  {
    wp(win, "%s* Error saving handlers%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  wp(win, "%s* Successfully saved handlers%s\n", BRIGHT(BLUE), WHITE);
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dclearhandler)
{
  handler_t *cur, *cur1=NULL;
  
  for (cur=hndhead;;cur=cur1)
  {
    if (!cur)
    {
      if (!cur1)
        hndhead = NULL;
      break;
    }
    cur1 = cur->next;
    free(cur->args);
    free(cur);
  }
  
  wp(win, "%s* Cleared all handlers%s\n", BRIGHT(BLUE), WHITE);
  drw(win);
  
  return(1);
}

/* /notify <user> - Attempts to add a user to your hotlist */
O_NAP_FUNC(dnotify)
{
  hotlist_t *elt;
  
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  list_find(elt, hlist, !strcasecmp(elt->nm, tok[1]));

  if (elt) {
    wp(win, ""BRIGHT(BLUE)"* %s is already on your hotlist"WHITE"\n", tok[1]);
    drw(win);
    return(1);
  }

  wp(win, ""BRIGHT(BLUE)"* Adding %s to your hotlist"WHITE"\n", tok[1]);
  drw(win);

  /* create new hotlist entry */
  elt = (hotlist_t *)malloc(sizeof(hotlist_t));
  elt->nm = strdup(tok[1]);
  elt->on = 0;
  elt->conn = 0;

  /* add it to the list */
  list_append(hotlist_t, hlist, elt);

  /* send request to server */
  sendpack(s, ADD_NOTIFY, "%s", elt->nm);

  /* and keep file up-to-date */
  save_hotlist(win);

  return(1);
}

/* save the hotlist to the standard file. Return 0 on success, 1 on
   error. Print error message to win. */
int save_hotlist(WINDOW *win) {
  FILE *f;
  char *fn;
  hotlist_t *elt;

  fn = glistn(info.user);
  f = fopen(fn, "w");
  if (!f) {
    wp(win, ""RED"* Error writing your hotlist file %s: %s"WHITE"\n", fn, sys_errlist[errno]);
    drw(win);
    return(1);
  }
  list_forall(elt, hlist) {
    fprintf(f, "%s\n", elt->nm);
  }
  fclose(f);
  return 0;
}
    

/* /unnotify <user> - Removes a user from your hotlist */
O_NAP_FUNC(dunnotify)
{
  char *t;
  hotlist_t *elt;
  
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  t = glistn(info.user);
  
  list_unlink_cond(hotlist_t, hlist, elt, !strcmp(elt->nm, tok[1]));

  if (!elt) {
    wp(win, ""BRIGHT(BLUE)" * %s was not on your hotlist"WHITE"\n", tok[1]);
    drw(win);
    return(1);
  }

  wp(win, ""BRIGHT(BLUE)"* Removing %s from your hotlist"WHITE"\n", tok[1]);
  drw(win); 
  
  free(elt->nm);
  free(elt);

  sendpack(s, REMOVE_NOTIFY, "%s", tok[1]);
  
  /* and keep file up-to-date */
  save_hotlist(win);

  return(1);
}

/* /hotlist - Lists users on your hotlist that are on */
O_NAP_FUNC(dhotlist)
{
  hotlist_t *elt;
  
  if (!hlist)
  {
    wp(win, "%s* No users on your hotlist%s\n", BRIGHT(BLUE), WHITE);
    drw(win);
    return(1);
  }

  wp(win, "%s* Hotlist:%s\n", BRIGHT(BLUE), WHITE);
  
  list_forall(elt, hlist) {
    wp(win, "%s* %s - %s%s\n", (elt->on)?BRIGHT(GREEN):BRIGHT(BLUE), elt->nm, (elt->on)?"On":"Off", WHITE);
  }
  
  drw(win);
  return(1);
}

O_NAP_FUNC(dwstats)
{
  wstats(win);
  
  return(1);
}

O_NAP_FUNC(dsay)
{
  if (!curchan)
  {
    wp(win, "%s* Not on a channel%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  sendpack(s, F_SAY, "%s %s", curchan->nm, fixquotes(cstr(str, 1)));
  
  return(1);
}

O_NAP_FUNC(dabout)
{
  wp(win, "%s* nap v%s by Ignitor (Kevin Sullivan), modified by Peter Selinger and others%s\n", BRIGHT(RED), VERSION, WHITE);
  wp(win, "%s* Thanks: nocarrier, napster, fizban, amputee, nytr0, fletch, Plugh, pea and the rest of the admins and mods%s\n", BRIGHT(RED), WHITE);
  wp(win, "%s* Thanks: nyt for info on ncurses and drscholl for opennap (useful for testing)%s\n", BRIGHT(RED), WHITE);
  wp(win, "%s* Special Thanks: everyone that has emailed me with bug reports, comments and suggestions%s\n", BRIGHT(RED), WHITE);
  if (num == 5)
    wp(win, "%s* Extra Special Thanks: Brigand for being such a bitch :P%s\n", BRIGHT(RED), WHITE);
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dirc)
{
  struct sockaddr_in dst;
  struct passwd *pw;
  char *t=NULL;
  int i;

  if (num < 2)
  {
    if (!ircsock)
      return(1);
    wp(win, "%s* IRC mode toggled: On%s\n", RED, WHITE);
    drw(win);
    ircmode = 1;
    return(1);
  }

  if (strcasecmp(tok[1], "server"))
  {
    if (!ircsock)
      return(1);
    if (!checkouts(ircsock, cstr(str, 1), tok+1, num-1, win))
      ssock(ircsock, "%s\n", cstr(str, 1));
    return(1);
  }
  
  if (num < 4) {
    usage(win, tok[0]);
    return(-3);
  }  

  if (ircsock)
  {
    wp(win, "%s* Disconnect from your current IRC server first%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  if (!resolve(tok[2], &dst.sin_addr)) {
    wp(win, "%s* Error resolving host %s: %s%s\n", RED, tok[2], hstrerror(h_errno), WHITE);
    drw(win);
    return(1);
  }
  dst.sin_port = htons(atoi(tok[3]));
  dst.sin_family = AF_INET;
  
  ircsock = socket(AF_INET, SOCK_STREAM, 0);
  
  if (connect(ircsock, (struct sockaddr *)&dst, sizeof(dst)) == -1)
  {
    wp(win, "%s* Error connecting socket: %s%s\n", RED, sys_errlist[errno], WHITE);
    drw(win);
    close(ircsock);
    ircsock = 0;
    return(1);
  }
  
  addsock(ircsock, "irc", S_R, inirc);
  
  pw = getpwuid(getuid());
  if (pw)
  {
    msprintf(&t, "%s", pw->pw_gecos);
    for (i=0;t[i];i++)
    {
      if (t[i] == ',')
      {
        t[i] = 0;
        break;
      }
    }
  }
  
  ssock(ircsock, "NICK %s\nUSER %s NULL NULL :%s\n", info.user, info.user, (t)?t:info.user);
  if (t)
    free(t);
  mnick = strdup(info.user);
  ircmode = 1;
  
  return(1);
}

O_NAP_FUNC(dkickall)
{
  chans_t *cur;

  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  if (!curchan)
  {
    wp(win, "%s* Not on a channel%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  for (cur=chanl;cur;cur=cur->next)
  {
    if (finduser(cur, tok[1]))
    {
      if (num == 2)
        sendpack(s, CHANNEL_KICK, "%s %s", cur->nm, tok[1]);
      else
        sendpack(s, CHANNEL_KICK, "%s %s \"%s\"", cur->nm, tok[1], cstr(str, 2));
    }
  }

  return(1);
}


O_NAP_FUNC(deval)
{
  unsigned char *p;

  if (num < 2) {
    usage(win, tok[0]);
    return(-3);
  }

  p = dovars(tok[1]);
  
  wp(win, "%s\n", p);
  drw(win);

  free(p);
  
  return(1);
}

O_NAP_FUNC(dset)
{
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  if (num > 2)
    chset(tok[1], cstr(str, 2));
  else /* num == 2 */
    chset(tok[1], "");

  if (nvar("noechosets") != 1)
  {
    wp(win, "%s* Set variable %s to \"%s\"%s\n", BRIGHT(BLUE), tok[1], getval(tok[1]), WHITE);
    drw(win);
  }
  
  return(1);
}

O_NAP_FUNC(dunset)
{
  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  if (!getval(tok[1]))
  {
    if (nvar("noechosets") != 1)
    {
      wp(win, "%s* %s was not set%s\n", RED, tok[1], WHITE);
      drw(win);
    }
    return(1);
  }
  
  delset(tok[1]);
  
  if (nvar("noechosets") != 1)
  {
    wp(win, "%s* Unset variable %s%s\n", BRIGHT(BLUE), tok[1], WHITE);
    drw(win);
  }
  
  return(1);
}

O_NAP_FUNC(dloadconfig)
{
  int r;
  char *fn;

  if (num < 2)
    fn = getval("configfile");
  else
    fn = tok[1];

  if (!fn) 
    fn = home_file(CONFIGFILE);
  
  fn = strdup(fn);  /* we have to make a copy, since loading a
		       configuration will also update "configfile",
		       and thus fn would get freed */

  r = loadsets(fn, win, 1, 1);

  if (r == -1) {
    wp(win, "%s* Error loading config file %s: %s%s\n", RED, fn, sys_errlist[errno], WHITE);
  } else if (r == 1) {
    wp(win, "%s* Loaded config file %s with some warnings%s\n", BRIGHT(BLUE), fn, WHITE);
  } else {  
    wp(win, "%s* Successfully loaded config file %s%s\n", BRIGHT(BLUE), fn, WHITE);
  }

  free(fn);

  drw(win);
  
  return(1);
}

O_NAP_FUNC(dsaveconfig)
{
  char *fn;  /* config filename */

  if (num < 2)
    fn = getval("configfile");
  else
    fn = tok[1];

  if (!fn) 
    fn = home_file(CONFIGFILE);
  
  if (savesets(fn) == -1)
  {
    wp(win, "%s* Error saving config file %s: %s%s\n", RED, fn, sys_errlist[errno], WHITE);
    drw(win);
    return(1);
  }
  
  wp(win, "%s* Successfully saved config file %s%s\n", BRIGHT(BLUE), fn, WHITE);
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dloadchannels)
{
  char *fn;
  sock_t *sk;

  if (num < 2) {
    char *tmp = NULL;
    msprintf(&tmp, CHANNELFILE, info.user);
    fn = home_file(tmp);
    free(tmp);
  }
  else {
    fn = strdup(tok[1]);
  }
  
  sk = findsock("server");
  if (sk)
  {
    if (loadchans(sk->fd, fn)==-1) {
      wp(win, ""RED"* Could not load channels from %s: %s"WHITE"\n", fn, \
	 sys_errlist[errno]);
    } else {
      wp(win, "* Joining channels from %s\n", fn);
    }
  } else 
  {
    wp(win, "* Not connected to server\n");
  }
  free(fn);

  drw(win);
  
  return(1);
}

O_NAP_FUNC(dsavechannels)
{
  char *fn;

  if (num < 2) {
    char *tmp = NULL;
    msprintf(&tmp, CHANNELFILE, info.user);
    fn = home_file(tmp);
    free(tmp);
  }
  else {
    fn = strdup(tok[1]);
  }
  
  if (savechans(chanl, fn)==-1) {
    wp(win, ""RED"* Could not save channels to %s: %s"WHITE"\n", fn, \
       sys_errlist[errno]);
  } else {
    wp(win, "* Saved channels to %s\n", fn);
  }
  free(fn);

  drw(win);
  
  return(1);
}

O_NAP_FUNC(decho)
{
  wp(win, "%s\n", cstr(str, 1));
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dresults)
{
  switchtoscreen(RESULT_SCREEN);
  
  return(1);
}

O_NAP_FUNC(dexec)
{
  char buf[512];
  FILE *f;
  int fl=0, i;

  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  if (tok[1][0] == '-' && tok[1][1] == 'o')
  {
    fl = 1;
    f = popen(cstr(str, 2), "r");
  }
  else
    f = popen(cstr(str, 1), "r");
  
  if (!f)
  {
    wp(win, "%s* Error executing command%s\n", RED, WHITE);
    drw(win);
    return(1);
  }
  
  memset(buf, 0, sizeof(buf));
  fread(&buf, sizeof(buf)-1, 1, f);
  pclose(f);
  for (i=0;buf[i];i++)
    if (buf[i] == '\n' || buf[i] == '\r')
    {
      strncpy(buf+i, buf+i+1, strlen(buf+i+1));
      buf[strlen(buf)-1] = 0;
    } 
  
  if (!fl)
  {
    wp(win, "%s\n", buf);
    drw(win);
    return(1);
  }
  
  if (!curchan)
  {
    wp(win, "%s* Not on a channel%s\n", RED, WHITE);
    drw(win);
  }
  else if (curchan->q == 1)
  {
    sendpack(s, F_TELL, "%s %s", curchan->nm, fixquotes(buf));
    recent = curchan;
    wp(win, "%s* --> (%s%s%s)%s %s\n", GREEN, WHITE, curchan->nm, GREEN, WHITE, buf);
    drw(win);
    recent = NULL;
  }
  else if (curchan->q == 2 && buf[0] != '\0')
  {
    ssock(ircsock, "PRIVMSG %s :%s\n", curchan->nm, buf);
    recent = curchan;
    wp(win, "%s<%s%s%s>%s %s\n", BRIGHT(MAGENTA), WHITE, mnick, BRIGHT(MAGENTA), WHITE, buf);
    drw(win);
    recent = NULL;
  }
  else if (buf[0] != '\0')
    sendpack(s, F_SAY, "%s %s", curchan->nm, fixquotes(buf));
  
  return(1);
}

/* /timer <[min:]sec> command */
O_NAP_FUNC(dtimer)
{
  int sec, min;
  char *data, *p, *q;

  if (num < 3)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  p = tok[1];

  /* parse minutes */
  if (strchr(p, ':')) {
    min = strtol(p, &q, 10);
    if (!q || *q != ':') {
      min = -1;
    } else {
      q++;
    }
  } else {
    min = 0;
    q = p;
  }
  
  /* parse seconds */
  if (min>=0) {
    sec = strtol(q, &p, 10);
    if (!p || *p || p==q) {
      sec = -1;
    }
  }

  if (min<0 || sec<0)
  {
    wp(win, "%s* Invalid time \"%s\". Usage: /%s <min:sec> <command>%s\n", RED, tok[0], tok[1], WHITE);
    drw(win);
    return(1);
  }
  
  sec = 60*min + sec;
  
  /* set timer */
  data = strdup(cstr(str, 2));
  addtimer(sec, timed_command, (void *)data, data);

  wp(win, ""BRIGHT(BLUE)"* Scheduled (in %i:%02i): "WHITE"%s\n", sec/60, sec%60, data);
  drw(win);

  return(1);
}

O_NAP_FUNC(dif)
{
  char *t, *b;
  int i, c=0, r, f = 0;
  
  if (num < 2) {
    usage(win, tok[0]);
    return(-3);
  }

  t = (char *)malloc(512);
  memset(t, 0, 512);
  strcpy(t, tok[1]);
  
  b = strev(t, '(', ')', cmp);
  if (b)
    r = atoi(b);
  else
    r = -1;
  if (b)
    free(b);
  
  if (r == -1)
  {
    wp(win, "%s* Error parsing script%s\n", RED, WHITE);
    drw(win);
    free(t);
    return(1);
  }
  else if (!r)
  {
    free(t);
    return(1);
  }
  
  memset(t, 0, 512);
  
  for (i=0,c=0;str[i];i++)
  {
    if (!c && f)
      break;
    if (c < 0)
      c = 0;
    if (str[i] == '(')
    {
      f = 1;
      c++;
    }
    else if (str[i] == ')')
      c--;
  }
  
  t[0] = '/';
  strcpy(t+1, str+i);
  
  for (i=0,c=0;t[i];i++)
  {
    if (c < 0)
      c = 0;
    if (t[i] == '{' && !c)
    {
      t[i] = ' ';
      c++;
    }
    else if (t[i] == '{')
      c++;
    else if (t[i] == '}' && c == 1)
    {
      t[i] = ' ';
      c--;
    }
    else if (t[i] == '}')
      c--;
  }
  
  r = parseout(s, t, win);
  
  free(t);
  
  return(r);
}

O_NAP_FUNC(dwhile)
{
  char *t, *b, *cm;
  int i, c=0, r, f = 0;
  
  if (num < 2) {
    usage(win, tok[0]);
    return(-3);
  }

  t = (char *)malloc(2048);
  memset(t, 0, 2048);
  
  cm = strdup(tok[1]);
  
  b = strev(cm, '(', ')', cmp);
  if (b)
    r = atoi(b);
  else
    r = -1;
  if (b)
    free(b);
  
  if (r == -1)
  {
    wp(win, "%s* Error parsing script%s\n", RED, WHITE);
    drw(win);
    free(cm);
    free(t);
    return(1);
  }
  else if (!r)
  {
    free(cm);
    free(t);
    return(1);
  }
  
  memset(t, 0, 512);
  
  for (i=0,c=0;str[i];i++)
  {
    if (!c && f)
      break;
    if (c < 0)
      c = 0;
    if (str[i] == '(')
    {
      f = 1;
      c++;
    }
    else if (str[i] == ')')
      c--;
  }
  
  t[0] = '/';
  strcpy(t+1, str+i);
  
  for (i=0,c=0;t[i];i++)
  {
    if (c < 0)
      c = 0;
    if (t[i] == '{' && !c)
    {
      t[i] = ' ';
      c++;
    }
    else if (t[i] == '{')
      c++;
    else if (t[i] == '}' && c == 1)
    {
      t[i] = ' ';
      c--;
    }
    else if (t[i] == '}')
      c--;
  }
  
  while (1)
  {
    b = strev(cm, '(', ')', cmp);
    if (b)
    {
      r = atoi(b);
      free(b);
    }
    else
      break;
    if (!r)
      break;
    r = parseout(s, t, win);
    if (r == -3)       /* break out of nested loops; return -3 again */
      break;
    else if (r == -4)  /* break out of this loop only; return 1 */
    {
      r = 1;
      break;
    }
    if (lpbrk) 
    {
      lpbrk = 0; 
      /* note: prevoiusly r=1 was returned here, making it impossible
	 to interrupt a nested loop. -3 will bring us to the top
	 level. -PS */
      r = -3;
      break;
    }
  }
  
  free(t);
  free(cm);
  
  return(r);
}

O_NAP_FUNC(dinc)
{
  int t;
  char *p;
  char *tp=NULL;

  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  p = getval_t(tok[1], &t);
  if (!p) {
    wp(win, "%s* Variable %s is not set%s\n", RED, tok[1], WHITE);
    drw(win);
    return(1);
  }

  if (t && strlen(p))
  {
    tp = strdup(p+1);
    chset(tok[1], tp);
    free(tp);
  }
  else if (!t)
  {
    msprintf(&tp, "%i", atoi(p)+1);
    chset(tok[1], tp);
    free(tp);
  }

  p = getval_t(tok[1], &t);
  
  if (nvar("noechosets") != 1)
  {
    wp(win, "%s* Set variable %s to \"%s\"%s\n", BRIGHT(BLUE), tok[1], p, WHITE);
    drw(win);
  }
  
  return(1);
}

O_NAP_FUNC(ddec)
{
  char *p;
  char *tp=NULL;
  int t;

  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  p = getval_t(tok[1], &t);
  if (!p) {
    wp(win, "%s* Variable %s is not set%s\n", RED, tok[1], WHITE);
    drw(win);
    return(1);
  }

  if (t && strlen(p))
  {
    tp = strdup(p);
    tp[strlen(tp)-1] = 0;
    chset(tok[1], tp);
    free(tp);
  }
  else if (!t)
  {
    msprintf(&tp, "%i", atoi(p)-1);
    chset(tok[1], tp);
    free(tp);
  }

  p = getval_t(tok[1], &t);
  
  if (nvar("noechosets") != 1)
  {
    wp(win, "%s* Set variable %s to \"%s\"%s\n", BRIGHT(BLUE), tok[1], p, WHITE);
    drw(win);
  }
  
  return(1);
}
  
O_NAP_FUNC(dstop)
{
  return(2);
}

O_NAP_FUNC(ddone)
{
  return(-3);
}

O_NAP_FUNC(dbreak)
{
  return(-4);
}

O_NAP_FUNC(dlastlog)
{
  scrls_t *cur, *cur1 = scrend(mscroll);
  char buf[32];

  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  wp(win, "%s* Matches:%s\n", BRIGHT(MAGENTA), WHITE);
  
  memset(buf, 0, sizeof(buf));
  memset(buf, ' ', TABSIZE);
  
  for (cur=mscroll;cur;cur=cur->next)
  {
    if (strstr(cur->ln, tok[1]) && cur->ddd != 2)
    {
      if (cur->own)
        cur = cur->own;
      
      wp(win, "%s\n", cur->ln);
      cur = cur->next;
      if (!cur)
        break;
      
      if (cur && cur->own)
      {
        while (cur && cur->own)
        {
          wp(win, "%s%s\n", buf, cur->ln);
          cur = cur->next;
        }
        if (!cur)
          break;
        cur = cur->prev;
      }
    }
    
    if (cur == cur1)
    {
      cur1 = cur1->next;
      break;
    }
  }
  
  wp(win, "%s* End of matches%s\n", BRIGHT(MAGENTA), WHITE);
  drw(win);
  
  for (cur=cur1;cur;cur=cur->next)
    cur->ddd = 2;
  
  return(1);
}

/* user command to rebuild the library */
O_NAP_FUNC(drebuild)
{
  wp(win, "* Rebuilding your library\n");
  drw(win);
  
  if (fork())
    return(1);
  
  if (rebuild(s, info.shared_filename, info.up) == -1)
  {
    ssock(ipcs[1], "%s* Error building your library%s\n", RED, WHITE);
    exit(1);
  }
  
  ssock(ipcs[1], "* Successfully rebuilt your library\n");
  exit(1);
}

/* user command to rebuild the library, but only if necessary */
O_NAP_FUNC(dupdate)
{
  int ret;

  ret = up_to_date(info.shared_filename, getval("upload"));
  if (ret) {
    wp(win, "* Your library is up-to-date.\n");
    drw(win);
    return(1);
  }
  wp(win, "* Your library is not up-to-date. Rebuilding...\n");
  drw(win);
  
  if (fork())
    return(1);
  
  if (rebuild(s, info.shared_filename, info.up) == -1)
  {
    ssock(ipcs[1], "%s* Error building your library%s\n", RED, WHITE);
    exit(1);
  }
  
  ssock(ipcs[1], "* Successfully rebuilt your library\n");
  exit(1);
}

O_NAP_FUNC(dchupload)
{
  if (num < 2)
  {
    wp(win, "* Current upload path: %s\n", info.up);
    drw(win);
    return(1);
  }
  
  free(info.up);
  info.up = strdup(tok[1]);
  
  wp(win, "* Set your upload path to: %s\n", info.up);
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dtlist)
{
  wp(win, "* Timers:\n");
  timerlist(win);
  
  return(1);
}

O_NAP_FUNC(dnoprint)
{
  if (noprint)
    noprint = 0;
  else
    noprint = 1;
  
  return(1);
}

O_NAP_FUNC(ddebug)
{
  if (num < 2) {
    usage(win, tok[0]);
    return(-3);
  }

  chset("debug", tok[1]);
  
  wp(win, "%s* Debugging level set to: %s%s\n", RED, getval("debug"), WHITE);
  drw(win);

  return(1);
}

O_NAP_FUNC(dserv)
{
  struct sockaddr_in dst;
  int frmlen = sizeof(dst);
  
  if (getpeername(s, (struct sockaddr *)&dst, &frmlen))
  {
    wp(win, "%s* Not connected to server%s\n", RED, WHITE);
    drw(win);
    return(1);
  }

  wp(win, "* Currently connected to: %s:%i\n", inet_ntoa(dst.sin_addr), ntohs(dst.sin_port));
  drw(win);
  
  return(1);
}

O_NAP_FUNC(dserver)
{
  int t;
  sock_t *sk;
  chans_t *cur;
  char *t1 = NULL;
  int c;

  if (num < 2)
  {
    usage(win, tok[0]);
    return(-3);
  }
  
  if (!strchr(tok[1], ':'))
  {
    wp(win, "%s* Invalid parameter%s\n", RED, WHITE);
    drw(win);
    return(1);
  }

  t = conn(tok[1], PORT);
  if (t == -1)
  {
    drw(win);
    return(1);
  }
  wp(win, "Logging in...\n");
  drw(win);
  sk = findsock("server");
  if (sk) {
    close(sk->fd);
    delsock(sk->fd);
  }
  
  c = info.conn ? atoi(info.conn) : 0;
  if (c < 0 || c > 10)
    c = 0;
  if (login(t, info.user, info.pass, info.port, CLIENT, c, info.email) == -1)
  {
    drw(win);
    close(t);
    return(1);
  }
  
  sk = addsock(t, "server", S_R, inserv);
  
  t1 = glistn(info.user);
  checkhotlist(t, t1);
  free(t1);
  
  curchan = NULL;
  
  if (chanl)
  {
    for (cur=chanl;cur;cur=cur->next)
      if (!cur->q)
        sendpack(sk->fd, F_JOIN, "%s", cur->nm);
    delnapchans();
  }
  
  upchan(chanl);
  
  return(1);
}

/* once a second or so check download list, both for queued items that
   might be requested, and for items that are expired. Also do the
   check for quit_after_transfers. */

/* these list manipulations are inefficient, but so is all the rest of
   nap, so we don't worry about it too much. */

void qevent(void) {
  download_t *dtask, *dnext, *a;
  upload_t *utask, *unext, *b;
  int c=0, d;
  int autopurge = nvar("autopurge");  
  sock_t *sk;
  time_t t = time(0);
  
  /* 1. check whether to quit if all transfers have completed */
  if (quit_after_transfers) {
    list_count(down, c, a, !STOPPED(a->state));
    list_count(up, d, b, !STOPPED(b->state));
    if (c==0 && d==0) {
      quit_now = 1;
      wp(wchan, "* All uploads and downloads have been completed. Quitting now.\n");
      drw(wchan);
      return;
    }
  }

  /* 2. Check for remotely queued items that should be queued */

  list_forall(dtask, down) {
    if (dtask->state == RQUEUED && t-dtask->r_time > 30) {
      dtask->state = QUEUED;
    }
  }

  /* 3. check for queued items that can be activated */
  sk = findsock("server");
  if (sk)
  {
    list_forall(dtask, down) {
      if (dtask->state == QUEUED || 
	  (dtask->state == RQUEUED && t-dtask->r_time>=15)) {
	if (!download_limit_reached(dtask->nick)) {
	  sendpack(sk->fd, F_DGET, "%s \"%s\"", dtask->nick, dtask->rfn);
	  if (dtask->state == QUEUED) {
	    wp(wchan, "* Getting \"%s\" from %s\n", dtask->fn, dtask->nick);
	    drw(wchan);
	    dtask->state = REQUESTED;
	  } else {
	    dtask->state = RRQUEUED;
	  }
	  dtask->c_time = t;
	}
      }
    }
  }  
  
  /* 4. check for items that are timed out */
  
  list_forall(dtask, down) {
    if (TIMED(dtask->state) && t - dtask->c_time > 90) {

      /* if server's lazyness timed us out, re-request the item */
      if (sk && (dtask->state == REQUESTED || dtask->state == RRQUEUED)) {
	sendpack(sk->fd, F_DGET, "%s \"%s\"", dtask->nick, dtask->rfn);
	dtask->c_time = t;
	continue;
      }

      /* if remote client's lazyness timed us out, or server not
         available, go to TIMED_OUT */
      if (dtask->state == CONNECTING) {
	if (dtask->sk) {   /* should always succeed */
	  delsock(dtask->sk->fd);
	}
      }
      if (dtask->state == CONNECTING || dtask->state == WAITING) {
	free(dtask->check);
      }
      dtask->state = TIMED_OUT;
      dtask->d_time = t;
      wp(wchan, "* Download of \"%s\" from %s timed out\n", dtask->fn, dtask->nick);
      drw(wchan);
    }
  }

  list_forall(utask, up) {
    if (TIMED(utask->state) && t - utask->c_time > 90) {
      /* go to TIMED_OUT */
      if (utask->state == CONNECTING || utask->state == CONNECTING1) {
	if (utask->sk) {  /* should always succeed */
	  delsock(utask->sk->fd);
	}
      }
      utask->state = TIMED_OUT;
      utask->d_time = t;
      wp(wchan, "* Upload of \"%s\" to %s timed out\n", utask->fn, utask->nick);
      drw(wchan);
    }
  }


  /* 5. Do autopurge */

  if (autopurge>=0) {
    for (dtask=down; dtask; dtask=dnext) {
      /* remember next in case we delete this item. This works because
         ddownload deletes precisely the one item and no others */
      dnext = dtask->next;  
      if (STOPPED(dtask->state) && t-dtask->d_time>=autopurge) {
	ddownload(wchan, dtask);
      }
    }
    for (utask=up; utask; utask=unext) {
      /* remember next in case we delete this item. This works because
         dupload deletes precisely the one item and no others */
      unext = utask->next; 
      if (STOPPED(utask->state) && t-utask->d_time>=autopurge) {
	dupload(utask);
      }
    }
  }	  
}
  
/* a hook called whenever a download completes: check immediately if
   there are remotely queued items by this user, and if yes, decrease
   its r_time so that it will be requested immediately (provided other
   criteria are also met, such as, there are no other downloads whose
   turn it is first). */
void rqueued_hook(char *nick) {
  download_t *task;
  sock_t *sk;
  
  sk = findsock("server");
  if (sk)
  {
    list_find(task, down, !strcasecmp(task->nick, nick) && task->state == RQUEUED);
    if (!task) 
      return;
    task->r_time -= 100;
  }
}
  
/* end user commands */
/* ---------------------------------------------------------------------- */
