/* Copyright (c) 2000  Kevin Sullivan <nite@gis.net>
 *
 * Please refer to the COPYRIGHT file for more information.
 */

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>

#include "md5.h"
#include "mp3s.h"

/* arrays are indexed using bitrate index from frame header */
/* for MPEG-1 Layer III */
int brate1[] = { 0, 32, 40, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320, -1 };
/* for MPEG-2 Layer II or III */
int brate2[] = { 0, 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160, -1 };

int freq1[] = { 44100, 48000, 32000, 50000 }; /* MPEG-1 */
int freq2[] = { 22050, 24000, 16000, 50000 }; /* MPEG-2 */


/* check for valid mp3 header. Return 1 if ok. */
int checkhdr(hdr_t *h)
{
  if (h->sync != 0xfff ||
      h->layer != 1 ||
      h->brate == 15 ||
      h->freq == 3 ||
      h->emph == 2)
    return(0);
  
  return(1);
}

/* get "real" header info from an mp3 file. The stream f is already
   positioned to skip any id3v2 header, if present */
hdr_t *getrhdr(int f)
{
  hdr_t *r = (hdr_t *)malloc(sizeof(hdr_t));
  unsigned char bw[4];
  unsigned char *xxx=(unsigned char *)r;

  read(f, bw, sizeof(bw));

  /* Note: to ensure portability, use explicit assignments and shift
     operators, rather than writing binary data directly into the hdr_t
     structure. The difference in efficiency is negligible. -PS */
  r->emph      = (bw[3] & 0x03) >> 0;  
  r->original  = (bw[3] & 0x04) >> 2;  
  r->copyright = (bw[3] & 0x08) >> 3;  
  r->mode_ext  = (bw[3] & 0x30) >> 4;
  r->mode      = (bw[3] & 0xc0) >> 6;
  r->pvt       = (bw[2] & 0x01) >> 0;
  r->pad       = (bw[2] & 0x02) >> 1;
  r->freq      = (bw[2] & 0x0c) >> 2;
  r->brate     = (bw[2] & 0xf0) >> 4;
  r->prot      = (bw[1] & 0x01) >> 0;
  r->layer     = (bw[1] & 0x06) >> 1;
  r->id        = (bw[1] & 0x08) >> 3;
  r->sync      = (bw[1] & 0xf0) >> 4;
  r->sync     |= (bw[0] & 0xff) << 4;

  if (!checkhdr(r))
  {
    free(r);
    return(NULL);
  }
  
  return(r);
}

/* get the mp3 information (bitrate, frequency, length in seconds,
   size in bytes, and checksum) from the named mp3 file */
mhdr_t *gethdr(char *fn)
{
  hdr_t *m;
  mhdr_t *r;
  int f, i;
  unsigned long size = 0;
  struct stat st;
  id3v2_t tag;
  char *buf, rdt[4];
  unsigned char ret[16];
  
  f = open(fn, O_RDONLY);
  if (f == -1)
    return(NULL);
  
  /* first check whether there is an id3v2 header */
  memset(rdt, 0, sizeof(rdt));
  read(f, &rdt, 3);
  if (!strcmp(rdt, "ID3"))  /* found id3v2 header */
  {
    lseek(f, 0, SEEK_SET);
    read(f, &tag, sizeof(tag));
    /* corrected a bug here: according to id3v2 spec, the size is
     * represented as a "synchsafe", or base-128, integer. 
     * This means 28 bits encode 32 bits as follows:
     *  0xxxxxxx 0xxxxxxx 0xxxxxxx 0xxxxxxx means
     *  0000xxxx xxxxxxxx xxxxxxxx xxxxxxxx.         -PS
     */
    size = (long)tag.size[0] << (24-3)|(long)tag.size[1] << (16-2)|(long)tag.size[2] << (8-1)|(long)tag.size[3];

    /* skip the id3v2 header */
    if (tag.flags & 0x10) {
      /* this is new in id3v2 version 4.0: tag.flags & 0x10 indicates
	 the presence of a "footer", which is 10 bytes in length, and which is
	 *not* included in the stored tag size */
      lseek(f, size+20, SEEK_SET);
    } else {
      lseek(f, size+10, SEEK_SET);
    }
  }
  else
  {
    memset(rdt, 0, sizeof(rdt));
    lseek(f, 0, SEEK_SET);
  }
  /* now the stream is positioned at the beginning of the "real" mp3 data. */
  m = getrhdr(f);
  if (!m)
  {
    close(f);
    return(NULL);
  }
  
  r = (mhdr_t *)malloc(sizeof(mhdr_t));
  memset(r, 0, sizeof(mhdr_t));
  
  if (m->id)
  {
    r->bitrate = brate1[m->brate];
    r->freq = freq1[m->freq];
  }
  else
  {
    r->bitrate = brate2[m->brate];
    r->freq = freq2[m->freq];
  }
  
  free(m);
  
  fstat(f, &st);
  
  r->len = ((st.st_size*8)/r->bitrate)/1000;
  r->sz = st.st_size;
  
  /* according to drscholl's napster spec, the official client hashes
  the first 299008 bytes, or 300032 bytes, *not* 300000 bytes as nap
  used to do. I did three random spot checks, and in all three cases,
  the hash matched at 300032. Thus, we now use 300032 bytes, hoping
  that this will be more conforming to general usage. By the way,
  299008 = 1024 * 292, and 300032 = 1024 * 293. */

  /* note: we no longer allocate 300032 bytes, since md5 operates on
     streams anyway. */

  lseek(f, -4, SEEK_CUR); /* rewind to beginning of mp3 header (frame synch) */
  if (md5_stream_blocks(f, 293, ret)) {
    free(r);
    close(f);
    return NULL;
  }
  close(f);

  for (i=0;i<16;i++)
    sprintf(r->check, "%s%02x", r->check, ret[i]);
  
  r->check[32] = '\0';
  
  return(r);
}

/* the next function really belongs in md5.c, but since that file is
   taken verbatim from GNU, we prefer not to change it. */

/* Compute MD5 message digest for 1024 * KB bytes read from file FD.
   The resulting message digest number will be written into the 16
   bytes beginning at RESBLOCK. Return 1 on error. */
int md5_stream_blocks (int fd, int kb, void *resblock)
{
  struct md5_ctx ctx;
  char buffer[1024 + 72];
  size_t sum;

  /* Initialize the computation context.  */
  md5_init_ctx (&ctx);

  /* Iterate over full file contents.  */
  while (kb>0)
    {
      /* We read the file in blocks of 1024 bytes.  One call of the
	 computation function processes the whole buffer so that with the
	 next round of the loop another block can be read.  */
      size_t n;
      sum = 0;

      /* Read block.  Take care for partial reads.  */
      do
	{
	  n = read (fd, buffer + sum, 1024 - sum);

	  sum += n;
	}
      while (sum < 1024 && n != 0 && n != -1);
      if (n == -1)
        return 1;

      /* If end of file is reached, end the loop.  */
      if (n == 0) {
	/* Add the last bytes if necessary.  */
	if (sum > 0)
	  md5_process_bytes (buffer, sum, &ctx);
	break;
      }
      
      /* Process buffer with 1024 bytes.  Note that 1024 % 64 == 0   */
      md5_process_block (buffer, 1024, &ctx);
      
      kb--;
    }

  /* Construct result in desired memory.  */
  md5_finish_ctx (&ctx, resblock);
  return 0;
}
