/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_font_manager.cpp 
 * @brief Implements the global font manager class that handles all text output related tasks.
 */

#ifdef __APPLE__
#include <CoreServices/CoreServices.h>
#include <Carbon/Carbon.h>
#endif

#include "myx_gc_font_manager.h"

//#define FONTCLASS FTGLPixmapFont
#define FONTCLASS FTGLBitmapFont

//----------------------------------------------------------------------------------------------------------------------

/**
 * Converts the given string into a font weight value.
 * Allowed values are: normal | bold | bolder | lighter | 100 | 200 | 300| 400 | 500 | 600 | 700 | 800 | 900 | inherit
 *
 * @param weight A string containing a font weight specification.
 * @return A font weight value corresponding to the given description.
 */
int convertFontWeight(const string weight)
{
  int result = 0;
#ifdef USE_FONTCONFIG
  if (weight == "normal")
    result = FC_WEIGHT_NORMAL;
  else if (weight == "bold")
    result = FC_WEIGHT_BOLD;
  else if (weight == "bolder")
    result = FC_WEIGHT_EXTRABOLD;
  else if (weight == "lighter")
    result = FC_WEIGHT_LIGHT;
  else if (weight == "inherit")
    result = FC_WEIGHT_NORMAL;
  else
    result = atoi(weight.c_str());
#else // !USE_FONTCONFIG
  if (weight == "normal")
    result = 400;
  else
    if (weight == "bold")
      result = 700;
    else
      if (weight == "bolder")
        result = 800;
      else
        if (weight == "lighter")
          result = 300;
        else
          if (weight == "inherit")
            result = 400;
          else
            result = atoi(weight.c_str());
#endif

  return result;
}

//----------------- CFontManager ---------------------------------------------------------------------------------------

static CFontManager* internalManager;
static int lockCount = 0;

// Returns the current font manager (there is always only one).
CFontManager* fontManager(void)
{
  return internalManager;
}

//----------------------------------------------------------------------------------------------------------------------

/*
 * Increases the lock count of the font manager. If the manager does not yet exist it is created.
 *
 */
void lockFontManager(void)
{
  if (internalManager == NULL)
  {
    internalManager = new CFontManager();
#ifdef USE_FONTCONFIG
    if (!FcInit())
      g_warning("Could not initialize Fontconfig");
#endif
  }
  ++lockCount;
}

//----------------------------------------------------------------------------------------------------------------------

// Returns the current font manager (there is always only one).
void unlockFontManager(void)
{
  if (lockCount > 0)
  {
    --lockCount;
    delete internalManager;
    internalManager = NULL;
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Constructor of the class.
 */
CFontManager::CFontManager(void)
{
  FUseLocalDisplayLists = true; // Set only to true if the text is as a whole not rendered in a display list.    

#ifndef USE_FONTCONFIG
  // Fill our font file list with some predefined font files.
  FontFileEntry* entry;

  // "Arial" normal, bold, italic, bold italic.
  entry = new FontFileEntry;
  entry->entries[false][false] = "Arial.ttf"; entry->entries[false][true] = "Arialbd.ttf"; 
  entry->entries[true][false] = "Ariali.ttf";  entry->entries[true][true] = "Arialbi.ttf"; 
  entry->useCount = 5;
  FFiles["Arial"] = entry;
  FFiles["Arial Normal"] = entry;
  FFiles["Arial-Normal"] = entry;
  FFiles["Arial Standard"] = entry;
  FFiles["Arial-Standard"] = entry;

  // Courier New
  entry = new FontFileEntry;
  entry->entries[false][false] = "Cour.ttf"; entry->entries[false][true] = "Courbd.ttf"; 
  entry->entries[true][false] = "Couri.ttf";  entry->entries[true][true] = "Courbi.ttf"; 
  entry->useCount = 3;
  FFiles["Courier New"] = entry;
  FFiles["Courier New Standard"] = entry;
  FFiles["Courier"] = entry;

  // Garamond
  entry = new FontFileEntry;
  entry->entries[false][false] = "Gara.ttf"; entry->entries[false][true] = "Garabd.ttf"; 
  entry->entries[true][false] = "Garait.ttf";  entry->entries[true][true] = "Garabi.ttf";
  entry->useCount = 3;
  FFiles["Garamond"] = entry;
  FFiles["Garamond Standard"] = entry;
  FFiles["Garamond-Standard"] = entry;

  // Palatino Linotype
  entry = new FontFileEntry;
  entry->entries[false][false] = "Pala.ttf"; entry->entries[false][true] = "Palab.ttf"; 
  entry->entries[true][false] = "Palai.ttf";  entry->entries[true][true] = "Palabi.ttf";
  entry->useCount = 1;
  FFiles["Palation Linotype"] = entry;

  // Tahoma
  entry = new FontFileEntry;
  entry->entries[false][false] = "Tahoma.ttf"; entry->entries[false][true] = "Tahomabd.ttf"; 
  entry->entries[true][false] = "Tahomai.ttf";  entry->entries[true][true] = "Tahomabi.ttf";
  entry->useCount = 1;
  FFiles["Tahoma"] = entry;

  // Tahoma-bold, not an own font but Adobe Illustrator denotes it so.
  entry = new FontFileEntry;
  entry->entries[false][false] = "Tahomabd.ttf"; entry->entries[false][true] = "Tahomabd.ttf"; 
  entry->entries[true][false] = "Tahomabdi.ttf";  entry->entries[true][true] = "Tahomabdi.ttf";
  entry->useCount = 1;
  FFiles["Tahoma-bold"] = entry;

  // Times New Roman
  entry = new FontFileEntry;
  entry->entries[false][false] = "Times.ttf"; entry->entries[false][true] = "Timesbd.ttf"; 
  entry->entries[true][false] = "Timesi.ttf";  entry->entries[true][true] = "Timesbi.ttf";
  entry->useCount = 2;
  FFiles["Times New Roman"] = entry;
  FFiles["Times"] = entry;

  // Trebuchet MS
  entry = new FontFileEntry;
  entry->entries[false][false] = "Trebuc.ttf"; entry->entries[false][true] = "Trebucbd.ttf"; 
  entry->entries[true][false] = "Trebucit.ttf";  entry->entries[true][true] = "Trebucbi.ttf";
  entry->useCount = 2;
  FFiles["Trebuchet MS"] = entry;
  FFiles["Trebuchet"] = entry;

  // Verdana
  entry = new FontFileEntry;
  entry->entries[false][false] = "Verdana.ttf"; entry->entries[false][true] = "Verdanab.ttf"; 
  entry->entries[true][false] = "Verdanai.ttf";  entry->entries[true][true] = "Verdanaz.ttf";
  entry->useCount = 1;
  FFiles["Verdana"] = entry;
#endif // !USE_FONTCONFIG
}


//----------------------------------------------------------------------------------------------------------------------

/**
 * Destructor of the class. Does some clean up.
 */
CFontManager::~CFontManager(void)
{
  clear();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Determines platform dependantly the full path of a font file depending on some characteristics.
 *
 * @param family The font family (e.g. Arial, Tahoma, Verdana).
 * @param style The font style (normal, italic).
 * @param weight The "boldness" of the font in the range of [100..900]. Currently values <= 400 are
 *               considered als normal font, everything else as bold.
 *
 * @return The full path and file name of a font that represents the given characteristics.
 *          A kind of intelligent replacement is done here, though. If there is no file with the given characteristics 
 *          (or cannot be found) then the one from the same family, but normal styles, is used instead. If there is no 
 *          entry for the family or even the standard style for a family cannot be found then Arial Standard is returned as 
 *          default. If this files is also not present on the system then the FTGL lib will throw an exception.
 * @note The returned file name is ANSI encoded as FTGL expects it so.
 */
string CFontManager::getFontFile(string family, string style, int weight)
{
  string filename;

#ifdef USE_FONTCONFIG
  FcPattern *pattern= FcPatternCreate();
  FcResult result;
  int slant;
  
  if (style == "italic")
    slant= FC_SLANT_ITALIC;
  else if (style == "oblique")
    slant= FC_SLANT_OBLIQUE;
  else
    slant= FC_SLANT_ROMAN;
  
  FcPatternAddString(pattern, FC_FAMILY, (FcChar8*)family.c_str());
  FcPatternAddInteger(pattern, FC_SLANT, slant);
  FcPatternAddInteger(pattern, FC_WEIGHT, weight);

  if (!FcConfigSubstitute(FcConfigGetCurrent(), pattern, FcMatchPattern))
  {
    g_warning("Error calling FcConfigSubstitute() on font '%s %s'",
              family.c_str(), style.c_str());
    FcPatternDestroy(pattern);
  }
  else
  {
    FcPattern *fpat;
    
    FcDefaultSubstitute(pattern);
  
    fpat= FcFontMatch(FcConfigGetCurrent(), pattern, &result);
    if (result != FcResultMatch)
      g_warning("Error looking up for font '%s %s'",
                family.c_str(), style.c_str());
    else
    {
      FcValue value;
      
      if (FcPatternGet(fpat, FC_FILE, 0, &value))
      {
        g_message("FILE --> %s", value.u.s);
      
        filename= (char*)value.u.s;
      }
      FcPatternDestroy(fpat);
    }
    FcPatternDestroy(pattern);
  }
#elif defined(__APPLE__)
  FSSpec pathSpec;
  FSRef fsref;
  FT_Long index;
  FMFontFamilyIterator famIter;
  OSStatus status;
  FMFont match= 0;
  FMFontFamily ffamily= 0;
  bool wants_bold= weight > 400;
  bool wants_italic= style == "italic" || style == "Oblique";

  status= FMCreateFontFamilyIterator(NULL, NULL, 
                                     kFMUseGlobalScopeOption,
                                     &famIter);
    
  index = 0;
  while (status == 0 && !match)
  {
    status = FMGetNextFontFamily(&famIter, &ffamily);
    if (status == 0)
    {
      int stat2;
      FMFontFamilyInstanceIterator instIter;
      Str255 famNameStr;
      char famName[256];

      /* get the family name */
      FMGetFontFamilyName(ffamily, famNameStr);
      CopyPascalStringToC(famNameStr, famName);
      
      if (strcasecmp(famName, family.c_str()) == 0)
      {      
        /* iterate through the styles */
        FMCreateFontFamilyInstanceIterator(ffamily, &instIter);
      
        index= 0;
        stat2= 0;
        while (stat2 == 0 && !match)
        {
          FMFontStyle style;
          FMFontSize size;
          FMFont font;
          
          stat2= FMGetNextFontFamilyInstance(&instIter, &font,
                                             &style, &size );
          if (stat2 == 0 && size == 0)
          {
            bool f_bold= (style & bold)!=0;
            bool f_italic= (style & italic)!=0;

            if (f_bold == wants_bold && f_italic == wants_italic)
            {
              match= font;
            }
            else
              index++;
          }
        }
      }
      FMDisposeFontFamilyInstanceIterator(&instIter);
    }
  }
  
  FMDisposeFontFamilyIterator(&famIter);
  
  if (match)
  {
    FMGetFontContainer(match, &pathSpec);
    if (FSpMakeFSRef(&pathSpec, &fsref) == 0)
    {  
      UInt8 thePath[1024];
      if (FSRefMakePath(&fsref, thePath, sizeof(thePath)) == 0)
      {
        filename= (char*)thePath;
      }
    }    
  }
  
//  if (filename.empty() && family != "Arial")
//    filename= getFontFile("Arial", "", 0);
  if (FFontResourcesPath.empty())
    filename="/Library/Fonts/Vera.ttf";
  else
    filename=FFontResourcesPath+"/Vera.ttf";
#else // !USE_FONTCONFIG
  FontFileEntry* entry;

  // Adobe Illustrator (and probably others too) surround the family name with quotes.
  if (family[0] == '"' || family[0] == '\'')
    family = family.substr(1, family.size() - 2);
  FontFiles::iterator iterator = FFiles.find(family);
  if (iterator == FFiles.end())
  {
    // Family is unknown, use Arial as default.
    entry = FFiles["Arial"];
  }
  else
  {
    entry = iterator->second;
  };

  bool italic = (style == "italic") || (style == "oblique");
  bool bold = weight > 400;

  LPITEMIDLIST pidl;
  wchar_t Path[MAX_PATH];
  if (SHGetSpecialFolderLocation(0, CSIDL_FONTS, &pidl) == NOERROR)
  {
    if (SHGetPathFromIDList(pidl, Path))
      filename = utf16ToANSI(Path);
    
    // We are required to free the returned pidl via the shell's IMalloc interface.
    LPMALLOC pMalloc;
    SHGetMalloc(&pMalloc);
    pMalloc->Free(pidl);
  }
  string fontFile = entry->entries[italic][bold];
  string testFile = filename + '/' + fontFile;

  FILE* file = fopen(testFile.c_str(), "rb");
  if (file != NULL)
  {
    fclose(file);
    filename = testFile;
  }
  else
  {
    // There is no file for the required style. Try the base entry.
    fontFile = entry->entries[false][false];
    testFile = filename + '/' + fontFile;
    file = fopen(testFile.c_str(), "rb");
    if (file != NULL)
    {
      fclose(file);
      filename = testFile;
    }
    else
    {
      // That's bad luck. No standard file either. Use the default font.
      entry = FFiles["Arial"];
      filename += '/' + entry->entries[false][false];
    };
  };
#endif // !USE_FONTCONFIG
  return filename;
}

//----------------------------------------------------------------------------------------------------------------------

void CFontManager::boundingBox(const wstring& Output, string FontFamily, string FontStyle, int weight, int FontSize, 
                               TBoundingBox* Box)
{
  FTFont* Font;
  string Key = createLookupKey(FontFamily, FontStyle, weight, FontSize);
  Fonts::iterator iterator = FFonts.find(Key);

  if (iterator == FFonts.end())
  {
    string fontFile = getFontFile(FontFamily, FontStyle, weight);
    Font = new FONTCLASS(fontFile.c_str());
    Font->FaceSize(FontSize);
    Font->UseDisplayList(FUseLocalDisplayLists);
    FFonts[Key] = Font;
  }
  else
    Font = iterator->second;

  Font->BBox(Output.c_str(), Box->upper.x, Box->upper.y, Box->upper.z, Box->lower.x, Box->lower.y, Box->lower.z);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Clears the font file list.
 */
void CFontManager::clear(void)
{
  for (Fonts::iterator iterator = FFonts.begin(); iterator != FFonts.end(); ++iterator)
  {
    FTFont* Font = iterator->second;
    delete Font;
  };
  FFonts.clear();

#ifndef USE_FONTCONFIG
  for (FontFiles::iterator iterator = FFiles.begin(); iterator != FFiles.end(); iterator++)
  {
    FontFileEntry* entry = iterator->second;
    if (--entry->useCount == 0)
      delete entry;
  };
  FFiles.clear();
#endif
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates a string that can be used for lookup in the font list.
 * Either parameter can be NULL (or 0 for weight) causing the manager to use the default values for each missing 
 * parameter. See header file for the list of default values.
 *
 * @param family The font family (Arial, Courier etc.)
 * @param style The font style.
 * @param weight The boldness of the font (400 for normal).
 * @param FontSize The font size.
 */
string CFontManager::createLookupKey(const string& family, const string& style, int weight, int FontSize)
{
  char WeightString[30];

  // Construct a lookup string out of the font properties.
  string Key = family + style;
  #ifdef _WINDOWS
    // Microsoft has confirmed that there is no guaranteed NULL termination (see MSDN for details).
    // Hence we better add one.
    _snprintf(WeightString, sizeof(WeightString), "%i:%i\0", weight, FontSize);
  #else
    snprintf(WeightString, sizeof(WeightString), "%i:%i", weight, FontSize);
  #endif // #ifdef _WINDOWS
  Key += WeightString;

  return Key;
}

//----------------------------------------------------------------------------------------------------------------------

void CFontManager::setFontResourcesPath(const string& path)
{
  FFontResourcesPath= path;
}

//----------------------------------------------------------------------------------------------------------------------

void CFontManager::textOut(const wstring& Output, string FontFamily, string FontStyle, int weight, int FontSize)
{
  FTFont* Font;
  string Key = createLookupKey(FontFamily, FontStyle, weight, FontSize);
  Fonts::iterator iterator = FFonts.find(Key); 

  if (iterator == FFonts.end())
  {
    string fontFile = getFontFile(FontFamily, FontStyle, weight);
    Font = new FONTCLASS(fontFile.c_str());
    Font->FaceSize(FontSize);
    Font->UseDisplayList(FUseLocalDisplayLists);
    
    FFonts[Key] = Font;
  }
  else
    Font = iterator->second;

  Font->Render(Output.c_str());
}

//----------------------------------------------------------------------------------------------------------------------

