/* Copyright (C) 2005 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include "MWMainWindow.h"
#include "myg_gtkutils.h"
#include "MGGladeXML.h"

#include "MGImageButton.h"
#include "MGImageCheckButton.h"

#include "MWBCanvas.h"
#include <MySQLGRT/MGRT.h>
#include <MySQLGRT/MGRTShell.h>
#include <MySQLGRT/MGRTSchemaEditor.h>
#include <MySQLGRT/MGRTViewEditor.h>
#include <MySQLGRT/MGRTRoutineGroupEditor.h>
#include <MySQLGRT/MGRTTableEditor.h>
#include "MWReverseEngineering.h"
#include "MWSynchronizeDatabase.h"
#include "MGAboutPanel.h"
#include "myg_utils.h"

#include "myx_grt_builtin_module_public_interface.h"

#include "MWLayerTree.h"


#define COPYRIGHT_STRING "(c) Copyright 2005, 2006 by MySQL AB. All rights reserved."



#define CREDITS_STRING "Michael G. Zinner: Graphic Design, Windows GUI, Library.\nAlfredo K. Kojima: Linux GUI, Library.\n"\
                       "Mike Lischke: Windows development, Generic canvas.\n"\
                       "Vladimir Kolesnikov: Windows development, Library.\n"\
                       "Mike Hillyer: Documentation."
#define MY_NAME "MySQL Workbench"

/**
 * @file  MWMainWindow.cc
 * @brief 
 */

extern char **wb_argv;
extern pthread_t main_thread;

static MGCanvas::Size MWDefaultCanvasSize(10000, 7000);


static struct {
  const char *icon;
  const char *icon_down;
  const char *cursor;
  MWMainWindow::ToolType type;
  int tool_palette;
  Glib::ustring accel;
  const char *tooltip;
} tool_buttons[]= {
  {"wb_arrow.png", "wb_arrow_down.png", NULL, MWMainWindow::MWToolCursor, 0, "Escape",
  N_("Object selection tool (Escape)")},
  {"wb_hand.png", "wb_hand_down.png", "hand.png", MWMainWindow::MWToolPanning, 0, "h",
  N_("Hand/pan tool (h)")},
  {"", "", NULL, MWMainWindow::MWToolCursor, 0, "", NULL},
  {"wb_layer.png", "wb_layer_down.png", "layer.png", MWMainWindow::MWToolLayer, 4, "l",
  N_("Layer tool (l)")},
  {"", "", NULL, MWMainWindow::MWToolCursor, 0, "", ""},
  {"wb_db_table.png", "wb_db_table_down.png", "table.png", MWMainWindow::MWToolTable, 1, "t",
  N_("Table tool (t)")},
  {"wb_db_view.png", "wb_db_view_down.png", "view.png", MWMainWindow::MWToolView, 2, "v",
  N_("View tool (v)")},
  {"wb_db_sp.png", "wb_db_sp_down.png", "routine.png", MWMainWindow::MWToolRoutine, 3, "g",
  N_("Routine group tool (g)")},
  {"", "", NULL, MWMainWindow::MWToolCursor, 9, "", ""},
  {"wb_rel_1n.png", "wb_rel_1n_down.png", "res1n.png", MWMainWindow::MWToolRelationship2, 0, "r",
  N_("1:n relationship tool (r)")},
  {"wb_rel_11.png", "wb_rel_11_down.png", "res11.png", MWMainWindow::MWToolRelationship1, 0, "<shift>r",
  N_("1:1 relationship tool (Shift-r)")},
  {"wb_rel_nm.png", "wb_rel_nm_down.png", "resnm.png", MWMainWindow::MWToolRelationship3, 0, "<control>r",
  N_("n:m relationship tool (Control-r)")},
  {"", "", NULL, MWMainWindow::MWToolCursor, 0, "", ""},
  {"wb_note.png", "wb_note_down.png", "note.png", MWMainWindow::MWToolNote, 5, "n",
  N_("Note tool (n)")},
//  {"wb_image.png", "wb_image_down.png", "image.png", MWMainWindow::MWToolImage, "", ""},
  {NULL}
};


static const char *table_engines[]= {
  "InnoDB",
    "MyISAM",
    "BDB",
    NULL
};




static int copy_to_clipboard_cb(const char *text, void *data)
{
  MWMainWindow *mainw= (MWMainWindow*)data;

  mainw->copy_to_clipboard(text);

  return 0;
}
  


MWMainWindow::MWMainWindow(GtkWindow *win)
  : Gtk::Window(win)
{
  _current_tool= MWToolCursor;
  _dirty= false;
  _untouched= true;
  _loading= false;
  _prop_xml= 0;
  
  _plugin_menu= 0;

  _canvas_view_index= 0;

  _idle_refresh_pending= false;
  _schema_refresh_pending= false;
  _selection_refresh_pending= false;
  _refreshing_selection= false;
  _refreshing_selection_tree= false;
  _adding_view_page= false;

  myx_grt_module_base_set_copy_to_clipboard_callback(copy_to_clipboard_cb, this);
  
  _dispatch_message.connect(sigc::mem_fun(*this,&MWMainWindow::handle_message));

  _clipboard_copy_dispatcher.connect(sigc::mem_fun(*this, &MWMainWindow::do_copy_to_clipboard));
  _input_req_dispatcher.connect(sigc::mem_fun(*this, &MWMainWindow::do_input_request));
}


void MWMainWindow::copy_to_clipboard(const char *text)
{
  _clipboard_copy_data= text;
  _clipboard_copy_dispatcher.emit();
}


void MWMainWindow::do_copy_to_clipboard()
{
  get_clipboard("CLIPBOARD")->set_text(_clipboard_copy_data);
}


MWMainWindow *MWMainWindow::create()
{
  MGGladeXML *xml= new MGGladeXML(get_app_file("document.glade"),
                                   "window");
  
  MWMainWindow *mainw= 0;
  xml->get_widget_derived("window", mainw);
  
  mainw->_xml= xml;
  mainw->create_canvas();

  return mainw;
}


void MWMainWindow::show_about()
{
  MGAboutPanel *about= new MGAboutPanel(get_app_file("about.png"),
                                        MY_NAME,
                                        VERSION"   ("BUILD_DATE")",
                                        COPYRIGHT_STRING,
                                        CREDITS_STRING);
//  about->set_credits();
  about->set_transient_for(*this);
  about->set_delete_on_close();
  about->show();
}


void MWMainWindow::quit()
{  
  if (_dirty)
  {
    Gtk::MessageDialog dlg(*this, _("<b>Save changes to model before quitting?</b>\n\nThere are changes to the model that will be lost if not saved."), true,
                           Gtk::MESSAGE_INFO, Gtk::BUTTONS_NONE);
    
    dlg.add_button(_("_Don't Save"), Gtk::RESPONSE_NO);
    dlg.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
    dlg.add_button(Gtk::Stock::SAVE, Gtk::RESPONSE_OK);
    
    switch (dlg.run())
    {
    case Gtk::RESPONSE_OK:
      save_document();
      if (_dirty) // cancelled save
        return;
      break;
    case Gtk::RESPONSE_NO:
      break;
    default:
      return;
    }
  }
  _grt->call_procedure("Workbench", "shutdownWorkbench", MGRTValue::createList());

  // save /app/options
  _grt->save_subtree("/app/options", prefs.build_path_to("workbench/workbench_app_options.xml"));

  _grt->save_subtree("/app/commonOptions", prefs.build_path_to("grt_common_options.xml"));

  delete _grt;
  
  Gtk::Main::instance()->quit();
}

void MWMainWindow::editor_closed(MGRTObjectEditor *editor)
{
  _editors.erase(_editors.find(editor->object_id()));
  delete editor;
}


void MWMainWindow::editor_saved(MGRTObjectEditor *editor)
{
  _canvas->queue_draw();
  
  schemata_changed();
}



void MWMainWindow::arrange_objects()
{
  MGRTValue args(MGRTValue::createList());
  
  args.append(MGRTValue(_grt->global_ref_value("/workbench/model/currentView")));
  
  args.append(MGRTValue(_has_selection ? 1 : 0));
  _grt->call_procedure("WorkbenchController", "arrangeObjects", args);
}


void MWMainWindow::create_schema()
{
  MGRTSchemaEditor *editor= MGRTSchemaEditor::create(_grt, _grt->global_value("/workbench/catalog"));

  editor->set_transient_for(*this);
  editor->create_new();
  editor->show();
  
  editor->signal_closed().connect(sigc::bind<MGRTObjectEditor*>(sigc::mem_fun(*this,&MWMainWindow::editor_closed), editor));
  editor->signal_saved().connect(sigc::bind<MGRTObjectEditor*>(sigc::mem_fun(*this,&MWMainWindow::editor_saved), editor));

  _editors[editor->object_id()]= editor;
  
  set_dirty();
}


MGRTValue MWMainWindow::edit_schema(MGRTValue args)
{
  MGRTSchemaEditor *editor= MGRTSchemaEditor::create(_grt, _grt->global_value("/workbench/catalog"));

  editor->set_transient_for(*this);
  editor->edit_object(args[0]);
  editor->show();

  editor->signal_closed().connect(sigc::bind<MGRTObjectEditor*>(sigc::mem_fun(*this,&MWMainWindow::editor_closed), editor));
  editor->signal_saved().connect(sigc::bind<MGRTObjectEditor*>(sigc::mem_fun(*this,&MWMainWindow::editor_saved), editor));

  _editors[editor->object_id()]= editor;

  return MGRTValue();
}


MGRTValue MWMainWindow::edit_table(MGRTValue args)
{
  if (_editors.find(args[0].dictId())==_editors.end())
  {
    MGRTTableEditor *editor= MGRTTableEditor::create(_grt, _grt->global_value("/workbench/catalog"));

    editor->set_transient_for(*this);
    editor->edit_object(args[0]);
    editor->show();
    
    editor->signal_closed().connect(sigc::bind<MGRTObjectEditor*>(sigc::mem_fun(*this,&MWMainWindow::editor_closed), editor));
    editor->signal_saved().connect(sigc::bind<MGRTObjectEditor*>(sigc::mem_fun(*this,&MWMainWindow::editor_saved), editor));

    _editors[editor->object_id()]= editor;
  }
  return MGRTValue();
}


MGRTValue MWMainWindow::edit_view(MGRTValue args)
{
  if (_editors.find(args[0].dictId())==_editors.end())
  {
    MGRTViewEditor *editor= MGRTViewEditor::create(_grt, _grt->global_value("/workbench/catalog"));

    editor->set_transient_for(*this);
    editor->edit_object(args[0]);
    editor->show();
    
    editor->signal_closed().connect(sigc::bind<MGRTObjectEditor*>(sigc::mem_fun(*this,&MWMainWindow::editor_closed), editor));
    editor->signal_saved().connect(sigc::bind<MGRTObjectEditor*>(sigc::mem_fun(*this,&MWMainWindow::editor_saved), editor));
    
    _editors[editor->object_id()]= editor;
  }
  return MGRTValue();
}


MGRTValue MWMainWindow::edit_routine_group(MGRTValue args)
{
  if (_editors.find(args[0].dictId())==_editors.end())
  {
    MGRTRoutineGroupEditor *editor= MGRTRoutineGroupEditor::create(_grt, _grt->global_value("/workbench/catalog"));

    editor->set_transient_for(*this);
    editor->edit_object(args[0]);
    editor->show();
    
    editor->signal_closed().connect(sigc::bind<MGRTObjectEditor*>(sigc::mem_fun(*this,&MWMainWindow::editor_closed), editor));
    editor->signal_saved().connect(sigc::bind<MGRTObjectEditor*>(sigc::mem_fun(*this,&MWMainWindow::editor_saved), editor));
    
    _editors[editor->object_id()]= editor;
  }
  return MGRTValue();
}



void MWMainWindow::build_plugin_popup_menu(MGRTValue rootObj, MGRTValue object, Gtk::Menu *&menu)
{
  if (!menu)
  {
    menu= new Gtk::Menu();
    for (unsigned int i= 0; i < rootObj.count(); i++)
    {
      MGRTValue group(rootObj[i]);
      build_plugin_popup_menu(group, object, menu);
    }
  }
  else
  {
    if (rootObj["plugins"].count() > 0)
    {
      MGRTValue plugins(rootObj["plugins"]);

      for (unsigned int j= 0; j < plugins.count(); j++)
      {
        MGRTValue plugin(MGRTValue::refObject(_grt->grt(), plugins[j].asString()));
        bool ok= false;

        for (unsigned int i= 0; i < plugin["objectStructNames"].count(); i++)
        {
          if (myx_grt_dict_struct_is_or_inherits_from(_grt->grt(), object.grtValue(),
                                                      plugin["objectStructNames"][i].asString()))
          {
            ok= true;
            break;
          }
        }
        if (ok)
        {
          Gtk::MenuItem *pluginItem= Gtk::manage(new Gtk::MenuItem(plugin["caption"].asString()));
          
          pluginItem->signal_activate().connect(sigc::bind<MGRTValue,MGRTValue>(
               sigc::mem_fun(*this, &MWMainWindow::call_plugin_for_object),plugin,object));
          
          menu->append(*pluginItem);
        }
      }
    }

    for (unsigned int i= 0; i < rootObj["subGroups"].count(); i++)
      build_plugin_popup_menu(rootObj["subGroups"][i], object, menu);
  }
}


MGRTValue MWMainWindow::popup_plugins_menu(MGRTValue args)
{
  if (_plugin_menu)
    delete _plugin_menu;
  _plugin_menu= 0;

  build_plugin_popup_menu(MGRTValue(_grt->global_value("/app/pluginGroups")), args[1], _plugin_menu);
  _plugin_menu->show_all();
  _plugin_menu->popup(1, 0);
  
  return MGRTValue();
}


MGRTValue MWMainWindow::popup_connection_menu(MGRTValue args)
{
  MGRTValue element(args[0]);
  MGRTValue action(args[1]);

  show_object_properties(element);
  if (action.asInt() == 1003)
  {
    _rel_menu.popup(1, 0);
  }
  
  return MGRTValue();
}


int MWMainWindow::get_available_element_index(const char *format, const char *objectListPath)
{
  char buffer[strlen(format)+20];
  MGRTValue objectList= MGRTValue::fromGlobal(_grt->grt(), objectListPath);
  if (objectList.isValid())
  {
    unsigned int n;
    for (n=1; ; n++)
    {
      sprintf(buffer, format, n);
      if (!objectList.listItemNamed(buffer).isValid())
        return n;
    }
  }
  return 123;
}


int MWMainWindow::get_available_layer_index()
{
  MGRTValue layerList= MGRTValue::fromGlobal(_grt->grt(), "/workbench/model/currentView/layers");
  if (layerList.isValid())
  {
    unsigned int n;
    for (n=1; ; n++)
    {
      char buffer[100];
      sprintf(buffer, "Layer %d", n);
      if (!layerList.listItemNamed(buffer).isValid())
        return n;
    }
  }
  return 123;
}


bool MWMainWindow::request_input(Glib::ustring caption, bool password,
                                 Glib::ustring *retval)
{
  _input_req_caption= caption;
  _input_req_password= password;

  _input_ready.lock();
  _input_req_dispatcher.emit();
  _input_ready.lock();
  
  *retval= _input_req_text;

  return _input_req_result;
}


void MWMainWindow::do_input_request()
{
  _input_req_result= myg_ask_string("Workbench", _input_req_caption, 
                                    _input_req_text, _input_req_password);

  _input_ready.unlock();  
}


void MWMainWindow::set_default_appoptions()
{
  MGRTValue options(MGRTValue::createTypedDict());
  
  options.set("DefaultModelBaseWidth", (float)10000.0);
  options.set("DefaultModelBaseHeight", (float)7000.0);
  options.set("PkColumnNameTemplate", "id%tablename%");
  options.set("DefaultRdbms", "Mysql");
  options.set("DefaultRdbmsVersion", "5.1.6");
  options.set("FKNameTemplate", "FK%table%");
  options.set("AuxTableTemplate", "%stable%_has_%dtable%");
  options.set("FKColumnNameTemplate", "FK%table%%column%");
  options.set("FKDeleteRule", "NO ACTION");
  options.set("FKUpdateRule", "NO ACTION");
  options.set("Rel1nSourceOptional", (int)1);
  options.set("Rel1nTargetOptional", 0);
  options.set("Rel11SourceOptional", 1);
  options.set("Rel11TargetOptional", 0);
  options.set("RelnmSourceOptional", 1);
  options.set("RelnmTargetOptional", 1);
  options.set("DockGrtShell", 1);
  options.set("DockGrtShellAutoAddSnippets", 0);
  options.set("DockEditors", 1);
  options.set("SideBarOneWidth", 1);
  options.set("SideBarWidthView", 230);
  options.set("SideBarWidthGrtShell", 350);
  options.set("EditorPanelHeight", 280);
  options.set("InitialZoom", (float)0.75);

  _grt->set_global_value("/app/options", options);
}


void MWMainWindow::create_canvas()
{
  show(); // force all contents to get realized so that the canvas
  hide(); // will have a realized XID to init opengl
  // initialize grt
  _grt= new MGRT();

  _grt->set_input_handler(sigc::mem_fun(*this, &MWMainWindow::request_input));
  
  _grt_shell= new MGRTShell(_grt);
  _grt->initialize_grt_thread(get_common_file(""), get_libexec_file(""));

  _grt->scan_structs_in_path(get_common_file("grt/"));

  myx_register_grt_wb_module(_grt->grt());

  std::map<Glib::ustring, sigc::slot<MGRTValue,MGRTValue> > funcs;
  funcs["editSchema"]= sigc::mem_fun(*this,&MWMainWindow::edit_schema);
  funcs["editTable"]= sigc::mem_fun(*this,&MWMainWindow::edit_table);
  funcs["editRoutineGroup"]= sigc::mem_fun(*this,&MWMainWindow::edit_routine_group);
  funcs["editView"]= sigc::mem_fun(*this,&MWMainWindow::edit_view);
  funcs["popupPluginsMenu"]= sigc::mem_fun(*this,&MWMainWindow::popup_plugins_menu);
  funcs["popupConnectionMenu"]= sigc::mem_fun(*this,&MWMainWindow::popup_connection_menu);
  _grt->register_builtin_module("WorkbenchUi", funcs);
  
//  _grt->scan_modules_in_path(get_common_file("lua/"));
  _grt->scan_modules_in_path(get_app_file("lua"));
  
  // make sure /app is set
  _grt->global_app_dict();

  _grt->load_common_options(prefs.build_path_to("grt_common_options.xml"));

  if (!_grt->load_subtree("/app/options", prefs.build_path_to("workbench/workbench_app_options.xml")))
    set_default_appoptions();

  _grt->call_procedure("Workbench", "registerEditors", MGRTValue());

  // init canvas
  _canvas_scroll= Gtk::manage(new MGCanvasScroller());
  _canvas= new MWBCanvas(_grt);

  _canvas->set_hand_cursor(Gdk::Cursor(Gdk::Display::get_default(),
                                       PIXCACHE->load("hand.png"),
                                       3, 3),
                           Gdk::Cursor(Gdk::Display::get_default(),
                                       PIXCACHE->load("hand_closed.png"),
                                       3, 3));

  _canvas->signal_key_press_event().connect(sigc::mem_fun(*this,&MWMainWindow::handle_canvas_key));
  
  _canvas->set_mouse_down_handler(sigc::mem_fun(*this,&MWMainWindow::handle_button_press));
  _canvas->set_mouse_up_handler(sigc::mem_fun(*this,&MWMainWindow::handle_button_release));
  _canvas->set_action_handler(sigc::mem_fun(*this,&MWMainWindow::handle_canvas_action));
  _canvas_scroll->add(_canvas);

  _canvas_scroll->show_all();
}


bool MWMainWindow::tree_display_filter(MGRTValue value)
{
  if (value.type() == MYX_DICT_VALUE)
  {
    if (myx_grt_dict_struct_is_or_inherits_from(_grt->grt(),
                                                value.grtValue(),
                                                "db.Schema")
        )
      return true;
  }
  else if (value.type() == MYX_LIST_VALUE)
  {
    if (value.listContentStruct()
        && strcmp(value.listContentStruct(), "db.mysql.Sequence")!=0
        && strcmp(value.listContentStruct(), "db.mysql.Synonym")!=0
        && strcmp(value.listContentStruct(), "db.mysql.StructuredDatatype")!=0)
    return true;
  }
  return false;
}


bool MWMainWindow::on_delete_event(GdkEventAny *ev)
{
  quit();
  return true;
}


void MWMainWindow::setup_toolbar()
{
  MGImageButton *btn;

  btn= Gtk::manage(new MGImageButton());
  btn->set_image(PIXCACHE->load("tiny_overview.png"));
  btn->set_alt_image(PIXCACHE->load("tiny_overview_down.png"));
  btn->signal_clicked().connect(sigc::mem_fun(*this,&MWMainWindow::toggle_overview));
  _xml->get_box("tools_hbox")->pack_start(*btn, false, false);
  
  _grid_check= Gtk::manage(new MGImageCheckButton(PIXCACHE->load("tiny_grid.png"),
                                                  PIXCACHE->load("tiny_grid_down.png")));
  _grid_check->set_active(true);
  _grid_check->signal_toggled().connect(sigc::mem_fun(*this,&MWMainWindow::toggle_grid));
  _xml->get_box("tools_hbox")->pack_start(*_grid_check, false, false);
  _xml->get_check_menu_item("display_grid1")->set_active(true);
    
  btn= Gtk::manage(new MGImageButton());
  btn->set_image(PIXCACHE->load("tiny_lock.png"));
  btn->set_alt_image(PIXCACHE->load("tiny_lock_down.png"));
  _xml->get_box("tools_hbox")->pack_start(*btn, false, false);
  btn->set_sensitive(false);
  
  _xml->get_box("tools_hbox")->pack_start(*Gtk::manage(new Gtk::VSeparator()), false, false);
  
  btn= Gtk::manage(new MGImageButton());
  btn->set_image(PIXCACHE->load("tiny_undo.png"));
  btn->set_alt_image(PIXCACHE->load("tiny_undo_down.png"));
  _xml->get_box("tools_hbox")->pack_start(*btn, false, false);
  btn->set_sensitive(false);
  
  btn= Gtk::manage(new MGImageButton());
  btn->set_image(PIXCACHE->load("tiny_redo.png"));
  btn->set_alt_image(PIXCACHE->load("tiny_redo_down.png"));
  _xml->get_box("tools_hbox")->pack_start(*btn, false, false);
  btn->set_sensitive(false);

  _xml->get_box("tools_hbox")->pack_start(*Gtk::manage(new Gtk::VSeparator()), false, false);

  btn= Gtk::manage(new MGImageButton());
  btn->set_image(PIXCACHE->load("tiny_save.png"));
  btn->set_alt_image(PIXCACHE->load("tiny_save_down.png"));
  btn->signal_clicked().connect(sigc::mem_fun(*this,&MWMainWindow::save_document));
  _xml->get_box("tools_hbox")->pack_start(*btn, false, false);

  _xml->get_box("tools_hbox")->pack_start(*Gtk::manage(new Gtk::VSeparator()), false, false);

  _xml->get_box("tools_hbox")->show_all();

  //
  
  _xml->get_box("cursor_toolbox")->pack_start(*Gtk::manage(new Gtk::VSeparator()), false, false);

  _xml->get_box("cursor_toolbox")->pack_start(*Gtk::manage(new Gtk::Label(_("Align:"))), false, false);

  btn= Gtk::manage(new MGImageButton());
  btn->set_image(PIXCACHE->load("tiny_align_v_top.png"));
  btn->set_alt_image(PIXCACHE->load("tiny_align_v_top_down.png"));
  btn->signal_clicked().connect(sigc::bind<int>(sigc::mem_fun(*this,&MWMainWindow::align_objects),1));
  _xml->get_box("cursor_toolbox")->pack_start(*btn, false, false);

  btn= Gtk::manage(new MGImageButton());
  btn->set_image(PIXCACHE->load("tiny_align_v_middle.png"));
  btn->set_alt_image(PIXCACHE->load("tiny_align_v_middle_down.png"));
  btn->signal_clicked().connect(sigc::bind<int>(sigc::mem_fun(*this,&MWMainWindow::align_objects),2));
  _xml->get_box("cursor_toolbox")->pack_start(*btn, false, false);

  btn= Gtk::manage(new MGImageButton());
  btn->set_image(PIXCACHE->load("tiny_align_v_bottom.png"));
  btn->set_alt_image(PIXCACHE->load("tiny_align_v_bottom_down.png"));
  btn->signal_clicked().connect(sigc::bind<int>(sigc::mem_fun(*this,&MWMainWindow::align_objects),3));
  _xml->get_box("cursor_toolbox")->pack_start(*btn, false, false);

  btn= Gtk::manage(new MGImageButton());
  btn->set_image(PIXCACHE->load("tiny_align_h_left.png"));
  btn->set_alt_image(PIXCACHE->load("tiny_align_h_left_down.png"));
  btn->signal_clicked().connect(sigc::bind<int>(sigc::mem_fun(*this,&MWMainWindow::align_objects),4));
  _xml->get_box("cursor_toolbox")->pack_start(*btn, false, false);

  btn= Gtk::manage(new MGImageButton());
  btn->set_image(PIXCACHE->load("tiny_align_h_middle.png"));
  btn->set_alt_image(PIXCACHE->load("tiny_align_h_middle_down.png"));
  btn->signal_clicked().connect(sigc::bind<int>(sigc::mem_fun(*this,&MWMainWindow::align_objects),5));
  _xml->get_box("cursor_toolbox")->pack_start(*btn, false, false);

  btn= Gtk::manage(new MGImageButton());
  btn->set_image(PIXCACHE->load("tiny_align_h_right.png"));
  btn->set_alt_image(PIXCACHE->load("tiny_align_h_right_down.png"));
  btn->signal_clicked().connect(sigc::bind<int>(sigc::mem_fun(*this,&MWMainWindow::align_objects),6));
  _xml->get_box("cursor_toolbox")->pack_start(*btn, false, false);

  _xml->get_box("cursor_toolbox")->pack_start(*Gtk::manage(new Gtk::VSeparator()), false, false);

  _xml->get_box("cursor_toolbox")->pack_start(*Gtk::manage(new Gtk::Label(_("Space:"))), false, false);

  btn= Gtk::manage(new MGImageButton());
  btn->set_image(PIXCACHE->load("tiny_align_v_middle2.png"));
  btn->set_alt_image(PIXCACHE->load("tiny_align_v_middle2_down.png"));
  btn->signal_clicked().connect(sigc::bind<int>(sigc::mem_fun(*this,&MWMainWindow::align_objects),10));
  _xml->get_box("cursor_toolbox")->pack_start(*btn, false, false);

  btn= Gtk::manage(new MGImageButton());
  btn->set_image(PIXCACHE->load("tiny_align_h_middle2.png"));
  btn->set_alt_image(PIXCACHE->load("tiny_align_h_middle2_down.png"));
  btn->signal_clicked().connect(sigc::bind<int>(sigc::mem_fun(*this,&MWMainWindow::align_objects),11));
  _xml->get_box("cursor_toolbox")->pack_start(*btn, false, false);

  _space_spin= Gtk::manage(new Gtk::SpinButton());
  _space_spin->set_range(0, 500);
  _space_spin->set_increments(5, 100);
  _space_spin->set_value(10);
  _xml->get_box("cursor_toolbox")->pack_start(*_space_spin, false, false);

  _xml->get_box("cursor_toolbox")->show_all();
}


void MWMainWindow::setup_ui()
{
  std::map<std::string, Glib::RefPtr<Gdk::Pixbuf> > tree_icons;
  
  setup_toolbar();
  
  signal_delete_event().connect(sigc::mem_fun(*this, &MWMainWindow::on_delete_event));
  
  _schema_tree= 0;
  _xml->get_widget_derived("schema_tree", _schema_tree);
    
  std::list<Gtk::TargetEntry> targets;
  targets.push_back(Gtk::TargetEntry("x-workbench-schema-object", Gtk::TARGET_SAME_APP, 0));
  _schema_tree->drag_source_set(targets, Gdk::ModifierType(GDK_BUTTON1_MASK),
                                Gdk::DragAction(GDK_ACTION_COPY));
  _schema_tree->signal_drag_data_get().connect(sigc::mem_fun(*this,&MWMainWindow::schema_drag_data_get));

  _schema_tree->signal_drag_begin().connect(sigc::mem_fun(*this,&MWMainWindow::schema_drag_begin));
  _schema_tree->signal_drag_end().connect(sigc::mem_fun(*this,&MWMainWindow::schema_drag_end));

  _canvas->signal_drag_motion().connect(sigc::mem_fun(*this,&MWMainWindow::canvas_drag_motion));
  _canvas->signal_drag_leave().connect(sigc::mem_fun(*this,&MWMainWindow::canvas_drag_leave));
  _canvas->signal_drag_drop().connect(sigc::mem_fun(*this,&MWMainWindow::canvas_drag_drop));

  _layer_tree= 0;
  _xml->get_widget_derived("layer_tree", _layer_tree);
  _layer_tree->set_grt(_grt);
  
  _layer_tree->get_selection()->set_mode(Gtk::SELECTION_MULTIPLE);
  _layer_tree->get_selection()->signal_changed().connect(sigc::mem_fun(*this,&MWMainWindow::layer_tree_selection_changed));
  
  _schema_tree->set_display_filter(sigc::mem_fun(*this,&MWMainWindow::tree_display_filter));

  tree_icons["db.mysql.Schema"]= PIXCACHE->load("db.Schema.16x16.png");
  tree_icons["db.Schema"]= PIXCACHE->load("db.Schema.16x16.png");

  tree_icons["db.mysql.Table"]= PIXCACHE->load("db.Table.16x16.png");
  tree_icons["db.Table"]= PIXCACHE->load("db.Table.16x16.png");

  tree_icons["db.mysql.View"]= PIXCACHE->load("db.View.16x16.png");
  tree_icons["db.View"]= PIXCACHE->load("db.View.16x16.png");

  _schema_tree->set_icon_info(tree_icons);
  _schema_tree->set_grt(_grt);
  _schema_tree->set_root_path("/workbench/catalog/schemata");

  Gtk::Box *box= _xml->get_box("tool_box");
  Gtk::RadioButtonGroup group;

  box->set_spacing(0);
  
  for (unsigned int i= 0; tool_buttons[i].icon; i++)
  {
    if (tool_buttons[i].cursor)
    {
      Gdk::Cursor cursor(Gdk::Display::get_default(), 
                         PIXCACHE->load(tool_buttons[i].cursor),
                         0, 0);

      _cursors.push_back(cursor);
    }
    else
      _cursors.push_back(Gdk::Cursor());
    
    if (*tool_buttons[i].icon)
    {
      MGImageRadioButton *img= new MGImageRadioButton(group,
                                                      Gdk::Pixbuf::create_from_file(get_app_file(tool_buttons[i].icon)),
                                                      Gdk::Pixbuf::create_from_file(get_app_file(tool_buttons[i].icon_down)));
      img->set_focus_on_click(false);
      img->signal_clicked().connect(sigc::bind<int>(sigc::mem_fun(*this,&MWMainWindow::change_tool), i));
      box->pack_start(*img, false, false);

      _tooltips.set_tip(*img, tool_buttons[i].tooltip);
      
      if (!tool_buttons[i].accel.empty())
        _shortcut_map[tool_buttons[i].accel]= img;

      if (i == 0)
        _default_tool_button= img;
    }
    else
    {
      Gtk::HSeparator *sep= new Gtk::HSeparator();
      box->pack_start(*sep, false, false);
    }
  }
  box->show_all();
  
  
  _xml->get_menu_item("new1")->signal_activate().connect(sigc::bind<bool>(sigc::mem_fun(*this,&MWMainWindow::new_document), true));
  _xml->get_menu_item("save1")->signal_activate().connect(sigc::mem_fun(*this,&MWMainWindow::save_document));
  _xml->get_menu_item("save_as1")->signal_activate().connect(sigc::mem_fun(*this,&MWMainWindow::save_document_as));
  _xml->get_menu_item("open1")->signal_activate().connect(sigc::bind<std::string>(sigc::mem_fun(*this,&MWMainWindow::open_document),""));
  _xml->get_menu_item("quit1")->signal_activate().connect(sigc::mem_fun(*this,&MWMainWindow::quit));

  _xml->get_menu_item("open_workbench_script1")->set_sensitive(false);

  _xml->get_menu_item("as_sql_create_script1")->signal_activate().connect(sigc::mem_fun(*this,&MWMainWindow::generate_script));
  //_xml->get_menu_item("as_sql_drop_script1")->signal_activate().connect(sigc::mem_fun(*this,&MWMainWindow::generate_drop_script));
  _xml->get_menu_item("as_sql_drop_script1")->set_sensitive(false);

  _xml->get_menu_item("as_png1")->signal_activate().connect(sigc::mem_fun(*this,&MWMainWindow::export_as_png));
  _xml->get_menu_item("as_pdf1")->signal_activate().connect(sigc::mem_fun(*this,&MWMainWindow::export_as_pdf));
  _xml->get_menu_item("as_png1")->set_sensitive(false);
  _xml->get_menu_item("as_pdf1")->set_sensitive(false);

  _xml->get_menu_item("import_from_dbd1")->signal_activate().connect(sigc::mem_fun(*this,&MWMainWindow::import_dbd4));
//  _xml->get_menu_item("sql_create_script1")->signal_activate().connect(sigc::mem_fun(*this,&MWMainWindow::import_create_script));
  _xml->get_menu_item("sql_create_script1")->set_sensitive(false);

  _xml->get_menu_item("new_view_tab1")->signal_activate().connect(sigc::mem_fun(*this,&MWMainWindow::new_view_tab));
  
  _xml->get_menu_item("align_to_grid1")->set_sensitive(false);
  _xml->get_menu_item("send_to_back1")->signal_activate().connect(sigc::bind<bool>(sigc::mem_fun(*this,&MWMainWindow::send_to_back), false));
  _xml->get_menu_item("send_to_back1")->set_sensitive(false);
  _xml->get_menu_item("bring_to_front1")->signal_activate().connect(sigc::bind<bool>(sigc::mem_fun(*this,&MWMainWindow::send_to_back), true));
  _xml->get_menu_item("bring_to_front1")->set_sensitive(false);
  _xml->get_menu_item("send_to_back1")->set_sensitive(false);
  _xml->get_menu_item("center_objects_here1")->set_sensitive(false);
  
  _xml->get_menu_item("select_all1")->signal_activate().connect(sigc::mem_fun(*this,&MWMainWindow::select_all));
  
  _xml->get_menu_item("properties1")->signal_activate().connect(sigc::mem_fun(*this,&MWMainWindow::show_properties));

  _xml->get_menu_item("delete1")->signal_activate().connect(sigc::mem_fun(*this,&MWMainWindow::delete_object));
  
  _xml->get_menu_item("toggle_overview1")->signal_activate().connect(sigc::mem_fun(*this,&MWMainWindow::toggle_overview));
  _xml->get_menu_item("default_zoom1")->signal_activate().connect(sigc::bind<int>(sigc::mem_fun(*this,&MWMainWindow::change_zoom), 0));
  _xml->get_menu_item("zoom_in1")->signal_activate().connect(sigc::bind<int>(sigc::mem_fun(*this,&MWMainWindow::change_zoom), 1));
  _xml->get_menu_item("zoom_out1")->signal_activate().connect(sigc::bind<int>(sigc::mem_fun(*this,&MWMainWindow::change_zoom), -1));
  
  for (int i= 1; i <= 9; i++)
  {
    Gtk::MenuItem *item;
    item= _xml->get_menu_item(ufmt("set_marker_%i", i));
    item->signal_activate().connect(sigc::bind<int>(sigc::mem_fun(*this,&MWMainWindow::set_marker),i));
    item= _xml->get_menu_item(ufmt("go_to_marker_%i", i));
    item->signal_activate().connect(sigc::bind<int>(sigc::mem_fun(*this,&MWMainWindow::go_marker),i));
    item->set_sensitive(false);
  }

  _xml->get_menu_item("grt_console_item")->signal_activate().connect(sigc::mem_fun(*this, &MWMainWindow::show_grt_console));
//  _xml->get_menu_item("new_view1")->signal_activate().connect(sigc::mem_fun(*this, &MWMainWindow::create_canvas_view));
  
//  _xml->get_menu_item("create_schema1")->signal_activate().connect(sigc::mem_fun(*this, &MWMainWindow::create_schema));
  
  _xml->get_menu_item("arrange_objects1")->signal_activate().connect(sigc::mem_fun(*this, &MWMainWindow::arrange_objects));
  
  _xml->get_menu_item("eer1")->signal_activate().connect(sigc::bind<const char*>(sigc::mem_fun(*this,&MWMainWindow::change_relationship_style), "eer"));
//  _xml->get_menu_item("eer2")->signal_activate().connect(sigc::bind<const char*>(sigc::mem_fun(*this,&MWMainWindow::change_relationship_style), "eer"));

  _xml->get_menu_item("traditional1")->signal_activate().connect(sigc::bind<const char*>(sigc::mem_fun(*this,&MWMainWindow::change_relationship_style), "default"));


  _xml->get_menu_item("expand_elements1")->signal_activate().connect(sigc::bind<bool,bool>(sigc::mem_fun(*this,&MWMainWindow::set_elements_expanded), false, true));
  _xml->get_menu_item("collapse_elements1")->signal_activate().connect(sigc::bind<bool,bool>(sigc::mem_fun(*this,&MWMainWindow::set_elements_expanded), false, false));
  _xml->get_menu_item("expand_element_compartments1")->signal_activate().connect(sigc::bind<bool,bool>(sigc::mem_fun(*this,&MWMainWindow::set_elements_expanded), true, true));
  _xml->get_menu_item("collapse_element_compartments1")->signal_activate().connect(sigc::bind<bool,bool>(sigc::mem_fun(*this,&MWMainWindow::set_elements_expanded), true, false));
  
  _xml->get_menu_item("table_visualization1")->set_sensitive(false);

//  _xml->get_menu_item("full1")->signal_activate().connect(sigc::bind<const char*>(sigc::mem_fun(*this,&MWMainWindow::change_detail_level), "default"));
//  _xml->get_menu_item("columns_only1")->signal_activate().connect(sigc::bind<const char*>(sigc::mem_fun(*this,&MWMainWindow::change_detail_level), "simple"));
//  _xml->get_menu_item("icons1")->signal_activate().connect(sigc::bind<const char*>(sigc::mem_fun(*this,&MWMainWindow::change_detail_level), "icon"));

// _xml->get_menu_item("snap_to_grid1")->signal_activate().connect(sigc::mem_fun(*this, &MWMainWindow::snap_grid));
  _xml->get_menu_item("snap_to_grid1")->set_sensitive(false);
  _xml->get_menu_item("display_grid1")->signal_activate().connect(sigc::mem_fun(*this, &MWMainWindow::toggle_grid_mi));

  
  _xml->get_menu_item("reverse_engineering1")->signal_activate().connect(sigc::mem_fun(*this, &MWMainWindow::reverse_engineer));
  
  _xml->get_menu_item("synchronize1")->signal_activate().connect(sigc::bind<bool>(sigc::mem_fun(*this, &MWMainWindow::synchronize_db),false));
  _xml->get_menu_item("resynchronize1")->signal_activate().connect(sigc::bind<bool>(sigc::mem_fun(*this, &MWMainWindow::synchronize_db),true));
  
  _xml->get_menu_item("about1")->signal_activate().connect(sigc::mem_fun(*this,&MWMainWindow::show_about));

  
  myg_menu_add(_rel_menu, _("Delete Relationship"),
               sigc::mem_fun(*this,&MWMainWindow::delete_relationship), "del");
  _rel_menu.show_all();


  while (_xml->get_note("content_note")->get_n_pages())
    _xml->get_note("content_note")->remove_page(0);
  _xml->get_note("content_note")->signal_switch_page().connect(sigc::mem_fun(*this,&MWMainWindow::page_switch));
  
  _xml->get_paned("main_paned")->set_position(_xml->get_paned("main_paned")->get_width()-200);
  _xml->get_paned("side_paned")->set_position(_xml->get_paned("side_paned")->get_height()/2);

 
  _table_colors= Gtk::ListStore::create(_color_columns);
  _xml->get_combo("table_color")->set_model(_table_colors);
  _xml->get_combo("table_color")->pack_start(_color_columns.color);
  
  _view_colors= Gtk::ListStore::create(_color_columns);
  _xml->get_combo("view_color")->set_model(_view_colors);
  _xml->get_combo("view_color")->pack_start(_color_columns.color);

  _routine_colors= Gtk::ListStore::create(_color_columns);
  _xml->get_combo("routine_color")->set_model(_routine_colors);
  _xml->get_combo("routine_color")->pack_start(_color_columns.color);
  
  _layer_colors= Gtk::ListStore::create(_color_columns);
  _xml->get_combo("layer_color")->set_model(_layer_colors);
  _xml->get_combo("layer_color")->pack_start(_color_columns.color);
  
  _note_colors= Gtk::ListStore::create(_color_columns);
  _xml->get_combo("note_color")->set_model(_note_colors);
  _xml->get_combo("note_color")->pack_start(_color_columns.color);

  _table_schemas= Gtk::ListStore::create(_grtvalue_columns);
  _xml->get_combo("table_schema")->set_model(_table_schemas);
  _xml->get_combo("table_schema")->pack_start(_grtvalue_columns.text);
  
  _table_charsets= Gtk::ListStore::create(_grtvalue_columns);
  _xml->get_combo("table_charset")->set_model(_table_charsets);
  _xml->get_combo("table_charset")->pack_start(_grtvalue_columns.text);
  
  _table_engines= Gtk::ListStore::create(_grtvalue_columns);
  _xml->get_combo("table_engine")->set_model(_table_engines);
  _xml->get_combo("table_engine")->pack_start(_grtvalue_columns.text);
  for (int i= 0; table_engines[i]; i++)
  {
    Gtk::TreeIter iter= _table_engines->append();
    Gtk::TreeRow row= *iter;
    row[_grtvalue_columns.text]= table_engines[i];
    row[_grtvalue_columns.value]= MGRTValue(table_engines[i]);
  }
  _xml->get_combo("table_engine")->set_active(0);

  _view_schemas= Gtk::ListStore::create(_grtvalue_columns);
  _xml->get_combo("view_schema")->set_model(_view_schemas);
  _xml->get_combo("view_schema")->pack_start(_grtvalue_columns.text);
  
  _routine_schemas= Gtk::ListStore::create(_grtvalue_columns);
  _xml->get_combo("routine_schema")->set_model(_routine_schemas);
  _xml->get_combo("routine_schema")->pack_start(_grtvalue_columns.text);
  
  _xml->get_entry("plugin_search_entry")->signal_changed().connect(sigc::mem_fun(*this,&MWMainWindow::plugin_search));
  
  _plugin_store= Gtk::ListStore::create(_plug_columns);
  _xml->get_tree("plugin_tree")->signal_row_activated().connect(sigc::mem_fun(*this,&MWMainWindow::plugin_tree_activated));
  _xml->get_tree("plugin_tree")->set_model(_plugin_store);
  _xml->get_tree("plugin_tree")->append_column("", _plug_columns.text);
  ((Gtk::CellRendererText*)_xml->get_tree("plugin_tree")->get_column(0)->get_first_cell_renderer())->property_background()= "#fff0f0";
  _xml->get_tree("plugin_tree")->get_column(0)->add_attribute(((Gtk::CellRendererText*)_xml->get_tree("plugin_tree")->get_column(0)->get_first_cell_renderer())->property_background_set(),
                                                              _plug_columns.supported);

  /*
      _toolCursors= [[NSArray arrayWithObjects:
      [NSCursor arrowCursor],
      [NSCursor openHandCursor],
      [[[NSCursor alloc] initWithImage:[NSImage imageNamed:@"rubber"] hotSpot:NSMakePoint(0,0)] autorelease],
      [[[NSCursor alloc] initWithImage:[NSImage imageNamed:@"table"] hotSpot:NSMakePoint(4,4)] autorelease],
      [[[NSCursor alloc] initWithImage:[NSImage imageNamed:@"view"] hotSpot:NSMakePoint(0,0)] autorelease],
      [[[NSCursor alloc] initWithImage:[NSImage imageNamed:@"routine"] hotSpot:NSMakePoint(0,0)] autorelease],
      [[[NSCursor alloc] initWithImage:[NSImage imageNamed:@"res11"] hotSpot:NSMakePoint(0,0)] autorelease],
      [[[NSCursor alloc] initWithImage:[NSImage imageNamed:@"res1n"] hotSpot:NSMakePoint(0,0)] autorelease],
      [[[NSCursor alloc] initWithImage:[NSImage imageNamed:@"resnm"] hotSpot:NSMakePoint(0,0)] autorelease],
      [[[NSCursor alloc] initWithImage:[NSImage imageNamed:@"image"] hotSpot:NSMakePoint(0,0)] autorelease],
      [[[NSCursor alloc] initWithImage:[NSImage imageNamed:@"note"] hotSpot:NSMakePoint(0,0)] autorelease],
      [[[NSCursor alloc] initWithImage:[NSImage imageNamed:@"layer"] hotSpot:NSMakePoint(0,0)] autorelease],
      nil] retain];
   
*/

  
  _xml->get_entry("x_entry")->signal_changed().connect(sigc::bind<const char*>(sigc::mem_fun(*this,&MWMainWindow::object_property_changed), "x"));
  _xml->get_entry("y_entry")->signal_changed().connect(sigc::bind<const char*>(sigc::mem_fun(*this,&MWMainWindow::object_property_changed), "y"));
  _xml->get_entry("w_entry")->signal_changed().connect(sigc::bind<const char*>(sigc::mem_fun(*this,&MWMainWindow::object_property_changed), "w"));
  _xml->get_entry("h_entry")->signal_changed().connect(sigc::bind<const char*>(sigc::mem_fun(*this,&MWMainWindow::object_property_changed), "h"));
  _xml->get_entry("caption_entry")->signal_changed().connect(sigc::bind<const char*>(sigc::mem_fun(*this,&MWMainWindow::object_property_changed), "caption"));
  _xml->get_text("comment_text")->get_buffer()->signal_changed().connect(sigc::bind<const char*>(sigc::mem_fun(*this,&MWMainWindow::object_property_changed), "comment"));
  _xml->get_text("note_text")->get_buffer()->signal_changed().connect(sigc::bind<const char*>(sigc::mem_fun(*this,&MWMainWindow::object_property_changed), "note"));
  _xml->get_combo("type_combo")->signal_changed().connect(sigc::bind<const char*>(sigc::mem_fun(*this,&MWMainWindow::object_property_changed), "type"));
  _xml->get_toggle("source_optional_check")->signal_toggled().connect(sigc::bind<const char*>(sigc::mem_fun(*this,&MWMainWindow::object_property_changed), "source"));
  _xml->get_toggle("target_optional_check")->signal_toggled().connect(sigc::bind<const char*>(sigc::mem_fun(*this,&MWMainWindow::object_property_changed), "target"));
  _xml->get_button("delete_rel_button")->signal_clicked().connect(sigc::bind<const char*>(sigc::mem_fun(*this,&MWMainWindow::object_property_changed), "delete"));


  // setup icons
  std::list<Glib::RefPtr<Gdk::Pixbuf> > icons;
  //icons.push_back(PIXCACHE->load("MySQLIcon_Workbench_16x16.png"));
  icons.push_back(PIXCACHE->load("MySQLIcon_Workbench_32x32.png")),
  icons.push_back(PIXCACHE->load("MySQLIcon_Workbench_48x48.png"));
  set_icon_list(icons);
}


void MWMainWindow::view_callback(MYX_GRT *grt, MYX_GRT_VALUE *grtView, CGCView *view, MYX_WB_GC_CHANGE change, void *data)
{
  MWMainWindow *me= (MWMainWindow*)data;
  Message msg;

  msg.type= 'V';
  msg.msg= change;
  msg.data.view= view;

//  if (change == MYX_WBGC_SWITCHED)
//    g_message("swiotch view to %p", view);
  
  switch (change)
  {
  case MYX_WBGC_ADDED:    
  case MYX_WBGC_SWITCHED:
  case MYX_WBGC_REMOVED:
    me->_messages.push_back(msg);
    me->handle_message();
    break;
    /*
    me->_messages_mutex.lock();
    me->_messages.push_back(msg);
    me->_messages_mutex.unlock();
    me->_dispatch_message.emit();
    break;
     */

  case MYX_WBGC_RUBBERBAND_STARTED:
  case MYX_WBGC_RUBBERBAND_STOPPED:
    me->refresh_when_idle();
    break;
    
    
  case MYX_WBGC_SELECTION_CHANGE:
    me->refresh_selection_when_idle();
    break;
    
  default:
    break;
  }
}


void MWMainWindow::schemata_changed()
{
  setup_schema_list(_table_schemas);
  _xml->get_combo("table_schema")->set_active(0);
  setup_charset_list(_table_charsets);
  for (Gtk::TreeIter iter= _table_charsets->children().begin();
       iter != _table_charsets->children().end(); ++iter)
  {
    Gtk::TreeRow row= *iter;
    MGRTValue grt= row[_grtvalue_columns.value];
    if (strcmp(grt.asString(), "utf8_general_ci")==0)
    {
      _xml->get_combo("table_charset")->set_active(iter);
      break;
    }
  }
  setup_schema_list(_view_schemas);
  _xml->get_combo("view_schema")->set_active(0);
  setup_schema_list(_routine_schemas);
  _xml->get_combo("routine_schema")->set_active(0);

  _schema_tree->refresh();
}


void MWMainWindow::layer_callback(MYX_GRT *grt, MYX_GRT_VALUE *layer, CFigureInstance *gcLayer, MYX_WB_GC_CHANGE change, void *data)
{
  MWMainWindow *self= (MWMainWindow*)data;
  Message msg;
  
  bool oldval= self->_refreshing_selection_tree;

  self->_refreshing_selection_tree= true;

  msg.type= 'L';
  msg.msg= change;
  msg.data.layer= layer;

  switch (change)
  {
  case MYX_WBGC_ADDED:    
  case MYX_WBGC_MODIFIED:
    self->_messages_mutex.lock();
    self->_messages.push_back(msg);
    self->_messages_mutex.unlock();
    self->_dispatch_message.emit();
    break;

  case MYX_WBGC_REMOVED:
    self->_layer_tree->handle_layer_delete(layer);
    break;
    
  default:
    break;
  }
  self->set_dirty();
  self->refresh_when_idle();
  
  self->_refreshing_selection_tree= oldval;
}
 

void MWMainWindow::element_callback(MYX_GRT *grt, MYX_GRT_VALUE *element, CFigureInstance *gcElem, MYX_WB_GC_CHANGE change, void *data)
{
  MWMainWindow *self= (MWMainWindow*)data;
  bool oldval= self->_refreshing_selection_tree;

  self->_refreshing_selection_tree= true;

  switch (change)
  {
  case MYX_WBGC_ADDED:
    self->_layer_tree->handle_element_add(element);
    break;
  case MYX_WBGC_REMOVED:
    self->_layer_tree->handle_element_delete(element);
    self->handle_element_delete(element);
    break;
  case MYX_WBGC_MODIFIED:
    if (element)
    {
      self->_layer_tree->handle_element_change(element);
    }
    break;
  case MYX_WBGC_CHANGED_LAYER:
    if (element)
    {
      self->_layer_tree->handle_element_delete(element);
      self->_layer_tree->handle_element_add(element);
    }
    break;
  default:
    break;
  }

  self->refresh_when_idle(false);

  self->set_dirty();
  
  self->_refreshing_selection_tree= oldval;
}


static void dummy(const MGRTValue&,bool,void*)
{
}


void MWMainWindow::handle_element_delete(MGRTValue element)
{
  if (element.isKindOf(_grt->grt(), "db.workbench.TableElement"))
  {
    MGRTValue args(MGRTValue::createList());
    args.append(element);
    _grt->call_async_function("Workbench", "deleteFKsWithTable", args, 0, 
                              sigc::ptr_fun(dummy), true);
  }
  
  if (_current_object.isValid() && strcmp(element.dictId(), _current_object.dictId())==0)
    show_object_properties(MGRTValue());
}


void MWMainWindow::change_view_page(int page)
{
  if (_adding_view_page)
    return;

  if (page < (int)_views.size())
  {
    // store the root_layer from the current page
    for (unsigned int i= 0; i < _views.size(); i++)
    {
      if (_canvas->canvas()->currentViewGet() == _views[i].view)
      {
        _views[i].root_layer= _layer_tree->root_layer();
        break;
      }
    }

    //  _canvas_scroll->reparent(*_views[p].frame);
    _layer_tree->change(_views[page].root_layer, _views[page].layer_tree);

    _canvas->switch_to_view(_views[page].view);

    _xml->get_combo("zoom_combo")->set_active(_canvas->zoom_level());
    _canvas->update_scrollers();
  }
}


void MWMainWindow::page_switch(GtkNotebookPage *page, guint n)
{
  change_view_page(n);
}



bool MWMainWindow::idle_refresh()
{
  if (_schema_refresh_pending)
    schemata_changed();
  
  _idle_refresh_pending= false;
  myx_grt_wb_bridge_process_pending(_grt->grt());

  _canvas->queue_draw();

  return false;
}


void MWMainWindow::refresh_when_idle(bool canvas_only)
{
  if (!_idle_refresh_pending)
  {
    _schema_refresh_pending= !canvas_only;
    _idle_refresh_pending= true;
    Glib::signal_idle().connect(sigc::mem_fun(*this,&MWMainWindow::idle_refresh));
  }
}


bool MWMainWindow::refresh_selection()
{
  MYX_GRT_VALUE **objects;
  int count;
  
  _selection_refresh_pending= false;
  
  _refreshing_selection_tree= true;


  _xml->get_menu_item("cut1")->set_sensitive(false);
  _xml->get_menu_item("copy1")->set_sensitive(false);
  _xml->get_menu_item("paste1")->set_sensitive(false);

  objects= myx_grt_wb_get_selected_objects(_grt->grt(), &count);
  if (objects)
  {
    _layer_tree->select_objects(objects, count);

    _xml->get_menu_item("delete1")->set_sensitive(true);

    _xml->get_menu_item("send_to_back1")->set_sensitive(true);
    _xml->get_menu_item("bring_to_front1")->set_sensitive(true);
    
    _has_selection= true;

    refresh_plugins_menu(objects, count);

    g_free(objects);
  }
  else
  {
    refresh_plugins_menu(NULL, 0);
    
    _layer_tree->get_selection()->unselect_all();
    _xml->get_menu_item("delete1")->set_sensitive(false);
    _xml->get_menu_item("send_to_back1")->set_sensitive(false);
    _xml->get_menu_item("bring_to_front1")->set_sensitive(false);
    
    _has_selection= false;
  }
  
  _refreshing_selection_tree= false;
  
  plugin_search();

  return false;
}


void MWMainWindow::refresh_selection_when_idle()
{
  if (!_selection_refresh_pending && !_refreshing_selection)
  {
    _selection_refresh_pending= false;
    Glib::signal_idle().connect(sigc::mem_fun(*this,&MWMainWindow::refresh_selection));
  }
}



void MWMainWindow::set_dirty()
{
  if (!_loading)
  {
    _dirty= true;
    _untouched= false;
  }
}


void MWMainWindow::set_status_text(const Glib::ustring &text)
{
  ((Gtk::Statusbar*)_xml->get_widget("statusbar1"))->pop();
  ((Gtk::Statusbar*)_xml->get_widget("statusbar1"))->push(text);
}


void MWMainWindow::send_to_back(bool front)
{
  if (front)
    _grt->call_async_function("Workbench", "bringSelectionToFront", MGRTValue::createList(),
                              0, sigc::ptr_fun(&MGRT::dummy), true);
  else
    _grt->call_async_function("Workbench", "sendSelectionToBack", MGRTValue::createList(),
                              0, sigc::ptr_fun(&MGRT::dummy), true);
}


void MWMainWindow::change_tool(int tool_index)
{
  _current_tool= tool_buttons[tool_index].type;
  _xml->get_note("tool_option_note")->set_current_page(tool_buttons[tool_index].tool_palette);

  if (_relationship_context.isValid())
  {
    MGRTValue args(MGRTValue::createList());
    args.append(_relationship_context);

    _grt->call_procedure("Workbench","relationshipCancel", args);
    
    _relationship_context.clear();
  }

  if (_cursors[tool_index].gobj())
    _canvas->get_window()->set_cursor(_cursors[tool_index]);
  else
    _canvas->get_window()->set_cursor();

  switch (_current_tool)
  {
  case MWToolCursor:
    _canvas->set_grab_panning(false);
    set_status_text("Cursor Tool");
    break;
  case MWToolPanning:
    _canvas->set_grab_panning(true);
    set_status_text("Panning Tool");
    break;
  case MWToolTable:
    _canvas->set_grab_panning(false);
    set_status_text("Table Tool: select position to place a new table.");
    break;
    
  case MWToolView:
    _canvas->set_grab_panning(false);
    set_status_text("View Tool: select position to place a new view.");
    break;
    
  case MWToolRoutine:
    _canvas->set_grab_panning(false);
    set_status_text("Routine Tool: select position to place new routine package.");
    break;
    
  case MWToolEraser:
    set_status_text("Eraser Tool: click on object to delete");
    break;

  case MWToolNote:
    set_status_text("Note Tool: select area to create note");
    break;

  case MWToolRelationship1:    
  case MWToolRelationship2:
  case MWToolRelationship3:
    {
      MGRTValue args(MGRTValue::createList());
      
      args.append(MGRTValue((_current_tool == MWToolRelationship1) ? "11" : ((_current_tool == MWToolRelationship2) ? "1n" : "nm")));
      args.append(MGRTValue(1));
      args.append(MGRTValue(1));

      _relationship_context= _grt->call_function("Workbench",
                                                 "relationshipStart",
                                                 args);
      if (!_relationship_context.isValid())
        g_message("error initiating relationship creation");
      
      if (_current_tool == MWToolRelationship1)
        set_status_text("1:n Relationship Tool: select the source table.");
      else if (_current_tool == MWToolRelationship2)
        set_status_text("1:1 Relationship Tool: select the source table.");
      else
        set_status_text("n:m Relationship Tool: select the source table.");
    }
    break;
    
  case MWToolLayer:
    set_status_text("Layer Tool: select an area for the new layer.");
    break;
    
  case MWToolImage:
    set_status_text("Image Tool: select position to place image.");
    break;
  }
}


void MWMainWindow::handle_canvas_action(CGCBase *object, TAction *action)
{
  MYX_GRT_VALUE *value= myx_grt_wb_object_for_element(object);

  if (value && action->type == GC_ACTION_APPLICATION+3)
  {
    MGRTValue grtv(value);

    g_message("%s", grtv["name"].asString());
  }
}


bool MWMainWindow::handle_canvas_key(GdkEventKey *event)
{
  guint key;
  Gdk::ModifierType mod;

  for (std::map<Glib::ustring,Gtk::Button*>::const_iterator iter= _shortcut_map.begin();
       iter != _shortcut_map.end(); ++iter)
  {
    Gtk::AccelGroup::parse(iter->first, key, mod);

    if ((event->keyval <= 0xff && isalnum(event->keyval) && (int)key == tolower(event->keyval) && mod == (int)(event->state & 0xf))
        || 
        ((event->keyval > 0xff || !isalnum(event->keyval)) && event->keyval == GDK_Escape && iter->first == "Escape"))
    {
      iter->second->clicked();
      break;
    }
  }
  return true;
}


bool MWMainWindow::handle_button_press(MGCanvas *canvas, int button, int modifiers, MGCanvas::Point pos)
{
  if (button == 1)
  {
    switch (_current_tool)
    {
    case MWToolLayer:
    case MWToolNote:
      _canvas->start_area_selection_at(pos);
      break;
    case MWToolCursor:
      break;
    default:
      return true;
    }
  }

  return false;
}


void MWMainWindow::get_selected_tool_schema(const char *combo, MGRTValue &schema, Glib::ustring &name)
{
  Gtk::TreeRow row;
  
  row= *_xml->get_combo(combo)->get_active();
  name= row[_grtvalue_columns.text];
  schema= row[_grtvalue_columns.value];
}


bool MWMainWindow::handle_button_release(MGCanvas *canvas, int button, int modifiers, MGCanvas::Point pos)
{
  bool handled= false;
  
  if (button == 1)
  {
    MGRTValue schema_value;
    Glib::ustring schema;
    
    switch (_current_tool)
    {
    case MWToolTable:
      {
        MGRTValue args(MGRTValue::createList());

        if (_xml->get_combo("table_schema")->get_active_row_number() < 0)
        {
          Gtk::MessageDialog dlg(*this, _("No schema selected for new table, please create one"));
          dlg.run();
        }
        else
        {
          Gtk::TreeRow row;
          get_selected_tool_schema("table_schema", schema_value, schema);

          char path[schema.length() + 50];
          sprintf(path, "/workbench/catalog/schemata/%s/tables", schema.c_str());
          
          args.append(MGRTValue(floor(pos.x)));
          args.append(MGRTValue(floor(pos.y)));
          args.append(MGRTValue(ufmt("table_%i", get_available_element_index("table_%i", path)).c_str()));

          args.append(schema_value);
          row= *_xml->get_combo("table_engine")->get_active();
          args.append(MGRTValue(row[_grtvalue_columns.value]));
          row= *_xml->get_combo("table_charset")->get_active();
          args.append(MGRTValue(row[_grtvalue_columns.value]));
          row= *_xml->get_combo("table_color")->get_active();
          args.append(MGRTValue(row[_color_columns.value]));
          
          MGRTValue result(_grt->call_function("Workbench",
                                               "placeNewTableElement",
                                               args));
        }
        _default_tool_button->clicked();
        handled= true;
      }
      break;

    case MWToolView:
      {
        MGRTValue args(MGRTValue::createList());
        
        if (_xml->get_combo("view_schema")->get_active_row_number() < 0)
        {
          Gtk::MessageDialog dlg(*this, _("No schema selected for new view, please create one"));
          dlg.run();
        }
        else
        {
          Gtk::TreeRow row;
          get_selected_tool_schema("view_schema", schema_value, schema);

          char path[schema.length() + 50];
          sprintf(path, "/workbench/catalog/schemata/%s/views", schema.c_str());

          args.append(MGRTValue(floor(pos.x)));
          args.append(MGRTValue(floor(pos.y)));
          args.append(MGRTValue(ufmt("view_%i",get_available_element_index("view_%i", path)).c_str()));
          args.append(schema_value);
          row= *_xml->get_combo("view_color")->get_active();
          args.append(MGRTValue(row[_color_columns.value]));
          
          MGRTValue result(_grt->call_function("Workbench",
                                               "placeNewViewElement",
                                               args));
        }
        _default_tool_button->clicked();
        handled= true;
      }
      break;

    case MWToolRoutine:
      {
        MGRTValue args(MGRTValue::createList());

        if (_xml->get_combo("routine_schema")->get_active_row_number() < 0)
        {
          Gtk::MessageDialog dlg(*this, _("No schema selected for new routine group, please create one"));
          dlg.run();
        }
        else
        {
          Gtk::TreeRow row;
          get_selected_tool_schema("routine_schema", schema_value, schema);

          char path[schema.length() + 50];
          sprintf(path, "/workbench/catalog/schemata/%s/routines", schema.c_str());

          args.append(MGRTValue(floor(pos.x)));
          args.append(MGRTValue(floor(pos.y)));
          args.append(MGRTValue(ufmt("routineGroup_%i",get_available_element_index("routineGroup_%i", path)).c_str()));
          args.append(schema_value);
          row= *_xml->get_combo("routine_color")->get_active();
          args.append(MGRTValue(row[_color_columns.value]));
          
          MGRTValue result(_grt->call_function("Workbench",
                                               "placeNewRoutineGroupElement",
                                               args));
        }
        _default_tool_button->clicked();
        handled= true;
      }
      break;
      
    case MWToolRelationship1:
    case MWToolRelationship2:
    case MWToolRelationship3:
      {
        std::string message;
        MGRTValue args(MGRTValue::createList());
        args.append(_relationship_context);
        args.append(MGRTValue(floor(pos.x)));
        args.append(MGRTValue(floor(pos.y)));
        message= _grt->call_string_function("Workbench",
                                            "relationshipClickedPoint",
                                            args);

        if (myx_grt_dict_item_get_as_int(_relationship_context.grtValue(), "step") == -1)
        {
          _default_tool_button->clicked();
        }

        if (!message.empty())
          set_status_text(message);
        handled= true;
      }
      break;

    case MWToolLayer:
      {
        MGCanvas::Rectangle rect= _canvas->finish_area_selection();
        MGRTValue args(MGRTValue::createList());
        MGRTValue result;
        Gtk::TreeRow row;

        args.append(MGRTValue(floor(rect.origin.x)));
        args.append(MGRTValue(floor(rect.origin.y)));
        args.append(MGRTValue(floor(rect.size.width)));
        args.append(MGRTValue(floor(rect.size.height)));
        args.append(MGRTValue(ufmt("Layer %i", get_available_layer_index()).c_str()));
        row= *_xml->get_combo("layer_color")->get_active();
        args.append(row[_color_columns.value]);

        result= _grt->call_function("Workbench", "placeNewLayerElement", args);

        _default_tool_button->clicked();
        handled= true;
      }
      break;

    case MWToolNote:
      {
        MGCanvas::Rectangle rect= _canvas->finish_area_selection();
        MGRTValue args(MGRTValue::createList());
        MGRTValue result;
        Gtk::TreeRow row;

        args.append(MGRTValue(floor(rect.origin.x)));
        args.append(MGRTValue(floor(rect.origin.y)));
        args.append(MGRTValue(floor(rect.size.width)));
        args.append(MGRTValue(floor(rect.size.height)));
        args.append(MGRTValue("Note"));
        row= *_xml->get_combo("note_color")->get_active();
        args.append(row[_color_columns.value]);

        result= _grt->call_function("Workbench",
                                    "placeNewNoteElement",
                                    args);
        
        _default_tool_button->clicked();
        handled= true;
      }
      break;
      
    case MWToolEraser:
      {
        MGRTValue result;
        MGRTValue args(MGRTValue::createList());
        
        args.append(MGRTValue(floor(pos.x)));
        args.append(MGRTValue(floor(pos.y)));
        result= _grt->call_function("Workbench",
                                    "deleteObjectAtPoint",
                                    args);
        handled= true;
      }
      break;

    case MWToolCursor:
      if (_last_click_pos.x == pos.x && _last_click_pos.y == pos.y)
      {
        MGRTValue args(MGRTValue::createList());
        
        args.append(MGRTValue(pos.x));
        args.append(MGRTValue(pos.y));
        
        // raise the object
        _grt->call_function("Workbench", "raiseObjectAtPoint", args);
      }
      // show clicked object properties
      {
        MGRTValue args(MGRTValue::createList());
        MGRTValue result;
        
        args.append(MGRTValue(pos.x));
        args.append(MGRTValue(pos.y));
        
        _last_click_pos= pos;
        result= _grt->call_function("WorkbenchController", "objectAtPoint", args);
        
        if (result.isValid())
          show_object_properties(result["element"]);
        else
          show_object_properties(MGRTValue());
      }

      myx_grt_wb_bridge_process_relocations(_grt->grt());

    default:
      return false;
    }
  }

  // handle any eventual layer/element relocations
  if (_current_tool == MWToolCursor || _current_tool == MWToolLayer)
  {
    myx_grt_wb_bridge_process_relocations(_grt->grt());
  }
  return handled;
}


bool MWMainWindow::create_view_in_canvas(CGCView *view)
{
  _canvas->create_view(view);

  _canvas->switch_to_view(view);
        
  _canvas->set_zoom_level();
  return false;
}


bool MWMainWindow::delayed_handle_view_add(CGCView *view)
{
  ViewInfo vinfo;
  Gtk::Frame *frame;
  bool created= false;

  if (_views.size() == 0 && _xml->get_note("content_note")->get_n_pages() > 0)
    frame= (Gtk::Frame*)_xml->get_note("content_note")->get_nth_page(0);
  else
  {
    frame= Gtk::manage(new Gtk::Frame());
    created= true;
  }
  frame->show();
  
  frame->set_shadow_type(Gtk::SHADOW_NONE);
  
  vinfo.frame= frame;
  vinfo.view= view;
  vinfo.root_layer= 0;
  vinfo.layer_tree= _layer_tree->create_model();
  
  create_view_in_canvas(view);
  
  _views.push_back(vinfo);
  _adding_view_page= true;
  if (created)
    _xml->get_note("content_note")->append_page(*frame, (string)view->propertyGet("name", 0));
  _adding_view_page= false;

  _layer_tree->change(vinfo.root_layer, vinfo.layer_tree);
  
  return false;
}


void MWMainWindow::handle_message()
{
  Message msg;
  
  _messages_mutex.lock();
  msg= *_messages.begin();
  _messages.pop_front();
  _messages_mutex.unlock();
  
  if (msg.type == 'V')
  {
    switch (msg.msg)
    {
    case MYX_WBGC_ADDED:      
      {
        // this is called from an aux thread, but the new view handler must
        // be called from the main thread (esp. because nvidia's libGL.so
        // doesn't seem to like threaded stuff)
        Glib::signal_idle().connect(sigc::bind<CGCView*>(sigc::mem_fun(*this,&MWMainWindow::delayed_handle_view_add),msg.data.view));
        break;
      }
      break;
    case MYX_WBGC_SWITCHED:
      for (unsigned int i= 0; i < _views.size(); i++)
      {
        if (msg.data.view == _views[i].view)
        {
          _xml->get_note("content_note")->set_current_page(i);
          break;
        }
      }
      break;
    case MYX_WBGC_REMOVED:
      {
        int i= 0;
        for (std::vector<ViewInfo>::iterator iter= _views.begin();
             iter != _views.end(); ++iter, i++)
        {
          if (iter->view == msg.data.view)
          {
            _views.erase(iter);
            _layer_tree->change(NULL, Glib::RefPtr<Gtk::TreeStore>());
            if (_views.size() == 0)
              iter->frame->hide();
            else
              _xml->get_note("content_note")->remove_page(i);
            break;
          }
        }
      }
      break;
    default:
      break;
    }
  }
  else if (msg.type == 'L')
  {
    switch (msg.msg)
    {
    case MYX_WBGC_ADDED:
      _layer_tree->handle_layer_add(msg.data.layer);
      break;

    case MYX_WBGC_MODIFIED:
      _layer_tree->handle_layer_change(msg.data.layer);
      break;
      
    default:
      break;
    }
  }
}

void MWMainWindow::setup_schema_list(Glib::RefPtr<Gtk::ListStore> list)
{
  MGRTValue schemata(_grt->global_value("/workbench/catalog/schemata"));

  list->clear();
  for (unsigned int i= 0; i < schemata.count(); i++)
  {
    MGRTValue schema(schemata[i]);
    Gtk::TreeIter iter= list->append();
    Gtk::TreeRow row= *iter;

    row[_grtvalue_columns.text]= schema["name"].asString();
    row[_grtvalue_columns.value]= schema;
  }
}


void MWMainWindow::setup_charset_list(Glib::RefPtr<Gtk::ListStore> list)
{
  MGRTValue charsets(_grt->global_value("/rdbmsMgmt/rdbms/Mysql/characterSets"));

  list->clear();
  for (unsigned int i= 0; i < charsets.count(); i++)
  {
    MGRTValue charset(charsets[i]);
    MGRTValue collations(charset["collations"]);
    
    for (unsigned int j= 0; j < collations.count(); j++)
    {
      const char *coll= collations[j].asString();
      
      Gtk::TreeIter iter= list->append();
      Gtk::TreeRow row= *iter;

      row[_grtvalue_columns.text]= coll;
      row[_grtvalue_columns.value]= collations[j];
    }
  }
}


void MWMainWindow::setup_color_list(Glib::RefPtr<Gtk::ListStore> list,
                                    const MGRTValue &colorList)
{
  unsigned int c= colorList.count();
  list->clear();
  
  for (unsigned i= 0; i < c; i++)
  {
    MGRTValue item(colorList[i]);
  
    Gtk::TreeIter iter= list->append();
    Gtk::TreeRow row= *iter;
    Glib::RefPtr<Gdk::Pixbuf> sample;
    unsigned int pixel;

    sscanf(item["color"].asString(), "#%x", &pixel);
    
    pixel<<= 8;
    
    sample= Gdk::Pixbuf::create(Gdk::COLORSPACE_RGB, false, 8, 22, 16);
    sample->fill(pixel);
//    sample->

    row[_color_columns.color]= sample;
    row[_color_columns.value]= item["colorName"];
  }
}


void MWMainWindow::set_layer_colors(const MGRTValue &colors)
{
  std::map<Glib::ustring,Glib::ustring> color_map;
  unsigned int i, c= colors.count();
  
  for (i= 0; i < c; i++)
  {
    color_map[colors[i]["colorName"].asString()]= colors[i]["color"].asString();
  }
  
  _layer_tree->set_color_values(color_map);
}


void MWMainWindow::setup()
{
  setup_ui();
  _xml->get_box("content_box")->pack_start(*_canvas_scroll);

  if (!_canvas->canvas())
  {
    hide();
    Gtk::MessageDialog dlg(_("<b>Could not create an OpenGL enabled window.</b>\n\nMySQL Workbench requires OpenGL to work. A 3D accelerated card with at least 32MB of memory is recommended."), true,
                           Gtk::MESSAGE_ERROR);
    dlg.run();
    exit(1);
  }
  
  if (!(MYX_GRT_VALUE*)_grt->global_value("/app/options/hideAlphaWarning") || !(int)_grt->global_value("/app/options/hideAlphaWarning"))
  {
    MGGladeXML *xml= new MGGladeXML(get_app_file("alpha_warning.glade"));
    
    if (((Gtk::Dialog*)xml->get_widget("dialog1"))->run() != Gtk::RESPONSE_OK)
    {
      exit(1);
    }
    xml->get_widget("dialog1")->hide();
    
    if (xml->get_toggle("checkbutton1")->get_active())
    {
      _grt->set_global_value("/app/options/hideAlphaWarning", 1);
    }

    _grt->save_subtree("/app/options", prefs.build_path_to("workbench/workbench_app_options.xml"));

    delete xml;
  }
  

  _canvas->canvas()->setTexturePath(get_common_file("").c_str());

  //_canvas->set_status_text(_xml->get_label("status_label"));
  myx_grt_wb_load_style_file(_canvas->canvas(),
                             get_common_file("grt/layout.styles.db.connection.xml").c_str(),
                             NULL);

  myx_grt_wb_load_style_file(_canvas->canvas(),
                             get_common_file("grt/layout.styles.db.table.xml").c_str(),
                             NULL);
  myx_grt_wb_load_style_file(_canvas->canvas(),
                             get_common_file("grt/layout.styles.db.view.xml").c_str(),
                             NULL);
  myx_grt_wb_load_style_file(_canvas->canvas(),
                             get_common_file("grt/layout.styles.db.routine.xml").c_str(),
                             NULL);
  myx_grt_wb_load_style_file(_canvas->canvas(),
                             get_common_file("grt/layout.styles.db.xml").c_str(),
                             NULL);
  {
    MGRTValue presets(_grt->call_function("Workbench", "getColorPresets", MGRTValue()));
    if (!presets.isValid())
      g_warning("Error getting color presets.");
    else
    {
      unsigned int j;
      struct {
        const char *name;
        const char *path;
        const char *combo_name;
        Glib::RefPtr<Gtk::ListStore> list_store;
      } colorPresetInfo[] =
      {
      {"table", "grt/layout.styles.db.table.tmpl.xml", "table_color", _table_colors},
      {"view", "grt/layout.styles.db.view.tmpl.xml", "view_color", _view_colors},
      {"layer", "grt/layout.styles.layer.tmpl.xml", "layer_color", _layer_colors},
      {"note", "grt/layout.styles.note.tmpl.xml", "note_color", _note_colors},
      {"package", "grt/layout.styles.package.tmpl", "routine_color", _routine_colors},
      {NULL, NULL, NULL}
      };
      
      for (j= 0; colorPresetInfo[j].name; j++)
      {
        MGRTValue colorList(presets[colorPresetInfo[j].name]);

        myx_grt_wb_load_style_file(_canvas->canvas(),
                                   get_common_file(colorPresetInfo[j].path).c_str(),
                                   colorList.grtValue());
        setup_color_list(colorPresetInfo[j].list_store, colorList);
        _xml->get_combo(colorPresetInfo[j].combo_name)->set_active(0);
        
        if (strcmp(colorPresetInfo[j].name, "layer")==0)
          set_layer_colors(colorList);
      }
    }
  }
  
  if (!_canvas->load_layouts(get_common_file("grt/layout.figures.db.xml")))
    g_warning("Error reading figures file.");

  _canvas->set_base_size(MWDefaultCanvasSize);

  {
    _zoom_store= Gtk::ListStore::create(_zoom_columns);
    std::vector<float> steps;
    _default_zoom_level= 0;

    _xml->get_combo("zoom_combo")->set_model(_zoom_store);

    steps= _canvas->get_zoom_steps();
    for (unsigned i= 0; i < steps.size(); i++)
    {
      Gtk::TreeIter iter= _zoom_store->append();
      Gtk::TreeRow row= *iter;
      row[_zoom_columns.text]= ufmt("%.0f%%", 100*steps[i]);
      row[_zoom_columns.zoom]= steps[i];
      if (steps[i] == 1)
      {
        _default_zoom_level= i;
        _canvas->set_zoom_level(i);
      }
    }

    _xml->get_combo("zoom_combo")->signal_changed().connect(sigc::mem_fun(*this,&MWMainWindow::zoom_changed));

    _xml->get_combo("zoom_combo")->set_active(_default_zoom_level);
  }

  MGRTValue args;

  args= MGRTValue::createList();

  // setup grt
  if (!_grt->call_procedure("Workbench", "initWorkbench", args))
  {
    Gtk::MessageDialog dlg("The GRT environment for the Workbench could not be initialized.\nPlease verify your installation.");
    dlg.run();
    exit(1);
  }

  myx_grt_wb_bridge_initialize(_grt->grt(), _canvas->canvas(), "/workbench/model");
  
  myx_grt_wb_bridge_set_callbacks(_grt->grt(), this,
                                  view_callback,
                                  layer_callback,
                                  element_callback);
  
  refresh_selection();
  
  _canvas->grab_focus();
}




void MWMainWindow::show_grt_console()
{
  _grt_shell->refresh();
  _grt_shell->show();
}


void MWMainWindow::create_canvas_view()
{
  MGRTValue args= MGRTValue::createList(MYX_ANY_VALUE);
  
  args.append(_grt->global_value("/workbench/model"));
  args.append(MGRTValue(ufmt("View %i", ++_canvas_view_index).c_str()));
  
  _grt->call_function("Workbench", "addView", args);
}


void MWMainWindow::zoom_changed()
{
  int level= _xml->get_combo("zoom_combo")->get_active_row_number();
  
  _canvas->set_zoom_level(level);
}


void MWMainWindow::reverse_engineer()
{
  MWReverseEngineering *rev= MWReverseEngineering::create(_grt);

  rev->set_transient_for(*this);
  rev->run();
}


void MWMainWindow::synchronize_db(bool resync)
{
  MWSynchronizeDatabase *sync= MWSynchronizeDatabase::create(_grt);

  sync->set_transient_for(*this);
  sync->run();
}


void MWMainWindow::generate_script()
{
  Gtk::FileSelection fsel(_("Generate SQL Script"));
  
  if (fsel.run() == Gtk::RESPONSE_OK)
  {
    std::string path= fsel.get_filename();
    
    Glib::ustring data= _grt->call_string_function("Workbench",
                                                   "generateSqlCreateScript",
                                                   MGRTValue());
    if (data.empty())
    {
      set_status_text(_("Error generating SQL script. See GRT console for more information."));
      myg_show_error(*this, _("Error generating SQL script. See GRT console for more information."));
      return;
    }

    FILE *f= fopen(path.c_str(), "w+");
    bool ok= false;
    if (f)
    {
      if (fwrite(data.c_str(), 1, data.size(), f) == data.size())
      {
        set_status_text(ufmt(_("Wrote SQL script to '%s'"), path.c_str()));
        ok= true;
      }
    }
    if (!ok)
    {
      myg_show_sys_error(*this, _("Error writing SQL script."), f?ferror(f):errno);
      
      set_status_text(_("Error writing SQL script."));
    }
    if (f)
      fclose(f);
  }
}


void MWMainWindow::update_title()
{
  if (_filename.empty())
    set_title(_("MySQL Workbench - untitled"));
  else
    set_title(ufmt(_("MySQL Workbench - %s"), Glib::path_get_basename(_filename).c_str()));
}


void MWMainWindow::update_marker_menu()
{
  for (int i= 1; i <= 9; i++)
  {
    if (_canvas->has_marker(i))
      _xml->get_menu_item(ufmt("go_to_marker_%i",i))->set_sensitive(true);
    else
      _xml->get_menu_item(ufmt("go_to_marker_%i",i))->set_sensitive(false);
  }
}


void MWMainWindow::dbd4_import_finished(const MGRTValue&result, bool error, void *data)
{
  if (!error)
    set_status_text(_("Imported DBDesigner4 document."));
  else
    set_status_text(_("Error importing DBDesigner4 document."));
}


void MWMainWindow::import_dbd4()
{
  Gtk::FileSelection dlg(_("Open Model"));
    
  if (dlg.run() == Gtk::RESPONSE_OK)
  {
    MGRTValue args(MGRTValue::createList());
    _filename= dlg.get_filename();
    
    args.append(MGRTValue(_filename.c_str()));

    if (Glib::str_has_suffix(_filename.c_str(), ".xml"))
      _filename= _filename.substr(0, _filename.size()-4)+".mwb";
    else
      _filename+= ".mwb";

    _grt->call_async_function("WorkbenchImport", "importDbd4", args,
                              NULL, sigc::mem_fun(*this,&MWMainWindow::dbd4_import_finished),
                              false);
    set_status_text(_("Importing DBDesigner4 document..."));
    update_title();
  }
}


void MWMainWindow::open_document(const std::string &filename)
{
  if (filename.empty())
  {
    Gtk::FileSelection dlg(_("Open Model"));
    
    if (dlg.run() == Gtk::RESPONSE_OK)
    {
      open_document(dlg.get_filename());
    }
  }
  else
  {
    MYX_GRT_VALUE *data;
    
    data= myx_grt_retrieve_from_file(_grt->grt(), filename.c_str());
    
    if (data)
    {      
      if (_untouched && _filename.empty())
      {
        MGRTValue args(MGRTValue::createList());
        
        args.append(MGRTValue(data));
        
        _loading= true;

        // open in this window
        _grt->call_function("Workbench", "loadDocumentData", args);
        
        _loading= false;
        
        update_marker_menu();
        
        schemata_changed();

        _layer_tree->rescan(_grt->global_ref_value("/workbench/model/currentView/rootLayer"));
        
        set_status_text(ufmt(_("Loaded model file '%s'"), filename.c_str()));

        _filename= filename;
        update_title();
      }
      else
      {
        // open a new instance and run it in there
        char **argv;
        int i;
        bool added= false;
        
        for (i= 0; wb_argv[i]; i++);
        argv= g_new0(char*, i+2);
        for (i= 0; wb_argv[i]; i++)
        {
          if (Glib::str_has_suffix(wb_argv[i], ".mwb"))
          {
            if (!added)
              argv[i]= (char*)filename.c_str();
            added= true;
          }
          else
            argv[i]= wb_argv[i];
        }
        if (!added)
          argv[i++]= (char*)filename.c_str();
        argv[i]= NULL;
        if (!g_spawn_async(NULL, argv, NULL, (GSpawnFlags)0, NULL, NULL, NULL, NULL))
          g_warning("failed spawning new instance");
        g_free(argv);
      }
      myx_grt_value_release(data);
    }
    else
    {
      Gtk::MessageDialog dlg(*this, ufmt(_("Could not load model file '%s'"), filename.c_str()),
                             Gtk::MESSAGE_ERROR);
      
      dlg.run();
      set_status_text(ufmt(_("Failed opening file '%s'"),
                           filename.c_str()));
    }
  }
}


void MWMainWindow::save_document()
{
  if (_filename.empty())
    save_document_as();
  else
  {
    MGRTValue value(_grt->global_value("/workbench/model/properties"));
    char date[100];
    time_t now= time(NULL);
    strftime(date, sizeof(date), "%x %X", localtime(&now));

    value.set("dateChanged", date);
    
    MYX_GRT_ERROR error= myx_grt_store_to_file(_grt->grt(),
                                               _grt->global_value("/workbench"),
                                               _filename.c_str());
    if (error != MYX_GRT_NO_ERROR)
    {
      Gtk::MessageDialog dlg(*this, 
                             ufmt(_("Error saving Workbench model: %s"),
                                  _grt->error_text(error).c_str()),
                             Gtk::MESSAGE_ERROR);
      dlg.run();
    }
    else
    {
      _dirty= false;

      set_status_text(ufmt(_("Model saved to '%s'"), _filename.c_str()));
      update_title();
    }
  }
}


void MWMainWindow::save_document_as()
{
  Gtk::FileSelection fsel(_("Save Model"));
    
  if (fsel.run() == Gtk::RESPONSE_OK)
  {
    fsel.hide();
    std::string filename= fsel.get_filename();
    
    if (!Glib::str_has_suffix(filename, ".mwb"))
      filename+= ".mwb";

    if (Glib::file_test(filename, Glib::FILE_TEST_EXISTS))
    {
      Gtk::MessageDialog dlg(*this, 
                             ufmt(_("<b>A file named '%s' already exists.</b>\n\nDo you want to replace it?"),
                                  filename.c_str()), true,
                             Gtk::MESSAGE_WARNING, Gtk::BUTTONS_NONE);
      dlg.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
      dlg.add_button(_("_Replace"), Gtk::RESPONSE_OK);
      if (dlg.run() == Gtk::RESPONSE_CANCEL)
        return;
    }
    _filename= filename;
    save_document();
  }
}



void MWMainWindow::new_document(bool new_instance)
{  
  if (new_instance)
  {
        _layer_tree->rescan(_grt->global_ref_value("/workbench/model/currentView/rootLayer"));
  return;

    
    // open a new instance and run it in there
    char **argv;
    int i, j;
    
    for (i= 0; wb_argv[i]; i++);
    argv= g_new0(char*, i+1);
    for (i= 0, j=0; wb_argv[i]; i++)
    {
      if (!Glib::str_has_suffix(wb_argv[i], ".mwb"))
        argv[j++]= wb_argv[i];
    }
    argv[j]= NULL;
    if (!g_spawn_async(NULL, argv, NULL, (GSpawnFlags)0, NULL, NULL, NULL, NULL))
      g_warning("failed spawning new instance");
    g_free(argv);
  }
  else
  {
    MGRTValue args= MGRTValue::createList(MYX_STRING_VALUE);
  
    args.append(_grt->global_value("/app/options").get("DefaultRdbms", "Mysql"));
    args.append(_grt->global_value("/app/options").get("DefaultRdbmsVersion", "5.1.6"));
    
    _loading= true;
    
    if (_grt->call_function("Workbench", "newDocument", args).isValid()==false)
      g_message("new doc fail");
    else
      set_status_text(_("New document."));
    
    char date[100];
    time_t now= time(NULL);
    strftime(date, sizeof(date), "%x %X", localtime(&now));
    
    // initialize some model properties
    MGRTValue value(_grt->global_value("/workbench/model/properties"));
    value.set("author", g_get_real_name());
    
    _loading= false;
    
    _filename= "";
    
    schemata_changed();
    
    update_title();
  }
}



void MWMainWindow::change_zoom(int dir)
{
  int zoom= _xml->get_combo("zoom_combo")->get_active_row_number();
  
  if (dir < 0 && zoom > 0)
    _xml->get_combo("zoom_combo")->set_active(zoom-1);
  else if (dir > 0 && zoom < (int)_zoom_store->children().size()-1)
    _xml->get_combo("zoom_combo")->set_active(zoom+1);
  else if (dir == 0)
    _xml->get_combo("zoom_combo")->set_active(_default_zoom_level);
  zoom_changed();

}


void MWMainWindow::select_all()
{
  _grt->call_procedure("Workbench", "selectAll", MGRTValue::createList());
}


void MWMainWindow::toggle_grid()
{
  _xml->get_check_menu_item("display_grid1")->set_active(_grid_check->get_active());
  _canvas->set_grid_enabled(_grid_check->get_active());
}


void MWMainWindow::toggle_grid_mi()
{
  bool active= _xml->get_check_menu_item("display_grid1")->get_active();
  _grid_check->set_active(active);
  _canvas->set_grid_enabled(active);
}
                                   

void MWMainWindow::toggle_overview()
{
  _canvas->toggle_overview();
}


void MWMainWindow::set_marker(int marker)
{
  _canvas->set_marker(marker);
  update_marker_menu();
}


void MWMainWindow::go_marker(int marker)
{
  _canvas->go_to_marker(marker);
}



void MWMainWindow::delete_object()
{
  _grt->call_async_function("Workbench", "deleteSelection", MGRTValue());
}


void MWMainWindow::delete_relationship()
{
  MGRTValue args(MGRTValue::createList());

  args.append(_current_object);
  _grt->call_procedure("Workbench", "deleteRelationship", args);
}


void MWMainWindow::align_objects(int type)
{
  bool vert= false;
  const char *side= NULL;
  bool align= true;
  
  switch (type)
  {
  case 1:
    vert= true;
    side= "T";
    break;
  case 2:
    vert= true;
    side= "C";
    break;
  case 3:
    vert= true;
    side= "B";
    break;
  case 4:
    vert= false;
    side= "L";
    break;
  case 5:
    vert= false;
    side= "C";
    break;
  case 6:
    vert= false;
    side= "R";
    break;
      
  case 10:
    align= false;
    vert= true;
    break;
  case 11:
    align= false;
    vert= false;
    break;
  }
  
  MGRTValue args(MGRTValue::createList(MYX_ANY_VALUE));
  
  if (align)
  {
    args.append(MGRTValue(side));
    if (vert)
    {      
      if (!_grt->call_procedure("Workbench", "verticalAlign", args))
        g_warning("error");
    }
    else
    {
      if (!_grt->call_procedure("Workbench", "horizontalAlign", args))
        g_warning("error");
    }
  }
  else
  {
    args.append(MGRTValue(vert?1:0));
    args.append(MGRTValue(_space_spin->get_value_as_int()));
    if (!_grt->call_procedure("Workbench", "autoSpaceElements", args))
      g_warning("error");
  }
}


void MWMainWindow::export_as_pdf()
{
  Gtk::FileSelection dlg(_("Export as PDF"));
  
  if (dlg.run() == Gtk::RESPONSE_OK)
  {
    std::string file= dlg.get_filename();
    MGRTValue value(_grt->global_value("/workbench/model/properties"));
    TGCViewport bounds;
    
    bounds.left= 0;
    bounds.top= 0;
    bounds.width= (int)_canvas->base_size().width;
    bounds.height= (int)_canvas->base_size().height;

    set_status_text(_("Exporting diagram as PDF..."));

    if (!_canvas->canvas()->renderToFile(file.c_str(), GC_FILE_FORMAT_PDF, value["modelName"].asString(), "MySQL Workbench",  TGCRenderContent(GC_RENDER_CONNECTIONS),
                      1.0, bounds))
    {
      Gtk::MessageDialog dlg(ufmt(_("Could not export to %s."), file.c_str()), Gtk::MESSAGE_ERROR);
      dlg.run();
      set_status_text(_("Error exporting file to PDF"));
    }
    else
      set_status_text(ufmt(_("Diagram exported to %s."), file.c_str()));
  }
}


void MWMainWindow::export_as_png()
{
  Gtk::FileSelection dlg(_("Export as PNG"));
  
  if (dlg.run() == Gtk::RESPONSE_OK)
  {
    std::string file= dlg.get_filename();
    MGRTValue value(_grt->global_value("/workbench/model/properties"));
    TGCViewport bounds;
    
    bounds.left= 0;
    bounds.top= 0;
    bounds.width= (int)_canvas->base_size().width;
    bounds.height= (int)_canvas->base_size().height;
    
    set_status_text(_("Exporting diagram as PNG..."));
        
    if (!_canvas->canvas()->renderToFile(file.c_str(), GC_FILE_FORMAT_PNG, value["modelName"].asString(), "MySQL Workbench",  TGCRenderContent(GC_RENDER_CONNECTIONS),
                      1.0, bounds))
    {
      Gtk::MessageDialog dlg(ufmt(_("Could not export to %s."), file.c_str()), Gtk::MESSAGE_ERROR);
      dlg.run();
      set_status_text(_("Error exporting file to PNG"));
    }
    else
      set_status_text(ufmt(_("Diagram exported to %s."), file.c_str()));
  }
}




void MWMainWindow::show_properties()
{
  if (!_prop_xml)
  {
    _prop_xml= new MGGladeXML(get_app_file("document.glade"),
                              "properties");

    _prop_xml->get_button("ok_button")->signal_clicked().connect(sigc::mem_fun(*this,&MWMainWindow::save_properties));
    _prop_xml->get_button("cancel_button")->signal_clicked().connect(sigc::mem_fun(*(Gtk::Window*)_prop_xml->get_widget("properties"),&Gtk::Widget::hide));
  }
  
  MGRTValue value(_grt->global_value("/workbench/model/properties"));
  
  _prop_xml->get_entry("name")->set_text(value["modelName"].asString());
  _prop_xml->get_entry("version")->set_text(value["version"].asString());
  _prop_xml->get_entry("author")->set_text(value["author"].asString());
  _prop_xml->get_entry("project")->set_text(value["project"].asString());
  _prop_xml->get_entry("cdate")->set_text(value["dateCreated"].asString());
  _prop_xml->get_entry("mdate")->set_text(value["dateChanged"].asString());
  
  _prop_xml->get_text("description_text")->get_buffer()->set_text(value["description"].asString());
  
  ((Gtk::Window*)_prop_xml->get_widget("properties"))->set_transient_for(*this);
  _prop_xml->get_widget("properties")->show();
}


void MWMainWindow::save_properties()
{
  MGRTValue value(_grt->global_value("/workbench/model/properties"));
  
  value.set("modelName", _prop_xml->get_entry("name")->get_text().c_str());
  value.set("version", _prop_xml->get_entry("version")->get_text().c_str());
  value.set("author", _prop_xml->get_entry("author")->get_text().c_str());
  value.set("project", _prop_xml->get_entry("project")->get_text().c_str());
  value.set("dateCreated", _prop_xml->get_entry("cdate")->get_text().c_str());
  value.set("dateChanged", _prop_xml->get_entry("mdate")->get_text().c_str());
  
  value.set("description", _prop_xml->get_text("description_text")->get_buffer()->get_text().c_str());
  

  _prop_xml->get_widget("properties")->hide();
}


void MWMainWindow::layer_tree_selection_changed()
{
  if (!_refreshing_selection_tree)
  {
    std::list<MYX_GRT_VALUE*> selected= _layer_tree->get_selected_objects();
    MGRTValue args(MGRTValue::createList());

    _refreshing_selection= true;
  
    _grt->call_function("WorkbenchController", "unselectObject", args);

    for (std::list<MYX_GRT_VALUE*>::iterator iter= selected.begin();
         iter != selected.end(); ++iter)
    {
      args.clear();
      args.append(MGRTValue(*iter));
    
      _grt->call_function("WorkbenchController", "selectObject",
                          args);
    }
  
    _refreshing_selection= false;
  }
}


bool MWMainWindow::tree_selection_is_draggable()
{
  Gtk::TreeIter iter= _schema_tree->get_selection()->get_selected();

  if (iter)
  {
    MGRTValue value= _schema_tree->value_at_row(Gtk::TreePath(iter));

    if (value.type() == MYX_DICT_VALUE)
    {
      if (value.inheritsFrom(_grt->grt(), "db.Table") ||
          value.inheritsFrom(_grt->grt(), "db.View") ||
          value.inheritsFrom(_grt->grt(), "db.RoutineGroup"))
        return true;
    }
  }
  return false;
}


void MWMainWindow::schema_drag_data_get(const Glib::RefPtr<Gdk::DragContext>& context, Gtk::SelectionData& selection_data, guint info, guint time)
{
  if (tree_selection_is_draggable())
  {
    Gtk::TreeIter iter= _schema_tree->get_selection()->get_selected();
    MGRTValue value= _schema_tree->value_at_row(Gtk::TreePath(iter));

    selection_data.set("x-workbench-schema-object", value.dictId());
  }
}


void MWMainWindow::schema_drag_begin(const Glib::RefPtr<Gdk::DragContext>& context)
{
  if (tree_selection_is_draggable())
  {
    std::list<Gtk::TargetEntry> targets;
    targets.push_back(Gtk::TargetEntry("x-workbench-schema-object", Gtk::TARGET_SAME_APP, 0));

    // hack... can't figure how to dynamically select whether a target
    // accepts a certain type of object
    _canvas->drag_dest_set(targets);//, Gtk::DEST_DEFAULT_ALL, Gdk::DragAction(GDK_ACTION_COPY));
  }
}


void MWMainWindow::schema_drag_end(const Glib::RefPtr<Gdk::DragContext>& context)
{
  _canvas->drag_dest_unset();
}


bool MWMainWindow::canvas_drag_motion(const Glib::RefPtr<Gdk::DragContext>& context, int x, int y, guint time)
{
  _canvas->drag_highlight();
  return true;
}


void MWMainWindow::canvas_drag_leave(const Glib::RefPtr<Gdk::DragContext>& context, guint time)
{
  _canvas->drag_unhighlight();
}


bool MWMainWindow::canvas_drag_drop(const Glib::RefPtr<Gdk::DragContext>& context, int x, int y, guint time)
{
  Gtk::TreeIter iter= _schema_tree->get_selection()->get_selected();

  if (iter)
  {
    MGRTValue value= _schema_tree->value_at_row(Gtk::TreePath(iter));

    if (value.type() == MYX_DICT_VALUE)
    {
      MGRTValue args(MGRTValue::createList());
      MGCanvas::Point pos= _canvas->convert_to_canvas_point(Gdk::Point(x,y));
      
      args.append(value);
      args.append(MGRTValue((int)pos.x));
      args.append(MGRTValue((int)pos.y));

      _grt->call_procedure("Workbench", "placeElementFromObject", args);
                           
      return true;
    }
  }
  return false;
}


void MWMainWindow::plugin_finished(const MGRTValue &result, bool error, void *data)
{
  MGRTValue plugin((MYX_GRT_VALUE*)data);
  
  if (_plugin_output_lines == 1)
  {
    if (Glib::str_has_suffix(_plugin_output, "\n"))
      set_status_text(_plugin_output.substr(0, _plugin_output.bytes()-1));
    else
      set_status_text(_plugin_output);
  }
  else if (error)
    set_status_text(ufmt(_("Error executing plugin '%s'."), plugin["caption"].asString()));
  else
    set_status_text(ufmt(_("Plugin '%s' executed."), plugin["caption"].asString()));
}


void MWMainWindow::plugin_output_handler(Glib::ustring text)
{
  _plugin_output_lines++;
  _plugin_output= text;
  
  Glib::RefPtr<Gtk::TextBuffer> buffer= _xml->get_text("plugin_output")->get_buffer();
  buffer->insert(buffer->end(), text);
}


void MWMainWindow::call_plugin(MGRTValue plugin)
{
  set_status_text(ufmt(_("Calling plugin '%s'..."), plugin["caption"].asString()));
  int count;
  MYX_GRT_VALUE **objects= myx_grt_wb_get_selected_objects(_grt->grt(), &count);
  if (objects)
  {
    MGRTValue args(MGRTValue::createList());
    MGRTValue objList(MGRTValue::createList());
    MGRT::AsyncRequest *req;

    for (int i= 0; i < count; i++)
      objList.append(MGRTValue(objects[i]));
    
    args.append(objList);

    _plugin_output= "";
    _plugin_output_lines= 0;
    _xml->get_text("plugin_output")->get_buffer()->set_text("");

    req= _grt->prepare_async_call(plugin["moduleName"].asString(),
                                  plugin["moduleFunctionName"].asString(),
                                  args,
                                  plugin.grtValue(),
                                  sigc::mem_fun(*this, &MWMainWindow::plugin_finished),
                                  false);
    req->set_output_handler(sigc::mem_fun(*this,&MWMainWindow::plugin_output_handler));
    _grt->queue_request(req);
  }
  else
    set_status_text(_("No objects selected for plugin."));
}


void MWMainWindow::call_plugin_for_object(MGRTValue plugin, MGRTValue target)
{
  set_status_text(ufmt(_("Calling plugin '%s'..."), plugin["caption"].asString()));

  MGRTValue args(MGRTValue::createList());
  MGRTValue objList(MGRTValue::createList());
  MGRT::AsyncRequest *req;

  objList.append(target);
  args.append(objList);

  _plugin_output= "";
  _plugin_output_lines= 0;
  if (_xml)
    _xml->get_text("plugin_output")->get_buffer()->set_text("");

  req= _grt->prepare_async_call(plugin["moduleName"].asString(),
                                plugin["moduleFunctionName"].asString(),
                                args,
                                plugin.grtValue(),
                                sigc::mem_fun(*this, &MWMainWindow::plugin_finished),
                                false);
  req->set_output_handler(sigc::mem_fun(*this,&MWMainWindow::plugin_output_handler));
  _grt->queue_request(req);
}


static bool matches_selection(MYX_GRT *grt, const MGRTValue &plugin, MYX_GRT_VALUE **selection, int count)
{
  MGRTValue validFor(plugin["objectStructNames"]);
  unsigned int i;
  int s;

  if (validFor.count() == 0)
    return true;

  for (s= 0; s < count; s++)
  {
    bool ok= false;
    for (i= 0; i < validFor.count(); i++)
    {
      if (myx_grt_dict_struct_is_or_inherits_from(grt, selection[s], validFor[i].asString()))
      {
        ok= true;
        break;
      }
    }
    if (!ok)
      return false;
  }
  return true;
}


void MWMainWindow::refresh_plugins_menu(MYX_GRT_VALUE **selection, int selection_count, const MGRTValue &root, Gtk::Menu *menu)
{
  MGRTValue rootObj;

  if (!root.isValid())
    rootObj= MGRTValue(_grt->global_value("/app/pluginGroups"));
  else
    rootObj= root;

  if (!menu)
  {
    Gtk::MenuItem *item= _xml->get_menu_item("plugins");
    menu= Gtk::manage(new Gtk::Menu());
    item->set_submenu(*menu);

    for (unsigned int i= 0; i < rootObj.count(); i++)
    {
      MGRTValue group(rootObj[i]);
      Gtk::MenuItem *groupItem= Gtk::manage(new Gtk::MenuItem(group["name"].asString()));
      Gtk::Menu *submenu= Gtk::manage(new Gtk::Menu());
      menu->append(*groupItem);
      groupItem->set_submenu(*submenu);
      refresh_plugins_menu(selection, selection_count, group, submenu);
    }
    menu->show_all();
  }
  else
  {      
    if (rootObj["plugins"].count() > 0)
    {
      MGRTValue plugins(rootObj["plugins"]);
      
      for (unsigned int j= 0; j < plugins.count(); j++)
      {
        MGRTValue plugin(MGRTValue::refObject(_grt->grt(), plugins[j].asString()));
        Gtk::MenuItem *pluginItem= Gtk::manage(new Gtk::MenuItem(plugin["caption"].asString()));

        pluginItem->signal_activate().connect(sigc::bind<MGRTValue>(
                      sigc::mem_fun(*this, &MWMainWindow::call_plugin),plugin));
        
        menu->append(*pluginItem);
        if (!selection || selection_count == 0 
            || !matches_selection(_grt->grt(), plugin, selection, selection_count))
          pluginItem->set_sensitive(false);
      }
    }

    for (unsigned int i= 0; i < rootObj["subGroups"].count(); i++)
    {
      MGRTValue subGroup(rootObj["subGroups"][i]);
      Gtk::MenuItem *groupItem= Gtk::manage(new Gtk::MenuItem(subGroup["name"].asString()));
      Gtk::Menu *submenu= Gtk::manage(new Gtk::Menu());
      menu->append(*groupItem);
      groupItem->set_submenu(*submenu);
      refresh_plugins_menu(selection, selection_count, subGroup, submenu);
    }
    menu->show_all();
  }
}


void MWMainWindow::object_property_changed(const char *prop)
{
  float d;

  if (!_current_object.isValid())
    return;

  if (strcmp(prop, "x")==0)
  {
    d= atof(_xml->get_entry("x_entry")->get_text().c_str());
    _current_object.set("x", d);
  }
  else if (strcmp(prop, "y")==0)
  {
    d= atof(_xml->get_entry("y_entry")->get_text().c_str());
    _current_object.set("y", d);
  }
  else if (strcmp(prop, "w")==0)
  {
    d= atof(_xml->get_entry("w_entry")->get_text().c_str());
    _current_object.set("width", d);
  }
  else if (strcmp(prop, "h")==0)
  {
    d= atof(_xml->get_entry("h_entry")->get_text().c_str());
    _current_object.set("height", d);
  }
  else if (strcmp(prop, "caption")==0)
  {
    _current_object.set("caption", _xml->get_entry("caption_entry")->get_text().c_str());
  }
  else if (strcmp(prop, "comment")==0)
  {
    _current_object.set("comment", _xml->get_text("comment_text")->get_buffer()->get_text().c_str());
  }
  else if (strcmp(prop, "note")==0)
  {
    _current_object.set("text", _xml->get_text("note_text")->get_buffer()->get_text().c_str());
  }
  else if (strcmp(prop, "source")==0)
  {
    _current_object.set("startMandatory", !_xml->get_toggle("source_optional_check")->get_active());
  }
  else if (strcmp(prop, "target")==0)
  {
    _current_object.set("endMandatory", !_xml->get_toggle("target_optional_check")->get_active());
  }
  else if (strcmp(prop, "type")==0)
  {
    switch (_xml->get_combo("type_combo")->get_active_row_number())
    {
    case 0:
      _current_object.set("startMany", 0);
      _current_object.set("endMany", 1);
      break;
    case 1:
      _current_object.set("startMany", 1);
      _current_object.set("endMany", 0);
      break;
    case 2:
      _current_object.set("startMany", 0);
      _current_object.set("endMany", 0);
      break;
    case 3:
      _current_object.set("startMany", 1);
      _current_object.set("endMany", 1);
      break;
    }
  }
  else if (strcmp(prop, "delete")==0)
  {
    MGRTValue args(MGRTValue::createList());
    args.append(_current_object);
    _grt->call_procedure("Workbench", "deleteRelationship", args);
  }
}


void MWMainWindow::show_object_properties(MGRTValue object)
{
  _current_object= MGRTValue();
  
  if (!object.isValid())
  {
    _xml->get_widget("geom_box")->set_sensitive(false);
    _xml->get_widget("relationship_options")->hide();
    _xml->get_widget("comments_frame")->hide();
    _xml->get_widget("note_frame")->hide();
  }
  else
  {
    _xml->get_widget("geom_box")->set_sensitive(true);

    if (object.isKindOf(_grt->grt(), "db.workbench.Relationship"))
    {
      _xml->get_entry("x_entry")->set_text("0");
      _xml->get_entry("y_entry")->set_text("0");
      _xml->get_entry("w_entry")->set_text("0");
      _xml->get_entry("h_entry")->set_text("0");
      _xml->get_entry("x_entry")->set_sensitive(false);
      _xml->get_entry("y_entry")->set_sensitive(false);
      _xml->get_entry("w_entry")->set_sensitive(false);
      _xml->get_entry("h_entry")->set_sensitive(false);
      
      _xml->get_entry("caption_entry")->set_text(object["caption"].asString());
      if (object["startMany"].asInt()==0 && object["endMany"].asInt()==1)
        _xml->get_combo("type_combo")->set_active(0);
      else if (object["startMany"].asInt()==1 && object["endMany"].asInt()==0)
        _xml->get_combo("type_combo")->set_active(1);
      else if (object["startMany"].asInt()==0 && object["endMany"].asInt()==1)
        _xml->get_combo("type_combo")->set_active(2);
      else if (object["startMany"].asInt()==1 && object["endMany"].asInt()==1)
        _xml->get_combo("type_combo")->set_active(3);
      
      _xml->get_toggle("source_optional_check")->set_active(!object["startMandatory"].asInt());
      _xml->get_toggle("target_optional_check")->set_active(!object["endMandatory"].asInt());

      _xml->get_widget("relationship_options")->show();
    }
    else
    {
      _xml->get_entry("x_entry")->set_text(ufmt("%.0f", object["left"].asDouble()));
      _xml->get_entry("y_entry")->set_text(ufmt("%.0f", object["top"].asDouble()));
      _xml->get_entry("w_entry")->set_text(ufmt("%.0f", object["width"].asDouble()));
      _xml->get_entry("h_entry")->set_text(ufmt("%.0f", object["height"].asDouble()));
      _xml->get_entry("x_entry")->set_sensitive(true);
      _xml->get_entry("y_entry")->set_sensitive(true);
      _xml->get_entry("w_entry")->set_sensitive(true);
      _xml->get_entry("h_entry")->set_sensitive(true);
      _xml->get_widget("relationship_options")->hide();
    }

    if (!object.isKindOf(_grt->grt(), "db.workbench.NoteElement"))
    {
      _xml->get_widget("comments_frame")->show();
      _xml->get_widget("note_frame")->hide();
      
      if (object["comment"].isValid())
        _xml->get_text("comment_text")->get_buffer()->set_text(object["comment"].asString());
      else
        _xml->get_text("comment_text")->get_buffer()->set_text("");
    }
    else
    {
      _xml->get_widget("comments_frame")->hide();
      _xml->get_widget("note_frame")->show();
      
      if (object["text"].isValid())
        _xml->get_text("note_text")->get_buffer()->set_text(object["text"].asString());
      else
        _xml->get_text("note_text")->get_buffer()->set_text("");
    }
  }
  _current_object= object;
}


void MWMainWindow::list_plugins_matching(const char *match, MGRTValue list,
                                         MYX_GRT_VALUE **selection, int selection_count)
{
  Gtk::TreeIter iter;
  Gtk::TreeRow row;

  for (unsigned int i= 0; i < list.count(); i++)
  {
    MGRTValue group(list[i]);

    if (group["plugins"].count() > 0)
    {
      MGRTValue plugins(group["plugins"]);

      for (unsigned int j= 0; j < plugins.count(); j++)
      {
        MGRTValue plugin(MGRTValue::refObject(_grt->grt(), plugins[j].asString()));
        bool supported;
        char *caption= g_strdup(plugin["caption"].asString());
        int k;
        
        for (k= 0; caption[k]; k++)
          caption[k]= tolower(caption[k]);
        if (!strstr(caption, match))
        {
          g_free(caption);
          continue;
        }
        g_free(caption);

        if (!selection || !selection_count)
          supported= false;
        else
          supported= matches_selection(_grt->grt(), plugin, selection, selection_count);
        
        iter= _plugin_store->append();
        row= *iter;
        row[_plug_columns.text]= plugin["caption"].asString();
        row[_plug_columns.plugin]= plugin;
        row[_plug_columns.supported]= !supported;
      }
    }

    list_plugins_matching(match, group["subGroups"], selection, selection_count);
  }
}


void MWMainWindow::plugin_search()
{
  Glib::ustring text= _xml->get_entry("plugin_search_entry")->get_text();
  MYX_GRT_VALUE **objects;
  int count;
  
  _plugin_store->clear();

  if (!text.empty())
  {    
    char lower_text[text.bytes()+1];
    unsigned int i;
    objects= myx_grt_wb_get_selected_objects(_grt->grt(), &count);
    
    for (i= 0; i < text.length(); i++)
      lower_text[i]= tolower(text[i]);
    lower_text[i]= 0;
    list_plugins_matching(lower_text, _grt->global_value("/app/pluginGroups"),
                          objects, count);
  }
}



void MWMainWindow::plugin_tree_activated(const Gtk::TreeModel::Path &path, Gtk::TreeViewColumn *column)
{
  Gtk::TreeIter iter= _plugin_store->get_iter(path);
  Gtk::TreeRow row= *iter;
  MGRTValue plugin= row[_plug_columns.plugin];
  
  call_plugin(plugin);
}



void MWMainWindow::change_detail_level(const char *layoutClass)
{
  MGRTValue view(_grt->global_ref_value("/workbench/model/currentView"));
  
  view.set("elementLayoutClass", layoutClass);

  _grt->call_procedure("WorkbenchController", "reconstructViewElements", 
                       MGRTValue::createList());
}


void MWMainWindow::change_relationship_style(const char *styleClass)
{
  MGRTValue view(_grt->global_ref_value("/workbench/model/currentView"));

  view.set("connectionLayoutClass", styleClass);
}


void MWMainWindow::set_elements_expanded(bool compartments, bool expanded)
{
  MGRTValue args(MGRTValue::createList());
  args.append(MGRTValue(compartments));
  
  _grt->call_async_function("Workbench", 
                            expanded?"expandElements":"collapseElements",
                            args,
                            0,
                            sigc::ptr_fun(MGRT::dummy),
                            true);
}



void MWMainWindow::new_view_tab()
{
  Glib::ustring name;
  
  if (myg_ask_string("Workbench", _("Name for New View Tab"), name, false))
  {
    MGRTValue args(MGRTValue::createList());

    args.append(_grt->global_value("/workbench/model"));
    args.append(MGRTValue(name.c_str()));
    
    _grt->call_async_function("Workbench", "addView", args, 0, sigc::ptr_fun(MGRT::dummy), true);
  }
}
