/* Copyright (C) 2002 MySQL AB & Jorge del Conde

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
    
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA 
*/

#include "CUserAdminWindow.h"
#include "CAdministrationPanel.h"
#include "CMySQLQuery.h"
#include <qmessagebox.h>

privateDatabaseItem::privateDatabaseItem (QListView * parent, const QString &dbname, int t)
: QCheckListItem(parent, dbname, QCheckListItem::CheckBox), m_dbname(dbname)  
{  
  setText(0, dbname);
  setPixmap(0, getPixmapIcon("databaseConnectedIcon"));
  init(t);  
}

privateDatabaseItem::privateDatabaseItem (privateDatabaseItem * parent)
: QCheckListItem(parent, 0, QCheckListItem::CheckBox), m_dbname(parent->getDatabaseName())
{
  init(TABLE_ITEM);  
}

void privateDatabaseItem::init(int t)
{
  setType(t);
  itemPrivileges.setAutoDelete(true);
  allPrivs = false;
  withGrant = false;
}

void privateDatabaseItem::stateChange(bool b)
{
  if (type == DATABASE_ITEM)
  {
    QCheckListItem *d = (QCheckListItem *)firstChild();
    listView()->setCursor(Qt::waitCursor);
    while (d != NULL)
    {    
      d->setEnabled(!b);
      d = (QCheckListItem *)d->nextSibling();
    }
    listView()->setCursor(Qt::ArrowCursor);
  }
}

privateTableItem::privateTableItem (privateDatabaseItem * parent, const QString &tablename)
: privateDatabaseItem(parent), m_tablename(tablename)
{  
  setText(0, getDatabaseName() + "." + tablename);
  setPixmap(0, getPixmapIcon("tableIcon"));  
}

CUserAdminWindow::CUserAdminWindow(QWidget* parent, CMyWindow *consoleWindow, CMySQLConnection *m, const QString &username, const QString &hostname, bool isediting, CDatabaseTreeItem *item)
: CMyWindow(parent, "CUserAdminWindow", WDestructiveClose, !g_isMDI), m_Username(username), m_Hostname(hostname), isEditing(isediting)
{  
  treeItem = item;
  m_refresh = false;
  setIcon(getPixmapIcon("applicationIcon"));
  setCaption((!isEditing ? tr("Add User") :(tr("Edit User") + " - '" + username + "@" + hostname + "'")) + " [" + m->getConnectionName() + "]");
  ConsoleWindow = consoleWindow;
  mysql = new CMySQLConnection(m->getConnectionName(), ConsoleWindow->messagePanel());  
  
  setCentralWidget(new QWidget(this, "qt_central_widget"));
  CUserAdminWindowLayout = new QGridLayout(centralWidget(), 1, 1, 4, 2, "CUserAdminWindowLayout");
  
  Password = new QLineEdit(centralWidget(), "Password");
  Password->setFrameShape(QLineEdit::LineEditPanel);
  Password->setFrameShadow(QLineEdit::Sunken);
  QWhatsThis::add(Password, tr("This is the Password for Username."));
  
  CUserAdminWindowLayout->addWidget(Password, 2, 1);
  
  hostLabel = new QLabel(centralWidget(), "hostLabel");
  hostLabel->setText(tr("Host"));
  
  CUserAdminWindowLayout->addWidget(hostLabel, 1, 0);
  
  passwordLabel = new QLabel(centralWidget(), "passwordLabel");
  passwordLabel->setText(tr("Password"));
  
  CUserAdminWindowLayout->addWidget(passwordLabel, 2, 0);
  
  Host = new QLineEdit(centralWidget(), "Host");
  Host->setFrameShape(QLineEdit::LineEditPanel);
  Host->setFrameShadow(QLineEdit::Sunken);
  QWhatsThis::add(Host, tr("This is the Hostname the Username will be connecting from.  Use '%' for referring to Any."));
  
  CUserAdminWindowLayout->addWidget(Host, 1, 1);
  
  ButtonGroup1 = new QButtonGroup(centralWidget(), "ButtonGroup1");
  ButtonGroup1->setTitle(tr("Privileges"));
  ButtonGroup1->setColumnLayout(0, Qt::Vertical);
  ButtonGroup1->layout()->setSpacing(6);
  ButtonGroup1->layout()->setMargin(6);
  ButtonGroup1Layout = new QGridLayout(ButtonGroup1->layout());
  ButtonGroup1Layout->setAlignment(Qt::AlignTop);
  
  allPrivileges = new QCheckBox(ButtonGroup1, "allPrivileges");
  allPrivileges->setText(tr("All Privileges"));
  QWhatsThis::add(allPrivileges, tr("This option will GRANT ALL PRIVILEGES to Username"));
  
  ButtonGroup1Layout->addWidget(allPrivileges, 0, 0);
  
  withGrantOption = new QCheckBox(ButtonGroup1, "withGrantOption");
  withGrantOption->setText(tr("With GRANT option"));
  QWhatsThis::add(withGrantOption, tr("This option will grant Username privileges for GRANT."));
  
  ButtonGroup1Layout->addWidget(withGrantOption, 2, 0);
  
  privilegeListBox = new QListBox(ButtonGroup1, "privilegeListBox");
  privilegeListBox->setSelectionMode(QListBox::Multi);
  QWhatsThis::add(privilegeListBox, tr("Select the desired Privileges for Username"));
  
  ButtonGroup1Layout->addWidget(privilegeListBox, 1, 0);
  
  CUserAdminWindowLayout->addMultiCellWidget(ButtonGroup1, 3, 3, 0, 1);
  
  Username = new QLineEdit(centralWidget(), "Username");
  QWhatsThis::add(Username, tr("This is the Username you will be Granting / Editing privileges to."));
  
  CUserAdminWindowLayout->addWidget(Username, 0, 1);
  
  usernameLabel = new QLabel(centralWidget(), "usernameLabel");
  usernameLabel->setText(tr("Username"));
  
  CUserAdminWindowLayout->addWidget(usernameLabel, 0, 0);

  databaseListView = new QListView(centralWidget(), "databaseListView");  
  databaseListView->addColumn(tr("Allow access to"));
  databaseListView->header()->setResizeEnabled(false, databaseListView->header()->count() - 1);  
  databaseListView->setShowSortIndicator(true);
  databaseListView->setResizeMode(QListView::AllColumns);
  databaseListView->setRootIsDecorated (true);
  QWhatsThis::add(databaseListView, tr("These are the Databases/Tables username will have access to."));  
  
  CUserAdminWindowLayout->addMultiCellWidget(databaseListView, 0, 3, 2, 2);    

  Layout6 = new QHBoxLayout(0, 0, 6, "Layout6");
  deleteButton = new QPushButton( centralWidget(), "deleteButton");
  deleteButton->setText( tr( "&Delete User" ) );
  if (!isEditing)    
    deleteButton->hide();
  Layout6->addWidget( deleteButton );

  QSpacerItem* spacer = new QSpacerItem(20, 20, QSizePolicy::Expanding, QSizePolicy::Minimum);
  Layout6->addItem(spacer);
  
  applyButton = new QPushButton(centralWidget(), "applyButton");
  applyButton->setMinimumSize(QSize(60, 0));
  applyButton->setText(tr(isEditing ? "&Apply" : "&Add"));
  applyButton->setDefault(true);
  QWhatsThis::add(applyButton, tr("Click here to apply changes any you have made."));
  Layout6->addWidget(applyButton);
  
  cancelButton = new QPushButton(centralWidget(), "cancelButton");
  cancelButton->setMinimumSize(QSize(60, 0));
  cancelButton->setText(tr("&Cancel"));
  QWhatsThis::add(cancelButton, tr("Close this Dialog without saving any changes you have made."));
  Layout6->addWidget(cancelButton);
  
  CUserAdminWindowLayout->addMultiCellLayout(Layout6, 4, 4, 0, 2);
  
  // tab order
  setTabOrder(Username, Host);
  setTabOrder(Host, Password);
  setTabOrder(Password, allPrivileges);
  setTabOrder(allPrivileges, privilegeListBox);
  setTabOrder(privilegeListBox, withGrantOption);
  setTabOrder(withGrantOption, databaseListView);  
  setTabOrder(databaseListView, cancelButton);
  setTabOrder(cancelButton, applyButton);
  if (isEditing)
  {    
    Username->setEnabled(false);
    Host->setEnabled(false);
  }
  selectedItems.setAutoDelete(true);
  userGrants.setAutoDelete(true);
  blockPrivilegeListBox = true;
  if (mysql->connect())
    reset();
  else
    applyButton->setEnabled(false);
  init();
  myResize(437, 321);
  autoPlace();
}

CUserAdminWindow::~CUserAdminWindow()
{
  delete mysql;
}

void CUserAdminWindow::reset()
{
  setCursor(Qt::waitCursor);
  applyButton->setEnabled(mysql->isConnected());
  selectedItems.clear();
  allPrivileges->setChecked(false);
  withGrantOption->setChecked(false);
  Username->setText(m_Username);
  Host->setText(m_Hostname);
  Password->setText(QString::null);
  listPrivileges();
  listDatabases();
  setCursor(Qt::ArrowCursor);
}

bool CUserAdminWindow::isRestrictedPrivilege(const QString &s)
{
  QString tmp = s.lower();
  return ((tmp == "reload") || (tmp == "shutdown") || (tmp == "process") || (tmp == "file"));
}

void CUserAdminWindow::parseUserGrants()
{
  userGrants.clear();
  QStringList grants;
  if (CMySQLQuery::ColumnQuery(grants, mysql, "SHOW GRANTS FOR '" + m_Username + "'@'" + m_Hostname + "'", 0, true, true))
  {
    for (QStringList::Iterator it = grants.begin(); it != grants.end(); ++it)
    {
      UserGrants *u = new UserGrants();
      int tmp = (*it).find(" ON ");
      QString tmp2 = (*it).mid((*it).find(' ') + 1, tmp - 6);
      u->allPrivs = (tmp2 == "ALL PRIVILEGES");
      if (!u->allPrivs)
      {
        u->privileges = QStringList::split (", ", tmp2);
        for (QStringList::Iterator i = u->privileges.begin(); i != u->privileges.end(); ++i)
        {
          if (*i == "REFERENCE")
            *i = "References";  //Work around for inconsistency in mysqld
          else
            *i = capitalize(*i);
        }
      }      
      (*it) = (*it).right((*it).length() - tmp - 4);
      u->item = (*it).left((*it).find(' '));      
      u->withGrant = ((*it).find("WITH GRANT OPTION") != -1);
      if ((tmp2 != "USAGE") || u->withGrant || u->allPrivs)
        userGrants.append(u);
      else
        delete u;
    }
  }
}

privateDatabaseItem *CUserAdminWindow::findItem(const QString &txt, int type)
{
  QListViewItemIterator it(databaseListView);    
  for ( ; it.current(); ++it )
  {
    if (type == 0)
      return (privateDatabaseItem *)it.current();
    else
      if ((it.current()->text(0) == txt) && (((privateDatabaseItem *)it.current())->getType() == type))
        return (privateDatabaseItem *)it.current();
  }
  return 0;
}

void CUserAdminWindow::selectFirstItem()
{
  if (isEditing)
  {
    QListViewItemIterator it(databaseListView);    
    for ( ; it.current(); ++it )
      if (((privateDatabaseItem *)it.current())->isOn())
      {
        databaseListView->setSelected(it.current(), true);
        return;
      }
  }  
  databaseListView->setSelected(databaseListView->firstChild(), true);
}

void CUserAdminWindow::setUserGrants()
{
  parseUserGrants();  
  UserGrants *i;    
  for (i = userGrants.first(); i; i = userGrants.next())
  {
    int type = TABLE_ITEM;
    if (i->item == "*.*")
      type = 0;
    else
    {
      int tmp = i->item.find(".*");
      if (tmp != -1)
      {
        type = DATABASE_ITEM;
        i->item = i->item.left(tmp);
      }
    }
    privateDatabaseItem *p = findItem(i->item, type);    
    if (p != 0)
    {      
      if (p->getType() == TABLE_ITEM)
        p->parent()->setOpen(true);      
      p->setOn(true);
      p->allPrivs = i->allPrivs;
      p->withGrant = i->withGrant;
      if (!p->allPrivs)
        for (QStringList::Iterator prv = i->privileges.begin(); prv != i->privileges.end(); ++prv)
        {
          bool *b = p->itemPrivileges.find(*prv);
          if (b != NULL)
            *b = true;          
        }
    }
  }
}

void CUserAdminWindow::listDatabases()
{  
  databaseListView->clear();
  CMySQLQuery *q = new CMySQLQuery(mysql);
  if (q->exec("SHOW DATABASES", false, false))
  {
    privateDatabaseItem * globalPrivilegesItem = new privateDatabaseItem(databaseListView, tr("Global Privileges"), 0);
    globalPrivilegesItem->setPixmap(0, getPixmapIcon("globalPrivsIcon"));
    for (unsigned int i = 0; i < privilegeListBox->count(); i++)
      globalPrivilegesItem->itemPrivileges.insert(privilegeListBox->text(i), new bool(false));
    while (q->next())
    {
      QString row = q->Row(0);
      if (mysql->setDatabaseName(row))
      {
        privateDatabaseItem * database = new privateDatabaseItem(databaseListView, row);
        initItemPrivileges(database);
        CMySQLQuery *q2 = new CMySQLQuery(mysql);
        if (q2->exec("SHOW TABLES", false, false))
        {
          while (q2->next())
            initItemPrivileges(new privateTableItem(database, q2->Row(0)));          
        }
        delete q2;
      }
    }
  }
  delete q;
  if (isEditing)
    setUserGrants();
  selectFirstItem();
  setCurrentItemPrivileges(databaseListView->currentItem());  
}

void CUserAdminWindow::initItemPrivileges(privateDatabaseItem *item)
{
  for (unsigned int i = 0; i < privilegeListBox->count(); i++)
  {
    if (isEditing)
      item->itemPrivileges.insert(privilegeListBox->text(i), new bool(false));
    else
      item->itemPrivileges.insert(privilegeListBox->text(i), new bool(!isRestrictedPrivilege(privilegeListBox->text(i))));
  }
}

void CUserAdminWindow::setCurrentItemPrivileges(QListViewItem *item)
{
  if (item == 0)
    return;
  blockPrivilegeListBox = true;
  int mytype = ((privateDatabaseItem *)item)->getType();  
  for (unsigned int i = 0; i < privilegeListBox->count(); i++)
  {
    privilegeListBox->item(i)->setSelectable(true);
    bool *b = ((privateDatabaseItem *)item)->itemPrivileges[privilegeListBox->text(i)];
    privilegeListBox->setSelected(i, *b);
    privilegeListBox->item(i)->setSelectable((mytype != 0) ? !isRestrictedPrivilege(privilegeListBox->text(i)) : true);
  }
  withGrantOption->setChecked(((privateDatabaseItem *)item)->withGrant);  
  allPrivileges->setChecked(((privateDatabaseItem *)item)->allPrivs);
  privilegeListBox->setEnabled(!allPrivileges->isChecked());
  blockPrivilegeListBox = false;
  privilegeListBoxChanged();
}

void CUserAdminWindow::privilegeListBoxChanged()
{
  if (blockPrivilegeListBox)
    return;
  privateDatabaseItem *p = (privateDatabaseItem *)databaseListView->currentItem();
  if (!p || (p == 0))
    return;
  for (unsigned int i = 0; i < privilegeListBox->count(); i++)  
    *(p->itemPrivileges[privilegeListBox->text(i)]) = privilegeListBox->isSelected(i);  
}

void CUserAdminWindow::listPrivileges()
{
  privilegeListBox->clear();
  privilegeListBox->insertItem("Select");
  privilegeListBox->insertItem("Insert");
  privilegeListBox->insertItem("Update");
  privilegeListBox->insertItem("Delete");
  privilegeListBox->insertItem("Create");
  privilegeListBox->insertItem("Drop");
  privilegeListBox->insertItem("Index");
  privilegeListBox->insertItem("Alter");
  privilegeListBox->insertItem("References");

  privilegeListBox->insertItem("Reload");
  privilegeListBox->insertItem("Shutdown");
  privilegeListBox->insertItem("Process");
  privilegeListBox->insertItem("File");  
}

bool CUserAdminWindow::prepareAccessTo()
{  
  privateDatabaseItem *d = (privateDatabaseItem *)databaseListView->firstChild();
  selectedItems.clear();  
  while (d != NULL)
  {
    if (d->isOn())  //database or global privs is selected          
        selectedItems.append(new SelectedItems((d->getType() == 0) ? (const QString) "*.*" : d->text(0) + ".*", d->allPrivs, d->withGrant, ((privateDatabaseItem *)d)->itemPrivileges));    
    else  //database was not selected...maybe tables are.
    {
      QCheckListItem *t = (QCheckListItem *)d->firstChild();
      while (t != NULL)
      {
        if (t->isOn())                  
          selectedItems.append(new SelectedItems(t->text(0), ((privateDatabaseItem *)t)->allPrivs, ((privateDatabaseItem *)t)->withGrant, ((privateDatabaseItem *)t)->itemPrivileges));
        t = (QCheckListItem *)t->nextSibling();
      }
    }
    d = (privateDatabaseItem *)d->nextSibling();
  }
  return (!selectedItems.isEmpty());    
}

bool CUserAdminWindow::removeUser(CMyWindow *consoleWindow, CMySQLConnection *m, const QString &username, const QString &hostname)
{
  if (g_confirmCritical)
    if ((QMessageBox::information(0, QObject::tr("Confirm Delete User"), 
      QObject::tr("Are you sure you want to Delete User") + ": '" + username + "@" + hostname + "' ?",
      QObject::tr("&Yes"), QObject::tr("&No")) != 0))
      return false;
  QStringList tables;
  tables += "user";
  tables += "db";
  tables += "tables_priv";
  tables += "columns_priv";
  bool ret = true;
  CMySQLQuery *q = new CMySQLQuery(m);    
  for ( QStringList::Iterator it = tables.begin(); it != tables.end(); ++it)  
    ret &= q->exec_static("DELETE FROM mysql." + *it + " WHERE User='" + username + "' AND Host = '" + hostname + "'");
  delete q;
  if (ret)
  {
    m->PrintError(INFORMATION, tr("User deleted successfully"));
    CAdministrationPanel::Flush_Privileges(consoleWindow, m);
  }
  return ret;
}

bool CUserAdminWindow::execute(const QString &access, bool allPrivs, bool withGrant, const QDict<bool> &Privileges)
{  
  QString sql = "GRANT ";
  if (!allPrivs)
  {
    bool hasPriv = false;
    QDictIterator<bool> it(Privileges);
    for( ; it.current(); ++it )
    {
      if (*it.current())
      {
        if (!hasPriv)
          hasPriv = true;
        sql += it.currentKey() + ", ";
      }
    }
    if (hasPriv)
      sql = sql.left(sql.length() - 2);
    else
      sql += "USAGE";
  }
  else
    sql += "ALL PRIVILEGES";
  sql += " ON " + access + " TO ";
  sql += "'" + Username->text() + "'@'" + (Host->text().isEmpty() ? (const QString) "%" : Host->text()) + "'";  
  if (!Password->text().isEmpty())
    sql += " IDENTIFIED BY '" + Password->text() + "'";
  if (withGrant)
    sql += " WITH GRANT OPTION";
  bool ret = false;
  CMySQLQuery *q = new CMySQLQuery(mysql);  
  if (q->exec_static(sql))
    ret = true;
  else
    mysql->PrintError(CRITICAL, sql);
  delete q;  
  return ret;
}

void CUserAdminWindow::deletePrivileges()
{
  CMySQLQuery *q = new CMySQLQuery(mysql);
  q->exec_static("REVOKE ALL PRIVILEGES ON *.* FROM '" + Username->text() + "'@'" + (Host->text().isEmpty() ? (const QString)"%" : Host->text()) + "'");
  q->exec_static("REVOKE GRANT OPTION ON *.* FROM '" + Username->text() + "'@'" + (Host->text().isEmpty() ? (const QString)"%" : Host->text()) + "'");
  QStringList tables;
  tables += "db";
  tables += "tables_priv";
  tables += "columns_priv";
  for ( QStringList::Iterator it = tables.begin(); it != tables.end(); ++it)
    q->exec_static("DELETE FROM mysql." + *it + " WHERE User='" + Username->text() + "' AND Host = '" + (Host->text().isEmpty() ? (const QString)"%" : Host->text()) + "'");  
  delete q;
  CAdministrationPanel::Flush_Privileges(ConsoleWindow, mysql);
}

void CUserAdminWindow::applyClicked()
{
  bool b = true;
  m_refresh = true;  
  if (prepareAccessTo())
  {
    if (isEditing)
      deletePrivileges();    
    SelectedItems *i;    
    for (i = selectedItems.first(); i; i = selectedItems.next())
    {
      b &= execute(i->item, i->allPrivs, i->withGrant, i->privs);
      if (!b)
        break;
    }    
    if (b)
    {
      if (!isEditing)
      {
        mysql->PrintError(INFORMATION, tr("User created successfully."));
        reset();
      }
      else
      {
        mysql->PrintError(INFORMATION, tr("User modified successfully."));
        close();
      }
    }
    else
      selectedItems.clear();
  }
  else
    mysql->PrintError(WARNING, tr("You must select the databases/tables that the user will have access to."));  
}

void CUserAdminWindow::allPrivilegesToggled(bool b)
{
  privateDatabaseItem *p = (privateDatabaseItem *)databaseListView->currentItem();
  if (!p || (p == 0))
    return;
  p->allPrivs = b;  
  privilegeListBox->setEnabled(!b);
}

void CUserAdminWindow::withGrantToggled(bool b)
{
  privateDatabaseItem *p = (privateDatabaseItem *)databaseListView->currentItem();
  if (!p || (p == 0))
    return;
  p->withGrant = b;  
}

void CUserAdminWindow::databaseListViewMenu(QListViewItem *, const QPoint &pos, int)
{
  QPopupMenu p_itemMenu;  
  p_itemMenu.insertItem(getPixmapIcon("refreshIcon"), tr("Refresh"), MENU_REFRESH);
  switch (p_itemMenu.exec(pos))
  {  
  case MENU_REFRESH:
    reset();
    break;
  }    
}

void CUserAdminWindow::closeEvent(QCloseEvent * e)
{
  beforeClose();
  if (m_refresh && (treeItem != 0))
  {
    treeItem->listView()->setCurrentItem(treeItem);
    treeItem->refresh();
  }
  e->accept();
}

void CUserAdminWindow::deleteClicked()
{
  m_refresh = true;
  if (removeUser(ConsoleWindow, mysql, Username->text(), Host->text()))
    close();
}

void CUserAdminWindow::init()
{
  connect(applyButton, SIGNAL(clicked()), this, SLOT(applyClicked()));
  connect(deleteButton, SIGNAL(clicked()), this, SLOT(deleteClicked()));
  connect(cancelButton, SIGNAL(clicked()), this, SLOT(close()));
  connect(allPrivileges, SIGNAL(toggled(bool)), this, SLOT(allPrivilegesToggled(bool)));
  connect(withGrantOption, SIGNAL(toggled(bool)), this, SLOT(withGrantToggled(bool)));
  connect(databaseListView, SIGNAL(contextMenuRequested(QListViewItem *, const QPoint &, int)), this, SLOT(databaseListViewMenu(QListViewItem *, const QPoint &, int))); 
  connect(databaseListView, SIGNAL(currentChanged(QListViewItem *)), this, SLOT(setCurrentItemPrivileges(QListViewItem *)));
  connect(privilegeListBox, SIGNAL(selectionChanged()), this, SLOT(privilegeListBoxChanged()));
}

