/* Copyright (C) 2002 MySQL AB & Jorge del Conde

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
    
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA 
*/

#include "CTableItem.h"
#include "CTableSpecific.h"
#include "syntaxhighliter_sql.h"

CTableItem::CTableItem(CMyWindow *consoleWindow, int serverid, CDatabaseTreeItem * parent, const QString &tablename, CShowTableProperties *tableproperties, int widgetid, QWidgetStack *w)
: CDatabaseTreeItem(parent), tableProperties(tableproperties)
{	  
  m_pMySQLDatabase = ((CDatabaseTreeItem *)this->parent())->m_pMySQLDatabase;
  ConsoleWindow = consoleWindow;
  serverID = serverid;
  widgetID = widgetid;
  widgetStack = w;
  setType(TABLE_ITEM);
  setText(0, tablename);
  setName("CTableItem");  
  setPixmap(0, getPixmapIcon("tableIcon"));  
  setRenameEnabled(0, true);
  DBKeyWord *dbk = g_DBKeywordDict->find(serverID);
  if (dbk != NULL)
  {
    dbk->addTable(tablename);
    dbk->addTable(m_pMySQLDatabase->getDatabaseName() + "." + tablename);
  }
}

CTableItem::~CTableItem()
{
  m_pMySQLDatabase = 0;
}

void CTableItem::startDrag()
{	
  itemDrag("TABLE");    
}

void CTableItem::clicked()
{
  setCurrentText(text(0));
  if (tableProperties->getTableName() != text(0))
    tableProperties->exec(text(0));
  if (widgetStack->visibleWidget() != (QWidget *)tableProperties)
    widgetStack->raiseWidget(widgetID); 
}

void CTableItem::renamed(const QString & fromText, const QString & toText)
{    
  if (!CTableSpecific::renameTable(m_pMySQLDatabase, fromText, toText))
    setText(0,fromText);
  else
  {
    listView()->setCurrentItem(this->parent());
    ((CDatabaseTreeItem *) this->parent())->refresh();
  }
}

void CTableItem::doubleClicked()
{    
  setCurrentText(text(0));
  CTableSpecific::processMenu(m_pMySQLDatabase, SQL_ALL_ROWS, text(0), serverID);  
}

inline int f_processMenu(int res, CDatabaseTreeItem *p, QWidgetStack *w, CMySQLConnection *m, const QString &txt, int serverid)
{  
  int r = 0;
  if (CTableTools::inRange(res))
    CTableTools::processMenu(w, m, res, txt, serverid);
  else
    if (CTableSpecific::inRange(res))
      CTableSpecific::processMenu(m, res, txt, serverid);
    else
      switch (res)
    {
      case MENU_EDIT: CTableSpecific::editTable(m, txt);
        break;
      case MENU_PROPERTIES: CTableSpecific::tableProperties(m, txt);
        break;
      case MENU_EXPORT: CTableSpecific::exportTable(m, txt);
        break;
      case MENU_EMPTY: CTableSpecific::emptyTable(m, txt);
        break;
      case MENU_DELETE:
        if (CTableSpecific::deleteTable(m, txt))
        {
          p->listView()->setCurrentItem(p);
          p->refresh();
        }
        break;
      default: r = res;
        break;
    }    
    return r;
}

void CTableItem::processMenu(int res)
{	
  int r = f_processMenu(res, ((CDatabaseTreeItem *) this->parent()), widgetStack, m_pMySQLDatabase, text(0), serverID);
  if (r != 0)
  switch (r)  
  {
    case MENU_RENAME:
      {        
        setCurrentText(text(0));
        startRename(0);
        break;
      }
    case MENU_REFRESH:
      {
        listView()->setCurrentItem((CDatabaseTreeItem *) this->parent());
        ((CDatabaseTreeItem *) this->parent())->refresh();
      }
      break;
  }
}

inline int f_displayMenu(const QPoint &pos, CMySQLConnection *mysql, bool disable)
{
  QPopupMenu *p_itemMenu = new QPopupMenu();
  Q_CHECK_PTR(p_itemMenu);
  
  QPopupMenu *p_openTableMenu = CTableSpecific::openTableMenu();  	
  p_itemMenu->insertItem(getPixmapIcon("openTableIcon"), qApp->translate("CTableItem","Open Table"), p_openTableMenu);  
  p_itemMenu->insertItem(getPixmapIcon("designIcon"), qApp->translate("CTableItem","Edit Table"), MENU_EDIT);  
  p_itemMenu->insertItem(getPixmapIcon("propertiesIcon"), qApp->translate("CTableItem","Properties"), MENU_PROPERTIES);  
  p_itemMenu->insertItem(getPixmapIcon("exportTableIcon"), qApp->translate("CTableItem","Export Table"), MENU_EXPORT);
  p_itemMenu->insertSeparator();
  QPopupMenu *p_toolsMenu = CTableTools::tableToolsMenu();  
  p_itemMenu->insertItem(getPixmapIcon("toolsIcon"), qApp->translate("CTableItem","Tools"), p_toolsMenu, MENU_TOOLS);
  p_itemMenu->insertSeparator();  
  p_itemMenu->insertItem(getPixmapIcon("trashIcon"), qApp->translate("CTableItem","Empty Table"), MENU_EMPTY);  
  p_itemMenu->insertItem(getPixmapIcon("deleteIcon"), qApp->translate("CTableItem","Delete Table"), MENU_DELETE);  
  p_itemMenu->insertItem(getPixmapIcon("renameTableIcon"), qApp->translate("CTableItem","Rename"), MENU_RENAME);
  p_itemMenu->insertSeparator();  
  p_itemMenu->insertItem(getPixmapIcon("refreshTablesIcon"), qApp->translate("CTableItem","Refresh Tables"), MENU_REFRESH);

   //Disable features that aren't currently working
  p_itemMenu->setItemEnabled(MENU_EDIT, false);
  p_itemMenu->setItemEnabled(MENU_EXPORT, false);

  if (disable)  
  {
    for (uint i = 0; i < p_itemMenu->count(); i++)  //disable everything except the refresh menu
    {
      int id = p_itemMenu->idAt(i);
      p_itemMenu->setItemEnabled(id, id == MENU_REFRESH);
    }    
  }
  else
  if (mysql != 0)
  {    
    bool notnew = ((mysql->getMySQLServerVersion().major <= 3) && (mysql->getMySQLServerVersion().minor <= 22));
    if (notnew)  //Disable unsupported features for mysql <= 3.22  
    {
      p_itemMenu->setItemEnabled(MENU_PROPERTIES, false);  //Disable 'Table Properties'
      p_itemMenu->setItemEnabled(MENU_TOOLS, false);  //Disable 'Table Tools'      
    }
  }
  int res = p_itemMenu->exec(pos);
  delete p_itemMenu;
  delete p_openTableMenu;
  delete p_toolsMenu;
  return res;	
}

int CTableItem::displayMenu(const QPoint &pos)
{
  return f_displayMenu(pos, m_pMySQLDatabase);
}

CTableItemToolbar::CTableItemToolbar(QMainWindow *parent)
 : CToolBar(TABLE_ITEM, tr("Table"), parent, "Table Toolbar")
{  
  p_openTableMenu = CTableSpecific::openTableMenu();
  addButton(p_openTableMenu, getPixmapIcon("openTableIcon"), tr("Open Table"));
  addButton(MENU_EDIT, getPixmapIcon("designIcon"), tr("Edit Table"));  
  addButton(MENU_PROPERTIES, getPixmapIcon("propertiesIcon"), tr("Properties"));
  addButton(MENU_EXPORT, getPixmapIcon("exportTableIcon"), tr("Export Table"));
  insertSeparator();
  p_toolsMenu = CTableTools::tableToolsMenu();
  addButton(p_toolsMenu, getPixmapIcon("toolsIcon"), tr("Tools"));
  insertSeparator();
  addButton(MENU_EMPTY, getPixmapIcon("trashIcon"), tr("Empty Table"));
  addButton(MENU_DELETE, getPixmapIcon("deleteIcon"), tr("Delete Table"));
  addButton(MENU_RENAME, getPixmapIcon("renameTableIcon"), tr("Rename Table"));
  insertSeparator();
  addButton(MENU_REFRESH, getPixmapIcon("refreshTablesIcon"), tr("Refresh Tables"));
}

void CTableItemToolbar::disableButtons(const int &maj, const int &min)
{
  //Disable features that aren't currently working
  Button(1)->setEnabled(false);  //Disable 'Edit Table'
  Button(3)->setEnabled(false);  //Disable 'Export Table'

  //Disable features not supported by mysql <= 3.22
  if ((maj <= 3) && (min <= 22))
  {
    Button(2)->setEnabled(false);  //Disable 'Table Properties'
    Button(4)->setEnabled(false);  //Disable 'Table Tools'
  }
}

CTableItemToolbar::~CTableItemToolbar()
{
  delete p_openTableMenu;
  delete p_toolsMenu;
}

