/* Copyright (C) 2002 MySQL AB & Jorge del Conde

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
    
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA 
*/

#include "CTableGroupItem.h"
#include "CTableSpecific.h"

CTableGroupItem::CTableGroupItem(CMyWindow *consoleWindow, int serverid, CDatabaseTreeItem * parent, QWidgetStack *w) : CDatabaseTreeItem(parent)
{	
  m_pMySQLServer = ((CDatabaseTreeItem *)this->parent())->m_pMySQLServer;
  m_pMySQLDatabase = ((CDatabaseTreeItem *)this->parent())->m_pMySQLDatabase;
  deleteMySQL = false;
  init_item(w, consoleWindow, serverid);
}

CTableGroupItem::CTableGroupItem(CMyWindow *consoleWindow, int serverid, QListView * parent, CMySQLConnection *mysql, QString dbname, CMySQLConnection *dbconn, QWidgetStack *w) : CDatabaseTreeItem(parent)
{
  m_pMySQLServer = mysql;
  m_pMySQLDatabase = dbconn;
  databaseName = dbname;
  deleteMySQL = true;
  init_item(w, consoleWindow, serverid);
  clicked();
}

void CTableGroupItem::refresh()
{
  if (isRefreshing)
    return;  
  isRefreshing = true;
  listView()->setCursor (Qt::waitCursor);
  tableView->setCursor (Qt::waitCursor);
  CMySQLQuery *query = new CMySQLQuery(m_pMySQLDatabase);
  QString sql;
  bool notnew = ((m_pMySQLDatabase->getMySQLServerVersion().major <= 3) && (m_pMySQLDatabase->getMySQLServerVersion().minor <= 22)) || !m_pMySQLDatabase->isRetrieveShowTableStatus();
  sql = notnew ? "SHOW TABLES" : "SHOW TABLE STATUS";
  if (query->exec(sql, false))
  {
    while (this->childCount() > 0)
      deleteChilds();
    tableView->reset();
    tableView->setNumRows(query->numRows());
    tableView->setNumCols(notnew ? 1 : 6);
    // TODO: get rid of magic numbers
    tableView->horizontalHeader()->setLabel(0,QObject::tr("Table"), 120);
    if (!notnew)
    {
      tableView->horizontalHeader()->setLabel(1,QObject::tr("Records"), 50);
      tableView->horizontalHeader()->setLabel(2,QObject::tr("Size (bytes)"), 65);
      tableView->horizontalHeader()->setLabel(3,QObject::tr("Created"), 110);
      tableView->horizontalHeader()->setLabel(4,QObject::tr("Type"), 65);
      tableView->horizontalHeader()->setLabel(5,QObject::tr("Comments"), 180);
    }    
    int j = 0;    
    while (query->next())
    {
#ifdef DEBUG
      QString dbg = QString::null;
      for (uint x = 0; x < query->numFields(); x++)      
        dbg += query->Row(x) + " | ";
      qDebug(dbg);
#endif
      new CTableItem(ConsoleWindow, serverID, this, query->Row(0), tableProperties, tablePropertiesWidgetID, widgetStack);      
      tableView->setPixmap(j,0, getPixmapIcon("tableIcon"));
      tableView->setText(j,0, query->Row(0));
      if (!notnew)
      {
        tableView->setText(j,1, query->Row(3));			
        tableView->setText(j,2, query->Row(5));
        tableView->setText(j,3, query->Row(10));
        tableView->setText(j,4, query->Row(1));
        tableView->setText(j,5, query->Row(14));
      }
      j++;
    }  
  }
  listView()->setCursor(Qt::ArrowCursor);
  tableView->setCursor(Qt::ArrowCursor);
  delete query;  
  isRefreshing = false;
}

void CTableGroupItem::init_item(QWidgetStack *w, CMyWindow *consoleWindow, int serverid)
{	
  isRefreshing = false;
  serverID = serverid;
  ConsoleWindow = consoleWindow;
  setType(TABLE_GROUP);
  setName("CTableGroupItem");
  setText(0, QObject::tr("Tables", ""));
  widgetStack = w;
    
  setPixmap(0, getPixmapIcon("tablesIcon"));
  
  tableView = new CShowTableGroupProperties(widgetStack, m_pMySQLDatabase, this, serverID);
  Q_CHECK_PTR(tableView);
  widgetID = getWidgetID();
  widgetStack->addWidget (tableView, widgetID);
  
  tableProperties = new CShowTableProperties((QWidget *) widgetStack, serverID, m_pMySQLDatabase);
  Q_CHECK_PTR(tableProperties); 
  tablePropertiesWidgetID = getWidgetID();
  widgetStack->addWidget(tableProperties, tablePropertiesWidgetID);  
  refresh();
}

CTableGroupItem::~CTableGroupItem()
{
  if (deleteMySQL)
  {
    delete m_pMySQLServer;
    delete m_pMySQLDatabase;
  }
  widgetStack->removeWidget(tableView);
  widgetStack->removeWidget(tableProperties);
  delete tableProperties;
  delete tableView;
}

void CTableGroupItem::clicked()
{
  widgetStack->raiseWidget(widgetID);	
}

void CTableGroupItem::processMenu(int res)
{
  if (isRefreshing)
    return; 
  if (CTableTools::inRange(res))
    CTableTools::processMenu(widgetStack, m_pMySQLDatabase, res, "", serverID);
  else
    switch (res)
		{
      case MENU_NEW: CTableSpecific::newTable(m_pMySQLDatabase);
        break;
      case MENU_QUERY:	CTableSpecific::QueryWindow(m_pMySQLDatabase, "", SQL_QUERY, serverID);
        break;
      case MENU_IMPORT: CTableSpecific::importTable(m_pMySQLDatabase);
        break;
      case MENU_REFRESH: refresh();
        break;
		}
}

int CTableGroupItem::displayMenu(const QPoint &pos)
{
  if (isRefreshing)
    return -1; 
  QPopupMenu *p_itemMenu = new QPopupMenu();
  Q_CHECK_PTR(p_itemMenu);  
  p_itemMenu->insertItem(getPixmapIcon("newTableIcon"), QObject::tr("New Table"), MENU_NEW);
  p_itemMenu->insertSeparator();  
  p_itemMenu->insertItem(getPixmapIcon("sqlIcon"), QObject::tr("Query"), MENU_QUERY);
  p_itemMenu->insertSeparator();  
  QPopupMenu *p_toolsMenu = CTableTools::tableToolsMenu();  
  p_itemMenu->insertItem(getPixmapIcon("toolsIcon"), QObject::tr("Tools"), p_toolsMenu, MENU_TOOLS);    
  p_itemMenu->insertItem(getPixmapIcon("importTableIcon"), QObject::tr("Import"), MENU_IMPORT);
  p_itemMenu->insertSeparator();  
  p_itemMenu->insertItem(getPixmapIcon("refreshTablesIcon"), QObject::tr("Refresh"), MENU_REFRESH);
  p_itemMenu->insertSeparator();  
  p_itemMenu->insertItem(getPixmapIcon("newWindowIcon"), QObject::tr("New Window from Here"),
			 NEW_WINDOW_FROM_HERE);
  
  //Disable features that aren't currently working
  p_itemMenu->setItemEnabled(MENU_NEW, false);
  p_itemMenu->setItemEnabled(MENU_IMPORT, false);

  bool notnew = ((m_pMySQLDatabase->getMySQLServerVersion().major <= 3) && (m_pMySQLDatabase->getMySQLServerVersion().minor <= 22));
  if (notnew)  //Disable unsupported features for mysql <= 3.22
    p_itemMenu->setItemEnabled(MENU_TOOLS, false);

  int res = p_itemMenu->exec(pos);
  delete p_itemMenu;
  delete p_toolsMenu;
  return res;	
}

CTableGroupToolbar::CTableGroupToolbar(QMainWindow * parent)
:CToolBar(TABLE_GROUP, QObject::tr("Table Group"), parent,
	  "Table Group Toolbar")
{  
  addButton(MENU_NEW, getPixmapIcon("newTableIcon"), QObject::tr("New Table"));     
  insertSeparator();  
  addButton(MENU_QUERY, getPixmapIcon("sqlIcon"), QObject::tr("Query"));   
  insertSeparator();
  p_toolsMenu = CTableTools::tableToolsMenu();
  addButton(p_toolsMenu, getPixmapIcon("toolsIcon"), QObject::tr("Tools"));
  addButton(MENU_IMPORT, getPixmapIcon("importTableIcon"), QObject::tr("Import"));
  insertSeparator();
  addButton(MENU_REFRESH, getPixmapIcon("refreshTablesIcon"), QObject::tr("Refresh"));
  insertSeparator();
  addButton(NEW_WINDOW_FROM_HERE, getPixmapIcon("newWindowIcon"), QObject::tr("Open in New Window"));
}

void CTableGroupToolbar::disableButtons(const int &maj, const int &min)
{
  //Disable features that aren't currently working
  Button(0)->setEnabled(false);  //Disable 'New Table'
  Button(3)->setEnabled(false);  //Disable 'Import Table'

  //Disable features not supported by mysql <= 3.22
  if ((maj <= 3) && (min <= 22))
    Button(2)->setEnabled(false);  //Disable 'Table Tools'
}

CTableGroupToolbar::~CTableGroupToolbar()
{
  delete p_toolsMenu;
}
