/* Copyright (C) 2002 MySQL AB & Jorge del Conde

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
    
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA

  Modified by: Jorge del Conde

  Original Code Copyrighted by Trolltech AS
*/

#include "editor.h"
#include "parenmatcher.h"
#include "CGeneralConfig.h"
#include "Config.h"
#include <qfile.h>
#include <qprinter.h>
#include "qrichtext_p.h"
#include <qapplication.h>
#include <qpopupmenu.h>
#include <qaccel.h>
#include <qsimplerichtext.h>

Editor::Editor(QWidget *parent, const char *name )
: QTextEdit( parent, name ), hasError( FALSE )
{
  document()->setFormatter( new QTextFormatterBreakInWords );
  setHScrollBarMode( QScrollView::AlwaysOff );
  setVScrollBarMode( QScrollView::AlwaysOn );
  document()->setUseFormatCollection( FALSE );
  parenMatcher = new ParenMatcher;
  connect( this, SIGNAL( cursorPositionChanged( QTextCursor * ) ),
    this, SLOT( cursorPosChanged( QTextCursor * ) ) );
  document()->addSelection( Error );
  document()->addSelection( Step );
  document()->setSelectionColor( Error, red );
  document()->setSelectionColor( Step, yellow );
  document()->setInvertSelectionText( Error, FALSE );
  document()->setInvertSelectionText( Step, FALSE );
  document()->addSelection( ParenMatcher::Match );
  document()->addSelection( ParenMatcher::Mismatch );
  SqlFormat *k = (SqlFormat *)g_SqlEditorStyles->find(SQLKeyWord::highlight_PARENTHESES_MATCHING);
  if (k != NULL)
    document()->setSelectionColor( ParenMatcher::Match, k->Color);
  
  document()->setSelectionColor( ParenMatcher::Mismatch, QColor( 255, 170, 170 ) );
  document()->setInvertSelectionText( ParenMatcher::Match, FALSE );
  document()->setInvertSelectionText( ParenMatcher::Mismatch, FALSE );
  
  accelComment = new QAccel( this );
  accelComment->connectItem( accelComment->insertItem( ALT + Key_C ),
    this, SLOT( commentSelection() ) );
  accelUncomment = new QAccel( this );
  accelUncomment->connectItem( accelUncomment->insertItem( ALT + Key_U ),
    this, SLOT( uncommentSelection() ) );  
}

Editor::~Editor()
{
  delete parenMatcher;
}

void Editor::cursorPosChanged( QTextCursor *c )
{
  if ( parenMatcher->match( c ) )
    repaintChanged();
  if ( hasError )
  {
    emit clearErrorMarker();
    hasError = FALSE;
  }
}

#include <qmessagebox.h>
void Editor::print()
{

#ifndef QT_NO_PRINTER
  bool b = true;  //Just to avioid compiler warnings
  QPrinter printer;
  printer.setFullPage(true);
  if ( printer.setup( this ) )
  {
    QPainter p( &printer );
    QPaintDeviceMetrics metrics( p.device() );
    int dpix = metrics.logicalDpiX();
    int dpiy = metrics.logicalDpiY();
    const int margin = 72;
    QRect body( margin * dpix / 72, margin * dpiy / 72,
      metrics.width() - margin * dpix / 72 * 2,
      metrics.height() - margin * dpiy / 72 * 2 );    
    QSimpleRichText richText(charReplace(text(), '\n', "<br>"), g_printerFont, context(), styleSheet(), mimeSourceFactory(), body.height() );
    richText.setWidth( &p, body.width() );
    QRect view( body );
    int page = 1;
    do
    {
      richText.draw( &p, body.left(), body.top(), view, colorGroup() );
      view.moveBy( 0, body.height() );
      p.translate( 0 , -body.height() );
      p.setFont( g_printerFont );
      p.drawText( view.right() - p.fontMetrics().width( QString::number( page ) ),
        view.bottom() + p.fontMetrics().ascent() + 5, QString::number( page ) );
      if ( view.top()  >= richText.height() )
        break;
      printer.newPage();
      page++;
    } while (b);
  }
#endif
}

void Editor::load( const QString &fn )
{
  filename = fn;
  QFile f( filename );
  if ( !f.open( IO_ReadOnly ) )
    return;
  QCString txt;
  txt.resize( f.size() );
  f.readBlock( txt.data(), f.size() );
  QString s( QString::fromLatin1( txt ) );
  setText( s );
}

void Editor::configChanged()
{
  document()->invalidate();
  viewport()->repaint( FALSE );
}

void Editor::setErrorSelection( int line )
{
  QTextParag *p = document()->paragAt( line );
  if ( !p )
    return;
  QTextCursor c( document() );
  c.setParag( p );
  c.setIndex( 0 );
  document()->removeSelection( Error );
  document()->setSelectionStart( Error, &c );
  c.gotoLineEnd();
  document()->setSelectionEnd( Error, &c );
  hasError = TRUE;
  viewport()->repaint( FALSE );
}

void Editor::setStepSelection( int line )
{
  QTextParag *p = document()->paragAt( line );
  if ( !p )
    return;
  QTextCursor c( document() );
  c.setParag( p );
  c.setIndex( 0 );
  document()->removeSelection( Step );
  document()->setSelectionStart( Step, &c );
  c.gotoLineEnd();
  document()->setSelectionEnd( Step, &c );
  viewport()->repaint( FALSE );
}

void Editor::clearStepSelection()
{
  document()->removeSelection( Step );
  viewport()->repaint( FALSE );
}

void Editor::doChangeInterval()
{
  emit intervalChanged();
  QTextEdit::doChangeInterval();
}

void Editor::commentSelection()
{
  QTextParag *start = document()->selectionStart( QTextDocument::Standard );
  QTextParag *end = document()->selectionEnd( QTextDocument::Standard );
  if ( !start || !end )
    start = end = textCursor()->parag();
  while ( start )
  {
    if ( start == end && textCursor()->index() == 0 )
      break;
    start->insert( 0, "//" );
    if ( start == end )
      break;
    start = start->next();
  }
  document()->removeSelection( QTextDocument::Standard );
  repaintChanged();
  setModified( TRUE );
}

void Editor::uncommentSelection()
{
  QTextParag *start = document()->selectionStart( QTextDocument::Standard );
  QTextParag *end = document()->selectionEnd( QTextDocument::Standard );
  if ( !start || !end )
    start = end = textCursor()->parag();
  while ( start )
  {
    if ( start == end && textCursor()->index() == 0 )
      break;
    while ( start->at( 0 )->c == '/' )
      start->remove( 0, 1 );
    if ( start == end )
      break;
    start = start->next();
  }
  document()->removeSelection( QTextDocument::Standard );
  repaintChanged();
  setModified( TRUE );
}

bool Editor::eventFilter( QObject *o, QEvent *e )
{  
  if ( ( e->type() == QEvent::FocusIn || e->type() == QEvent::FocusOut ) &&
    ( o == this || o == viewport() ) )
  {
    accelUncomment->setEnabled( e->type() == QEvent::FocusIn );
    accelComment->setEnabled( e->type() == QEvent::FocusIn );
  }    
  return QTextEdit::eventFilter( o, e ); 
}

