/* Copyright (C) 2002 MySQL AB & Jorge del Conde

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
    
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA

  Modified by: Jorge del Conde

  Original Code Copyrighted by Trolltech AS
*/

#include "completion.h"
#include "editor.h"
#include "paragdata.h"
#include <qlistbox.h>
#include <qvbox.h>
#include <qrichtext_p.h>
#include <qapplication.h>
#include <qregexp.h>
#include <qsizegrip.h>
#include <qtimer.h>

static QColor getColor( const QString &type )
{
  if ( type == "function" || type == "slot")
    return Qt::blue;
  if ( type == "variable" )
    return Qt::darkRed;
  if ( type == "property" )
    return Qt::darkGreen;
  return Qt::black;
}

class CompletionItem : public QListBoxItem
{
public:
    CompletionItem( QListBox *lb, const QString &txt, const QString &t, const QString &p,
		    const QString &pre, const QString &p2 )
	: QListBoxItem( lb ), type( t ), postfix( p ), prefix( pre ), postfix2( p2 ),
	  parag( 0 ), lastState( FALSE ) { setText( txt ); }
    ~CompletionItem() { delete parag; }
    void paint( QPainter *painter ) {
	if ( lastState != isSelected() ) {
	    delete parag;
	    parag = 0;
	}
	lastState = isSelected();
	if ( !parag )
	    setupParag();
	parag->paint( *painter, listBox()->colorGroup() );
    }

    int height( const QListBox * ) const {
	if ( !parag )
	    ( (CompletionItem*)this )->setupParag();
	return parag->rect().height();
    }
    int width( const QListBox * ) const {
	if ( !parag )
	    ( (CompletionItem*)this )->setupParag();
	return parag->rect().width() - 2;
    }
    QString text() const { return QListBoxItem::text() + postfix; }

private:
    void setupParag() {
	if ( !parag ) {
	    QTextFormatter *formatter;
	    formatter = new QTextFormatterBreakWords;
	    formatter->setWrapEnabled( FALSE );
	    parag = new QTextParag( 0 );
	    parag->pseudoDocument()->pFormatter = formatter;
	    parag->insert( 0, " " + type + ( type.isEmpty() ? " " : "\t" ) + prefix +
			   QListBoxItem::text() + postfix + postfix2 );
	    bool selCol = isSelected() && listBox()->colorGroup().highlightedText() != listBox()->colorGroup().text();
	    QColor sc = listBox()->colorGroup().highlightedText();
	    QTextFormat *f1 = parag->formatCollection()->format( listBox()->font(), selCol ? sc : getColor( type ) );
	    QTextFormat *f3 = parag->formatCollection()->format( listBox()->font(), isSelected() ?
								 listBox()->colorGroup().highlightedText() :
								 listBox()->colorGroup().text() );
	    QFont f( listBox()->font() );
	    f.setBold( TRUE );
	    QTextFormat *f2 =
		parag->formatCollection()->format( f, isSelected() ? listBox()->colorGroup().highlightedText() :
								 listBox()->colorGroup().text() );
	    parag->setFormat( 1, type.length() + 1, f1 );
	    parag->setFormat( type.length() + 2, prefix.length() + QListBoxItem::text().length(), f2 );
	    if ( !postfix.isEmpty() )
		parag->setFormat( type.length() + 2 + prefix.length() + QListBoxItem::text().length(),
				  postfix.length(), f3 );
	    parag->setFormat( type.length() + 2 + prefix.length() + QListBoxItem::text().length() + postfix.length(),
			      postfix2.length(), f3 );
	    f1->removeRef();
	    f2->removeRef();
	    f3->removeRef();
	    parag->format();
	}
    }
    QString type, postfix, prefix, postfix2;
    QTextParag *parag;
    bool lastState;

};

EditorCompletion::EditorCompletion( Editor *e )
{
  enabled = TRUE;
  lastDoc = 0;
  completionPopup = new QVBox( 0, 0, WType_Popup );
  completionPopup->setFrameStyle( QFrame::Box | QFrame::Plain );
  completionPopup->setLineWidth( 1 );
  completionListBox = new QListBox( completionPopup, "editor_completion_lb" );
  completionListBox->setFrameStyle( QFrame::NoFrame );
  completionListBox->installEventFilter( this );
  completionListBox->setHScrollBarMode( QScrollView::AlwaysOn );
  completionListBox->setVScrollBarMode( QScrollView::AlwaysOn );
  completionListBox->setCornerWidget( new QSizeGrip( completionListBox, "editor_cornerwidget" ) );
  completionPopup->installEventFilter( this );
  completionPopup->setFocusProxy( completionListBox );
  completionOffset = 0;
  curEditor = e;
  curEditor->installEventFilter( this );
  caseDict.setAutoDelete(true);
}

EditorCompletion::~EditorCompletion()
{
  delete completionPopup;
}

void EditorCompletion::addCompletionEntry( const QString &s, QTextDocument *, bool strict )
{
  QString mys = s.lower();

  if (caseDict.find(mys) == NULL)
    caseDict.insert(mys, new QString(s));

  QChar key( mys[ 0 ] );
  QMap<QChar, QStringList>::Iterator it = completionMap.find( key );
  if ( it == completionMap.end() )
  {
    completionMap.insert(key, QStringList( mys ) );
  }
  else
  {
    if ( strict )
    {
      QStringList::Iterator sit;
      for ( sit = (*it).begin(); sit != (*it).end(); )
      {
        QStringList::Iterator it2 = sit;
        ++sit;
        if ( (*it2).length() > s.length() && (*it2).left( s.length() ) == s )
        {
          if ( (*it2)[ (int)mys.length() ].isLetter() && (*it2)[ (int)mys.length() ].upper() != (*it2)[ (int)mys.length() ] )
            return;
        }
        else
          if ( mys.length() > (*it2).length() && mys.left( (*it2).length() ) == *it2 )
          {
            if ( mys[ (int)(*it2).length() ].isLetter() && mys[ (int)(*it2).length() ].upper() != mys[ (int)(*it2).length() ] )
              (*it).remove( it2 );
          }
      }
    }
    (*it).append( mys );
  }
}

QValueList<CompletionEntry> EditorCompletion::completionList( const QString &s, QTextDocument *) const
{   
  QChar key( s[ 0 ] );
  QMap<QChar, QStringList>::ConstIterator it = completionMap.find( key );
  if ( it == completionMap.end() )
    return QValueList<CompletionEntry>();
  QStringList::ConstIterator it2 = (*it).begin();
  QValueList<CompletionEntry> lst;
  int len = s.length();
  for ( ; it2 != (*it).end(); ++it2 )
  {
    CompletionEntry c;
    c.type = "";    
    QString *tmp = caseDict.find((*it2));
    c.text = ((tmp != NULL) ? *tmp : (*it2));
    c.postfix = "";
    c.prefix = "";
    c.postfix2 = "";
    if ( (int)(*it2).length() > len && (*it2).left( len ) == s && lst.find( c ) == lst.end() )
      lst << c;
  }
  
  return lst;
}

bool EditorCompletion::doCompletion()
{
  searchString = "";
  if ( !curEditor )
    return FALSE;
  
  QTextCursor *cursor = curEditor->textCursor();
  QTextDocument *doc = curEditor->document();
  int idx = cursor->index();
  if ( idx == 0 )
    return FALSE;
  QChar c = cursor->parag()->at( idx - 1 )->c;
  if ( !c.isLetter() && !c.isNumber() && c != '_' && c != '#' && c != '.')
    return FALSE;
  
  QString s;
  idx--;
  completionOffset = 1;
  bool b = true;
  while (b)
  {
    s.prepend( QString( cursor->parag()->at( idx )->c ) );
    idx--;
    if ( idx < 0 )
      break;
    if ( !cursor->parag()->at( idx )->c.isLetter() &&
	     !cursor->parag()->at( idx )->c.isNumber() &&
       cursor->parag()->at( idx )->c != '_' &&
       cursor->parag()->at( idx )->c != '.' &&
       cursor->parag()->at( idx )->c != '#' )
       break;
    completionOffset++;
  }
  
  s = s.lower();
  searchString = s;
  
  QValueList<CompletionEntry> lst( completionList( s, doc ) );
  if ( lst.count() > 1 )
  {
    QTextStringChar *chr = cursor->parag()->at( cursor->index() );
    int h = cursor->parag()->lineHeightOfChar( cursor->index() );
    int x = cursor->parag()->rect().x() + chr->x;
    int y, dummy;
    cursor->parag()->lineHeightOfChar( cursor->index(), &dummy, &y );
    y += cursor->parag()->rect().y();
    completionListBox->clear();
    for ( QValueList<CompletionEntry>::ConstIterator it = lst.begin(); it != lst.end(); ++it )
      (void)new CompletionItem( completionListBox, (*it).text, (*it).type, (*it).postfix,
      (*it).prefix, (*it).postfix2 );
    cList = lst;
    completionListBox->sort();
    completionPopup->resize( completionListBox->sizeHint() +
      QSize( completionListBox->verticalScrollBar()->width() + 4,
      completionListBox->horizontalScrollBar()->height() + 4 ));
    completionListBox->setCurrentItem( 0 );
    completionListBox->setFocus();
    if ( curEditor->mapToGlobal(QPoint(0, y)).y() + h + completionPopup->height() < QApplication::desktop()->height())
      completionPopup->move(curEditor->mapToGlobal( curEditor->
      contentsToViewport(QPoint(x, y + h))));
    else
      completionPopup->move( curEditor->mapToGlobal( curEditor->
      contentsToViewport(QPoint( x, y - completionPopup->height()))));
    completionPopup->show();
  }
  else
    
    if ( lst.count() == 1 )
    {      
      for (unsigned int x = 1; x <= searchString.length(); x++)
        curEditor->doKeyboardAction (QTextEdit::ActionBackspace);  
      curEditor->insert( lst.first().text, TRUE );      
    }	
    else
    {
      return FALSE;
    }
    
    return TRUE;
}

bool EditorCompletion::eventFilter( QObject *o, QEvent *e )
{
  if ( !enabled )
    return FALSE;  
  if ( o->inherits( "Editor" ) && e->type() == QEvent::KeyPress )
  {
    curEditor = (Editor*)o;
    QKeyEvent *ke = (QKeyEvent*)e;    
    if ( ke->key() == Key_Tab ) {
      QString s = curEditor->textCursor()->parag()->string()->toString().
        left( curEditor->textCursor()->index() );
      if ( s.simplifyWhiteSpace().isEmpty() )
      {
        curEditor->indent();
        int i = 0;
        for ( ; i < curEditor->textCursor()->parag()->length() - 1; ++i )
        {
          if ( curEditor->textCursor()->parag()->at( i )->c != ' ' &&
            curEditor->textCursor()->parag()->at( i )->c != '\t' )
            break;
        }
        curEditor->drawCursor( FALSE );
        curEditor->textCursor()->setIndex( i );
        curEditor->drawCursor( TRUE );
        return TRUE;
      }
    }    
    if ( ke->text().length() && !( ke->state() & AltButton ) &&
	     ( !ke->ascii() || ke->ascii() >= 32 ) ||
       ( ke->text() == "\t" && !( ke->state() & ControlButton ) ) )
    {      
      if ( ke->key() == Key_Tab )
      {
        if ( curEditor->textCursor()->index() == 0 &&
          curEditor->textCursor()->parag()->style() &&
          curEditor->textCursor()->parag()->style()->displayMode() ==
          QStyleSheetItem::DisplayListItem )
          return FALSE;
        if ( doCompletion() )
          return TRUE;
      }
    }    
  }
  else
    if ( o == completionPopup || o == completionListBox ||
      o == completionListBox->viewport() )
    {
      if ( e->type() == QEvent::KeyPress )
      {
        QKeyEvent *ke = (QKeyEvent*)e;        
        if ( ke->key() == Key_Enter || ke->key() == Key_Return || ke->key() == Key_Tab )
        {
          if ( ke->key() == Key_Tab && completionListBox->count() > 1 &&
            completionListBox->currentItem() < (int)completionListBox->count() - 1 )
          {
            completionListBox->setCurrentItem( completionListBox->currentItem() + 1 );
            return TRUE;
          }
          completeCompletion();
          return TRUE;
        }
        else
          if ( ke->key() == Key_Left || ke->key() == Key_Right ||
            ke->key() == Key_Up || ke->key() == Key_Down ||
            ke->key() == Key_Home || ke->key() == Key_End ||
            ke->key() == Key_Prior || ke->key() == Key_Next )
          {
            return FALSE;
          }
          else
            if ( ke->key() != Key_Shift && ke->key() != Key_Control &&
              ke->key() != Key_Alt )
            {              
              if ( ke->key() == Key_Backspace )             
                searchString.remove( searchString.length() - 1, 1 );              
              else              
                searchString += ke->text();                         
              completionPopup->close();
              curEditor->setFocus();              
              QApplication::sendEvent( curEditor, e );
              return TRUE;
            }
      }
      else
        if ( e->type() == QEvent::MouseButtonDblClick )
        {
          completeCompletion();
          return TRUE;
        }
    }
    return FALSE;
}


void EditorCompletion::completeCompletion()
{ 
  QString s = completionListBox->currentText();
  
  for (unsigned int x = 1; x <= searchString.length(); x++)
    curEditor->doKeyboardAction (QTextEdit::ActionBackspace);  
  
  curEditor->insert(s, TRUE );
  completionPopup->close();
  curEditor->setFocus();
}

void EditorCompletion::showCompletion( const QValueList<CompletionEntry> &lst )
{
  QTextCursor *cursor = curEditor->textCursor();
  QTextStringChar *chr = cursor->parag()->at( cursor->index() );
  int h = cursor->parag()->lineHeightOfChar( cursor->index() );
  int x = cursor->parag()->rect().x() + chr->x;
  int y, dummy;
  cursor->parag()->lineHeightOfChar( cursor->index(), &dummy, &y );
  y += cursor->parag()->rect().y();
  completionListBox->clear();
  for ( QValueList<CompletionEntry>::ConstIterator it = lst.begin(); it != lst.end(); ++it )
    (void)new CompletionItem( completionListBox, (*it).text, (*it).type,
    (*it).postfix, (*it).prefix, (*it).postfix2 );
  completionListBox->sort();
  cList = lst;
  completionPopup->resize( completionListBox->sizeHint() +
    QSize( completionListBox->verticalScrollBar()->width() + 4,
    completionListBox->horizontalScrollBar()->height() + 4 ) );
  completionListBox->setCurrentItem( 0 );
  completionListBox->setFocus();
  if ( curEditor->mapToGlobal( QPoint( 0, y ) ).y() + h + completionPopup->height() < QApplication::desktop()->height() )
    completionPopup->move( curEditor->mapToGlobal( curEditor->
    contentsToViewport( QPoint( x, y + h ) ) ) );
  else
    completionPopup->move( curEditor->mapToGlobal( curEditor->
    contentsToViewport( QPoint( x, y - completionPopup->height() ) ) ) );
  completionPopup->show();
}

