/* Copyright (C) 2002 MySQL AB & Jorge del Conde

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
    
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA 
*/

#include "CSqlEditor.h"
#include "completion.h"
#include "parenmatcher.h"

CSqlEditor::CSqlEditor (QWidget * parent,  int serverid, const char * name, CMySQLConnection *m)
: Editor(parent, name), serverID(serverid)
{
  mysql = m;
  tmpFileName = QString::null;
  setAcceptDrops(true);
  parenMatcher->setEnabled(g_parenthesesMatching);
  hasSaveTitle = false;
  saveTitle = QString::null;

  if (g_syntaxHighlight || serverID != -1)
  {    
    SyntaxHighlighter_SQL *syntax = new SyntaxHighlighter_SQL();
    if (g_syntaxHighlight)
      syntax->loadSqlKeywords(g_KeywordDict);
    if (serverID != -1)
      syntax->loadDatabaseKeywords(serverID);
    document()->setPreProcessor(syntax);    
  }  

  delete_completion = false;

  if (g_completion || serverID > -1)
  {
    completion = new EditorCompletion(this);
    delete_completion = true;  
  }

  if (g_completion)
  {   
    QDictIterator<SQLKeyWord> it( *g_KeywordDict );
    for( ; it.current(); ++it )
    {      
      QString Keyword = it.current()->Keyword;
      if (it.current()->Highlight == SQLKeyWord::highlight_FUNCTIONS)
        Keyword += "()";
      completion->addCompletionEntry(Keyword, 0, false);
    }
  }

  if (serverID != -1)
  {    
    //has field completion
    DBKeyWord *l= g_DBKeywordDict->find(serverid);
    if (l != NULL)
    {
      QDictIterator<QString> it(l->Databases);    
      for( ; it.current(); ++it )
        completion->addCompletionEntry(*(it.current()), 0, false);        

      QDictIterator<QString> it2(l->Tables);
      for( ; it2.current(); ++it2 )
        completion->addCompletionEntry(*(it2.current()), 0, false);        

      QDictIterator<QString> it3(l->Fields);
      for( ; it3.current(); ++it3 )
        completion->addCompletionEntry(*(it3.current()), 0, false);        
    }
  }
}

CSqlEditor::~CSqlEditor()
{
  if (delete_completion)
    delete completion;
}

void CSqlEditor::contentsDragEnterEvent (QDragEnterEvent *event)
{
  event->accept(QTextDrag::canDecode(event));	
}


void CSqlEditor::contentsDropEvent( QDropEvent * event)
{
  if (isReadOnly())
    return;
  QString txt;
  if (QTextDrag::decode(event, txt))
  {
    if (txt.contains('\\'))
      txt = txt.mid(txt.findRev ('\\') + 1, txt.length()); 
    insert(txt, false, true, false);
  }
}

QString CSqlEditor::getSaveContents()
{
  return getSaveTitle() + text();
}

void CSqlEditor::setSaveTitle(const QString &title)
{
  saveTitle = title;
  hasSaveTitle = true;
}

QString CSqlEditor::getSaveTitle()
{
  QString ret = QString::null;
  if (hasSaveTitle)
  {
    ret = "# " + saveTitle + mysql->getLineTerminator(true);
    ret += "# " + tr("Connection") + ": " + mysql->getConnectionName() + mysql->getLineTerminator(true);
    ret += "# " + tr("Host") + ": " + mysql->getHostName() + mysql->getLineTerminator(true);
    ret += "# " + tr("Saved") + ": " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") + mysql->getLineTerminator(true);
    ret += "#" + mysql->getLineTerminator(true);
  }
  return ret;
}

void CSqlEditor::save()
{
  saveToFile(tmpFileName, "sql", tr("SQL Files (*.sql);;All Files(*.*)" ), getSaveContents(), mysql->messagePanel);
}

void CSqlEditor::openFile()
{
  QString fn = QFileDialog::getOpenFileName( QString::null, tr("SQL Files") + " (*.sql);;" + tr("Text Files") + " (*.txt);;" + tr("All Files") + " (*.*)", this);
  if ( !fn.isEmpty() )  
    load(fn);  
}

bool CSqlEditor::LoadSyntaxFile()
{  
  g_KeywordDict->clear();
  QFile f_strm(*g_SyntaxFile);
  if (f_strm.open(IO_ReadOnly))
  {
    QTextStream t_strm(&f_strm);
    SQLKeyWord::Keyword_Type keywordType = (SQLKeyWord::Keyword_Type) 0;
    SQLKeyWord::Highlight_Type highlightType = (SQLKeyWord::Highlight_Type) 0;
    bool hasSection = false;
    while ( !t_strm.atEnd() )
    {
      QString str = t_strm.readLine();
      str = str.stripWhiteSpace();
      QRegExp rx("^\\[(\\d+)\\s(\\d+)\\]$");
      if (rx.search(str) != -1)
      {
        keywordType = (SQLKeyWord::Keyword_Type) rx.cap(1).toUShort();
        highlightType = (SQLKeyWord::Highlight_Type) rx.cap(2).toUShort();
        hasSection = true;        
      }
      else
        if (hasSection && !str.isEmpty())
          if (g_KeywordDict->find(str.lower()) == NULL)
            new SQLKeyWord(g_KeywordDict, str, keywordType, highlightType);
    }
    f_strm.close();
    return true;
  }
  else      
    return false;  
}


