/* Copyright (C) 2002 MySQL AB & Jorge del Conde

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
    
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA 
*/

#include "CMyWindow.h"
#include "Config.h"
#include <qapplication.h>
#include <qregexp.h>

CMyWindow::CMyWindow (QWidget * parent, const char * name, WFlags f, bool destroy, bool hasmessages) 
: QMainWindow(parent, name, f), m_hasMessages(hasmessages), m_destroy(destroy)
{
  windowState = 0;
  m_canSaveMe = false;
  m_isAppWidget = false;
  m_forceDelete = false;
  p_messagePanel = 0;  
  if (m_hasMessages)
  {
    p_messageWindow = new CMessageWindow(QDockWindow::InDock, this);
    Q_CHECK_PTR(p_messageWindow);
    p_messagePanel = new CMessagePanel(tr("Messages"));
    Q_CHECK_PTR(p_messagePanel);
    p_messageWindow->addPanel(p_messagePanel);
  }
  if ((!g_isMDI) && (destroy))
    g_WindowList.append(this);

  if (m_hasMessages)
  {
    moveDockWindow(p_messageWindow, DockBottom);
    p_messageWindow->setFixedExtentHeight(90);
  }
}

int CMyWindow::getWindowState()
{
  if (isMinimized())  
    windowState = WState_Minimized;
  else
    if (isMaximized())
      windowState = WState_Maximized;
    else
      windowState = 0;
  return windowState;
}

void CMyWindow::myResize(int w, int h)
{
  if (!loadWindowSettings())
    resize(w, h);
}

void CMyWindow::closeEvent(QCloseEvent *e)
{
  beforeClose();
  e->accept();
}

void CMyWindow::beforeClose()
{
  if (!m_isAppWidget)
    setCurrentPosition((g_isMDI ? parentWidget()->pos() : myPos()));
  else
    setCurrentPosition(myPos());
  saveWindowSettings();
}

void CMyWindow::setCurrentPosition(const QPoint &p)
{
  getWindowState();
  position = p;
  m_canSaveMe = true;
}
 
bool CMyWindow::loadWindowSettings(bool b)
{  
  CConfig *cfg = new CConfig(g_CONFIG_FILE);
  bool ret = false;
  cfg->prepare();
  QString s = cfg->readStringEntry("Window " + QString(name()), QString::null);  
  if (!s.isNull())  
  {
    s = charReplace(s, '|', "\n");    
    QTextStream ts( &s, IO_ReadOnly);
    ts >> *this;    
  }
  s = cfg->readStringEntry(name(), QString::null);
  if (!s.isNull())  
  {
    QRegExp rx("^(\\d+),(\\d+|-\\d+),(\\d+|-\\d+),(\\d+),(\\d+)$");
    if (rx.search(s) != -1)
    {
      windowState = rx.cap(1).toInt();
      if (g_isMDI || (windowState == 0))
      {
        if (b)
          setGeometry(rx.cap(2).toInt(), rx.cap(3).toInt(), rx.cap(4).toInt(), rx.cap(5).toInt());
        else
          resize(rx.cap(4).toInt(), rx.cap(5).toInt());
      }      
      ret = true;
    }    
  }
  delete cfg;
  return ret;
}

void CMyWindow::saveWindowSettings()
{
  if (m_canSaveMe)
  {
    CConfig *cfg = new CConfig(g_CONFIG_FILE);
    cfg->prepare();
    QString s;
    if (position.y() < 0)
      position.setY(0);
    s.sprintf("%d,%d,%d,%d,%d", windowState, position.x(), position.y(), width(), height());
    cfg->writeEntry(name(), s);
    s = QString::null;
    QTextStream ts( &s, IO_WriteOnly );
    ts << *(CMyWindow *)this;
    s = charReplace(s, '\n', "|");
    cfg->writeEntry("Window " + QString(name()), s);  
    cfg->flush();
    delete cfg;
  }
}

bool CMyWindow::findDockWindow(Dock *d, CMyWindow *wnd, QDockWindow *dockWnd)
{
  if (wnd->leftDock()->hasDockWindow(dockWnd))
    *d = Qt::DockLeft;
  else
    if (wnd->rightDock()->hasDockWindow(dockWnd))
      *d = Qt::DockRight;
    else
      if (wnd->topDock()->hasDockWindow(dockWnd))
        *d = Qt::DockTop;
      else
        if (wnd->bottomDock()->hasDockWindow(dockWnd))
          *d = Qt::DockBottom;
        else
          *d = (Qt::Dock)0;
  return !dockWnd->place();
}

CMyWindow::~CMyWindow()
{
  if (m_hasMessages)
    if (p_messageWindow != 0)
      p_messageWindow->close();

  if ((!m_forceDelete) && (m_destroy))
    g_WindowList.remove(this);
}

void CMyWindow::forceDelete(bool d)
{
  m_forceDelete = d;
}

void CMyWindow::autoPlace()
{
  if (g_isMDI)
    return;
  int overlap, minOverlap = 0;
  int possible;
  
  QRect r1(0, 0, 0, 0);
  QRect r2(0, 0, 0, 0);
  QRect maxRect = qApp->desktop()->rect();
  int x = maxRect.left(), y = maxRect.top();
  QPoint wpos(maxRect.left(), maxRect.top());
  
  bool firstPass = true;
  
  do {
    if ( y + height() > maxRect.bottom() )
      overlap = -1;
    else
      if( x + width() > maxRect.right() )
        overlap = -2;
      else
      {
        overlap = 0;
        
        r1.setRect(x, y, width(), height());
        
        CMyWindow *l;
        for (l = g_WindowList.first(); l; l = g_WindowList.next())
        {
          if (!l->isHidden() && l != this )
          {
            r2.setRect(l->x(), l->y(), l->width(), l->height());
            
            if (r2.intersects(r1))
            {
              r2.setCoords(QMAX(r1.left(), r2.left()),
                QMAX(r1.top(), r2.top()),
                QMIN(r1.right(), r2.right()),
                QMIN(r1.bottom(), r2.bottom())
                );
              
              overlap += (r2.right() - r2.left()) * (r2.bottom() - r2.top());
            }
          }
        }
      }
      
      if (overlap == 0)
      {
        wpos = QPoint(x, y);
        break;
      }
      
      if (firstPass)
      {
        firstPass = false;
        minOverlap = overlap;
      }
      else
        if ( overlap >= 0 && overlap < minOverlap)
        {
          minOverlap = overlap;
          wpos = QPoint(x, y);
        }
        
        if ( overlap > 0 )
        {
          possible = maxRect.right();
          if ( possible - width() > x)
            possible -= width();
          
          CMyWindow *l;
          for (l = g_WindowList.first(); l; l = g_WindowList.next())
          {
            if (!l->isHidden() && l != this )
            {
              r2.setRect(l->x(), l->y(), l->width(), l->height());
              
              if( ( y < r2.bottom() ) && ( r2.top() < height() + y ) )
              {
                if( r2.right() > x )
                  possible = possible < r2.right() ? possible : r2.right();
                
                if( r2.left() - width() > x )
                  possible = possible < r2.left() - width() ? possible : r2.left() - width();
              }
            }
          }
          
          x = possible;
        }
        else
          if ( overlap == -2 )
          {
            x = maxRect.left();
            possible = maxRect.bottom();
            
            if ( possible - height() > y )
              possible -= height();
            
            CMyWindow *l;
            for (l = g_WindowList.first(); l; l = g_WindowList.next())
            {
              if (l != this)
              {
                r2.setRect(l->x(), l->y(), l->width(), l->height());
                
                if( r2.bottom() > y)
                  possible = possible < r2.bottom() ? possible : r2.bottom();
                
                if( r2.top() - height() > y )
                  possible = possible < r2.top() - height() ? possible : r2.top() - height();
              }
            }
            
            y = possible;
          }
  }
  while( overlap != 0 && overlap != -1 );  
  move(wpos);
}

CSmallMyWindow::CSmallMyWindow (QWidget * parent, const char * name, WFlags f, bool destroy, int flags) 
: CMyWindow(parent, name, f, destroy, flags)
{    
}

CSmallMyWindow::~CSmallMyWindow()
{
}

QSize CSmallMyWindow::sizeHint() const
{
  return QSize (0,0);
}

static void saveDockArea( QTextStream &ts, QDockArea *a )
{
  QPtrList<QDockWindow> l = a->dockWindowList();
  for ( QDockWindow *dw = l.first(); dw; dw = l.next() )
    if (!dw->inherits("CToolBar") && !dw->inherits("QToolBar"))
    {
      ts << QString( dw->caption() );
      ts << ",";
    }
  ts << endl;
  ts << *a;
}

QTextStream &operator<<( QTextStream &ts, const CMyWindow &mainWindow )
{  
  QPtrList<QDockWindow> l = mainWindow.dockWindows( Qt::DockMinimized );
  QDockWindow *dw = 0;
  for ( dw = l.first(); dw; dw = l.next() )
    if (!dw->inherits("CToolBar") && !dw->inherits("QToolBar"))
    {    
      ts << dw->caption();
      ts << ",";
    }
  ts << endl;  
  l = mainWindow.dockWindows( Qt::DockTornOff );
  for ( dw = l.first(); dw; dw = l.next() )
    if (!dw->inherits("CToolBar") && !dw->inherits("QToolBar"))
    {
      ts << dw->caption();
      ts << ",";
    }
  ts << endl;
  for ( dw = l.first(); dw; dw = l.next() )
    if (!dw->inherits("CToolBar") && !dw->inherits("QToolBar"))
    {
      ts << "[" << dw->caption() << ","
         << (int)dw->geometry().x() << ","
         << (int)dw->geometry().y() << ","
         << (int)dw->geometry().width() << ","
         << (int)dw->geometry().height() << ","
         << (int)dw->isVisible() << "]";
    }
  ts << endl;
  saveDockArea(ts, mainWindow.topDock());
  saveDockArea(ts, mainWindow.bottomDock());
  saveDockArea(ts, mainWindow.rightDock());
  saveDockArea(ts, mainWindow.leftDock());
  return ts;
}

static void initDockAreas(QTextStream &ts, QDockArea &dockArea)
{
  QString s = ts.readLine();
  
  QString name, offset, newLine, width, height, visible;
  
  enum State { Pre, Name, Offset, NewLine, Width, Height, Visible, Post };
  int state = Pre;
  QChar c;
  QPtrList<QDockWindow> l = dockArea.dockWindowList();
  
  for ( int i = 0; i < (int)s.length(); ++i )
  {
    c = s[ i ];
    if ( state == Pre && c == '[' )
    {
      state++;
      continue;
    }
    if ( c == ',' && ( state == Name || state == Offset || state == NewLine || state == Width || state == Height ) )
    {
      state++;
      continue;
    }
    if ( state == Visible && c == ']' )
    {
      for ( QDockWindow *dw = l.first(); dw; dw = l.next() )
      {
        if ( QString( dw->caption() ) == name )
        {
          if (!dw->inherits("CToolBar") && !dw->inherits("QToolBar"))
          {
            dw->setNewLine( (bool)newLine.toInt() );
            dw->setOffset( offset.toInt() );
          }
          dw->setFixedExtentWidth( width.toInt() );
          dw->setFixedExtentHeight( height.toInt() );
          if ( !(bool)visible.toInt() )
            dw->hide();
          else
            dw->show();
          break;
        }
      }
      
      name = offset = newLine = width = height = visible = "";
      
      state = Pre;
      continue;
    }
    if ( state == Name )
      name += c;
    else
      if ( state == Offset )
        offset += c;
      else
        if ( state == NewLine )
          newLine += c;
        else
          if ( state == Width )
            width += c;
          else
            if ( state == Height )
              height += c;
            else
              if ( state == Visible )
                visible += c;
  }
  
  dockArea.QWidget::layout()->invalidate();
  dockArea.QWidget::layout()->activate();
}

static void loadDockArea( const QStringList &names, QDockArea *a, Qt::Dock d, QPtrList<QDockWindow> &l, QMainWindow *mw, QTextStream &ts )
{
  for ( QStringList::ConstIterator it = names.begin(); it != names.end(); ++it )
  {
    for ( QDockWindow *dw = l.first(); dw; dw = l.next() )
    {
      if (!dw->inherits("CToolBar") && !dw->inherits("QToolBar"))
        if ( dw->caption() == *it )
        {
          mw->addDockWindow( dw, d );
          break;
        }
    }
  }
  if ( a )
  {
    initDockAreas(ts, *a);
  }
  else
    if ( d == Qt::DockTornOff )
    {
      QString s = ts.readLine();
      enum State { Pre, Name, X, Y, Width, Height, Visible, Post };
      int state = Pre;
      QString name, x, y, w, h, visible;
      QChar c;
      for ( int i = 0; i < (int)s.length(); ++i )
      {
        c = s[ i ];
        if ( state == Pre && c == '[' )
        {
          state++;
          continue;
        }
        if ( c == ',' && ( state == Name || state == X || state == Y || state == Width || state == Height ) )
        {
          state++;
          continue;
        }
        if ( state == Visible && c == ']' )
        {
          for ( QDockWindow *dw = l.first(); dw; dw = l.next() )
          {
            if ( QString( dw->caption() ) == name )
            {
              if (!dw->inherits("CToolBar") && !dw->inherits("QToolBar"))              
                dw->setGeometry( x.toInt(), y.toInt(), w.toInt(), h.toInt() );
              else
                dw->setGeometry( x.toInt(), y.toInt(), dw->width(), dw->height() );              
              if ( !(bool)visible.toInt() )
              {
                dw->hide();
              }
              else
              {
                if ( mw->appropriate( dw ) )
                  dw->show();
                else
                  dw->hide();
              }              
              break;
            }
          }
          
          name = x = y = w = h = visible = "";
          
          state = Pre;
          continue;
        }
        if ( state == Name )
          name += c;
        else
          if ( state == X )
            x += c;
          else
            if ( state == Y )
              y += c;
            else
              if ( state == Width )
                w += c;
              else
                if ( state == Height )
                  h += c;
                else
                  if ( state == Visible )
                    visible += c;
      }
    }
}

QTextStream &operator>>( QTextStream &ts, CMyWindow &mainWindow )
{
  QPtrList<QDockWindow> l = mainWindow.dockWindows();
  
  QString s = ts.readLine();
  QStringList names = QStringList::split( ',', s );
  loadDockArea( names, 0, Qt::DockMinimized, l, &mainWindow, ts );
  
  s = ts.readLine();
  names = QStringList::split( ',', s );
  loadDockArea( names, 0, Qt::DockTornOff, l, &mainWindow, ts );
  
  int i = 0;
  QDockArea *areas[] = { mainWindow.topDock(), mainWindow.bottomDock(), mainWindow.rightDock(), mainWindow.leftDock() };
  for ( int d = (int)Qt::DockTop; d != (int)Qt::DockMinimized; ++d, ++i )
  {
    s = ts.readLine();
    names = QStringList::split( ',', s );
    loadDockArea( names, areas[ i ], (Qt::Dock)d, l, &mainWindow, ts );
  }
  return ts;
}

