/* Copyright (C) 2002 MySQL AB & Jorge del Conde

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
    
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA 
*/

#include "CMySQLConnection.h"
#include <qinputdialog.h>
#include <qmessagebox.h>
#include <qregexp.h>
#include "CMySQLQuery.h"

CMySQLConnection::CMySQLConnection(const QString & connectionName, CMessagePanel * msgpanel, const QString &path)
{	
  m_ConnectionName = connectionName;
  m_isConnected = false;
  m_isSocket = false;
  refresh(path);
  setName("CMySQLConnection");
  messagePanel = msgpanel; 
  m_ServerVersion.major = 0;
  m_ServerVersion.minor = 0;
}

void CMySQLConnection::PrintError(int type, const QString & msg, bool p)
{ 
  QString errmsg = (p ? "[" + getConnectionName() + "] " : QString::null) + ((msg.isEmpty()) ? getError() : msg);  
  switch (type)
  {
    case CRITICAL:
    {
      if (messagePanel != 0)
        messagePanel->Critical(errmsg);
      else
        QMessageBox::critical(0,"Critical",errmsg);
      break;
    }
    case WARNING:
    {
      if (messagePanel != 0)
        messagePanel->Warning(errmsg);
      else
        QMessageBox::warning(0,"Warning",errmsg);
      break;
    }
    case INFORMATION:
    {
      if (messagePanel != 0)
        messagePanel->Information(errmsg);
      else
        QMessageBox::information(0,"Information",errmsg);
      break;
    }
  }  
}

QString CMySQLConnection::Quote(const QString &f)
{
  return ((getMySQLServerVersion().major <= 3) && (getMySQLServerVersion().minor <= 22)) ? f : "`" + f + "`";    
}

void CMySQLConnection::initServerVersion()
{
  if (!m_isConnected)
  {
    m_ServerVersion.major = 0;
    m_ServerVersion.minor = 0;
    m_ServerVersion.rel = "0";
  }
  else
  {
    QRegExp rx("^(\\d+)\\.(\\d+)\\.(.+)$");
    QString str = getServerVersion();
    if (!str.isEmpty() && (rx.search(str) != -1 ))
    {
      m_ServerVersion.major = rx.cap(1).toInt();
      m_ServerVersion.minor = rx.cap(2).toInt();
      m_ServerVersion.rel = rx.cap(3).toInt();
    }
    else
    {
      m_ServerVersion.major = 0;
      m_ServerVersion.minor = 0;
      m_ServerVersion.rel = "0";
    }
  }
}

QString CMySQLConnection::getError()
{
  QString err = QObject::tr("ERROR") + " ";
  if (m_isConnected)
    err += QString::number(getErrno()) + ": " + QString::fromLatin1(mysql_error(mysql));
  else
    err += "10061: Connection to MySQL Server failed";
  return err;
}

CMySQLConnection::~CMySQLConnection()
{  
  disconnect();
}

bool CMySQLConnection::shutdown()
{
  if ((m_isConnected) && (mysql_shutdown(mysql) == 0))
  {    
    disconnect();
    return true;
  }
  PrintError(CRITICAL);
  return false;
}

bool CMySQLConnection::flush(const QString & f)
{ 
  QString sql = "FLUSH ";
  sql += f;
  CMySQLQuery qry(this);
  bool printedMessage = false;
  if ((m_isConnected) && (qry.exec_static(sql, false, &printedMessage)))
    return true;
  else
    if (!printedMessage)
      PrintError(CRITICAL);
  return false;
}

bool CMySQLConnection::ping()
{
  if ((m_isConnected) && (mysql_ping(mysql) == 0))
    return true;
  else
    PrintError(CRITICAL);
  return false;
}

bool CMySQLConnection::killProcess(unsigned long pid)
{
  if ((m_isConnected) && (mysql_kill(mysql, pid) == 0))
    return true;
  else
    PrintError(CRITICAL);
  return false;
}

bool CMySQLConnection::setConnected(bool c, const QString & dbname, bool del)
{		
  if (c)
    return connect(dbname);
  else
  {
    disconnect();
    if (del)
      g_PasswordDict->remove(m_ConnectionName);
    return false;
  }
}

void CMySQLConnection::signalConnect(const char *sig, QObject *receiver, const char *member)
{
  /*
    This member is used to connect Signals.  It has nothing to do with mysql.
  */
  QObject::connect(this, sig, receiver, member);
}

void CMySQLConnection::update()
{
  refresh();  
}

void CMySQLConnection::refresh(const QString &path)
{  
  CConfig *Settings = new CConfig(m_ConnectionName, path);
  Q_CHECK_PTR(Settings);
  Settings->prepare();
  m_UserName = Settings->readStringEntry("User");
  m_Password = Settings->readStringEntry("Password");  
  m_Compress = strtobool(Settings->readStringEntry("Compress", "false"));
  m_PromptPassword = strtobool(Settings->readStringEntry("Prompt Password", "false"));  
  m_BlockingQueries = strtobool(Settings->readStringEntry("Blocking Queries"));
  m_OneConnection = strtobool(Settings->readStringEntry("One Connection"));
  m_SocketFile = Settings->readStringEntry("Socket");
  m_HostName = Settings->readStringEntry("Host");
  m_Port = Settings->readNumberEntry("Port");
  m_Timeout = Settings->readNumberEntry("Port");
  m_Created = Settings->readStringEntry("Created");
  m_Modified = Settings->readStringEntry("Modified");
  m_Completion = strtobool(Settings->readStringEntry("Completion", "false"));
  m_fieldSeparated = Settings->readStringEntry("Field Separate", ",");
  m_lineTerminated = Settings->readStringEntry("Line Terminate", "\\r\\n");
  m_fieldEnclosed = Settings->readStringEntry("Field Enclosed", "'");
  m_replaceEmpty = Settings->readStringEntry("Replace Empty", QString::null);
  m_retrieveShowTableStatus = strtobool(Settings->readStringEntry("Retrieve Method", "true"));
  initServerVersion();
  delete Settings;
  emit statuschanged();
  emit refreshed();  
}

void CMySQLConnection::disconnect()
{  
  if (m_isConnected)
    mysql_close(mysql);
  m_isConnected = false;
  initServerVersion();
  emit statuschanged();
  emit disconnected();  
}

bool CMySQLConnection::setDatabaseName(const QString & dbname)
{	
  if (dbname.isNull())
    return false;
  if (m_isConnected)
    if (mysql_select_db(mysql, dbname.latin1()) == 0)
    {
      m_DatabaseName = dbname;
      return true;
    }
    return false;
}

QString CMySQLConnection::doEscape(const QString &str)
{  
  QString tmp = str;
  tmp = tmp.replace(QRegExp("\\\\n"), "\n");
  tmp = tmp.replace(QRegExp("\\\\r"), "\r");  
  return tmp.replace(QRegExp("\\\\t"), "\t");  
}

QString CMySQLConnection::getFieldSeparator(bool escape)
{
  if (!escape)
    return m_fieldSeparated;
  else
    return doEscape(m_fieldSeparated);
}

QString CMySQLConnection::getReplaceEmpty(bool escape)
{
  if (!escape)
    return m_replaceEmpty;
  else
    return doEscape(m_replaceEmpty);
}

QString CMySQLConnection::getFieldEncloser(bool escape)
{
  if (!escape)
    return m_fieldEnclosed;
  else
    return doEscape(m_fieldEnclosed);
}

QString CMySQLConnection::getLineTerminator(bool escape)
{
  if (!escape)
    return m_lineTerminated;
  else
    return doEscape(m_lineTerminated);
}

QString CMySQLConnection::getServerVersion()
{  
  return ((m_isConnected) ? QString(mysql_get_server_info(mysql)) : QString::null);
}

QString CMySQLConnection::getStatus()
{  
  return ((m_isConnected) ? QString(mysql_stat(mysql)) : QString::null);
}

QString CMySQLConnection::getHostInfo()
{  
  return ((m_isConnected) ? QString(mysql_get_host_info(mysql)) : QString::null);
}

QString CMySQLConnection::getProtoInfo()
{  
  return ((m_isConnected) ? QString::number(mysql_get_proto_info(mysql)) : QString::null);
}

QString CMySQLConnection::getUptime()
{
  if (!m_isConnected)
    return QString::null;
  if ((m_ServerVersion.major <= 3) && (m_ServerVersion.minor <= 22))
    return QString::null;

  QString ret = QString::null;
  CMySQLQuery *qry = new CMySQLQuery(this);  
  if (qry->exec("SHOW STATUS LIKE 'Uptime'", false))
    if (qry->next(false, true))
      ret = nice_time(qry->Row(1).toULong());
  delete qry;
  return ret;
}

bool CMySQLConnection::connect(const QString & dbname, bool p)
{	     	
  disconnect();  
  m_DatabaseName = dbname;
  unsigned int flags = 0;
  if (m_Compress)
	  flags |= CLIENT_COMPRESS;

  if (m_PromptPassword)
  {            
    QString *tmpPass = g_PasswordDict->find(m_ConnectionName);
	  if (g_PasswordDict->isEmpty() || (tmpPass == NULL))
    {
      bool ok = false;
      m_Password = CTextDialog::getText( tr( "Password" ),
        tr("Please enter the Password for") + ": " + m_UserName + "@" + m_HostName,
        QLineEdit::Password, QString::null, &ok, 0);
      if (!ok)  //user didn't click Accept in the dialog
      {
        initServerVersion();
        return false;
      }
      
      if (g_PasswordDict->size() <= g_PasswordDict->count() + 1)
        g_PasswordDict->resize(g_PasswordDict->count() + 1);
      g_PasswordDict->insert(m_ConnectionName, &m_Password); //insert the password g_PasswordDict      
    }
    else    
      m_Password = *tmpPass;  //password is already in g_PasswordDict
  }

#ifndef WIN32
  if (m_SocketFile.length() != 0)  //Set options to use a Unix Socket file instead of TCP/IP
  {
	  m_Port = 0;
    m_HostName = QString::null;
    m_isSocket = true;
  }
  else
#endif
  {
    m_SocketFile = QString::null;
    m_isSocket = false;
  }
  mysql = mysql_init(0);
  if (m_Compress)
    mysql_options(mysql, MYSQL_OPT_COMPRESS,0);
  unsigned int t = 5;
  mysql_options(mysql, MYSQL_OPT_CONNECT_TIMEOUT, (char*) &t);

  if(!(mysql_real_connect(mysql, m_HostName.latin1(), m_UserName.latin1(), m_Password.latin1(), dbname.latin1(), m_Port, m_SocketFile.latin1(), flags)))
  {
    PrintError(CRITICAL, getError(), p); 
    if (m_PromptPassword && !g_PasswordDict->isEmpty())  //Need to remove the password from g_PasswordDict
      g_PasswordDict->remove(m_ConnectionName);
    m_isConnected = false;		
  }
  else
  {
    m_isConnected = true;
    initServerVersion();
    emit statuschanged();
    emit connected();    
  }  
  return m_isConnected;	
}

void CMySQLConnection::setConnectionName(const QString & connectionName)
{  
  g_PasswordDict->remove(m_ConnectionName); 
  m_ConnectionName = connectionName;
}

