/* Copyright (C) 2002 MySQL AB & Jorge del Conde

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
    
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA 
*/

#include "CMessageWindow.h"
#include "Config.h"
#include "CHistoryView.h"
#include <qpopupmenu.h>
#include <qclipboard.h>
#include <qtooltip.h>
#include <qapplication.h>

CPanelListBox::CPanelListBox(QWidget * parent, const char * name, WFlags f)
: QListBox(parent, name, f), maxdisplay(-1), enter("\r\n")
{
  msgPanel = 0;
  realtext.setAutoDelete(true);  
  connect(this, SIGNAL(rightButtonPressed(QListBoxItem *, const QPoint &)), this, SLOT(displayMenu(QListBoxItem *, const QPoint &)));  
}

void CPanelListBox::setCarriageReturn(const QString &s)
{
  enter = s;
}

QString CPanelListBox::insertRealText(const QString &s)
{
  if (realtext.size() >= (uint)(numRows() + 1))
    realtext.resize(realtext.size() + 1);
  realtext.insert(numRows() , new QString(s));
  QString tmp(s);
  if ((maxdisplay > -1) && (tmp.length() >= (uint) maxdisplay))
  {
    tmp.truncate(maxdisplay - 3);
    tmp += "...";
  }
  return tmp.simplifyWhiteSpace();  
}

void CPanelListBox::myInsertItem (const QPixmap & pixmap, const QString & text, int index)
{  
  insertItem(pixmap, insertRealText(text), index);
}

void CPanelListBox::myInsertItem ( const QString & text, int index)
{  
  insertItem(insertRealText(text), index);
}

QString CPanelListBox::realText(int idx)
{
  if (realtext.find(idx) != NULL)
    return *realtext.find(idx);
  else
    return QString::null;
}

void CPanelListBox::Clear()
{
  clear();
  realtext.clear();
}

void CPanelListBox::setMessagePanel(CMessagePanel *m)
{
  msgPanel = m;
}

void CPanelListBox::save()
{
  QString txt;
  for (int i=0; i < numRows(); i++)
    txt += realText(i) + enter;    
  saveToFile(tmpFile, "txt", tr("Text Files (*.txt);;All Files(*.*)" ), txt, (msgPanel == 0) ? (CMessagePanel *)parentWidget() : msgPanel);
}

void CPanelListBox::copy(QListBoxItem *item)
{
#ifndef QT_NO_CLIPBOARD
  if (item != 0)
    QApplication::clipboard()->setText(realText(index(item)));
#endif  
}

CMessageListBox::CMessageListBox(QWidget * parent, const char * name, WFlags f)
: CPanelListBox(parent, name, f)
{  
}

void CMessageListBox::displayMenu(QListBoxItem *item, const QPoint &pos)
{
  QPopupMenu *p_itemMenu = new QPopupMenu();
  Q_CHECK_PTR(p_itemMenu);  
  p_itemMenu->insertItem(getPixmapIcon("copyIcon"), tr("Copy"), 1);
#ifdef QT_NO_CLIPBOARD
  p_itemMenu->setItemEnabled (1, false);
#else
  p_itemMenu->setItemEnabled (1, (item != 0));
#endif 
  p_itemMenu->insertSeparator();
  p_itemMenu->insertItem(getPixmapIcon("saveIcon"), tr("Save"), 2);
  p_itemMenu->insertSeparator();
  p_itemMenu->insertItem(tr("Clear"), 3);
  if (numRows() == 0)
  {    
    p_itemMenu->setItemEnabled (2, false);
    p_itemMenu->setItemEnabled (3, false);
  } 
  int res = p_itemMenu->exec(pos);
  delete p_itemMenu;
  switch (res) {
    case 1: copy(item);
      break;
    case 2: save();
      break;
    case 3: Clear();
      break;
  }    
}

CHistoryListBox::CHistoryListBox(QWidget * parent, const char * name, WFlags f)
: CPanelListBox(parent, name, f)
{
  itemsEnabled = true;
  connect(this, SIGNAL(doubleClicked(QListBoxItem *)), this, SLOT(emitDblClick(QListBoxItem *)));
}

void CHistoryListBox::emitDblClick(QListBoxItem * item)
{
  emit dblClick(realText(index(item)));
}

void CHistoryListBox::displayMenu(QListBoxItem *item, const QPoint &pos)
{
  QPopupMenu *p_itemMenu = new QPopupMenu();
  Q_CHECK_PTR(p_itemMenu);
  p_itemMenu->insertItem(getPixmapIcon("executeQueryIcon"), tr("Execute Query"), 1);
  p_itemMenu->insertSeparator();
  p_itemMenu->insertItem(getPixmapIcon("copyIcon"), tr("Copy"), 2);  

#ifdef QT_NO_CLIPBOARD
  p_itemMenu->setItemEnabled (2, false);
#else
  p_itemMenu->setItemEnabled (2, (item != 0));
#endif  

  p_itemMenu->insertSeparator();
  p_itemMenu->insertItem(getPixmapIcon("saveIcon"), tr("Save"), 3);
  p_itemMenu->insertSeparator();
  p_itemMenu->insertItem(getPixmapIcon("refreshIcon"), tr("Refresh"), 4);
  
  if (numRows() == 0)  
    p_itemMenu->setItemEnabled (3, false);  

  p_itemMenu->setItemEnabled(1, itemsEnabled);
  p_itemMenu->setItemEnabled(4, itemsEnabled);
  
  int res = p_itemMenu->exec(pos);
  delete p_itemMenu;
  switch (res) {
    case 1: emit executequery();
      break;
    case 2: copy(item);
      break;
    case 3: save();
      break;
    case 4: emit realoadhistorylist();
      break;
  }    
}

CSqlDebugListBox::CSqlDebugListBox(QWidget * parent, const char * name, WFlags f)
: CPanelListBox(parent, name, f)
{
}

void CSqlDebugListBox::copy(QListBoxItem *item)
{
#ifndef QT_NO_CLIPBOARD
  QString tmp = realText(index(item));  
  if (!tmp.isEmpty())
  {
    int p = tmp.find("] ");
    if (p > -1)
    {
      p += 2;
      QApplication::clipboard()->setText( tmp.mid(p, tmp.length() - p) );
    }
  }
#endif  
}

void CSqlDebugListBox::displayMenu(QListBoxItem *item, const QPoint &pos)
{
  QPopupMenu *p_itemMenu = new QPopupMenu();
  Q_CHECK_PTR(p_itemMenu);
  p_itemMenu->insertItem(getPixmapIcon("copyIcon"), tr("Copy"), 1);

#ifdef QT_NO_CLIPBOARD
  p_itemMenu->setItemEnabled (1, false);
#else
  p_itemMenu->setItemEnabled (1, (item != 0));
#endif

  p_itemMenu->insertItem(getPixmapIcon("saveIcon"), tr("Save"), 2);
  p_itemMenu->insertSeparator();
  p_itemMenu->insertItem(tr("Clear"), 3);  

  if (numRows() == 0)
  {
    p_itemMenu->setItemEnabled (2, false);
    p_itemMenu->setItemEnabled (3, false);
  }  

  int res = p_itemMenu->exec(pos);
  delete p_itemMenu;
  switch (res) {
    case 1: copy(item);
      break;
    case 2: save();
      break;
    case 3: Clear();
      break;
  }    
}

CMyPanel::CMyPanel(const QString &caption)
: QWidget(0, caption.latin1())
{
  if (!caption)
    setName("CMyPanel");  
  else
    setCaption(caption);  
}

void CMyPanel::setPanelWidget(CPanelListBox *p)
{
  panel = p;
  Layout = new QVBoxLayout(this, 1, 1, "Layout");
  Q_CHECK_PTR(Layout);  
  Layout->addWidget(panel);
}

void CMyPanel::clear()
{
  panel->clear();
}

void CMyPanel::setMessagePanel(CMessagePanel *m)
{
  panel->setMessagePanel(m);
}

void CMyPanel::setCarriageReturn(const QString &s)
{
  panel->setCarriageReturn(s);
}

void CMyPanel::setMaxDisplaySize(int len)
{
  panel->setMaxDisplaySize(len);
}

void CMyPanel::showMessage(const QString & m)
{
  panel->myInsertItem(m);
  panel->setBottomItem (panel->count() - 1);
}

void CMyPanel::showMessage(const QPixmap & w, const QString & m)
{
  panel->myInsertItem(w, m);
  panel->setBottomItem (panel->count() - 1);
}

CHistoryPanel::CHistoryPanel(const QString &caption)
: CMyPanel(caption.latin1())
{
  if (!caption)
  {
    setCaption(tr("History"));
    setName("CHistoryPanel");
  }
  else
    setCaption(caption);  
  Panel = new CHistoryListBox(this, "HistoryListBox");
  Q_CHECK_PTR(Panel);
  setPanelWidget(Panel);
  connect (Panel, SIGNAL(executequery()), this, SLOT(emitExecuteQuery()));
  connect (Panel, SIGNAL(dblClick(const QString &)), this, SLOT(emitDoubleClicked(const QString &)));
  connect (Panel, SIGNAL(realoadhistorylist()), this, SLOT(emitRealoadHistoryList()));
  connect (Panel, SIGNAL(onItem(QListBoxItem *)), this, SLOT(setCurrentHint(QListBoxItem * )));
  connect (Panel, SIGNAL(mouseButtonClicked(int, QListBoxItem *, const QPoint &)), this, SLOT(mouseClicked(int, QListBoxItem *, const QPoint &)));
  connect (Panel, SIGNAL(doubleClicked(QListBoxItem *)), this, SLOT(DoubleClicked(QListBoxItem *)));
}

void CHistoryPanel::emitExecuteQuery()
{  
  emit executeQuery(Panel->realText(Panel->currentItem()));
}

void CHistoryPanel::emitDoubleClicked(const QString &t)
{  
  if (!t.isEmpty())
    emit doubleClicked(t);
}

void CHistoryPanel::emitRealoadHistoryList()
{
  emit reloadHistoryList();
}

void CHistoryPanel::disableMenuItems(bool e)
{
  Panel->enableMenuItems(!e);
}

void CHistoryPanel::History(const QString & m, bool append)
{  
  if (append)
    g_HistoryList->append(m);
  showMessage(getPixmapIcon("historyScriptIcon"),m);  
}

void CHistoryPanel::DoubleClicked(QListBoxItem * i)
{
  setCurrentHint(i);
  QString hint = Panel->realText(Panel->index(i));
  if (!hint.isEmpty())
    emit itemSelected(hint);
}

void CHistoryPanel::mouseClicked(int, QListBoxItem * i, const QPoint &)
{
  setCurrentHint(i); 
}

QString CHistoryPanel::getSelectedHintText()
{  
  return Panel->realText(Panel->currentItem());
}

void CHistoryPanel::setCurrentHint(QListBoxItem * i)
{
  if (i == 0)
    return;
  QToolTip::remove(this);
  QString tmp = Panel->realText(Panel->index(i));
  if (!tmp.isEmpty())  
    QToolTip::add(this, tmp);  
}

CSqlDebugPanel::CSqlDebugPanel(const QString &caption)
: CMyPanel(caption.latin1())
{
  if (!caption)
  {
    setCaption(tr("SQL Debug"));
    setName("CSqlDebugPanel");
  }
  else
    setCaption(caption);  
  Panel = new CSqlDebugListBox(this, "SqlDebugListBox");
  Q_CHECK_PTR(Panel);
  setPanelWidget(Panel);  
  connect (Panel, SIGNAL(onItem(QListBoxItem *)), this, SLOT(setCurrentHint(QListBoxItem * )));
  connect (Panel, SIGNAL(mouseButtonClicked(int, QListBoxItem *, const QPoint &)), this, SLOT(mouseClicked(int, QListBoxItem *, const QPoint &)));  
}

void CSqlDebugPanel::SqlDebug(const QString & m)
{  
  showMessage(getPixmapIcon("historyScriptIcon"),m);  
}

void CSqlDebugPanel::mouseClicked(int, QListBoxItem * i, const QPoint &)
{
  setCurrentHint(i); 
}

QString CSqlDebugPanel::getSelectedHintText()
{  
  return Panel->realText(Panel->currentItem());
}

void CSqlDebugPanel::setCurrentHint(QListBoxItem * i)
{
  QToolTip::remove(this);
  QString tmp = Panel->realText(Panel->index(i));
  if (!tmp.isEmpty())  
    QToolTip::add(this, tmp);  
}

CMessagePanel::CMessagePanel(const QString &caption)
: CMyPanel(caption.latin1())
{
  if (!caption)
  {
    setCaption(tr("Messages"));
    setName("CMessagePanel");
  }
  else
    setCaption(caption);
  Panel = new CMessageListBox(this, "MessageListBox");
  Q_CHECK_PTR(Panel);
  setPanelWidget(Panel);  
}

void CMessagePanel::Warning(const QString & m)
{  
  showMessage(getPixmapIcon("warningIcon"),m);
  playSound(g_warningSoundFile);
}

void CMessagePanel::Critical(const QString & m)
{  
  showMessage(getPixmapIcon("criticalIcon"),m);
  playSound(g_errorSoundFile);
}

void CMessagePanel::Information(const QString & m)
{  
  showMessage(getPixmapIcon("informationIcon"),m);
  playSound(g_informationSoundFile);
}

CMessageWindow::CMessageWindow(Place p, QWidget * parent, const char * name, WFlags f)
:QDockWindow (p, parent, name, f)
{  
  if ( !name )
    setName("CMessageWindow");
  cr = QString::null;
  setCaption(tr("Message Panel"));
  Tabs = new QTabWidget(this, "Tabs" );
  Q_CHECK_PTR(Tabs);
  Tabs->setTabPosition( QTabWidget::Bottom );
  Tabs->setTabShape( QTabWidget::Triangular );  
  setWidget (Tabs);
  setMovingEnabled (true);
  setResizeEnabled (true);
  setHorizontallyStretchable (true);
  setVerticallyStretchable (true);
  setCloseMode(CMessageWindow::Always);  
}

void CMessageWindow::setCarriageReturn(const QString &s)
{
  cr = s;
  for (int i = 0; i < Tabs->count(); i++)
    ((CMyPanel *)Tabs->page(i))->setCarriageReturn(s);
}

void CMessageWindow::addPanel(CMyPanel *panel)
{
  if (panel != 0)
  {
    panel->reparent(Tabs, 0, QPoint(0,0));
    if (!cr.isEmpty())
      panel->setCarriageReturn(cr);
    Tabs->insertTab(panel, panel->caption());
  }
}

