/* Copyright (C) 2002 MySQL AB & Jorge del Conde

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
    
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA 
*/

#include "CListViewProperties.h"
#include <qdatetime.h>

CListViewProperties::CListViewProperties(QWidget* parent,  CMySQLConnection *m, const char *name)
: QWidget(parent, name)
{  
  if (!name)
    setName("CListViewProperties");
  tmpFileName = QString::null;
  CListViewPropertiesLayout = new QVBoxLayout( this, 0, 1, "CListViewPropertiesLayout"); 
  title = new QLabel( this, "connectionName" );
  Q_CHECK_PTR(title);
  QFont title_font(title->font());
  title_font.setPointSize(9);  
  title->setFont(title_font);
  title->setMinimumSize(0, 20);
  CListViewPropertiesLayout->addWidget(title);
  details = new QListView(this, "details");
  Q_CHECK_PTR(details);

  details->setFrameShape(QListView::Box);
  details->setLineWidth( 1 );
  details->setSelectionMode(QListView::NoSelection);
  details->setShowSortIndicator(true);
  details->setShowToolTips(false);
  details->header()->setClickEnabled(true, 0);
  details->header()->setClickEnabled(false, 1);
  details->header()->setStretchEnabled(true, 1);
  details->header()->setCellSize (0, 150);  
  
  refreshAction = new QAction(this, "refreshAction" );
  Q_CHECK_PTR(refreshAction);
  refreshAction->setText(tr("Refresh", "" ) );
  refreshAction->setIconSet( QIconSet(getPixmapIcon("refreshIcon")) );
  refreshAction->setMenuText(tr("&Refresh", "" ) );
  refreshAction->setAccel( 0 );
  
  connect(details, SIGNAL(contextMenuRequested(QListViewItem *, const QPoint &, int)), this, SLOT(RightButtonClicked( QListViewItem *, const QPoint &, int )));
  
  CListViewPropertiesLayout->addWidget( details );
  mysql = m;
  hasSaveTitle = false;
  saveTitle = QString::null;
}

void CListViewProperties::refresh()
{
  details->clear();
  insertData();
}

void CListViewProperties::setSaveTitle(const QString &title)
{
  saveTitle = title;
  hasSaveTitle = true;
}

QString CListViewProperties::getSaveTitle()
{
  QString ret = QString::null;
  if (hasSaveTitle)
  {
    ret = "# " + saveTitle + mysql->getLineTerminator(true);
    ret += "# " + tr("Connection") + ": " + mysql->getConnectionName() + mysql->getLineTerminator(true);
    ret += "# " + tr("Host") + ": " + mysql->getHostName() + mysql->getLineTerminator(true);
    ret += "# " + tr("Saved") + ": " + QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") + mysql->getLineTerminator(true);
    ret += "#" + mysql->getLineTerminator(true);
  }
  return ret;
}

QString CListViewProperties::getSaveContents()
{
  QListViewItemIterator it(details);
  QString contents = getSaveTitle();
  for (int i = 0; i < details->columns(); i++)
    contents += mysql->getFieldEncloser(true) + details->columnText(i) + mysql->getFieldEncloser(true) 
    + ((i < details->columns() - 1) ? mysql->getFieldSeparator(true) : mysql->getLineTerminator(true));
  for ( ; it.current(); ++it )
  {
    QString line = QString::null;
    for (int i = 0; i < details->columns(); i++)
    {
      QString txt = ((it.current()->text(i).isEmpty()) ? mysql->getReplaceEmpty(true) : it.current()->text(i));      
      line += mysql->getFieldEncloser(true) + txt + mysql->getFieldEncloser(true) 
      + ((i < details->columns() - 1) ? mysql->getFieldSeparator(true) : mysql->getLineTerminator(true));
    }
    contents += line;
  }
  return contents;
}

void CListViewProperties::save()
{
  saveToFile(tmpFileName, "txt", tr("Text Files (*.txt);;All Files(*.*)" ), getSaveContents(), mysql->messagePanel);
}

void CListViewProperties::RightButtonClicked( QListViewItem *, const QPoint & pos, int)
{
  QPopupMenu p_itemMenu;  
  p_itemMenu.insertItem(getPixmapIcon("saveIcon"), tr("Save As"), MENU_SAVE);
  p_itemMenu.insertSeparator();
  p_itemMenu.insertItem(refreshAction->iconSet(), refreshAction->text(), MENU_REFRESH);
  switch (p_itemMenu.exec(pos))
  {
  case MENU_SAVE:
    save();
    break;
  case MENU_REFRESH:
    refresh();
    break;
  }  
}

void CListViewProperties::insertItem(const QString & property, const QString & value)
{
  if (property.isEmpty() || value.isEmpty())
    return;
  QListViewItem * item = new QListViewItem(details);
  Q_CHECK_PTR(item);
  item->setText( 0, property);
  item->setText( 1, value);
  details->insertItem(item);
}


