/* Copyright (C) 2002 MySQL AB & Jorge del Conde

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
    
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA 
*/

#include <qvariant.h>
#include <qdialog.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qpushbutton.h>
#include <qspinbox.h>
#include <qlayout.h>
#include <qtooltip.h>
#include <qwhatsthis.h>
#include <qimage.h>
#include <qpixmap.h>
#include <qfiledialog.h>
#include <qcolordialog.h>
#include <qfontdatabase.h>
#include <qmessagebox.h>

#include "CGeneralConfig.h"
#include "CSqlEditor.h"
#include "CConfig.h"
#include "Config.h"

p_SyntaxHighlight::p_SyntaxHighlight( QWidget* parent,  const char* name, WFlags fl )
: privateTabConfig(parent, name, fl )
{
  if ( !name )
    setName( "p_SyntaxHighlight" );
  setCaption( tr( "Syntax Highlighting" ) );
  p_SyntaxHighlightLayout = new QHBoxLayout( this, 4, 2, "p_SyntaxHighlightLayout"); 
  
  Sections = new QListBox( this, "Sections" );
  Sections->setSelectionMode(QListBox::Extended);
  QWhatsThis::add( Sections, tr( "Available Sections for the SQL Editor" ) );
  p_SyntaxHighlightLayout->addWidget( Sections );
  
  Layout12 = new QGridLayout( 0, 1, 1, 0, 6, "Layout12"); 
  
  Preview = new QLineEdit( this, "Preview" );
  Preview->setMinimumSize( QSize( 0, 50 ) );
  Preview->setMargin( 1 );
  Preview->setText( tr( "Jorge@mysql.com" ) );
  QWhatsThis::add( Preview, tr( "Preview of this section." ) );
  
  Layout12->addMultiCellWidget( Preview, 3, 3, 1, 5 );
  
  Underline = new QCheckBox( this, "Underline" );
  Underline->setText( tr( "Underline" ) );
  QWhatsThis::add( Underline, tr( "The font for this section will be Underlined." ) );
  
  Layout12->addWidget( Underline, 2, 3 );
  
  sizeLabel = new QLabel( this, "sizeLabel" );
  sizeLabel->setText( tr( "Size" ) );
  
  Layout12->addWidget( sizeLabel, 1, 0 );
  
  Size = new QSpinBox( this, "Size" );  

  Size->setMaxValue( 30 );
  Size->setMinValue( 0 );
  Size->setValue( 8 );
  Size->setSpecialValueText(" ");
  QWhatsThis::add( Size, tr( "This is the font Size that will be used for this particular section." ) );
  
  Layout12->addWidget( Size, 1, 1 );
  
  Italic = new QCheckBox( this, "Italic" );
  Italic->setText( tr( "Italic" ) );
  QWhatsThis::add( Italic, tr( "The font for this section will be Italic." ) );
  
  Layout12->addMultiCellWidget( Italic, 2, 2, 4, 5 );
  
  preveiwLabel = new QLabel( this, "preveiwLabel" );
  preveiwLabel->setText( tr( "Preview" ) );
  
  Layout12->addWidget( preveiwLabel, 3, 0 );
  
  Bold = new QCheckBox( this, "Bold" );
  Bold->setText( tr( "Bold" ) );
  QWhatsThis::add( Bold, tr( "The font for this section will be Bold." ) );
  
  Layout12->addMultiCellWidget( Bold, 2, 2, 1, 2 );
  
  Font = new QComboBox( false, this, "Font" );
  QWhatsThis::add( Font, tr( "This is the Font that will be used for this particular section." ) );
  
  Layout12->addMultiCellWidget( Font, 0, 0, 1, 5 );
  
  Color = new QPushButton( this, "Color" );
  Color->setMinimumSize( QSize( 19, 19 ) );
  Color->setMaximumSize( QSize( 19, 19 ) );  
  Color->setText( tr( "" ) );
  QWhatsThis::add( Color, tr( "This is the font Color that will be used for this particular section." ) );
  
  Layout12->addWidget( Color, 1, 5 );
  
  fontLabel = new QLabel( this, "fontLabel" );
  fontLabel->setText( tr( "Font" ) );
  
  Layout12->addWidget( fontLabel, 0, 0 );
  QSpacerItem* spacer = new QSpacerItem( 20, 20, QSizePolicy::Minimum, QSizePolicy::Expanding );
  Layout12->addItem( spacer, 5, 1 );
  
  colorLabel = new QLabel( this, "colorLabel" );
  colorLabel->setText( tr( "Color" ) );
  colorLabel->setAlignment( int( QLabel::AlignVCenter ) );
  
  Layout12->addWidget( colorLabel, 1, 4 );
  
  DefaultPushButton = new QPushButton( this, "DefaultPushButton" );
  DefaultPushButton->setText( tr( "Restore Defaults" ) );
  
  Layout12->addMultiCellWidget( DefaultPushButton, 4, 4, 4, 5 );
  QSpacerItem* spacer_2 = new QSpacerItem( 20, 20, QSizePolicy::Expanding, QSizePolicy::Minimum );
  Layout12->addMultiCell( spacer_2, 4, 4, 2, 3 );
  p_SyntaxHighlightLayout->addLayout( Layout12 );

  setTabOrder( Sections, Font );
  setTabOrder( Font, Size );
  setTabOrder( Size, Color );
  setTabOrder( Color, Bold );
  setTabOrder( Bold, Underline );
  setTabOrder( Underline, Italic );
  setTabOrder( Italic, Preview );
  setTabOrder( Preview, DefaultPushButton );
  
  init();
  setDefaultValues();  
}

void p_SyntaxHighlight::setDefaultValues(CConfig *)
{
  QFontDatabase fdb;
  QStringList families = fdb.families();
  Font->insertStringList(families);
  QIntDictIterator<SqlFormat> it(*g_SqlEditorStyles);
  for ( ; it.current(); ++it )
  {
    Sections->insertItem(it.current()->displayName, (int) it.current()->ID);
    it.current()->setDefaults();
  }
  Sections->setSelected (0, true);
  Refresh();
}

bool p_SyntaxHighlight::save(CConfig *conn)
{
  bool ret = true;
  QIntDictIterator<SqlFormat> it(*g_SqlEditorStyles);
  for ( ; it.current(); ++it )
    ret &= it.current()->saveToCfg(conn);
  return ret;
}

SqlFormat *p_SyntaxHighlight::findItem(uint i)
{
  return (SqlFormat *)g_SqlEditorStyles->find(i);
}

void p_SyntaxHighlight::refreshPreview(const QFont & fnt, const QColor & clr)
{
  Preview->setFont(fnt);
  Preview->setPaletteForegroundColor(clr);
}

void p_SyntaxHighlight::Refresh()
{
  QString font = QString::null;
  bool bold = false;
  bool underline = false;
  bool italic = false;
  int size = 0;
  QColor color = paletteBackgroundColor();
  
  bool first = true;
  for (uint i=0; i < Sections->count(); i++)
  {
    if (Sections->isSelected(i))
    {
      SqlFormat *item = findItem(i);
      if (item != NULL)
      {
        if (first)
        {
          font = item->m_font.family();
          bold = item->m_font.bold();
          underline = item->m_font.underline();
          italic = item->m_font.italic();
          size = item->m_font.pointSize();
          color = item->m_color;
          first = false;
        }
        else
        {
          if (item->m_font.family() != font)
            font = "";
          
          if (item->m_font.bold() != bold)
            bold = false;
          
          if (item->m_font.underline() != underline)
            underline = false;
          
          if (item->m_font.italic() != italic)
            italic = false;
          
          if (item->m_font.pointSize() != size)
            size = 0;
          
          if (item->m_color != color)
            color = paletteBackgroundColor();
        }
      }
    }
  }
  
  Font->setCurrentText(font);
  Bold->setChecked(bold);
  Underline->setChecked(underline);
  Italic->setChecked(italic);
  Size->setValue(size);
  Color->setPaletteBackgroundColor(color);
  QFont f(font, size, QFont::Normal, italic);
  f.setBold(bold);
  f.setUnderline(underline);
  refreshPreview(f, color);
}

void p_SyntaxHighlight::setFontBoolValue(void (QFont::*member)(bool), bool value)
{  
  for (uint i=0; i < Sections->count(); i++)
  {
    if (Sections->isSelected(i))
    {
      SqlFormat *item = findItem(i);
      if (item != NULL)
        (item->m_font.*member)(value);      
    }
  }
  Refresh();
}

void p_SyntaxHighlight::BoldToggled(bool b)
{
  void (QFont::*member)(bool) = &QFont::setBold;
  setFontBoolValue(member, b);
}

void p_SyntaxHighlight::ItalicToggled(bool b)
{
  void (QFont::*member)(bool) = &QFont::setItalic;
  setFontBoolValue(member, b);
}

void p_SyntaxHighlight::UnderlineToggled(bool b)
{
  void (QFont::*member)(bool) = &QFont::setUnderline;
  setFontBoolValue(member, b); 
}

void p_SyntaxHighlight::ValueChanged(int s)
{
  for (uint i=0; i < Sections->count(); i++)
  {
    if (Sections->isSelected(i))
    {
      SqlFormat *item = findItem(i);
      if (item != NULL)
        item->m_font.setPointSize(s);   
    }
  }
  Refresh();
}

void p_SyntaxHighlight::FontChanged(const QString &f)
{
  for (uint i=0; i < Sections->count(); i++)
  {
    if (Sections->isSelected(i))
    {
      SqlFormat *item = findItem(i);
      if (item != NULL)
        item->m_font.setFamily(f);       
    }
  }
  Refresh();
}

void p_SyntaxHighlight::setSectionColor()
{
  QColor c = QColorDialog::getColor(Color->paletteBackgroundColor(), this );
  if (c.isValid())
  {
    for (uint i=0; i < Sections->count(); i++)
    {
      if (Sections->isSelected(i))
      {
        SqlFormat *item = findItem(i);
        if (item != NULL)
          item->m_color = c;    
      }
    }
    Color->setPaletteBackgroundColor(c);
    Refresh();
  }
}

void p_SyntaxHighlight::DefaultPushButtonClicked()
{ 
  for (uint i=0; i < Sections->count(); i++)
  {
    if (Sections->isSelected(i))
    {
      SqlFormat *item = findItem(i);
      if (item != NULL)
        item->setDefaultValues();
    }
  }  
  Refresh();
}

void p_SyntaxHighlight::init()
{
  connect(Sections, SIGNAL(selectionChanged()), this, SLOT(Refresh()));  
  connect(Bold, SIGNAL(toggled(bool)), this, SLOT(BoldToggled(bool)));  
  connect(Underline, SIGNAL(toggled(bool)), this, SLOT(UnderlineToggled(bool)));
  connect(Italic, SIGNAL(toggled(bool)), this, SLOT(ItalicToggled(bool)));
  connect(Font, SIGNAL(activated(const QString &)), this, SLOT(FontChanged(const QString &)));
  connect(Size, SIGNAL(valueChanged(int)), this, SLOT(ValueChanged(int)));  
  connect(Color, SIGNAL(clicked()), this, SLOT(setSectionColor()));
  connect(DefaultPushButton, SIGNAL(clicked()), this, SLOT(DefaultPushButtonClicked()));
}

p_SqlEditor::p_SqlEditor( QWidget* parent,  const char* name, WFlags fl )
: privateTabConfig(parent, name, fl )
{
  if ( !name )
    setName( "p_SqlEditor" );
  
  setCaption( tr( "Sql Editor" ) );
  p_SqlEditorLayout = new QGridLayout( this, 1, 1, 4, 2, "p_SqlEditorLayout"); 
  
  Parentheses = new QCheckBox( this, "Parentheses" );
  Parentheses->setText( tr( "Enable Parentheses Matching" ) );
  QWhatsThis::add( Parentheses, tr( "When enabled, the SQL Editor will use Parentheses Matching." ) );
  
  p_SqlEditorLayout->addMultiCellWidget( Parentheses, 2, 2, 0, 2 );
  
  Completion = new QCheckBox( this, "Completion" );
  Completion->setText( tr( "Enable Completion" ) );
  QWhatsThis::add( Completion, tr( "When enabled, the SQL Editor will support Completion for SQL Keywords." ) );
  
  p_SqlEditorLayout->addMultiCellWidget( Completion, 3, 3, 0, 2 );
  
  SyntaxFileLabel = new QLabel( this, "SyntaxFileLabel" );
  SyntaxFileLabel->setText( tr( "Syntax File" ) );
  
  p_SqlEditorLayout->addWidget( SyntaxFileLabel, 0, 0 );
  
  SyntaxFile = new QLineEdit( this, "SyntaxFile" );
  SyntaxFile->setFrameShape( QLineEdit::StyledPanel );
  SyntaxFile->setFrameShadow( QLineEdit::Sunken );
  QWhatsThis::add( SyntaxFile, tr( "This is the File that contains the list of keywords that will be highlighted and completed by the SQL Editor." ) );
  
  p_SqlEditorLayout->addWidget( SyntaxFile, 0, 1 );
  
  SyntaxFileBrowse = new QPushButton( this, "SyntaxFileBrowse" );
  SyntaxFileBrowse->setMinimumSize( QSize( 22, 22 ) );
  SyntaxFileBrowse->setMaximumSize( QSize( 22, 22 ) );
  SyntaxFileBrowse->setText( tr( "" ) );
  SyntaxFileBrowse->setPixmap(getPixmapIcon("openIcon"));
  QWhatsThis::add( SyntaxFileBrowse, tr( "Select the Folder that contains the Translation Files." ));
  
  p_SqlEditorLayout->addWidget( SyntaxFileBrowse, 0, 2 );
  
  SyntaxHighlight = new QCheckBox( this, "SyntaxHighlight" );
  SyntaxHighlight->setText( tr( "Enable SQL Syntax Highlighting" ) );
  QWhatsThis::add( SyntaxHighlight, tr( "When enabled, the SQL Editor will use Syntax Highlighting for SQL Keywords." ) );
  
  p_SqlEditorLayout->addMultiCellWidget( SyntaxHighlight, 1, 1, 0, 2 );
  QSpacerItem* spacer = new QSpacerItem( 20, 20, QSizePolicy::Minimum, QSizePolicy::Expanding );
  p_SqlEditorLayout->addItem( spacer, 4, 1 );
  setTabOrder( SyntaxFile, SyntaxFileBrowse );
  setTabOrder( SyntaxFileBrowse, SyntaxHighlight );
  setTabOrder( SyntaxHighlight, Parentheses );
  setTabOrder( Parentheses, Completion );
  setDefaultValues();
  init();
}

void p_SqlEditor::setDefaultValues(CConfig *)
{
  if (g_syntaxHighlight)  
    Parentheses->setChecked(g_parenthesesMatching && g_syntaxHighlight);  
  else
    Parentheses->setEnabled(false);
  Completion->setChecked(g_completion);
  SyntaxFile->setText(*g_SyntaxFile);
  SyntaxHighlight->setChecked(g_syntaxHighlight);
}

bool p_SqlEditor::save(CConfig *conn)
{    
  bool ret = conn->writeEntry("Syntax File", SyntaxFile->text().stripWhiteSpace());
  ret &= conn->writeEntry("Completion", booltostr(Completion->isChecked()));
  ret &= conn->writeEntry("Syntax Highlighting", booltostr(SyntaxHighlight->isChecked()));
  ret &= conn->writeEntry("Parentheses Matching", booltostr(Parentheses->isChecked()));  
  return ret;
}

void p_SqlEditor::setSyntaxFile()
{
  QString tmp = QFileDialog::getOpenFileName ("syntax.txt", tr("Text Files") + " (*.txt);;" + tr("All Files") + " (*.*)", this, "setSyntaxFile", tr("Select the Syntax File"));    
  if (!tmp.isEmpty())
  {
    tmp = charReplace(tmp.stripWhiteSpace(),'\\', "/");    
    SyntaxFile->setText(tmp);        
  }
}

void p_SqlEditor::init()
{
  connect(SyntaxFileBrowse, SIGNAL(clicked()), this, SLOT(setSyntaxFile()));
  connect(SyntaxHighlight, SIGNAL(toggled(bool)), Parentheses, SLOT(setEnabled(bool)));
}

p_GeneralConfig::p_GeneralConfig( QWidget* parent,  const char* name, WFlags fl )
: privateTabConfig(parent, name, fl ), p_needRestart(false)
{
  if ( !name )
    setName( "p_GeneralConfig" );
  setCaption( tr( "General" ) );
  p_GeneralConfigLayout = new QGridLayout( this, 1, 1, 4, 2, "p_GeneralConfigLayout"); 
  
  errorSoundFile = new QLineEdit( this, "errorSoundFile" );
  errorSoundFile->setFrameShape( QLineEdit::StyledPanel );
  errorSoundFile->setFrameShadow( QLineEdit::Sunken );
  QWhatsThis::add( errorSoundFile, tr( "This is a WAV file that will be played each time an Error message occures.  If you don't want to play a sound, leave this field empty." ) );
  
  p_GeneralConfigLayout->addMultiCellWidget( errorSoundFile, 1, 1, 1, 4 );
  
  translationsPath = new QLineEdit( this, "translationsPath" );
  translationsPath->setFrameShape( QLineEdit::StyledPanel );
  translationsPath->setFrameShadow( QLineEdit::Sunken );
  QWhatsThis::add( translationsPath, tr( "This is the Path which will be used by the application to find Language Files." ) );
  
  p_GeneralConfigLayout->addMultiCellWidget( translationsPath, 0, 0, 1, 4 );
  
  saveWorkspace = new QCheckBox( this, "saveWorkspace" );
  saveWorkspace->setText( tr( "Restore last Workspace on startup" ) );
  QWhatsThis::add( saveWorkspace, tr( "When enabled, the application will restore the last Workspace of the previous execution by automatically opening all windows and connections which where open prior shutdown." ) );
  
  p_GeneralConfigLayout->addMultiCellWidget( saveWorkspace, 6, 6, 0, 2 );
  
  informationSoundFile = new QLineEdit( this, "informationSoundFile" );
  informationSoundFile->setFrameShape( QLineEdit::StyledPanel );
  informationSoundFile->setFrameShadow( QLineEdit::Sunken );
  QWhatsThis::add( informationSoundFile, tr( "This is a WAV file that will be played each time a Information message occures.  If you don't want to play a sound, leave this field empty." ) );
  
  p_GeneralConfigLayout->addMultiCellWidget( informationSoundFile, 3, 3, 1, 4 );
  
  confirmCritical = new QCheckBox( this, "confirmCritical" );
  confirmCritical->setText( tr( "Confirm critical opperations" ) );
  QWhatsThis::add( confirmCritical, tr( "This option will confirm critical opperations done by the client.  Such critical opperations include shutting down the application, shutting down the Server, truncate table, etc ..." ) );
  
  p_GeneralConfigLayout->addMultiCellWidget( confirmCritical, 7, 7, 0, 1 );
  
  warningSoundFile = new QLineEdit( this, "warningSoundFile" );
  warningSoundFile->setFrameShape( QLineEdit::StyledPanel );
  warningSoundFile->setFrameShadow( QLineEdit::Sunken );
  QWhatsThis::add( warningSoundFile, tr( "This is a WAV file that will be played each time a Warning message occures.  If you don't want to play a sound, leave this field empty." ) );
  
  p_GeneralConfigLayout->addMultiCellWidget( warningSoundFile, 2, 2, 1, 4 );
  QSpacerItem* spacer = new QSpacerItem( 20, 20, QSizePolicy::Minimum, QSizePolicy::Expanding );
  p_GeneralConfigLayout->addItem( spacer, 8, 2 );
  
  language = new QComboBox( FALSE, this, "language" );
  language->setAutoCompletion( TRUE );
  language->setDuplicatesEnabled( FALSE );
  QWhatsThis::add( language, tr( "This will change the display Language of the Application.  If changed, one will have to restart the application for changes to take place." ) );
  
  p_GeneralConfigLayout->addMultiCellWidget( language, 4, 4, 1, 3 );
  
  historySize = new QSpinBox( this, "historySize" );
  historySize->setMaxValue( 1000 );
  historySize->setValue( 200 );
  QWhatsThis::add( historySize, tr( "This number specifies how many Queries will be saved by the History Panel in  the Query Windows." ) );
  
  p_GeneralConfigLayout->addMultiCellWidget( historySize, 5, 5, 1, 2 );
  QSpacerItem* spacer_2 = new QSpacerItem( 20, 20, QSizePolicy::Expanding, QSizePolicy::Minimum );
  p_GeneralConfigLayout->addItem( spacer_2, 5, 3 );
  QSpacerItem* spacer_3 = new QSpacerItem( 20, 20, QSizePolicy::Expanding, QSizePolicy::Minimum );
  p_GeneralConfigLayout->addItem( spacer_3, 4, 4 );
  
  historyNumberLabel = new QLabel( this, "historyNumberLabel" );
  historyNumberLabel->setText( tr( "History size for Queries" ) );
  
  p_GeneralConfigLayout->addWidget( historyNumberLabel, 5, 0 );
  
  languageLabel = new QLabel( this, "languageLabel" );
  languageLabel->setText( tr( "Language" ) );
  
  p_GeneralConfigLayout->addWidget( languageLabel, 4, 0 );
  
  informationSoundLabel = new QLabel( this, "informationSoundLabel" );
  informationSoundLabel->setText( tr( "Information Sound File" ) );
  
  p_GeneralConfigLayout->addWidget( informationSoundLabel, 3, 0 );
  
  warningSoundLabel = new QLabel( this, "warningSoundLabel" );
  warningSoundLabel->setText( tr( "Warning Sound File" ) );
  
  p_GeneralConfigLayout->addWidget( warningSoundLabel, 2, 0 );
  
  errorSoundLabel = new QLabel( this, "errorSoundLabel" );
  errorSoundLabel->setText( tr( "Error Sound File" ) );
  
  p_GeneralConfigLayout->addWidget( errorSoundLabel, 1, 0 );
  
  translationsPathLabel = new QLabel( this, "translationsPathLabel" );
  translationsPathLabel->setText( tr( "Translations Path" ) );
  
  p_GeneralConfigLayout->addWidget( translationsPathLabel, 0, 0 );
  
  warningSoundBrowse = new QPushButton( this, "warningSoundBrowse" );
  warningSoundBrowse->setMinimumSize( QSize( 22, 22 ) );
  warningSoundBrowse->setMaximumSize( QSize( 22, 22 ) );
  warningSoundBrowse->setText( tr( "" ) );
  warningSoundBrowse->setPixmap(getPixmapIcon("openIcon"));
  QWhatsThis::add( warningSoundBrowse, tr( "Click to browse." ) );
  
  p_GeneralConfigLayout->addWidget( warningSoundBrowse, 2, 5 );
  
  informationSoundBrowse = new QPushButton( this, "informationSoundBrowse" );
  informationSoundBrowse->setMinimumSize( QSize( 22, 22 ) );
  informationSoundBrowse->setMaximumSize( QSize( 22, 22 ) );
  informationSoundBrowse->setText( tr( "" ) );
  informationSoundBrowse->setPixmap(getPixmapIcon("openIcon"));
  QWhatsThis::add( informationSoundBrowse, tr( "Click to browse." ) );
  
  p_GeneralConfigLayout->addWidget( informationSoundBrowse, 3, 5 );
  
  errorSoundBrowse = new QPushButton( this, "errorSoundBrowse" );
  errorSoundBrowse->setMinimumSize( QSize( 22, 22 ) );
  errorSoundBrowse->setMaximumSize( QSize( 22, 22 ) );
  errorSoundBrowse->setText( tr( "" ) );
  errorSoundBrowse->setPixmap(getPixmapIcon("openIcon"));
  QWhatsThis::add( errorSoundBrowse, tr( "Click to browse." ) );
  
  p_GeneralConfigLayout->addWidget( errorSoundBrowse, 1, 5 );
  
  translationsBrowse = new QPushButton( this, "translationsBrowse" );
  translationsBrowse->setMinimumSize( QSize( 22, 22 ) );
  translationsBrowse->setMaximumSize( QSize( 22, 22 ) );
  translationsBrowse->setText( tr( "" ) );
  translationsBrowse->setPixmap(getPixmapIcon("openIcon"));
  QWhatsThis::add( translationsBrowse, tr( "Click to browse." ) );
  
  p_GeneralConfigLayout->addWidget( translationsBrowse, 0, 5 );
    
  setTabOrder( translationsPath, translationsBrowse );
  setTabOrder( translationsBrowse, errorSoundFile );
  setTabOrder( errorSoundFile, errorSoundBrowse );
  setTabOrder( errorSoundBrowse, warningSoundFile );
  setTabOrder( warningSoundFile, warningSoundBrowse );
  setTabOrder( warningSoundBrowse, informationSoundFile );
  setTabOrder( informationSoundFile, informationSoundBrowse );
  setTabOrder( informationSoundBrowse, language );
  setTabOrder( language, historySize );
  setTabOrder( historySize, saveWorkspace );
  setTabOrder( saveWorkspace, confirmCritical );

  setDefaultValues();
  init();
}

bool p_GeneralConfig::save(CConfig *conn)
{  
  bool ret = conn->writeEntry("Translations Path", translationsPath->text().stripWhiteSpace() );
  ret &= conn->writeEntry("Language File", language->currentText().stripWhiteSpace());
  ret &= conn->writeEntry("History Size", historySize->value());  
  ret &= conn->writeEntry("Save Workspace", booltostr(saveWorkspace->isChecked()));
  ret &= conn->writeEntry("Confirm Critical", booltostr(confirmCritical->isChecked()));
  ret &= conn->writeEntry("Error Sound", errorSoundFile->text().stripWhiteSpace() );
  ret &= conn->writeEntry("Warning Sound", warningSoundFile->text().stripWhiteSpace() );
  ret &= conn->writeEntry("Information Sound", informationSoundFile->text().stripWhiteSpace() );

  if (*g_TranslationsPath != translationsPath->text().stripWhiteSpace())  
    p_needRestart = true;  

  if (*g_CurrentLanguage != language->currentText().stripWhiteSpace())  
    p_needRestart = true;  

  return ret;
}

void p_GeneralConfig::setDefaultValues(CConfig *)
{
  translationsPath->setText(*g_TranslationsPath);
  historySize->setValue(g_historySize);
  confirmCritical->setChecked(g_confirmCritical);
  saveWorkspace->setChecked(g_saveWorkspace);
  errorSoundFile->setText(g_errorSoundFile);
  warningSoundFile->setText(g_warningSoundFile);
  informationSoundFile->setText(g_informationSoundFile);
  refreshLanguageCombo();
}

void p_GeneralConfig::refreshLanguageCombo()
{
  refreshLanguageCombo(translationsPath->text());   
}

void p_GeneralConfig::refreshLanguageCombo(const QString &path)
{
  language->clear();
  QDir dir = QDir(path, "*.qm");
  dir.setFilter(QDir::Files);
  dir.setSorting( QDir::Name);
  language->insertItem("English");
  language->insertStringList(dir.entryList());
  if ((dir.count() > 0) && (dir.exists(*g_CurrentLanguage)))
    language->setCurrentText(*g_CurrentLanguage);  
}

void p_GeneralConfig::setTranslationsPath()
{
  QString tmp = QFileDialog::getExistingDirectory("", this, "setTranslationFile", tr("Select the Translations Path"));
  if (!tmp.isEmpty())
  {
    tmp = charReplace(tmp.stripWhiteSpace(),'\\', "/");
    if (!tmp.endsWith ("/"))
      tmp += "/";
    translationsPath->setText(tmp);    
    refreshLanguageCombo();
  }
}

void p_GeneralConfig::setErrorFile()
{
  QString tmp = QFileDialog::getOpenFileName(QString::null, tr("Wav Files(*.wav)"), this, "BrowseErrorFile", tr("Select Error Sound"));
  if (!tmp.isEmpty())
  {
    tmp = charReplace(tmp.stripWhiteSpace(),'\\', "/");  
    errorSoundFile->setText(tmp);    
  }
}

void p_GeneralConfig::setWarningFile()
{
  QString tmp = QFileDialog::getOpenFileName(QString::null, tr("Wav Files(*.wav)"), this, "BrowseWarningFile", tr("Select Warning Sound"));
  if (!tmp.isEmpty())
  {
    tmp = charReplace(tmp.stripWhiteSpace(),'\\', "/");  
    warningSoundFile->setText(tmp);    
  }
}

void p_GeneralConfig::setInformationFile()
{
  QString tmp = QFileDialog::getOpenFileName(QString::null, tr("Wav Files(*.wav)"), this, "BrowseInformationFile", tr("Select Information Sound"));
  if (!tmp.isEmpty())
  {
    tmp = charReplace(tmp.stripWhiteSpace(),'\\', "/");  
    informationSoundFile->setText(tmp);    
  }
}

void p_GeneralConfig::init()
{
  connect(translationsBrowse, SIGNAL(clicked()), this, SLOT(setTranslationsPath()));
  connect(errorSoundBrowse, SIGNAL(clicked()), this, SLOT(setErrorFile()));
  connect(warningSoundBrowse, SIGNAL(clicked()), this, SLOT(setWarningFile()));
  connect(informationSoundBrowse, SIGNAL(clicked()), this, SLOT(setInformationFile()));
  connect(translationsPath, SIGNAL( textChanged(const QString &)), this, SLOT(refreshLanguageCombo(const QString &)));
}

CGeneralConfig::CGeneralConfig(QWidget* parent,  const char* name)
:CConfigDialog(parent, name)
{
  if ( !name )
    setName( "CGeneralConfig" );    
  setMinimumHeight(263);  
  setCaption( QString(CODENAME) + " - " + tr("General Configuration Dialog"));
  General = new p_GeneralConfig(Tab);
  insertTab(General);
  insertTab(new p_SqlEditor(Tab));
  insertTab(new p_SyntaxHighlight(Tab));
  okPushButton->setText(tr( "&Apply"));
  QWhatsThis::add(okPushButton, tr("Click to Apply changes."));
  myResize(421, 263);
}

CGeneralConfig::~CGeneralConfig()
{
}

void CGeneralConfig::setStylePalette(QString style)
{
  if (style== "Platinum")
  {
    QPalette p( QColor( 239, 239, 239 ) );
    qApp->setPalette(p, true);
  }
  else
    if (style == "CDE")
    {
      QPalette p( QColor( 75, 123, 130 ) );
      p.setColor( QPalette::Active, QColorGroup::Base, QColor( 55, 77, 78 ) );
      p.setColor( QPalette::Inactive, QColorGroup::Base, QColor( 55, 77, 78 ) );
      p.setColor( QPalette::Disabled, QColorGroup::Base, QColor( 55, 77, 78 ) );
      p.setColor( QPalette::Active, QColorGroup::Highlight, Qt::white );
      p.setColor( QPalette::Active, QColorGroup::HighlightedText, QColor( 55, 77, 78 ) );
      p.setColor( QPalette::Inactive, QColorGroup::Highlight, Qt::white );
      p.setColor( QPalette::Inactive, QColorGroup::HighlightedText, QColor( 55, 77, 78 ) );
      p.setColor( QPalette::Disabled, QColorGroup::Highlight, Qt::white );
      p.setColor( QPalette::Disabled, QColorGroup::HighlightedText, QColor( 55, 77, 78 ) );
      p.setColor( QPalette::Active, QColorGroup::Foreground, Qt::white );
      p.setColor( QPalette::Active, QColorGroup::Text, Qt::white );
      p.setColor( QPalette::Active, QColorGroup::ButtonText, Qt::white );
      p.setColor( QPalette::Inactive, QColorGroup::Foreground, Qt::white );
      p.setColor( QPalette::Inactive, QColorGroup::Text, Qt::white );
      p.setColor( QPalette::Inactive, QColorGroup::ButtonText, Qt::white );
      p.setColor( QPalette::Disabled, QColorGroup::Foreground, Qt::lightGray );
      p.setColor( QPalette::Disabled, QColorGroup::Text, Qt::lightGray );
      p.setColor( QPalette::Disabled, QColorGroup::ButtonText, Qt::lightGray );
      qApp->setPalette( p, true);
    }
    else
      if (style == "Motif")
      {
        QPalette p( QColor( 192, 192, 192 ) );
        qApp->setPalette( p, true);
      }
      else
        if (style == "MotifPlus")
        {
          QPalette p( QColor( 192, 192, 192 ) );

          qApp->setPalette(p, true);
        }
}

void CGeneralConfig::Configure(bool startupSpecific)
{
  CConfig *cfg = new CConfig(g_CONFIG_FILE);
  Q_CHECK_PTR(cfg);
  QString mydir = QDir::currentDirPath() + "/";  
  if (startupSpecific)
  {
    if (!cfg->exists())  //create the configuration file if it doesn't exist
    {
      bool err = cfg->prepare();

      err &= cfg->writeEntry("MDI", booltostr(IS_MDI));
      err &= cfg->writeEntry("Translations Path", mydir + "translations/");
      err &= cfg->writeEntry("Language", "English");
      err &= cfg->writeEntry("History Size", 200);
      err &= cfg->writeEntry("Save Workspace", "true");
      err &= cfg->writeEntry("Confirm Critical", "true");

      err &= cfg->writeEntry("Syntax File", mydir + "syntax.txt");
      err &= cfg->writeEntry("Error Sound", mydir + "error.wav");
      err &= cfg->writeEntry("Warning Sound", mydir + "warning.wav");

      err &= cfg->writeEntry("Completion", "true");
      err &= cfg->writeEntry("Syntax Highlighting", "true");
      err &= cfg->writeEntry("Parentheses Matching", "true");
      
      QFont font(qApp->font().family(), 8, QFont::Normal, true);
      
      font.setBold(false);
      err &= cfg->writeEntry("Comments Highlight", font.toString());
      err &= cfg->writeEntry("Comments Highlight Color", "192,0,0");
      
      font.setItalic(false);
      
      err &= cfg->writeEntry("Standard Highlight", font.toString());
      err &= cfg->writeEntry("Standard Highlight Color", "0,0,0");
      
      err &= cfg->writeEntry("Custom Highlight", font.toString());
      err &= cfg->writeEntry("Custom Highlight Color", "0,0,0");
      
      err &= cfg->writeEntry("Functions Highlight", font.toString());
      err &= cfg->writeEntry("Functions Highlight Color", "0,0,192");
      
      err &= cfg->writeEntry("Variables Highlight", font.toString());
      err &= cfg->writeEntry("Variables Highlight Color", "255,0,0");
      
      err &= cfg->writeEntry("Strings Highlight", font.toString());
      err &= cfg->writeEntry("Strings Highlight Color", "0,192,0");
      
      err &= cfg->writeEntry("Numbers Highlight", font.toString());
      err &= cfg->writeEntry("Numbers Highlight Color", "128,0,255");
      
      font.setBold(true);
      err &= cfg->writeEntry("Keywords Highlight", font.toString());
      err &= cfg->writeEntry("Keywords Highlight Color", "0,0,255");
      
      err &= cfg->writeEntry("Field Types Highlight", font.toString());
      err &= cfg->writeEntry("Field Types Highlight Color", "128,0,128");
      
      err &= cfg->writeEntry("Tables and Fields Highlight", font.toString());
      err &= cfg->writeEntry("Tables and Fields Highlight Color", "0,0,0");
      
      err &= cfg->writeEntry("Parentheses Matching Highlight Color", "255,255,192");
      
      err &= cfg->flush();
      if (!err)
        QMessageBox::warning(0, tr("Error"), tr("An error occured while writing the configuration file"));
    }
  }
  cfg->prepare();
  if (startupSpecific)
    g_isMDI = strtobool(cfg->readStringEntry("MDI", "false"));
  *g_TranslationsPath = cfg->readStringEntry("Translations Path", mydir + "translations/");
  *g_CurrentLanguage = cfg->readStringEntry("Language File", "English");
  g_saveWorkspace = strtobool(cfg->readStringEntry("Save Workspace", "true"));
  g_confirmCritical = strtobool(cfg->readStringEntry("Confirm Critical", "true"));
  g_historySize = cfg->readNumberEntry("History Size", 200);
  QFont font;
  font.fromString(cfg->readStringEntry("Font", QFont().toString()));
  qApp->setFont((font), true);
  g_printerFont.fromString(cfg->readStringEntry("Printer Font", QFont("Times", 10).toString()));  
  *g_CurrentStyle = cfg->readStringEntry("Style", "Windows");
  *g_SyntaxFile = cfg->readStringEntry("Syntax File", mydir + "syntax.txt");
  g_errorSoundFile = cfg->readStringEntry("Error Sound", mydir + "error.wav");
  g_warningSoundFile = cfg->readStringEntry("Warning Sound", mydir + "warning.wav");
  g_informationSoundFile = cfg->readStringEntry("Information Sound", QString::null);
  g_completion = strtobool(cfg->readStringEntry("Completion", "true"));
  g_syntaxHighlight = strtobool(cfg->readStringEntry("Syntax Highlighting", "true"));
  g_parenthesesMatching = strtobool(cfg->readStringEntry("Parentheses Matching", "true"));  
  qApp->setStyle(*g_CurrentStyle);
  setStylePalette(*g_CurrentStyle);  
  PrepareSqlEditor(cfg);
  delete cfg;
}

void CGeneralConfig::PrepareSqlEditor(CConfig *cfg)
{
  g_SqlEditorStyles->clear();
  
  QString f, c;    
  QFont font(qApp->font().family(), 8, QFont::Normal);
  font.setBold(false);
  //Standard should ALWAYS be the first.
  new SqlFormat(cfg, tr("Standard"), "Standard", SQLKeyWord::highlight_STANDARD, font.toString(), "0,0,0");  
  font.setBold(true);  
  new SqlFormat(cfg, tr("Keywords"), "Keywords", SQLKeyWord::highlight_KEYWORDS, font.toString(), "0,0,255");
  new SqlFormat(cfg, tr("Field Types"), "Field Types", SQLKeyWord::highlight_FIELD_TYPES, font.toString(), "128,0,128");
  new SqlFormat(cfg, tr("Tables and Fields"), "Tables and Fields", SQLKeyWord::highlight_TABLES_AND_FIELDS, font.toString(), "0,0,0");  

  font.setBold(false);
  new SqlFormat(cfg, tr("Functions"), "Functions", SQLKeyWord::highlight_FUNCTIONS, font.toString(), "0,0,192");  
  new SqlFormat(cfg, tr("Variables"), "Variables", SQLKeyWord::highlight_VARIABLES, font.toString(), "255,0,0");
  font.setItalic(true);  
  new SqlFormat(cfg, tr("Comments"), "Comments", SQLKeyWord::highlight_COMMENTS, font.toString(), "192,0,0");
  font.setItalic(false);
  new SqlFormat(cfg, tr("Custom"), "Custom", SQLKeyWord::highlight_CUSTOM, font.toString(), "0,0,0");  
  new SqlFormat(cfg, tr("Strings"), "Strings", SQLKeyWord::highlight_STRINGS, font.toString(), "0,192,0");;
  new SqlFormat(cfg, tr("Numbers"), "Numbers", SQLKeyWord::highlight_NUMBERS, font.toString(), "128,0,255");  
  new SqlFormat(cfg, tr("Parentheses Matching"), "Parentheses Matching", SQLKeyWord::highlight_PARENTHESES_MATCHING, "", "255,255,192");
}

void CGeneralConfig::okButtonClicked()
{
  CConfig *conn = new CConfig(g_CONFIG_FILE);
  Q_CHECK_PTR(conn);
  bool ret;
  ret = conn->prepare();
  ret &= save(conn);

  needRestart = General->needRestart();  

  if (!ret)
  {
    QMessageBox::critical(0,QString(CODENAME) + " Error", "An Error ocurred while saving the Configuration.");
    return;
  }
  else
    if (needRestart)
      QMessageBox::information(0,QString(CODENAME) + " Restart", "You will need to restart MyCC before you notice your changes.");
  Configure();
  if (g_syntaxHighlight || g_completion)
    CSqlEditor::LoadSyntaxFile();
  delete conn;
  close();
}
