/* Copyright (C) 2002 MySQL AB & Jorge del Conde

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
    
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA 
*/

#include "CDatabaseItem.h"
#include "CTableGroupItem.h"
#include "CTableSpecific.h"
#include "syntaxhighliter_sql.h"
#include <qmessagebox.h>

CDatabaseItem::CDatabaseItem(CMyWindow *consoleWindow, int serverid, CDatabaseTreeItem * parent, const QString &dbname, bool isconnected, QWidgetStack *w)
: CDatabaseTreeItem(parent)
{		
  m_pMySQLServer = ((CDatabaseTreeItem *)this->parent())->m_pMySQLServer;  
  deleteMySQL = false;
  init_item(isconnected, dbname, w, consoleWindow, serverid);
}

CDatabaseItem::CDatabaseItem(CMyWindow *consoleWindow, int serverid, QListView * parent, CMySQLConnection *mysql, const QString & dbname, bool isconnected, QWidgetStack *w)
: CDatabaseTreeItem(parent)
{	
  m_pMySQLServer = mysql;
  deleteMySQL = true;	
  init_item(isconnected, dbname, w, consoleWindow, serverid);
  setOpen(true);
}

CDatabaseItem::~CDatabaseItem()
{  
  if (deleteMySQL)
    delete m_pMySQLServer;
  delete m_pMySQLDatabase;
}

void CDatabaseItem::init_item(bool isconnected, const QString &dbname, QWidgetStack *w, CMyWindow *consoleWindow, int serverid)
{
  isRefreshing = false;
  serverID = serverid;
  ConsoleWindow = consoleWindow;
  setType(DATABASE_ITEM);
  setName("CDatabaseItem");
  setText(0,dbname);	
  widgetStack = w;
  m_pMySQLDatabase = new CMySQLConnection(m_pMySQLServer->getConnectionName(), ConsoleWindow->messagePanel());
  Q_CHECK_PTR(m_pMySQLDatabase);
  setConnected(isconnected);
}

void CDatabaseItem::startDrag()
{	
  itemDrag("DATABASE");
}

void CDatabaseItem::setConnected(bool isconnected)
{
  if (isRefreshing)
    return;
  isRefreshing = true;
  m_pMySQLDatabase->setUserName(m_pMySQLServer->getUserName());
  m_isConnected = m_pMySQLDatabase->setConnected(isconnected, text(0));
  setOpen(isconnected);  
  if (!m_isConnected)  
    deleteChilds();    
  else
  {
    DBKeyWord *dbk = g_DBKeywordDict->find(serverID);
    if (dbk != NULL)
      dbk->addDB(text(0));    
    new CTableGroupItem(ConsoleWindow, serverID, this, widgetStack);
  }
  setPixmap(0, getPixmapIcon(!m_isConnected ? "databaseDisconnectedIcon" : "databaseConnectedIcon"));
  isRefreshing = false;
}

void CDatabaseItem::processMenu(int res)
{
  if (isRefreshing)
    return;
  switch (res) {
		case MENU_CONNECT: setConnected(true);
      break;
    case MENU_DISCONNECT: setConnected(false);
      break;
    case MENU_QUERY: CTableSpecific::QueryWindow(m_pMySQLDatabase, "", SQL_QUERY, serverID);
      break;
    case MENU_DELETE:
    {
      if (g_confirmCritical)
        if ((QMessageBox::information(0, QObject::tr("Confirm Delete Database"), 
          QObject::tr("Are you sure you want to Delete Database") + ":\"" + text(0) + "\"?",
          QObject::tr("&Yes"), QObject::tr("&No")) != 0))
          return;
      CMySQLQuery query(m_pMySQLServer);
      QString sql = "DROP DATABASE IF EXISTS " + text(0);
      if (query.exec_static(sql))
      {
        setConnected(false);
        setVisible(false);
        delete this;
      }			
      break;
    }
  }
}

void CDatabaseItem::doubleClicked()
{
  if (isRefreshing)
    return;
  if (!m_isConnected)
    setConnected(true);	
  setOpen(true);
}

int CDatabaseItem::displayMenu(const QPoint &pos)
{
  if (isRefreshing)
    return -1;
  QPopupMenu *p_itemMenu = new QPopupMenu();
  Q_CHECK_PTR(p_itemMenu);
  
  if (!m_isConnected)      
    p_itemMenu->insertItem(getPixmapIcon("databaseConnectIcon"), QObject::tr("Connect"),
			   MENU_CONNECT);  
  else
  {    
    p_itemMenu->insertItem(getPixmapIcon("databaseDisconnectIcon"), QObject::tr("Disconnect"),
			   MENU_DISCONNECT); 
    p_itemMenu->insertSeparator();    
    p_itemMenu->insertItem(getPixmapIcon("sqlIcon"), QObject::tr("Query"), MENU_QUERY);
  }
  p_itemMenu->insertSeparator();  
  p_itemMenu->insertItem(getPixmapIcon("deleteIcon"), QObject::tr("Delete Database"),
			 MENU_DELETE);
  p_itemMenu->insertSeparator();  
  p_itemMenu->insertItem(getPixmapIcon("newWindowIcon"), QObject::tr("New Window from Here"),
			 NEW_WINDOW_FROM_HERE);
  
  int res = p_itemMenu->exec(pos);
  delete p_itemMenu;
  return res;
}

CDatabaseItemToolbar::CDatabaseItemToolbar(QMainWindow * parent)
: CToolBar(DATABASE_ITEM, QObject::tr("Database"), parent, "Database Toolbar")
{  
  addButton(MENU_CONNECT, getPixmapIcon("databaseConnectIcon"), QObject::tr("Connect"));
  addButton(MENU_DISCONNECT, getPixmapIcon("databaseDisconnectIcon"), QObject::tr("Disconnect"));
  insertSeparator();
  addButton(MENU_QUERY, getPixmapIcon("sqlIcon"), QObject::tr("Query"));
  insertSeparator();
  addButton(MENU_DELETE, getPixmapIcon("deleteIcon"), QObject::tr("Delete Database"));
  insertSeparator();
  addButton(NEW_WINDOW_FROM_HERE, getPixmapIcon("newWindowIcon"),
	    QObject::tr("Open in New Window"));
}

void CDatabaseItemToolbar::setConnectedToolbar(bool connected)
{
  Button(0)->setEnabled(!connected);
  Button(1)->setEnabled(connected);
  Button(2)->setEnabled(connected);
}

