/* Copyright (C) 2002 MySQL AB & Jorge del Conde

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
    
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA 
*/

#include "CDatabaseGroupItem.h"
#include "CDatabaseItem.h"

CDatabaseGroupItem::CDatabaseGroupItem(CMyWindow *consoleWindow, int serverid, CDatabaseTreeItem * parent, bool isopen, QWidgetStack *w)
: CDatabaseTreeItem(parent)
{  
  m_pMySQLServer = ((CDatabaseTreeItem *)this->parent())->m_pMySQLServer;	
  init_item(isopen, w, consoleWindow, serverid);
  deleteMySQL = false;
}

CDatabaseGroupItem::CDatabaseGroupItem(CMyWindow *consoleWindow, int serverid, QListView * parent, CMySQLConnection *mysql, QWidgetStack *w)
: CDatabaseTreeItem(parent)
{
  m_pMySQLServer = mysql;
  m_pMySQLServer->setConnected(true);
  setOpen(true);
  deleteMySQL = true;
  init_item(true, w, consoleWindow, serverid);
}

CDatabaseGroupItem::~CDatabaseGroupItem()
{
  if (deleteMySQL)
    delete m_pMySQLServer;
  widgetStack->removeWidget(databaseGroupProperties);
  delete databaseGroupProperties;
}

void CDatabaseGroupItem::clicked()
{
  databaseGroupProperties->refresh();
  widgetStack->raiseWidget(widgetID);  
}

void CDatabaseGroupItem::init_item(bool isopen, QWidgetStack *w, CMyWindow *consoleWindow, int serverid)
{
  isRefreshing = false;
  serverID = serverid;
  ConsoleWindow = consoleWindow;
  setName("CDatabaseGroupItem");
  setType(DATABASE_GROUP);
  widgetStack = w;
  if (m_pMySQLServer->isConnected())
    mySetOpen(isopen);  
  setText( 0, QObject::tr("Databases"));

  databaseGroupProperties = new CShowDatabaseGroupProperties((QWidget *) widgetStack, m_pMySQLServer);
  Q_CHECK_PTR(databaseGroupProperties);
  widgetID = getWidgetID();
  widgetStack->addWidget (databaseGroupProperties, widgetID);

  addDatabases(false);  
}

void CDatabaseGroupItem::addDatabases(bool isRefresh)
{
  if (isRefreshing)
    return;
  isRefreshing = true;
  listView()->setCursor(Qt::waitCursor);
  QStringList dbList;  
  CMySQLQuery::ColumnQuery(dbList, m_pMySQLServer, "SHOW DATABASES", 0, true);
  if (isRefresh)
  {
    
    QPtrList<CDatabaseTreeItem> RemovedDatabasesList;
    
    QListViewItemIterator c( firstChild() );
    for ( ; c.current(); ++c )
    {
      if (c.current()->parent() == this)
      {
        if (dbList.findIndex(c.current()->text(0)) != -1)
          dbList.remove (c.current()->text(0));
        else
        {
          CDatabaseTreeItem *p = (CDatabaseTreeItem *)c.current();
          p->setConnected(false);
          p->setVisible(false);
          RemovedDatabasesList.append(p);
        }
      }
    }
    if (!RemovedDatabasesList.isEmpty())
    {
      CDatabaseTreeItem *p;
      for (p = RemovedDatabasesList.first(); p; p = RemovedDatabasesList.next())
        delete p;
    }
  }
  if (!dbList.isEmpty())
  {
    QStringList::Iterator i = dbList.begin();
    while ( i != dbList.end() )
    {
      new CDatabaseItem(ConsoleWindow, serverID, this, (*i), false, widgetStack);
      i++;
    }
  }
  listView()->setCursor(Qt::ArrowCursor);
  isRefreshing = false;
}

void CDatabaseGroupItem::expanded()
{
  mySetOpen(true);	
}

void CDatabaseGroupItem::collapsed()
{
  mySetOpen(false);
}

void CDatabaseGroupItem::mySetOpen(bool isopen)
{
  m_isOpen = isopen;  
  setPixmap(0, getPixmapIcon(!m_isOpen ? "closedFolderIcon" : "openFolderIcon"));
}

void CDatabaseGroupItem::processMenu(int res)
{
  if (isRefreshing)
    return;
  switch (res) {
		case MENU_NEW: {
      bool ok = false;
      QString dbname = QInputDialog::getText(QObject::tr("Create Database"),
				   QObject::tr("Enter new Database name") +
					     ":",
        QLineEdit::Normal, QString::null, &ok, 0, getName());      
      if ( ok && !dbname.isEmpty())
      {
        CMySQLQuery query(m_pMySQLServer);
        QString qry = "CREATE DATABASE IF NOT EXISTS " + dbname;
        if (query.exec_static(qry))
          new CDatabaseItem(ConsoleWindow, serverID, this, dbname, false, widgetStack);
      }
      break;
            }

    case MENU_REFRESH: addDatabases(true);
      break;
  }
}

int CDatabaseGroupItem::displayMenu(const QPoint &pos)
{
  if (isRefreshing)
    return -1;
  QPopupMenu *p_itemMenu = new QPopupMenu();
  Q_CHECK_PTR(p_itemMenu);  
  p_itemMenu->insertItem(getPixmapIcon("newDatabaseIcon"), QObject::tr("New Database"), MENU_NEW);  
  p_itemMenu->insertSeparator();    
  p_itemMenu->insertItem(getPixmapIcon("refreshIcon"), QObject::tr("Refresh"), MENU_REFRESH);  
  p_itemMenu->insertSeparator();  
  p_itemMenu->insertItem(getPixmapIcon("newWindowIcon"), QObject::tr("New Window from Here"),
			 NEW_WINDOW_FROM_HERE);
  int res =  p_itemMenu->exec(pos);
  delete p_itemMenu;
  return res;
}

CDatabaseGroupToolbar::CDatabaseGroupToolbar(QMainWindow * parent)
 :CToolBar(DATABASE_GROUP, QObject::tr("Database Group"), parent,
	   "Database Group Toolbar")
{  
  addButton(MENU_NEW, getPixmapIcon("newDatabaseIcon"), QObject::tr("New Database"));
  insertSeparator();
  addButton(MENU_REFRESH, getPixmapIcon("refreshIcon"), QObject::tr("Refresh"));  
  insertSeparator();
  addButton(NEW_WINDOW_FROM_HERE, getPixmapIcon("newWindowIcon"),
	    QObject::tr("Open in New Window"));
}
