/* Copyright (C) 2002 MySQL AB & Jorge del Conde

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
    
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA 
*/

#include "CConnectionDialog.h"
#include <qfile.h>
#include <qfiledialog.h>

p_SaveOptions::p_SaveOptions(QWidget * parent, const char *name, WFlags fl)
  : privateTabConfig(parent, name, fl)
{  
  if ( !name )
    setName( "p_SaveOptions" );
  resize( 464, 171 ); 
  setCaption(tr("Save Options" ) );
  p_SaveOptionsLayout = new QVBoxLayout(this, 4, 2, "p_SaveOptionsLayout"); 
  Q_CHECK_PTR(p_SaveOptionsLayout);
  
  Layout20 = new QGridLayout( 0, 1, 1, 0, 2, "Layout20"); 
  QSpacerItem* spacer = new QSpacerItem( 20, 20, QSizePolicy::Expanding, QSizePolicy::Minimum );
  Q_CHECK_PTR(spacer);
  Layout20->addItem( spacer, 2, 2 );
  QSpacerItem* spacer_2 = new QSpacerItem( 20, 20, QSizePolicy::Expanding, QSizePolicy::Minimum );
  Q_CHECK_PTR(spacer_2);
  Layout20->addItem( spacer_2, 0, 2 );
  
  Separated = new QLineEdit(this, "Separated" );
  Q_CHECK_PTR(Separated);
  Separated->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)4, (QSizePolicy::SizeType)0, 0, 0, Separated->sizePolicy().hasHeightForWidth() ) );
  QWhatsThis::add( Separated,tr("The specified character will be used to Separate fields whenever you save a text file." ) );
  
  Layout20->addWidget( Separated, 0, 1 );
  Q_CHECK_PTR(Layout20);
  QSpacerItem* spacer_3 = new QSpacerItem( 20, 20, QSizePolicy::Expanding, QSizePolicy::Minimum );
  Layout20->addItem( spacer_3, 1, 2 );
  
  Terminated = new QLineEdit(this, "Terminated" );
  Q_CHECK_PTR(Terminated);
  Terminated->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)1, (QSizePolicy::SizeType)0, 0, 0, Terminated->sizePolicy().hasHeightForWidth() ) );
  QWhatsThis::add( Terminated,tr("The specified character will be appended to all the lines whenever you save a text file." ) );
  
  Layout20->addWidget( Terminated, 2, 1 );
  
  separateLable = new QLabel(this, "separateLable" );
  Q_CHECK_PTR(separateLable);
  separateLable->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)0, (QSizePolicy::SizeType)5, 0, 0, separateLable->sizePolicy().hasHeightForWidth() ) );
  separateLable->setText(tr("Fields Separated by" ) );
  
  Layout20->addWidget( separateLable, 0, 0 );
  
  enclosedLabel = new QLabel(this, "enclosedLabel" );
  Q_CHECK_PTR(enclosedLabel);
  enclosedLabel->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)0, (QSizePolicy::SizeType)5, 0, 0, enclosedLabel->sizePolicy().hasHeightForWidth() ) );
  enclosedLabel->setText(tr("Fields enclosed by" ) );
  
  Layout20->addWidget( enclosedLabel, 1, 0 );
  
  terminatedLabel = new QLabel(this, "terminatedLabel" );
  Q_CHECK_PTR(terminatedLabel);
  terminatedLabel->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)0, (QSizePolicy::SizeType)5, 0, 0, terminatedLabel->sizePolicy().hasHeightForWidth() ) );
  terminatedLabel->setText(tr("Lines terminated by" ) );
  
  Layout20->addWidget( terminatedLabel, 2, 0 );
  
  Enclosed = new QLineEdit(this, "Enclosed" );
  Q_CHECK_PTR(Enclosed);
  Enclosed->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)1, (QSizePolicy::SizeType)0, 0, 0, Enclosed->sizePolicy().hasHeightForWidth() ) );
  QWhatsThis::add( Enclosed,tr("The specified character will be used to Enclose each field whenever you save a text file." ) );
  
  Layout20->addWidget( Enclosed, 1, 1 );
  p_SaveOptionsLayout->addLayout( Layout20 );
  

  replaceEmptyLabel = new QLabel(this, "replaceEmptyLabel" );
  Q_CHECK_PTR(replaceEmptyLabel);
  replaceEmptyLabel->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)0, (QSizePolicy::SizeType)5, 0, 0, replaceEmptyLabel->sizePolicy().hasHeightForWidth() ) );
  replaceEmptyLabel->setText(tr("Replace emtpy fields with" ) );


  Layout20->addWidget(replaceEmptyLabel, 3, 0 );

  ReplaceEmpty = new QLineEdit(this, "ReplaceEmpty" );
  Q_CHECK_PTR(ReplaceEmpty);
  ReplaceEmpty->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)1, (QSizePolicy::SizeType)0, 0, 0, ReplaceEmpty->sizePolicy().hasHeightForWidth() ) );
  QWhatsThis::add(ReplaceEmpty,tr("All Empty fields will be replaced with what you type in this box for whenever you save a text file." ) );
  Layout20->addWidget(ReplaceEmpty, 3, 1 );

  TextLabel4 = new QLabel(this, "TextLabel4" );
  Q_CHECK_PTR(TextLabel4);
  TextLabel4->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)5, (QSizePolicy::SizeType)0, 0, 0, TextLabel4->sizePolicy().hasHeightForWidth() ) );
  TextLabel4->setMargin( 3 );
  TextLabel4->setText(tr("You can use the following escaped symbols to represent certain escaped characters:<br>\n"
    "<b>\\n</b>    New Line<br>\n"
    "<b>\\r</b>     Carriege Return<br>\n"
    "<b>\\t</b>     Tab" ) );
  TextLabel4->setAlignment( int( QLabel::WordBreak | QLabel::AlignTop ) );
  p_SaveOptionsLayout->addWidget( TextLabel4 );
  QSpacerItem* spacer_4 = new QSpacerItem( 20, 20, QSizePolicy::Minimum, QSizePolicy::Expanding );
  Q_CHECK_PTR(spacer_4);
  p_SaveOptionsLayout->addItem( spacer_4 );

  Separated->setText(",");
  Terminated->setText("\\r\\n");
  Enclosed->setText("'");  

  setTabOrder( Separated, Enclosed );
  setTabOrder( Enclosed, Terminated );
  setTabOrder( Terminated, ReplaceEmpty );
}

bool p_SaveOptions::save(CConfig *conn)
{
  bool ret = conn->writeEntry("Field Separate", Separated->text().stripWhiteSpace());  
  ret &= conn->writeEntry("Line Terminate", Terminated->text().stripWhiteSpace());
  ret &= conn->writeEntry("Field Enclosed", Enclosed->text().stripWhiteSpace());  
  ret &= conn->writeEntry("Replace Empty", ReplaceEmpty->text().stripWhiteSpace());
  return ret;
}

void p_SaveOptions::setDefaultValues(CConfig *conn)
{  
  Separated->setText(conn->readStringEntry("Field Separate", ","));
  Terminated->setText(conn->readStringEntry("Line Terminate", "\\r\\n"));
  Enclosed->setText(conn->readStringEntry("Field Enclosed", "'"));
  ReplaceEmpty->setText(conn->readStringEntry("Replace Emtpy", QString::null));
}

p_GeneralConnection::p_GeneralConnection(bool isediting, QWidget * parent, const char *name, WFlags fl)
  : privateTabConfig(parent, name, fl), isEditing(isediting)
{  
  if ( !name )
    setName( "p_GeneralConnection" );
  setCaption(tr("General" ) );
  p_GeneralConnectionLayout = new QVBoxLayout( this, 4, 2, "p_GeneralConnectionLayout");
  Q_CHECK_PTR(p_GeneralConnectionLayout);
  
  Layout15 = new QGridLayout( 0, 1, 1, 0, 2, "Layout15");
  Q_CHECK_PTR(Layout15);
  
  PortBox = new QSpinBox( this, "PortBox" );
  Q_CHECK_PTR(PortBox);
  PortBox->setMaxValue( 100000 );
  PortBox->setMinValue( 100 );
  PortBox->setValue( 3306 );
  QWhatsThis::add( PortBox,tr("TCP/IP Port Number to which the Server is listening to." ) );
  
  Layout15->addWidget( PortBox, 4, 1 );
  if (isEditing)
  {
    connectionLabel = new QLabel( this, "connectionLabel" );
    Q_CHECK_PTR(connectionLabel);
    connectionLabel->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)0, (QSizePolicy::SizeType)5, 0, 0, connectionLabel->sizePolicy().hasHeightForWidth() ) );
    connectionLabel->setText(tr("Name" ) );
    
    Layout15->addWidget( connectionLabel, 0, 0 );
    
    ConnectionNameBox = new QLineEdit( this, "ConnectionNameBox" );
    Q_CHECK_PTR(ConnectionNameBox);
    QWhatsThis::add( ConnectionNameBox,tr("This is an alias to your connection.  Whatever you type here it will be displayed in the Left Tree." ) );
    
    Layout15->addMultiCellWidget( ConnectionNameBox, 0, 0, 1, 3 );
  }
  
  portLabel = new QLabel( this, "portLabel" );
  Q_CHECK_PTR(portLabel);
  portLabel->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)0, (QSizePolicy::SizeType)5, 0, 0, portLabel->sizePolicy().hasHeightForWidth() ) );
  portLabel->setText(tr("Port" ) );
  
  Layout15->addWidget( portLabel, 4, 0 );
  
  passwordLabel = new QLabel( this, "passwordLabel" );
  Q_CHECK_PTR(passwordLabel);
  passwordLabel->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)0, (QSizePolicy::SizeType)5, 0, 0, passwordLabel->sizePolicy().hasHeightForWidth() ) );
  passwordLabel->setText(tr("Password" ) );
  
  Layout15->addWidget( passwordLabel, 3, 0 );
  
  timeOutLabel = new QLabel( this, "timeOutLabel" );
  Q_CHECK_PTR(timeOutLabel);
  timeOutLabel->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)3, (QSizePolicy::SizeType)5, 0, 0, timeOutLabel->sizePolicy().hasHeightForWidth() ) );
  timeOutLabel->setText(tr("Connect Timeout (sec)" ) );
  timeOutLabel->setAlignment( int( QLabel::AlignVCenter | QLabel::AlignRight ) );
  
  Layout15->addWidget( timeOutLabel, 4, 2 );
  
  userNameLabel = new QLabel( this, "userNameLabel" );
  Q_CHECK_PTR(userNameLabel);
  userNameLabel->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)0, (QSizePolicy::SizeType)5, 0, 0, userNameLabel->sizePolicy().hasHeightForWidth() ) );
  userNameLabel->setText(tr("User Name" ) );
  
  Layout15->addWidget( userNameLabel, 2, 0 );
  
  HostNameBox = new QLineEdit( this, "HostNameBox" );
  Q_CHECK_PTR(HostNameBox);
  QWhatsThis::add( HostNameBox,tr("Specifies the Database Server you'll be connecting to.  A valid Host Name or IP address is required to establish a connection." ) );
  
  Layout15->addMultiCellWidget( HostNameBox, 1, 1, 1, 3 );
  
  PasswordBox = new QLineEdit( this, "PasswordBox" );
  Q_CHECK_PTR(PasswordBox);
  PasswordBox->setEchoMode( QLineEdit::Password );
  QWhatsThis::add( PasswordBox,tr("Specifies the Password associated with the User Name which will be used to connect to the Server." ) );
  
  Layout15->addMultiCellWidget( PasswordBox, 3, 3, 1, 3 );
  
  connectTimeout = new QSpinBox( this, "connectTimeout" );
  Q_CHECK_PTR(connectTimeout);
  QWhatsThis::add(connectTimeout, tr("Specifies the amount of time the client should wait to receive a response from the server." ) );
  connectTimeout->setMaxValue( 300 );
  connectTimeout->setMinValue( 0 );
  connectTimeout->setValue( 30 );
  
  Layout15->addWidget( connectTimeout, 4, 3 );
  
  hostLabel = new QLabel( this, "hostLabel" );
  Q_CHECK_PTR(hostLabel);
  hostLabel->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)0, (QSizePolicy::SizeType)5, 0, 0, hostLabel->sizePolicy().hasHeightForWidth() ) );
  hostLabel->setText(tr("Host Name" ) );
  
  Layout15->addWidget( hostLabel, 1, 0 );
  
  UserNameBox = new QLineEdit( this, "UserNameBox" );
  Q_CHECK_PTR(UserNameBox);
  QWhatsThis::add( UserNameBox,tr("Specifies the User Name which will be used to connect to the Server." ) );
  
  Layout15->addMultiCellWidget( UserNameBox, 2, 2, 1, 3 );
  p_GeneralConnectionLayout->addLayout( Layout15 );
  
  tableRetrievalGroup = new QButtonGroup( this, "tableRetrievalGroup" );
  tableRetrievalGroup->setTitle(tr("Table retrieval method" ) );
  tableRetrievalGroup->setColumnLayout(0, Qt::Vertical );
  tableRetrievalGroup->layout()->setSpacing( 2 );
  tableRetrievalGroup->layout()->setMargin( 4 );
  tableRetrievalGroupLayout = new QGridLayout( tableRetrievalGroup->layout() );
  tableRetrievalGroupLayout->setAlignment( Qt::AlignTop );

  retrieveShowTableStatus = new QRadioButton( tableRetrievalGroup, "retrieveShowTableStatus" );
  retrieveShowTableStatus->setText(tr("SHOW TABLE STATUS" ) );
  QWhatsThis::add( retrieveShowTableStatus,tr("This method will make the GUI better looking but it will be slower than SHOW TABLES.  This option is recommended if the server you'll be connecting to doesn't have databases with many tables (<= 300)." ) );
  if (!isEditing)
    retrieveShowTableStatus->setChecked(true);

  tableRetrievalGroupLayout->addWidget( retrieveShowTableStatus, 0, 0 );

  retrieveShowTables = new QRadioButton( tableRetrievalGroup, "retrieveShowTables" );
  retrieveShowTables->setText(tr("SHOW TABLES" ) );
  QWhatsThis::add( retrieveShowTables,tr("This method is recommended when you're connecting to a server where the databases have many tables ( > 300).  It's less appealing in terms of visual effects as opposed to SHOW TABLE STATUS, but it will retrieve tables much faster.  This method will be used automatically when connecting to MySQL Servers < 3.23." ) );

  tableRetrievalGroupLayout->addWidget( retrieveShowTables, 0, 2 );
  QSpacerItem* spacer = new QSpacerItem( 20, 20, QSizePolicy::Expanding, QSizePolicy::Minimum );
  tableRetrievalGroupLayout->addItem( spacer, 0, 1 );

  Layout15->addMultiCellWidget( tableRetrievalGroup, 5, 5, 0, 3 );

  Options = new QGroupBox( this, "Options" );
  Q_CHECK_PTR(Options);
  Options->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)5, (QSizePolicy::SizeType)0, 0, 0, Options->sizePolicy().hasHeightForWidth() ) );
  Options->setTitle(tr("Options" ) );
  Options->setColumnLayout(0, Qt::Vertical );
  Options->layout()->setSpacing( 2 );
  Options->layout()->setMargin( 4 );
  OptionsLayout = new QGridLayout( Options->layout() );
  Q_CHECK_PTR(OptionsLayout);
  OptionsLayout->setAlignment( Qt::AlignTop );
  
  blockingCheckBox = new QCheckBox( Options, "blockingCheckBox" );
  Q_CHECK_PTR(blockingCheckBox);
  blockingCheckBox->setText(tr("Blocking Queries" ) );
  QWhatsThis::add( blockingCheckBox,tr("Use Blocking Queries when enabled.  This option will allow the user to keep doing things while processing a query.  If disabled, the application will block until the query is concluded. If you're unsure about this option, leave it disabled." ) );
  
  OptionsLayout->addMultiCellWidget( blockingCheckBox, 1, 1, 0, 1 );
  
  socketLabel = new QLabel( Options, "socketLabel" );
  Q_CHECK_PTR(socketLabel);
  socketLabel->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)0, (QSizePolicy::SizeType)5, 0, 0, socketLabel->sizePolicy().hasHeightForWidth() ) );
  socketLabel->setText(tr("Socket File" ) );
  
  OptionsLayout->addWidget( socketLabel, 3, 0 );
  
  socketFile = new QLineEdit( Options, "socketFile" );
  Q_CHECK_PTR(socketFile);
  socketFile->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)7, (QSizePolicy::SizeType)0, 0, 0, socketFile->sizePolicy().hasHeightForWidth() ) );
  QWhatsThis::add( socketFile,tr("Use a Unix Socket File instead of a TCP/IP connection to the Server.  This option will override the Host Name and Port." ) );
  
  OptionsLayout->addMultiCellWidget( socketFile, 3, 3, 1, 2 );
  
  socketBrowse = new QPushButton( Options, "socketBrowse" );
  Q_CHECK_PTR(socketBrowse);
  socketBrowse->setMinimumSize( QSize( 22, 22 ) );
  socketBrowse->setMaximumSize( QSize( 22, 22 ) );
  socketBrowse->setText(tr("" ) );
  socketBrowse->setPixmap(getPixmapIcon("openIcon"));
  QWhatsThis::add( socketBrowse,tr("Click to browse for a Socket File" ) );
  
  OptionsLayout->addWidget( socketBrowse, 3, 3 );
  
  oneConnectionCheckBox = new QCheckBox( Options, "oneConnectionCheckBox" );
  Q_CHECK_PTR(oneConnectionCheckBox);
  oneConnectionCheckBox->setText(tr("Use a single connection" ) );
  QWhatsThis::add( oneConnectionCheckBox,tr("Enable this option if you only want to use One connection rather than having MyCC use as many as required.  This option is recommended for when connecting to a high-load server." ) );
  
  OptionsLayout->addMultiCellWidget( oneConnectionCheckBox, 1, 1, 2, 3 );
  
  enableCompletionCheckBox = new QCheckBox( Options, "enableCompletionCheckBox" );
  Q_CHECK_PTR(enableCompletionCheckBox);
  enableCompletionCheckBox->setText(tr("Enable Completion and Syntax Highlighting" ) );
  QWhatsThis::add( enableCompletionCheckBox,tr("Enable Completion and Syntax Highlighting for the Database, Tables and Fields in this connection.") );    
  
  OptionsLayout->addMultiCellWidget( enableCompletionCheckBox, 2, 2, 0, 3 );
  
  compressCheckBox = new QCheckBox( Options, "compressCheckBox" );
  Q_CHECK_PTR(compressCheckBox);
  compressCheckBox->setText(tr("Use Compression" ) );
  QWhatsThis::add( compressCheckBox,tr("Use the compressed client/server protocol." ) );
  
  OptionsLayout->addMultiCellWidget( compressCheckBox, 0, 0, 0, 1 );
  
  promptPasswordCheckBox = new QCheckBox( Options, "promptPasswordCheckBox" );
  Q_CHECK_PTR(promptPasswordCheckBox);
  promptPasswordCheckBox->setText(tr("Prompt for Password" ) );
  QWhatsThis::add( promptPasswordCheckBox,tr("When enabled, a Password Dialog will appear before a connection to the Server is attempted.  This is useful for when you don't want to store your password in the settings file." ) );
  
  OptionsLayout->addMultiCellWidget( promptPasswordCheckBox, 0, 0, 2, 3 );
  p_GeneralConnectionLayout->addWidget( Options );
  QSpacerItem* spacer2 = new QSpacerItem( 20, 20, QSizePolicy::Minimum, QSizePolicy::Expanding );
  Q_CHECK_PTR(spacer2);
  p_GeneralConnectionLayout->addItem( spacer2 );
  
#ifdef WIN32
  socketLabel->setEnabled(false);
  socketBrowse->setEnabled(false);
  socketFile->setEnabled(false);
#endif
  if (isEditing)
     setTabOrder( ConnectionNameBox, HostNameBox );
  setTabOrder( HostNameBox, UserNameBox );
  setTabOrder( UserNameBox, PasswordBox );
  setTabOrder( PasswordBox, PortBox );
  setTabOrder( PortBox, connectTimeout );
  setTabOrder( connectTimeout, retrieveShowTableStatus );
  setTabOrder( retrieveShowTableStatus, retrieveShowTables );
  setTabOrder( retrieveShowTables, compressCheckBox );
  setTabOrder( compressCheckBox, promptPasswordCheckBox );
  setTabOrder( promptPasswordCheckBox, blockingCheckBox );
  setTabOrder( blockingCheckBox, oneConnectionCheckBox );
  setTabOrder( oneConnectionCheckBox, enableCompletionCheckBox );
  setTabOrder( enableCompletionCheckBox, socketFile );
  setTabOrder( socketFile, socketBrowse );
  init();
}

void p_GeneralConnection::setDefaultValues(CConfig *Settings)
{
  ConnectionNameBox->setText(mysql->getConnectionName());
  HostNameBox->setText(Settings->readStringEntry("Host"));
  UserNameBox->setText(Settings->readStringEntry("User"));
  PasswordBox->setText(Settings->readStringEntry("Password"));
  PortBox->setValue(Settings->readNumberEntry("Port"));
  connectTimeout->setValue(Settings->readNumberEntry("Timeout", 30));
  compressCheckBox->setChecked(Settings->readStringEntry("Compress", "false"));
  promptPasswordCheckBox->setChecked(strtobool(Settings->readStringEntry("Prompt Password", "false")));
  blockingCheckBox->setChecked(strtobool(Settings->readStringEntry("Blocking Queries")));
  oneConnectionCheckBox->setChecked(strtobool(Settings->readStringEntry("One Connection")));
  enableCompletionCheckBox->setChecked(strtobool(Settings->readStringEntry("Completion")));
  bool r = strtobool(Settings->readStringEntry("Retrieve Method", "true"));
  retrieveShowTableStatus->setChecked(r);
  retrieveShowTables->setChecked(!r);
  
#ifndef WIN32  
  socketFile->setText(Settings->readStringEntry("Socket"));
#endif
  if (mysql->isConnected())
  {
    HostNameBox->setEnabled(false);
    PortBox->setEnabled(false);
    if (socketFile->isEnabled())
      socketFile->setEnabled(false);

    if ((mysql->getMySQLServerVersion().major <= 3) && (mysql->getMySQLServerVersion().minor <= 22))
    {
      retrieveShowTableStatus->setChecked(false);
      retrieveShowTables->setChecked(true);
      tableRetrievalGroup->setEnabled(false);
    }
  }  
}

QString p_GeneralConnection::validate()
{  
  QString Err = QString::null;
  if (socketFile->text().isEmpty() && HostNameBox->text().isEmpty())
  {
    HostNameBox->setText("localhost");    
    if (socketFile->isEnabled())
      Err += " " + tr("or a Unix Socket");    
  }
  else
  {
    if (socketFile->text().isEmpty() && !validateAlphaNum(HostNameBox->text()))
      Err = hostLabel->text() + ", ";	
  
    if (!validateAlphaNum(UserNameBox->text()))
      Err += userNameLabel->text() + ", ";

    if (!Err.isEmpty())          
      Err = tr("The following fields contain invalid characters") + ": " + Err.left(Err.length()-2);    
  }
  return Err;
}

bool p_GeneralConnection::save(CConfig *conn)
{
  bool ret = conn->writeEntry("Host", HostNameBox->text().stripWhiteSpace() );  
  ret &= conn->writeEntry("Port", PortBox->value());
  ret &= conn->writeEntry("Timeout", connectTimeout->value());
  ret &= conn->writeEntry("User", UserNameBox->text().stripWhiteSpace());
  ret &= conn->writeEntry("Password", PasswordBox->text().stripWhiteSpace());
  ret &= conn->writeEntry("Compress", booltostr(compressCheckBox->isChecked()));
  ret &= conn->writeEntry("Prompt Password", booltostr(promptPasswordCheckBox->isChecked()));
  ret &= conn->writeEntry("Blocking Queries", booltostr(blockingCheckBox->isChecked()));
  ret &= conn->writeEntry("One Connection", booltostr(oneConnectionCheckBox->isChecked()));
  ret &= conn->writeEntry("Completion", booltostr(enableCompletionCheckBox->isChecked()));
  ret &= conn->writeEntry("Retrieve Method", booltostr(retrieveShowTableStatus->isChecked()));
#ifndef WIN32
  ret &= conn->writeEntry("Socket", socketFile->text().stripWhiteSpace());
#endif
  return ret;
}

void p_GeneralConnection::setSocketFile()
{
  socketFile->setText(
	QFileDialog::getOpenFileName( "", tr("All Files (*)"), this, "setSocketFile", tr("Select the MySQL Socket file")) );
}

void p_GeneralConnection::init()
{  
  connect(socketBrowse, SIGNAL(clicked()), this, SLOT(setSocketFile())); 
}

void CConnectionDialog::initConnectionDialog(CMyWindow *consoleWindow)
{
  ConsoleWindow = consoleWindow;  
  setCaption( QString(CODENAME) + " - " + tr("Database Connection Dialog"));
  if (g_isMDI)
    move(mapFromGlobal(QCursor::pos()).x(), mapFromGlobal(QCursor::pos()).y());
  else
    move(QCursor::pos().x(), QCursor::pos().y());

  //Tabs
  General = new p_GeneralConnection(isEditing, Tab, "General");  
  insertTab(General);  
  insertTab(new p_SaveOptions(Tab, "SaveOptions"));  
  customButton->setText(tr( "&Test"));
  customButton->setIconSet(getPixmapIcon("pingIcon"));
  QWhatsThis::add(customButton, tr("Click here to test the connection."));
  connect(customButton, SIGNAL(clicked()), this, SLOT(testConnection()));
  customButton->show();

  if (isEditing)
  {
    General->mysql = mysql;
    cfgname = mysql->getConnectionName();
    CConfig *Settings = new CConfig(cfgname, g_CONNECTION_PATH);
    Q_CHECK_PTR(Settings);
    Settings->prepare();
    setDefaultValues(Settings);
    delete Settings;    
    okPushButton->setText(tr( "&Apply"));
    QWhatsThis::add(okPushButton, tr("Click to Apply changes to this connection." ) );
  }
  else
  {
    okPushButton->setText(tr("&Add"));
    QWhatsThis::add(okPushButton, tr("Click to Add a new connection." ) );  
  }
  int h = (isEditing) ? 323 : 302;
  setMinimumSize( QSize( 0, h ) );
  myResize(421, h);
}

CConnectionDialog::CConnectionDialog(CMyWindow *consoleWindow, QWidget* parent,  const char* name)
:CConfigDialog( parent, name), isEditing(false)
{
  if ( !name )
    setName( "CConnectionDialog" );  
  initConnectionDialog(consoleWindow);
}

CConnectionDialog::CConnectionDialog(CMyWindow *consoleWindow, QWidget* parent,  const char* name, CMySQLConnection *conn, CDatabaseTreeItem *it, int serverid)
:CConfigDialog(parent, name), isEditing(true), serverID(serverid)
{
  if ( !name )
    setName( "CConnectionDialogEdit" );  
  mysql = conn;
  item = it;  
  initConnectionDialog(consoleWindow);
}

CConnectionDialog::~CConnectionDialog()
{
}

void CConnectionDialog::testConnection()
{
  if (validate())
  {
    QString tmpConnectionName = "mycctmp";
    if (!tempFileName(CConfig::getMyCCPath() + "/tmp/", tmpConnectionName))
    {
      ConsoleWindow->messagePanel()->Critical(tr("An error occured while creating a temporary file name."));
      return;
    }    
    CConfig connection(tmpConnectionName, "tmp");
    if (!save(&connection))
    {
      ConsoleWindow->messagePanel()->Critical(tr("An error occured while saving the temporary connection."));      
      return;
    }    
    CMySQLConnection *testConn = new CMySQLConnection(tmpConnectionName, ConsoleWindow->messagePanel(), "tmp");    
    if (testConn->connect(QString::null, false))
      ConsoleWindow->messagePanel()->Information(tr("Connection Successful."));
    delete testConn;
    connection.remove();
  }
}

void CConnectionDialog::okButtonClicked()
{
  if (validate())
  {	
    QString tmpConnectionName;
    if (isEditing)
      tmpConnectionName = General->ConnectionNameBox->text();
    else
    {
      if (General->HostNameBox->text().isEmpty() && !General->socketFile->text().isEmpty())  //socket connection
         General->HostNameBox->setText("localhost");

       tmpConnectionName = General->UserNameBox->text() + "@" + General->HostNameBox->text() + ";";
       tmpConnectionName += (General->socketFile->text().isEmpty()) ? QString::number(General->PortBox->value()) : getFileName(General->socketFile->text());
    }

    CConfig connection(tmpConnectionName, g_CONNECTION_PATH);

    if (!isEditing && connection.exists())
    {
      for (unsigned int i = 1; i; i++)
      {
        connection.setConfingName(tmpConnectionName + "_" + QString::number(i));        
        if (!connection.exists())
          break;
        qApp->processEvents();
      }
      tmpConnectionName = connection.getConfigName();
    }
    else
      if (isEditing && (mysql->getConnectionName() != tmpConnectionName) && connection.exists())
      {        
        if (tmpConnectionName.isEmpty())
          ConsoleWindow->messagePanel()->Warning(tr("You must enter a valid Connection Name."));
        else
          ConsoleWindow->messagePanel()->Critical(tr("A Connection with that name already exists."));        
        return;     
      }

    if (!save(&connection))
    {
      ConsoleWindow->messagePanel()->Critical(tr("An Error ocurred while saving the Connection."));
      return;
    }

    if (isEditing)
    {
      if (mysql->getConnectionName() != tmpConnectionName)
      {
        CConfig::Remove(cfgname, g_CONNECTION_PATH);
        emit connectionRenamed(item, tmpConnectionName);
      }
      QString *o = g_ServersDict->find(serverID);      
      g_ServersDict->replace(serverID, new QString(tmpConnectionName));
      delete o;
      mysql->setConnectionName(tmpConnectionName);
      mysql->update();
      ConsoleWindow->messagePanel()->Information(tr("Connection edited successfully."));
    }
    else
    {
      ConsoleWindow->messagePanel()->Information(tr("Connection added successfully."));
      emit newConnection(false);
    }
    close();
  }
}

bool CConnectionDialog::validate()
{
  QString Err = CConfigDialog::validate();  
  if (!Err.isEmpty())
  {    
    ConsoleWindow->messagePanel()->Warning(Err);
    return false;
  }  
  return true;
}

bool CConnectionDialog::save(CConfig *conn)
{  
  bool ret = conn->prepare();  
  ret &= CConfigDialog::save(conn, false);  
  ret &= conn->writeEntry("Modified", QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss"));  
  if (!isEditing)
    ret &= conn->writeEntry("Created", QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss"));  
  ret &= conn->flush();  
  return ret;
}


