/* Copyright (C) 2002 MySQL AB & Jorge del Conde

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
    
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA 
*/

#include "CConfig.h"
#include "Globals.h"
#include "Config.h"
#include <qdir.h>
#include <qfile.h>
#include <qmessagebox.h>

QString CConfig::Path;

void CConfig::list(QStringList &list, const QString &path)
{
#ifdef DEBUG
    qDebug( "CConfig::list(QString)");
#endif  
  Path = path;
  QDir dir = QDir(getConfigPath());
  const QFileInfoList *filist = dir.entryInfoList( QDir::DefaultFilter, QDir::DirsFirst | QDir::Name );
  QFileInfoListIterator it( *filist );
  QFileInfo *fi;    
  while ( ( fi = it.current() ) != 0 )
  {
    ++it;
    if ( fi && fi->fileName() != ".." && fi->fileName() != ".")
      list.append(charReplace(fi->fileName(), ';', ":"));			
  }  
}

bool CConfig::exists(const QString & name)
{
  return QFile::exists(getConfigPath() + charReplace(name, ':', ";"));
}

QString CConfig::getMyCCPath()
{
  QString myccPath = QDir::homeDirPath();
  if (myccPath.isNull())
  {
    QString errStr = "Config directory does not exist.\n";
#ifndef WIN32
    errStr += "Is the HOME environment variable set?\nDoes $HOME directory have write/read permissions ?";
#else
    errStr += "Does the directory " + charReplace(QDir::currentDirPath(), '/', "\\") + " exist ?";
#endif
    QMessageBox::critical(0,"Error",errStr);
    return QString();
  }
  myccPath += "/." + QString(CODENAME);
  QDir cfg = QDir(myccPath);
  if (!cfg.exists())
  {
#ifdef DEBUG
    qDebug( "CConfig::init() - " + myccPath + " does not exist ... trying to create it.");
#endif
    cfg.mkdir (myccPath);
  }
  return myccPath + "/";
}

QString CConfig::getConfigPath()
{
  QString configPath = getMyCCPath() + Path;
  QDir cfgCon = QDir(configPath);
  if (!cfgCon.exists())
  {
#ifdef DEBUG
    qDebug( "CConfig::init() - " + configPath + " does not exist ... trying to create it.");
#endif
    cfgCon.mkdir (configPath);
  }
  
  if (!cfgCon.exists())
  {
    QMessageBox::critical(0,"Error", "The path " + configPath + " could not be read.");
    return QString::null;
  }
  configPath += "/";
  return configPath;
}


CConfig::CConfig(const QString &name, const QString & path)
{
  configName = charReplace(name, ':', ";");
  Path = path;
  entries.setAutoDelete(true);
  entries.resize(60);
}

CConfig::~CConfig()
{	
}

bool CConfig::writeEntry(const QString & key, int value)
{
  return writeEntry(key, QString::number(value));  
}

bool CConfig::prepare()
{
  QFile f_strm(getConfigPath() + charReplace(configName, ':', ";"));
  if (f_strm.open(IO_ReadWrite))
  {
    QTextStream t_strm(&f_strm);
    if (!entries.isEmpty())
      entries.clear();
    uint i = 1;
    while ( !t_strm.atEnd() )
    {
      QString str = t_strm.readLine();    
      if (str.contains('=') == 1)  //is this a valid entry ?
      {
        int pos = str.find('=');
        QString key = str.left(pos).stripWhiteSpace();
        if (++i >= entries.size() - 1)
          entries.resize(i+5);
        entries.insert(key, new QString(str.mid(pos + 1).stripWhiteSpace()));
      }    
    }
    f_strm.close();
    return true;
  }
  else
    return false;
}

bool CConfig::flush()
{
  QFile f_strm(getConfigPath() + charReplace(configName, ':', ";"));
  if (f_strm.open(IO_WriteOnly))
  {
    QTextStream t_strm(&f_strm);
    QDictIterator<QString> it(entries);
    for( ; it.current(); ++it )
      t_strm << it.currentKey() << "\t=\t" << *it.current() << "\r\n";
    f_strm.close();
    return true;    
  }
  else
    return false;  
}

bool CConfig::writeEntry(const QString & key, const QString & value)
{   
  if (entries.find(key))
    entries.remove( key );
  else
    entries.resize(entries.size() + 1);
  entries.insert( key, new QString(value)); 
  return true;
}

int CConfig::readNumberEntry(const QString & key, int def)
{
  return readStringEntry(key, QString::number(def)).toInt();
}

QString CConfig::readStringEntry(const QString & key, const QString & def)
{  
  return (entries.isEmpty() ? def : entries.find(key) ? *entries.find(key) : def);
}

bool CConfig::exists()
{
  return QFile::exists(getConfigPath() + charReplace(configName, ':', ";"));
}

bool CConfig::remove()
{	
  return QFile::remove(getConfigPath() + charReplace(configName, ':', ";"));
}

void CConfig::Remove(const QString & name, const QString & path)
{
  CConfig connection(name, path);
  connection.remove();
}

