/* Copyright (C) 2002 MySQL AB & Jorge del Conde

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
    
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
  MA 02111-1307, USA 
*/

#include "CAdministrationPanel.h"
#include "CQueryWindow.h"
#include "CTextFieldEditor.h"
#include <qmessagebox.h>

privateGrid::privateGrid(QWidget * parent, CMySQLConnection *conn)
:CSqlTable(parent, conn), serverID(-1), create(true)
{
  setName("privateGrid");
  setAutoCloseEditors(false);
  columnSizes.setAutoDelete(true);
}

privateGrid::~privateGrid()
{
  closeEditors(true);
}

void privateGrid::finishEdit()
{  
  endEdit (currEditRow(), currEditCol(), true, false);
}

void privateGrid::reset()
{
  if (isExecuting)
    return;
  isExecuting = true;  
  for (int i = 0; i < numRows(); i++)
  {
    CCheckTableItem *tmp = (CCheckTableItem *) item (i, 0);
    if (tmp != NULL)
      delete tmp;
  }  
  isExecuting = false;
  CSqlTable::reset();
}

void privateGrid::exec(const QString & qry)
{
  if (isExecuting)
    return;  
  setCursor (Qt::waitCursor);  
  if (!create)
    for (int i = 0; i < numCols(); i++)    
      *columnSizes[i] = horizontalHeader()->sectionSize(i);    
  reset();
  isExecuting = true;
  setQuery(qry);
  if (Query->exec(m_Query, false))
  {		
    setNumRows(Query->numRows());
    setNumCols(Query->numFields());
    if (create)
    {
      columnSizes.clear();
      columnSizes.resize(Query->numFields());
    }    
    for (unsigned int i = 0; i < Query->numFields(); i++)
    {
      horizontalHeader()->setLabel(i, QString::fromLatin1(Query->Fields[i].name));
      setColumnReadOnly (i, (i != 0));
      if (create)
        columnSizes.insert(i, new int (horizontalHeader()->sectionSize(i)));      
      else
        horizontalHeader()->resizeSection (i, *columnSizes[i]);
    }
    create = false;
    unsigned long j = 0;
    while (Query->next())
    {      
      for (unsigned int i = 0; i < Query->numFields(); i++)
      {
        if (i == 0)
          setItem (j, i, new CCheckTableItem(this, Query->Row(i)));
         else        
          setText(j, i, Query->Row(i));        
      }
      j++;   
    }
  }
  Query->freeResult();
  isExecuting = false;  
  setCursor(Qt::ArrowCursor);
}

void privateGrid::DoubleClicked (int row, int col, int button, const QPoint &)
{
  if (isExecuting || (button != 1) || (col < 1))
    return;
  QString txt = text(row,col);
  if (txt.isEmpty() || txt.isNull())
    return;

  if (col == 7) //info field in show processlist
  {
    CQueryWindow *w = new CQueryWindow(g_WorkSpace, tr("Query"), m_pDatabaseConnection, txt, SQL_PANEL, serverID);     
    w->setFocus();
    w->raise();
    myShowWindow(w);
  }
  else
  {
     CTextFieldEditor *TextEditor = new CTextFieldEditor(g_WorkSpace, EditorList, "CTextFieldEditor", m_pDatabaseConnection, true);
     TextEditor->setCaption(tr("View Text"));
     TextEditor->setText(txt);
     TextEditor->setFocus();
     TextEditor->raise();
     myShowWindow(TextEditor);
  }
}

privateStatus::privateStatus(QWidget * parent, CMySQLConnection *conn)
:CSqlTable(parent, conn)
{
  setName("privateStatus");
  setReadOnly(true);
  enableColumnsWindow(true);
  setQuery("SHOW STATUS");
  reset();
  m_traditional = false;
}

privateStatus::~privateStatus()
{
}

void privateStatus::reset()
{
  if (isExecuting)
    return;
  CSqlTable::reset();
  m_firstTime = true;
}

void privateStatus::exec()
{
  if (isExecuting)
    return;  
  if (m_traditional)
  {
    if (ColumnsWindow->isVisible())
      ColumnsWindow->hide();
    CSqlTable::exec(false);
  }
  else
  {
    isExecuting = true;    
    setCursor (Qt::waitCursor);
    Query->freeResult();
    if (Query->exec(m_Query, false))
    {      
      if (m_firstTime)
      {
        columnWindowFirstTimeShown = false;
        if (isColumnsWindowEnabled())
          ColumnsWindow->Columns->clear();
        setNumRows(0);
        setNumCols(Query->numRows());
        headerDict.clear();
        headerDict.resize(Query->numRows());        
        unsigned long i = 0;
        while (Query->next(true, false))
        {
          horizontalHeader()->setLabel(i, getPixmapIcon("nothingIcon"), Query->Row(0));
          if (isColumnsWindowEnabled())
            new privateCheckListItem(ColumnsWindow->Columns, Query->Row(0), i);
          i++;
        }
        m_firstTime = false;
        Query->dataSeek(0);
      }
      setNumRows(numRows() + 1);      
      unsigned long i = 0;    
      while (Query->next())      
        setText(numRows() - 1, i++, Query->Row(1));
    }
    setCursor(Qt::ArrowCursor);
  }
  Query->freeResult();  
  isExecuting = false;
}

void privateStatus::setTraditionalMode(bool t, bool r)
{
  m_traditional = t;
  if (!m_traditional)
    m_firstTime = true;
  if (r)
    exec();
}

void privateStatus::processMenu(int res, int, int)
{
  switch (res)
  {
    case MENU_SAVE:
      save();
      break;
    case MENU_TRADITIONAL:      
      setTraditionalMode(!m_traditional);    
    break;
    case MENU_CLEAR_GRID:
      reset();
      break;
    case MENU_REFRESH:
      exec();
      break;
    case MENU_COLUMNS_WINDOW:
      if (isColumnsWindowEnabled())
      {
        if (!columnWindowFirstTimeShown)
        {
          ColumnsWindow->move(QCursor::pos().x(), QCursor::pos().y());
          columnWindowFirstTimeShown = true;
        }
        ColumnsWindow->show();       
        break;
      }
  }
}

int privateStatus::displayMenu(const QPoint &pos)
{
  QPopupMenu *p_itemMenu = new QPopupMenu();

  if (isColumnsWindowEnabled())
  {    
    p_itemMenu->insertItem(getPixmapIcon("chooseFieldsIcon"), tr("Choose Fields"), MENU_COLUMNS_WINDOW);
    if (ColumnsWindow->isVisible() || m_traditional)
      p_itemMenu->setItemEnabled (MENU_COLUMNS_WINDOW, false);
    p_itemMenu->insertSeparator();
  }  
  p_itemMenu->insertItem(getPixmapIcon("saveIcon"), tr("Save"), MENU_SAVE);  
  p_itemMenu->insertSeparator();  
  p_itemMenu->insertItem(getPixmapIcon("refreshTablesIcon"), tr("Refresh"), MENU_REFRESH);
  p_itemMenu->insertSeparator();  
  p_itemMenu->insertItem(getPixmapIcon("clearGridIcon"), tr("Clear Grid"), MENU_CLEAR_GRID);
  p_itemMenu->setItemEnabled (MENU_CLEAR_GRID, !isExecuting);
  p_itemMenu->insertSeparator();
  QString t;
  if (m_traditional)
    t = tr("Extended Mode");
  else
    t = tr("Traditional Mode");
  p_itemMenu->insertItem(t, MENU_TRADITIONAL);
  int res = p_itemMenu->exec(pos);
  delete p_itemMenu;	  
  return res;  
}

privateVariables::privateVariables(QWidget* parent,  const QString & caption, const QString & sql, CMySQLConnection *m)
: CListViewProperties(parent, m, "privateVariables")
{
  details->addColumn(tr("Property" ) );
  details->addColumn(tr("Value" ) );
  details->header()->setClickEnabled(false, 1);
  m_Sql = sql;
  m_Caption = caption;
}

void privateVariables::refresh()
{
  title->setText(" " + m_Caption);
  CListViewProperties::refresh();
}

void privateVariables::insertData()
{  
  CMySQLQuery query(mysql);
  if (query.exec(m_Sql, false))
    while (query.next())
    {
      QString row0 = query.Row(0);      
      insertItem(row0, query.Row(1));
    }
}

CAdministrationPanel::CAdministrationPanel( QWidget* parent, int type, CMySQLConnection *m, int serverid)
: CMyWindow( parent, 0, WDestructiveClose, !g_isMDI, true)
{
  mysql = new CMySQLConnection(m->getConnectionName(), messagePanel());
  Q_CHECK_PTR(mysql);
  mysql->connect(m->getDatabaseName());
  messageWindow()->setCarriageReturn(mysql->getLineTerminator(true));
  (void)statusBar();   
  
  setCaption(tr("Administration Panel") + " [" + mysql->getConnectionName() + "]"); 
  setName( "CAdministrationPanel" );  
  setIcon( getPixmapIcon("applicationIcon") );
  
  setCentralWidget( new QWidget( this, "qt_central_widget" ) );
  CAdministrationPanelLayout = new QHBoxLayout( centralWidget() );
  Q_CHECK_PTR(CAdministrationPanelLayout);
  CAdministrationPanelLayout->setSpacing( 2 );
  CAdministrationPanelLayout->setMargin( 2 );
  QHBox * pTop = new QHBox(centralWidget());
  Q_CHECK_PTR(pTop);
  
  TabWidget = new QTabWidget(pTop, "TabWidget" );
  Q_CHECK_PTR(TabWidget);
  
  processList = new QWidget( TabWidget, "processList" );
  Q_CHECK_PTR(processList);
  processListLayout = new QHBoxLayout(processList, 2, 2, "processListLayout");
  Q_CHECK_PTR(processListLayout);
  ProcessList = new privateGrid(processList, mysql);
  ProcessList->setServerID(serverid);
  ProcessList->setSaveTitle(tr("Process List"));
  Q_CHECK_PTR(ProcessList);
  processListLayout->addWidget(ProcessList);
  
  TabWidget->insertTab(processList, getPixmapIcon("showProcessListIcon"), "  " + tr("Process List"), SHOW_PROCESSLIST);
  
  status = new QWidget( TabWidget, "status" );
  Q_CHECK_PTR(status);
  
  statusLayout = new QHBoxLayout(status, 2, 2, "statusLayout");
  Q_CHECK_PTR(statusLayout);
  
  Status = new privateStatus(status, mysql);
  Status->setSaveTitle(tr("Server Status"));
  Q_CHECK_PTR(Status);
  statusLayout->addWidget(Status);
  TabWidget->insertTab(status, getPixmapIcon("showStatusIcon"), "  " +tr("Status" ), SHOW_STATUS);
     
  variables = new QWidget( TabWidget, "variables" );
  Q_CHECK_PTR(variables);
  variablesLayout = new QHBoxLayout(variables, 2, 2, "variablesLayout");
  Q_CHECK_PTR(variablesLayout);
  Variables = new privateVariables(variables, tr("Server Variables"), "SHOW VARIABLES", mysql);
  Variables->setSaveTitle(tr("Server Variables"));
  Q_CHECK_PTR(Variables);
  variablesLayout->addWidget(Variables);
  
  TabWidget->insertTab(variables, getPixmapIcon("showVariablesIcon"), "  " +tr("Variables" ), SHOW_VARIABLES );
  
  CAdministrationPanelLayout->addWidget(pTop);
  ProcessList->setMinimumSize(10, 50);	
  actionCloseAction = new QAction( this, "actioinCloseAction" );
  Q_CHECK_PTR(actionCloseAction);
  actionCloseAction->setText(tr("Close", "" ) );
  actionCloseAction->setMenuText(tr("&Close", "" ) );
  actionCloseAction->setIconSet( QIconSet( getPixmapIcon("closeIcon") ) );
  actionCloseAction->setAccel( 0 );
  
  showMessagesAction = new QAction(this, "showMessagesAction" );
  Q_CHECK_PTR(showMessagesAction);
  showMessagesAction->setText(tr("Show Messages Panel", "" ) );  
  showMessagesAction->setMenuText(tr("Show &Messages Panel", "" ) );
  showMessagesAction->setAccel( 0 );
  showMessagesAction->setToggleAction(true);
  showMessagesAction->setOn(messageWindow()->isVisible());

  refreshAction = new QAction( this, "refreshAction" );
  Q_CHECK_PTR(refreshAction);
  refreshAction->setText(tr("Refresh"));
  refreshAction->setMenuText(tr("&Refresh", "" ) );
  refreshAction->setIconSet(QIconSet(getPixmapIcon("refreshIcon")));
  refreshAction->setAccel( 0 );
  
  killProcessAction = new QAction( this, "killProcessAction" );
  Q_CHECK_PTR(killProcessAction);
  killProcessAction->setText(tr("Kill Process"));
  killProcessAction->setMenuText(tr("&Kill Process", "" ) );
  killProcessAction->setIconSet( QIconSet(getPixmapIcon("killProcessIcon") ) );
  killProcessAction->setAccel( 0 );

  flushAction = new QAction( this, "flushAction");
  Q_CHECK_PTR(flushAction);
  flushAction->setText(tr("Flush"));
  flushAction->setMenuText(tr("&Flush") );
  flushAction->setIconSet( QIconSet(getPixmapIcon("flushIcon") ) );
  flushAction->setAccel( 0 );
  
  saveAction = new QAction( this, "saveAction");
  Q_CHECK_PTR(flushAction);
  saveAction->setText(tr("Save"));
  flushAction->setMenuText(tr("&Save") );
  saveAction->setIconSet(QIconSet(getPixmapIcon("saveIcon") ) );
  saveAction->setAccel( 0 );

  pingAction = new QAction( this, "pingAction" );
  Q_CHECK_PTR(pingAction);
  pingAction->setText(tr("Ping"));
  pingAction->setMenuText(tr("&Ping", "" ) );
  pingAction->setIconSet( QIconSet(getPixmapIcon("pingIcon")));
  pingAction->setAccel( 0 );

  shutdownAction = new QAction( this, "shutdownAction" );
  Q_CHECK_PTR(shutdownAction);
  shutdownAction->setText(tr("Shutdown"));
  shutdownAction->setMenuText(tr("&Shutdown"));
  shutdownAction->setIconSet( QIconSet(getPixmapIcon("serverShutdownIcon")));
  shutdownAction->setAccel( 0 );  

  timerAction = new QAction( this, "timerAction" );
  Q_CHECK_PTR(timerAction);
  timerAction->setText(tr("Start Timer", "" ) );  
  timerAction->setMenuText(tr("&Start Timer", "" ) );
  timerAction->setIconSet( QIconSet(getPixmapIcon("timerIcon")) );
  timerAction->setAccel( 0 );
  timerAction->setToggleAction(true);  

  commandToolBar = new QToolBar( "", this, Top );
  Q_CHECK_PTR(commandToolBar);
  commandToolBar->setLabel(tr("Command Toolbar"));  

  menubar = new QMenuBar(this, "menubar");
  Q_CHECK_PTR(menubar);
  actionMenu = new QPopupMenu(this);
  Q_CHECK_PTR(actionMenu);

  saveTypeMenu = new QPopupMenu(this);
  Q_CHECK_PTR(saveTypeMenu);
  saveTypeMenu->insertItem(tr("&All"), MENU_SAVE_ALL);
  saveTypeMenu->insertItem(getPixmapIcon("showProcessListIcon"), tr("&Process List"), MENU_SAVE_PROCESSLIST);
  saveTypeMenu->insertItem(getPixmapIcon("showStatusIcon"), tr("&Status"), MENU_SAVE_STATUS);
  saveTypeMenu->insertItem(getPixmapIcon("showVariablesIcon"), tr("&Variables"), MENU_SAVE_VARIABLES);
  actionMenu->insertItem(getPixmapIcon("saveIcon"),tr("Save"), saveTypeMenu);

  refreshAction->addTo(actionMenu);
  actionMenu->insertSeparator();
  timerAction->addTo(actionMenu);
  actionMenu->insertSeparator();
  showMessagesAction->addTo( actionMenu );
  actionMenu->insertSeparator();
  actionCloseAction->addTo( actionMenu );
  menubar->insertItem(tr("&Action"), actionMenu );

  commandMenu = new QPopupMenu(this);
  Q_CHECK_PTR(commandMenu);    

  saveTypeButton = new QToolButton(commandToolBar);
  Q_CHECK_PTR(saveTypeButton);
  saveTypeButton->setPopup(saveTypeMenu);
  saveTypeButton->setPixmap(getPixmapIcon("saveIcon"));
  saveTypeButton->setTextLabel("Save", true);
  saveTypeButton->setPopupDelay (0);
  

  killProcessAction->addTo(commandMenu);
  killProcessAction->addTo(commandToolBar); 

  flushTypeMenu = new QPopupMenu(this);
  Q_CHECK_PTR(flushTypeMenu);
  FlushMenu(flushTypeMenu);
  flushTypeButton = new QToolButton(commandToolBar);
  Q_CHECK_PTR(flushTypeButton);
  flushTypeButton->setPopup(flushTypeMenu);
  flushTypeButton->setPixmap(getPixmapIcon("flushIcon"));
  flushTypeButton->setTextLabel("Flush", true);
  flushTypeButton->setPopupDelay (0);    

  commandMenu->insertItem(getPixmapIcon("flushIcon"),tr("Flush"), flushTypeMenu);
  pingAction->addTo(commandToolBar);
  shutdownAction->addTo(commandToolBar);
  
  pingAction->addTo(commandMenu);
  shutdownAction->addTo(commandMenu);

  toolBar = new QToolBar( "", this, Top );
  Q_CHECK_PTR(toolBar);
  toolBar->setLabel(tr("Refresh Toolbar"));  

  refreshAction->addTo(toolBar);
  
  toolBar->addSeparator();
  
  delayLabel = new QLabel(toolBar, "delayLabel" );
  Q_CHECK_PTR(delayLabel);
  delayLabel->setMinimumSize( QSize( 65, 0 ) );
  delayLabel->setText(tr("Refresh Rate (sec)" ) );
  
  delay = new QSpinBox(toolBar, "delayBox" );
  Q_CHECK_PTR(delay);
  delay->setMinimumSize( QSize( 50, 0 ) ); 
  delay->setMaxValue( 7200 );
  delay->setMinValue( 1 );
 
  cfgname = mysql->getConnectionName();
  CConfig *cfg = new CConfig(cfgname, g_CONNECTION_PATH);
  Q_CHECK_PTR(cfg);
  cfg->prepare();  
  delay->setValue( cfg->readNumberEntry("Refresh Rate", 30) );
  Status->setTraditionalMode(strtobool(cfg->readStringEntry("Status Traditional Mode", "false")), false);  
  delete cfg;  
  timerAction->addTo(toolBar); 

  menubar->insertItem(tr("&Command"), commandMenu);

  ProcessList->verticalHeader()->hide();
  ProcessList->setLeftMargin(0);
  Timer = new QTimer(this);
  Q_CHECK_PTR(Timer);
  TabWidget->setCurrentPage(type);
  tmpFileName = QString::null;
  init(); 
  Refresh();
  myResize( 520, 419 );
  autoPlace();
}

bool CAdministrationPanel::Shutdown(CMyWindow *parent, CMySQLConnection *m)
{
  if (g_confirmCritical)
    if ((QMessageBox::information(0, tr("Confirm Shutdown"), tr("Are you sure you want to Shutdown mysqld ?"),
      tr("&Yes"), tr("&No")) != 0))
      return false;

  if (m->shutdown())
  {
    parent->messagePanel()->Warning("[" + m->getConnectionName() + "] " + tr("mysqld was shutdown successfully"));
    return true;
  }
  return false;
}

void CAdministrationPanel::Ping(CMyWindow *parent, CMySQLConnection *m)
{  
  if (m->ping())
    parent->messagePanel()->Information("[" + m->getConnectionName() + "] " + tr("mysqld is alive"));
}

void CAdministrationPanel::Flush_Hosts(CMyWindow *parent, CMySQLConnection *m)
{
  if (m->flush("HOSTS"))
    parent->messagePanel()->Information("[" + m->getConnectionName() + "] " + tr("Flush Hosts successful"));
}

void CAdministrationPanel::Flush_Logs(CMyWindow *parent, CMySQLConnection *m)
{
  if (m->flush("LOGS"))
    parent->messagePanel()->Information("[" + m->getConnectionName() + "] " + tr("Flush Logs successful"));
}

void CAdministrationPanel::Flush_Privileges(CMyWindow *parent, CMySQLConnection *m)
{
  if (m->flush("PRIVILEGES"))
    parent->messagePanel()->Information("[" + m->getConnectionName() + "] " + tr("Flush Privileges successful"));
}

void CAdministrationPanel::Flush_Tables(CMyWindow *parent, CMySQLConnection *m)
{
  if (m->flush("TABLES"))
    parent->messagePanel()->Information("[" + m->getConnectionName() + "] " + tr("Flush Tables successful"));
}

void CAdministrationPanel::Flush_Status(CMyWindow *parent, CMySQLConnection *m)
{
  if (m->flush("Status"))
    parent->messagePanel()->Information("[" + m->getConnectionName() + "] " + tr("Flush Status successful"));
}

void CAdministrationPanel::enableTimer(bool e)
{
  timerAction->setEnabled(e);
  delayLabel->setEnabled(e);
  delay->setEnabled(e);
}

CAdministrationPanel::~CAdministrationPanel()
{
  if (timerAction->isEnabled())
  {
    CConfig *cfg = new CConfig(cfgname, g_CONNECTION_PATH);
    Q_CHECK_PTR(cfg);
    cfg->prepare();  
    cfg->writeEntry("Refresh Rate", delay->value());
    cfg->writeEntry("Status Traditional Mode", booltostr(Status->isTraditionalMode()));
    cfg->flush();
    delete cfg;
  }
  delete mysql;
}

void CAdministrationPanel::Toggled(bool b)
{      
  delay->setEnabled(!b);
  if (b)
  {
    timerAction->setText(tr( "Stop Timer"));  
    timerAction->setMenuText(tr( "&Stop Timer"));
    Timer->start(delay->value() * 1000);    
  }
  else
  {
    timerAction->setText(tr( "Start Timer"));  
    timerAction->setMenuText(tr( "&Start Timer"));
    Timer->stop();
  }
}

void CAdministrationPanel::Refresh()
{
  Variables->refresh();
  ExecuteQuery();
}

void CAdministrationPanel::RefreshProcessList()
{
  bool notnew = ((mysql->getMySQLServerVersion().major <= 3) && (mysql->getMySQLServerVersion().minor <= 22));
  QString qry = "SHOW " + (!notnew ? QString("FULL ") : QString::null) + "PROCESSLIST";
  ProcessList->exec(qry);
}

void CAdministrationPanel::ExecuteQuery()
{    
  refreshAction->setEnabled(false);
  RefreshProcessList();
  Status->exec();
  refreshAction->setEnabled(true);
}

void CAdministrationPanel::TabChanged(QWidget *)
{
  killProcessAction->setEnabled(TabWidget->currentPageIndex() == SHOW_PROCESSLIST);  
}

void CAdministrationPanel::KillProcess()
{    
  ProcessList->finishEdit();
  bool found = false;
  for (int i = 0; i < ProcessList->numRows(); i++)
  {     
    if (((ProcessList->item(i,0) != 0) ) && (((CCheckTableItem *) ProcessList->item(i,0))->isChecked()))
      if (mysql->killProcess(ProcessList->text(i,0).toLong()))
      {
        messagePanel()->Information(tr("Process killed successfully") + " :" +  ProcessList->text(i,0));
        if (!found)
          found = true;
      }
  }
  if (found)
    RefreshProcessList();    
}


void CAdministrationPanel::doSave(int res)
{    
  switch (res)
  {
  case MENU_SAVE_ALL: {
    QString s = ProcessList->getSaveContents() + mysql->getLineTerminator(true);
    s += Status->getSaveContents() + mysql->getLineTerminator(true);
    s += Variables->getSaveContents() + mysql->getLineTerminator(true);    
    saveToFile(tmpFileName, "txt", tr("Text Files (*.txt);;All Files(*.*)" ), s, mysql->messagePanel);
    break;
                 }
  case MENU_SAVE_PROCESSLIST:
    ProcessList->save();
    break;
  case MENU_SAVE_STATUS:
    Status->save();
    break;
  case MENU_SAVE_VARIABLES:
    Variables->save();
  }
}

void CAdministrationPanel::processMenu(CMyWindow *parent, int res, CMySQLConnection *m)
{    
  switch (res)
  {
  case FLUSH_HOSTS:
    Flush_Hosts(parent, m);
    break;
  case FLUSH_LOGS:
    Flush_Logs(parent, m);
    break;
  case FLUSH_PRIVILEGES:
    Flush_Privileges(parent, m);
    break;
  case FLUSH_TABLES:
    Flush_Tables(parent, m);
    break;  
  case FLUSH_STATUS:
    Flush_Status(parent, m);
    break;
  case MENU_PING:
    Ping(parent, m);
    break;
  case MENU_SHUTDOWN:
    Shutdown(parent, m);
    break;
  }
}

void CAdministrationPanel::FlushMenu(QPopupMenu *p_flushTypeMenu)
{  
  p_flushTypeMenu->insertItem(tr("&Hosts"), FLUSH_HOSTS);
  p_flushTypeMenu->insertItem(tr("&Logs"), FLUSH_LOGS);
  p_flushTypeMenu->insertItem(tr("&Privileges"), FLUSH_PRIVILEGES);
  p_flushTypeMenu->insertItem(tr("&Tables"), FLUSH_TABLES);  
  p_flushTypeMenu->insertItem(tr("&Status"), FLUSH_STATUS);  
}

void CAdministrationPanel::s_Ping()
{
  Ping(this, mysql);
}

void CAdministrationPanel::s_Shutdown()
{
  if (Shutdown(this, mysql))
    if (Timer->isActive())
      {
        Timer->stop();
        timerAction->setOn(false);
        Toggled(false);
      }
}

bool CAdministrationPanel::inRange(int res)
{  
  return (((res >= FLUSH_HOSTS) && (res <= FLUSH_STATUS)) || (res == MENU_PING) || (res == MENU_SHUTDOWN));
}

void CAdministrationPanel::s_ProcessMenu(int res)
{    
  CAdministrationPanel::processMenu(this, res, mysql);
}

void CAdministrationPanel::ProcessListMenu(int row, int, const QPoint &pos)
{
  QPopupMenu *p_itemMenu = new QPopupMenu();  
  p_itemMenu->insertItem(getPixmapIcon("killProcessIcon"), "Kill Process", MENU_KILL_PROCESS);
  p_itemMenu->insertSeparator();
  p_itemMenu->insertItem(getPixmapIcon("saveIcon"), "Save", MENU_SAVE);
  p_itemMenu->insertSeparator();  
  p_itemMenu->insertItem(getPixmapIcon("refreshTablesIcon"), "Refresh", MENU_REFRESH);
  
  int res = p_itemMenu->exec(pos);
  delete p_itemMenu;
  switch (res)
  {
    case MENU_REFRESH:
      RefreshProcessList();
      break;

    case MENU_SAVE:
      ProcessList->save();
      break;

    case MENU_KILL_PROCESS: {
      if (mysql->killProcess(ProcessList->text(row,0).toLong()))
      {
        messagePanel()->Information(tr("Process killed successfully") + " :" +  ProcessList->text(row,0));
        RefreshProcessList();
      }      
      break;
                            }
  }  
}

void CAdministrationPanel::processShowMessagesSlot()
{  
  if (!messageWindow()->isHidden())
    messageWindow()->hide();  
  else  
    messageWindow()->show();  
  showMessagesAction->setOn(!messageWindow()->isHidden());
}

void CAdministrationPanel::init()
{
  connect(actionCloseAction, SIGNAL(activated()), this, SLOT(close()));
  connect(showMessagesAction, SIGNAL(activated()), this, SLOT(processShowMessagesSlot()));
  connect(messageWindow(), SIGNAL(visibilityChanged(bool)), showMessagesAction, SLOT(setOn(bool)));
  connect(refreshAction, SIGNAL(activated()), this, SLOT(Refresh()));
  connect(killProcessAction, SIGNAL(activated()), this, SLOT(KillProcess()));

  connect(pingAction, SIGNAL(activated()), this, SLOT(s_Ping()));
  connect(shutdownAction, SIGNAL(activated()), this, SLOT(s_Shutdown()));

  connect(timerAction, SIGNAL(toggled(bool)), this, SLOT(Toggled(bool)));
  connect(Timer, SIGNAL(timeout()), this, SLOT(ExecuteQuery()) );
  connect(TabWidget, SIGNAL(currentChanged (QWidget *)), this, SLOT(TabChanged(QWidget *)));
  connect(flushTypeMenu, SIGNAL(activated(int)), this, SLOT (s_ProcessMenu(int)));
  connect(saveTypeMenu, SIGNAL(activated(int)), this, SLOT (doSave(int)));
  connect(ProcessList, SIGNAL(contextMenuRequested (int, int, const QPoint &)), this, SLOT (ProcessListMenu(int, int, const QPoint &)));  
}
